/**********************************************************************************
 *
 * @file    irq.c
 * @brief   Interrupt handler
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "main.h"
#include "md_utils.h"
#include "string.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
static uint32_t s_recv_len = sizeof(g_recv_buf);
static uint8_t  s_recv_ind = 0U;
static uint32_t s_send_len = sizeof(g_send_buf);
static uint8_t  s_send_ind = 0U;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples_I2C_interrupt
  * @{
  */

/** @addtogroup Core Processor Exceptions and Peripheral Interrupt HANDLER
  * @{
  */

/****************** Core Processor Exceptions HANDLER ************************/
/**
  * @brief  NMI IRQ handler
  * @retval None
  */
void NMI_Handler(void)
{
    /* Added Emergency operation */
    return;
}

/**
  * @brief  Hardfault IRQ handler
  * @retval None
  */
void HardFault_Handler(void)
{
    /* Added debug information */
    while (1);
}

/**
  * @brief  Supervisor Call IRQ handler
  * @retval None
  */
void SVC_Handler(void)
{
    /* Added system callback */
    return;
}

/**
  * @brief  Debug Monitor IRQ handler
  * @retval None
  */
void DebugMon_Handler(void)
{
    /* Added debug operation */
    return;
}

/**
  * @brief  PendSV IRQ handler
  * @retval None
  */
void PendSV_Handler(void)
{
    /* Added thread switching operation */
    return;
}

/**
  * @brief  SysTick IRQ handler
  * @retval None
  */
void SysTick_Handler(void)
{
    md_inc_tick();
    return;
}

/****************** Peripheral Interrupt HANDLER ************************/
/*I2C interrupt service function*/
void I2C1_Handler()
{
    /*Start bit is sent*/
    if (md_i2c_is_active_flag_sb(I2C1) && md_i2c_is_enable_it_evt(I2C1))
    {
        md_i2c_enable_it_buf(I2C1);

        if (g_read_or_write == I2C_REQUEST_READ)
        {
            md_i2c_transmit_data8(I2C1, (SLAVE_ADDR << 1) | I2C_REQUEST_READ);      /*Send the target from the machine address and direction of the communication*/
            md_i2c_enable_ack(I2C1);
        }
        else
            md_i2c_transmit_data8(I2C1, (SLAVE_ADDR << 1) | I2C_REQUEST_WRITE);
    }

    /*The host address has been sent*/
    if (md_i2c_is_active_flag_addr(I2C1) && md_i2c_is_enable_it_evt(I2C1))
    {
        md_i2c_clear_flag_addr(I2C1);       /*Remove the address sign*/

        return;
    }

    /*Receive buffer is not empty*/
    if (md_i2c_is_active_flag_rxne(I2C1) && md_i2c_is_enable_it_evt(I2C1) && md_i2c_is_enable_it_buf(I2C1))
    {
        if (s_recv_ind < s_recv_len)
        {
            g_recv_buf[s_recv_ind++] = md_i2c_receive_data8(I2C1);

            if (s_recv_ind == (s_recv_len - 1)) /*The last but one byte*/
            {
                md_i2c_disable_ack(I2C1);       /*Ban response*/
                md_i2c_generate_stop_condition_by_master(I2C1);     /*To generate the stop bit*/
            }
            else if (s_recv_ind == s_recv_len)
            {
                s_recv_ind = 0;
                g_finish = 1;
                md_i2c_disable_it_buf(I2C1);
                memcpy(g_send_buf, g_recv_buf, sizeof(g_send_buf));
            }
            else
                return;
        }

        return;
    }

    /*Send buffer empty*/
    if (md_i2c_is_active_flag_txbe(I2C1) && md_i2c_is_enable_it_evt(I2C1) && md_i2c_is_enable_it_buf(I2C1))
    {
        md_i2c_transmit_data8(I2C1, g_send_buf[s_send_ind++]);

        if (s_send_ind == s_send_len)
        {
            while (!md_i2c_is_active_flag_txbe(I2C1));

            md_i2c_generate_stop_condition_by_master(I2C1);
            s_send_ind = 0;
            g_finish = 1;
            md_i2c_disable_it_buf(I2C1);
        }
    }
}

/**
  * @} Core Processor Exceptions and Peripheral Interrupt HANDLER
  */

/**
  * @} Examples_interrupt
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
