/**********************************************************************************
 *
 * @file    i2c.c
 * @brief   i2c C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "i2c.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/* Private Variables --------------------------------------------------------- */
static i2c_handle_t s_i2c;
/* Private function prototypes ----------------------------------------------- */
static void init_i2c_pin(void);

/* Private Macros ------------------------------------------------------------ */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/**
  * @brief Write an amount of data in blocking mode to a specific memory address
  * @param  dev_addr: Target device address
  * @param  mem_addr: Internal memory address
  * @param  add_size: size of internal memory address
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @param  timeout: Timeout duration
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t i2c_write_device(uint16_t dev_addr, uint16_t mem_addr, i2c_addr_size_t add_size,
                              char *buf, uint32_t size, uint32_t timeout)
{
    return ald_i2c_mem_write(&s_i2c, dev_addr, mem_addr, add_size, (uint8_t *)buf, size, timeout);
}

/**
  * @brief  Read an amount of data in blocking mode from a specific memory address
  * @param  dev_addr: Target device address
  * @param  mem_addr: Internal memory address
  * @param  add_size: size of internal memory address
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @param  timeout: Timeout duration
   * @retval Status, see @ref ald_status_t.
  */
ald_status_t i2c_read_device(uint16_t dev_addr, uint16_t mem_addr, i2c_addr_size_t add_size,
                             char *buf, uint32_t size, uint32_t timeout)
{
    return ald_i2c_mem_read(&s_i2c, dev_addr, mem_addr, add_size, (uint8_t *)buf, size, timeout);
}

/**
  * @brief i2c function
  * @retval None.
  */
void init_mcu_i2c(void)
{
    init_i2c_pin();

    s_i2c.perh = I2C1;
    s_i2c.init.addr_mode    = I2C_ADDR_7BIT;
    s_i2c.init.clk_speed    = 100000;
    s_i2c.init.dual_addr    = I2C_DUALADDR_ENABLE;
    s_i2c.init.general_call = I2C_GENERALCALL_ENABLE;
    s_i2c.init.no_stretch   = I2C_NOSTRETCH_ENABLE;
    s_i2c.init.own_addr1    = 0xA0;

    ald_i2c_init(&s_i2c);  /*According to the initialization parameter I2C module*/
}

/**
  * @brief  Initializate i2c pin
  * @retval None.
  */
static void init_i2c_pin(void)
{
    gpio_init_t gpio_config;

    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.odos = GPIO_OPEN_DRAIN;
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;

    gpio_config.func = GPIO_FUNC_5;
    gpio_config.mode = GPIO_MODE_OUTPUT;
    ald_gpio_init(GPIOB, GPIO_PIN_10, &gpio_config);  /*Initialize PB8 is SCL pins*/

    ald_gpio_init(GPIOB, GPIO_PIN_11, &gpio_config);  /*Initialize PB9 for SDA pin*/
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
