/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for Demo
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "utils.h"

/* Private Macros ------------------------------------------------------------ */

#define ALD_DMA     /*Configuration to the DMA mode*/

/* Private Variables --------------------------------------------------------- */
static uint8_t s_slave_addr = 0x5AU;  /* real address = 0x2D */

/* Public Variables ---------------------------------------------------------- */
i2c_handle_t g_i2c_handle;

uint8_t  g_recv_buf[20];
uint8_t  g_send_buf[20]; 
uint32_t g_recv_len = sizeof(g_recv_buf);
uint32_t g_send_len = sizeof(g_send_buf);

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples_I2C
  * @{
  */
/*From the machine pin initialization function of communication*/
void init_i2c_pin(void)
{
    gpio_init_t gpio_config;

    /*Initialization SCL pins*/
    gpio_config.mode = GPIO_MODE_OUTPUT;
    gpio_config.odos = GPIO_OPEN_DRAIN;   /*Open drain output*/
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.func = GPIO_FUNC_5;
    ald_gpio_init(I2C1_SCL_PORT, I2C1_SCL_PIN, &gpio_config);

    /*Initialize the SDA pin*/
    gpio_config.mode = GPIO_MODE_OUTPUT;
    gpio_config.odos = GPIO_OPEN_DRAIN;   /*Open drain output*/
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.func = GPIO_FUNC_5;
    ald_gpio_init(I2C1_SDA_PORT, I2C1_SDA_PIN, &gpio_config);
}

/*Completion of the host*/
void slave_tx_complete(i2c_handle_t *arg)
{
    ald_i2c_slave_recv_by_dma(&g_i2c_handle, g_recv_buf, sizeof(g_recv_buf), 0);
    return;
}

/*The host receives the complete*/
void slave_rx_complete(i2c_handle_t *arg)
{
    memcpy(g_send_buf, g_recv_buf, sizeof(g_recv_buf));
    ald_i2c_slave_send_by_dma(&g_i2c_handle, g_send_buf, sizeof(g_send_buf), 1);
    return;
}


/*The main function*/
int main()
{
    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    
    /* Enable peripheral clock */
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    /*Interrupt priority level and make configuration can interrupt*/
    ald_mcu_irq_config(DMA_IRQn, 1, ENABLE);
    ald_mcu_irq_config(I2C1_IRQn, 1, ENABLE);

    /*Pin initialization of communication*/
    init_i2c_pin();

    /*Empty the I2C initialization structure*/
    memset(&g_i2c_handle, 0, sizeof(i2c_handle_t)); 
    /*User configuration I2C module*/
    g_i2c_handle.perh              = I2C1;
    g_i2c_handle.mode              = I2C_MODE_SLAVE;
    g_i2c_handle.init.own_addr1    = s_slave_addr;
    g_i2c_handle.init.clk_speed    = 400000;
    g_i2c_handle.init.addr_mode    = I2C_ADDR_7BIT;
    g_i2c_handle.init.duty         = I2C_DUTYCYCLE_16_9;
    g_i2c_handle.init.general_call = I2C_GENERALCALL_DISABLE;
    g_i2c_handle.init.no_stretch   = I2C_NOSTRETCH_DISABLE;
    g_i2c_handle.slave_rx_cplt_cbk = slave_rx_complete;
    g_i2c_handle.slave_tx_cplt_cbk = slave_tx_complete; 
    ald_i2c_init(&g_i2c_handle);
    
    ald_i2c_slave_recv_by_dma(&g_i2c_handle, g_recv_buf, sizeof(g_recv_buf), 0);
    
    while (1)
    {
        ald_delay_ms(1000);
    }
}

/**
  * @} Examples_I2C
  */

/**
  * @} Projects_Examples_ALD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
