/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for Demo
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *          3 Sep. 2021     Lisq            I2C引脚改为TTL模式
 *          29 Nov. 2022    AE Team         I2C引脚改为CMOS模式
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "main.h"
#include "utils.h"
#include <stdlib.h>
#include <string.h>

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
static uint8_t  s_slave_addr = 0x5AU;  /* real address = 0x2D */
static volatile uint8_t s_rx_complete;
static volatile uint8_t s_tx_complete;

uint8_t  g_recv_buf[20];
uint8_t  g_send_buf[20];

/* Public Variables ---------------------------------------------------------- */

i2c_handle_t g_i2c_handle;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/*Host mode communication pins initialization function*/
void init_i2c_pin(void)
{
    gpio_init_t gpio_config;

    /*Initialization SCL pins*/
    gpio_config.mode = GPIO_MODE_OUTPUT;
    gpio_config.odos = GPIO_OPEN_DRAIN;   /*Open drain output*/
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.func = GPIO_FUNC_5;
    ald_gpio_init(I2C1_SCL_PORT, I2C1_SCL_PIN, &gpio_config);

    /*Initialize the SDA pin*/
    gpio_config.mode = GPIO_MODE_OUTPUT;
    gpio_config.odos = GPIO_OPEN_DRAIN;   /*Open drain output*/
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.func = GPIO_FUNC_5;
    ald_gpio_init(I2C1_SDA_PORT, I2C1_SDA_PIN, &gpio_config);
}

/*Completion of the host*/
void master_tx_complete(i2c_handle_t *arg)
{
    s_tx_complete = 1;
    return;
}

/*The host receives the complete*/
void master_rx_complete(i2c_handle_t *arg)
{
    s_rx_complete = 1;
    return;
}

/*The main function*/
int main()
{
    uint16_t cnt;

    /*Initialize the ALD*/
    ald_cmu_init();
    ald_cmu_clock_config(CMU_CLOCK_HRC, 24000000);
    ald_cmu_perh_clock_config(CMU_PERH_I2C1, ENABLE);

    /*Pin initialization of communication*/
    init_i2c_pin();

    /*Interrupt priority level and make configuration can interrupt*/
    ald_mcu_irq_config(I2C1_IRQn, 1, ENABLE);

    /*Empty the I2C initialization structure*/
    memset(&g_i2c_handle, 0, sizeof(i2c_handle_t));

    /*User configuration I2C module*/
    g_i2c_handle.perh              = I2C1;
    g_i2c_handle.mode              = I2C_MODE_MASTER;
    g_i2c_handle.init.clk_speed    = 100000;
    g_i2c_handle.init.duty         = I2C_DUTYCYCLE_16_9;
    g_i2c_handle.init.own_addr1    = 0x5A;
    g_i2c_handle.init.addr_mode    = I2C_ADDR_7BIT;
    g_i2c_handle.init.general_call = I2C_GENERALCALL_DISABLE;
    g_i2c_handle.init.no_stretch   = I2C_NOSTRETCH_DISABLE;
    g_i2c_handle.master_rx_cplt_cbk = master_rx_complete;
    g_i2c_handle.master_tx_cplt_cbk = master_tx_complete;
    ald_i2c_reset(&g_i2c_handle);
    ald_i2c_init(&g_i2c_handle);

    while (1)
    {
        /*Host interrupts transceiver, from a real address 0x2d machine*/
        /* receive */
        s_rx_complete = 0;
        cnt = 40000;
        ald_i2c_master_recv_by_it(&g_i2c_handle, s_slave_addr, g_recv_buf, sizeof(g_recv_buf));
        while (s_rx_complete != 1 && --cnt);
        ald_delay_ms(1000);
        
        memcpy(g_send_buf, g_recv_buf, sizeof(g_recv_buf));
        
        /* send */
        s_tx_complete = 0;
        cnt = 40000;
        ald_i2c_master_send_by_it(&g_i2c_handle, s_slave_addr, g_send_buf, sizeof(g_send_buf));
        while (s_tx_complete != 1 && --cnt);
        ald_delay_ms(3000);
    }
}

/**
  * @} Examples_I2C
  */

/**
  * @} Projects_Examples_ALD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
