/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    2 Aug 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2 Aug 2021      biyq          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "ald_conf.h"


/* Private Macros ----------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */


/**
  * @brief  Nand pin init.
  * @param  None
  * @retval None
  */
void nand_pin_init(void)
{
    gpio_init_t gpio_config;

    gpio_config.mode  = GPIO_MODE_OUTPUT;
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.podrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_7;

    /* Data pin init */
    ald_gpio_init(GPIOD, GPIO_PIN_14 | GPIO_PIN_15 | GPIO_PIN_0 | GPIO_PIN_1, &gpio_config);
    ald_gpio_init(GPIOE, GPIO_PIN_7 | GPIO_PIN_8 | GPIO_PIN_9 | GPIO_PIN_10, &gpio_config);

    /* Control pin init */
    ald_gpio_init(GPIOD, GPIO_PIN_7 | GPIO_PIN_4 | GPIO_PIN_5 | GPIO_PIN_11 | GPIO_PIN_12, &gpio_config);
    /* Ready/Busy pin init */
    gpio_config.mode = GPIO_MODE_INPUT;
    gpio_config.func = GPIO_FUNC_7;
    ald_gpio_init(GPIOD, GPIO_PIN_6, &gpio_config);
}


/* Public Function---------------------------------------------------------- */
/**
  * @brief  Translate logical address into a phy one.
  * @param  addr: Address
  * @param  n_addr: NAND addr.
  * @retval none
  */
void nand_addr_get(uint32_t addr, nand_address_t *n_addr)
{
    const uint32_t NAND_PAGE_SIZE       = 0x0800;  /* 2048 bytes per page w/o Spare Area */
    const uint32_t NAND_BLOCK_SIZE      = 0x0040;  /* 64x2048 bytes pages per block */
    const uint32_t NAND_PLANE_SIZE      = 0x0400;  /* 1024 Block per plane */
    const uint32_t NAND_SPARE_AREA_SIZE = 0x0040;  /* last 64 bytes as spare area */

    n_addr->page  = (addr % (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE);
    n_addr->block = (addr % (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));
    n_addr->plane = addr / (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));
}

/**
  * @brief  nand_init
  * @retval none.
  */
void nand_init(nand_handle_t *nand_handle)
{
    const uint32_t NAND_PAGE_SIZE       = 0x0800;  /* 2048 bytes per page w/o Spare Area */
    const uint32_t NAND_BLOCK_SIZE      = 0x0040;  /* 64x2048 bytes pages per block */
    const uint32_t NAND_PLANE_SIZE      = 0x0400;  /* 1024 Block per plane */
    const uint32_t NAND_SPARE_AREA_SIZE = 0x0040;  /* last 64 bytes as spare area */
    const uint32_t NAND_BLOCK_NBR       = 0x0400;  /* 1 planes of 1024 block */
    const uint32_t NAND_PLANE_NBR       = 1;       /* 1 device of 1 planes */

    ald_ebi_nand_timing_t nand_timing;

    nand_pin_init();

    nand_handle->instance          = EBI_NAND_DEVICE;
    nand_handle->init.bank         = EBI_NAND_BANK2;
    nand_handle->init.wait         = EBI_NAND_WAIT_FEATURE_ENABLE;
    nand_handle->init.width        = EBI_NAND_MEM_BUS_WIDTH_8;
    nand_handle->init.ecc          = EBI_NAND_ECC_ENABLE;
    nand_handle->init.size         = EBI_NAND_ECC_PAGE_SIZE_2048BYTE;
    nand_handle->init.cle_time     = 0x0;
    nand_handle->init.ale_time     = 0x0;

    nand_handle->config.page_size  = NAND_PAGE_SIZE;
    nand_handle->config.spare_size = NAND_SPARE_AREA_SIZE;
    nand_handle->config.block_size = NAND_BLOCK_SIZE;
    nand_handle->config.block_nbr  = NAND_BLOCK_NBR;
    nand_handle->config.plane_nbr  = NAND_PLANE_NBR;
    nand_handle->config.plane_size = NAND_PLANE_SIZE;

    /*限制条件为：nand flash 手册中的相关参数 + 芯片外设限制。HCLK2最大频率72M，需求<= 13ns的不考虑*/
    nand_timing.hiz_time    = (((ald_cmu_get_hclk2_clock()) * 7) / 50000000);
    nand_timing.wait_time   = ((nand_timing.hiz_time) * 4) / 3;
    nand_timing.hold_time   = nand_timing.wait_time / 2;
    nand_timing.time        = nand_timing.hold_time;

    if ((nand_timing.hiz_time - nand_timing.time) < 2)
    {
        nand_timing.wait_time++;
        nand_timing.hold_time++;
    }

    /*将实际时序转为寄存器值+时序寄存器范围限制*/
    nand_timing.hiz_time     = (nand_timing.hiz_time < 0x01) ?  0x00 : (nand_timing.hiz_time - 1); /*MEMHIZTx/ATTHIZTx   reg范围 0x00--0xFE  HCLK周期 = reg+1 */
    nand_timing.hiz_time     = (nand_timing.hiz_time > 0xFE) ?  0xFE : nand_timing.hiz_time;
    nand_timing.hold_time    = (nand_timing.hold_time < 0x01) ? 0x01 : nand_timing.hold_time;      /*MEMHOLDx/ATTHOLDx   reg范围 0x01--0xFE  HCLK周期 = reg   */
    nand_timing.hold_time    = (nand_timing.hold_time > 0xFE) ? 0xFE : nand_timing.hold_time;
    nand_timing.wait_time    = (nand_timing.wait_time < 0x01) ? 0x00 : (nand_timing.wait_time - 1); /*MEMWAITx/ATTWAITx   reg范围 0x01--0xFF  HCLK周期 = reg+1 */
    nand_timing.wait_time    = (nand_timing.wait_time > 0xFF) ? 0xFF : nand_timing.wait_time;
    nand_timing.time         = (nand_timing.time < 0x01) ?      0x00 : (nand_timing.time - 1);     /*MEMSETUPx/ATTSETUPx reg范围 0x00--0xFE  HCLK周期 = reg+1 */
    nand_timing.time         = (nand_timing.time > 0xFE) ?      0xFE : nand_timing.time;

    /* NAND Flash initialization */
    ald_nand_init(nand_handle, &nand_timing, &nand_timing);
    ald_nand_reset(nand_handle);

    return;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
