/**********************************************************************************
 *
 * @file    lcd_driver.c
 * @brief   LCD file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    zhuxf          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "lcd_driver.h"
#include "font.h"


/* Private Macros ----------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */

static nor_handle_t s_lcd;
static ald_ebi_nor_sram_timing_t s_ebi_timing;

/* Public Variables -------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

static void ebi_pin_init(void);

/* Private Function---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

void delay(uint32_t cnt)
{
    while (cnt--)
        ;
}

/**
  * @brief  Lcd write reg
  * @param  reg: lcd reg
  * @param  para: parameter
  * @retval None
  */
inline void lcd_write_reg(uint16_t reg, uint16_t para)
{
    LCD_SET_REG(reg);
    LCD_SET_RAM(para);
}

/**
  * @brief  Lcd read reg
  * @param  reg: lcd reg
  * @retval uint8_t data
  */
uint8_t lcd_read_reg(uint16_t reg)
{
    uint16_t data;

    LCD->REG = reg;
    data = LCD_READ_RAM();    /* dummy data */
    data = LCD_READ_RAM();

    return data;
}

/**
  * @brief  Lcd draw point function
  * @param  x: The position of x axis
  * @param  y: The position of y axis
  * @retval None
  */
void lcd_set_cursor(uint16_t x, uint16_t y)
{
    lcd_write_reg(0x2A00, x >> 8);
    lcd_write_reg(0x2A00 + 1, x & 0xFF);

    lcd_write_reg(0x2B00, y >> 8);
    lcd_write_reg(0x2B00 + 1, y & 0xFF);
}

/**
  * @brief  Lcd draw point function
  * @param  x: The point of x axis
  * @param  y: The point of y axis
  * @param  rgb888: the color of point
  * @retval None
  */
void lcd_draw_point(uint16_t x, uint16_t y, uint32_t rgb888)
{
    lcd_set_cursor(x, y);

    LCD_WRRAM_PREPARE();

    LCD_SET_RAM(GET_RGB565_FROM_RGB888(rgb888));
}

/**
  * @brief  Lcd draw point function
  * @param  x: The point of x axis
  * @param  y: The point of y axis
  * @param  width: The width of image
  * @param  height: The height of image
  * @param  buf: the address of image
  * @retval None
  */
void lcd_draw_image(uint16_t x, uint16_t y, \
                    uint16_t width, uint16_t height, uint16_t *buf)
{
    uint32_t index;
    uint32_t totalpoint = width * height;

    lcd_write_reg(0x2A00, x >> 8);
    lcd_write_reg(0x2A00 + 1, x & 0xFF);
    lcd_write_reg(0x2A00 + 2, (x + width - 1) >> 8);
    lcd_write_reg(0x2A00 + 3, (x + width - 1) & 0xFF);

    lcd_write_reg(0x2B00, y >> 8);
    lcd_write_reg(0x2B00 + 1, y & 0xFF);
    lcd_write_reg(0x2B00 + 2, (y + height - 1) >> 8);
    lcd_write_reg(0x2B00 + 3, (y + height - 1) & 0xFF);

    LCD_WRRAM_PREPARE();

    for (index = 0; index < totalpoint; index++)
    {
        LCD_SET_RAM(*buf);
        buf++;
    }
}

/**
  * @brief  Lcd draw point function
  * @param  x: The point of x axis
  * @param  y: The point of y axis
  * @param  width: width of background
  * @param  height: height of background
  * @param  rgb888: The background color
  * @retval None
  */
void lcd_set_background(uint16_t x, uint16_t y, uint16_t width, uint16_t height, \
                        uint32_t rgb888)
{
    uint32_t i;

    lcd_write_reg(0x2A00, x >> 8);
    lcd_write_reg(0x2A00 + 1, x & 0xFF);
    lcd_write_reg(0x2A00 + 2, (x + width - 1) >> 8);
    lcd_write_reg(0x2A00 + 3, (x + width - 1) & 0xFF);

    lcd_write_reg(0x2B00, y >> 8);
    lcd_write_reg(0x2B00 + 1, y & 0xFF);
    lcd_write_reg(0x2B00 + 2, (y + height - 1) >> 8);
    lcd_write_reg(0x2B00 + 3, (y + height - 1) & 0xFF);

    LCD_WRRAM_PREPARE();

    for (i = 0; i < width * height; i++)
        LCD_SET_RAM(GET_RGB565_FROM_RGB888(rgb888));
}

/**
  * @brief  Lcd show char function
  * @param  x: The point of x axis
  * @param  y: The point of y axis
  * @param  ch: char
  * @param  size: The size of char
  * @param  rgb_ch: The color of char
  * @param  bg: BG_ON or BG_OFF
  * @param  rgb_bg: The background color
  * @retval None
  */
void lcd_show_char(uint16_t x, uint16_t y, char ch, char_size_t size, \
                   uint32_t rgb_ch, char_bg_t bg, uint32_t rgb_bg)
{
    uint8_t index_char, index_bit, index_byte;
    uint16_t y0 = y;
    /* 计算一个字符编码所需的字节数 */
    uint8_t ch_byte = (size / 8 + ((size % 8) ? 1 : 0)) * (size / 2);
    ch = ch - ' ';    /* 重定位字符编码 */

    for (index_byte = 0; index_byte < ch_byte; index_byte++)
    {
        if (size == C1206)
            index_char = ascii_1206[ch][index_byte];

        else if (size == C1608)
            index_char = ascii_1608[ch][index_byte];

        else if (size == C2412)
            index_char = ascii_2412[ch][index_byte];

        else
            return;

        for (index_bit = 0; index_bit < 8; index_bit++)
        {
            if (index_char & 0x80)
                lcd_draw_point(x, y, rgb_ch);

            else if (bg == BG_ON)
                lcd_draw_point(x, y, rgb_bg);
            else;

            index_char <<= 1;
            y++;

            if (y >= HEIGHT)    /* 超出范围 */
                return;

            if ((y - y0) == size)
            {
                y = y0;
                x++;

                if (x >= WIDTH) /* 超出范围 */
                    return;

                break;
            }
        }
    }
}

/**
  * @brief  Lcd show char function
  * @param  x: The point of x axis
  * @param  y: The point of y axis
  * @param  width:  |﹋﹋﹋﹋﹋﹋﹋﹋|
  * @param  height: |________________|
  * @param  rgb_ch: The color of char
  * @param  bg: BG_ON or BG_OFF
  * @param  rgb_bg: The background color
  * @retval None
  */
void lcd_show_string(uint16_t x, uint16_t y, uint16_t width, uint16_t height, \
                     char *p, uint32_t color, char_size_t size, \
                     char_bg_t bg, uint32_t rgb_bg)
{
    uint8_t x0 = x;

    width += x;
    height += y;

    while ((*p <= '~') && (*p >= ' ')) /* 合法字符 */
    {
        if (x >= width)
        {
            x = x0;
            y += size;
        }

        if (y >= height)
            break;  /* 超出区域 */

        lcd_show_char(x, y, *p, size, color, bg, rgb_bg);
        x += size / 2;
        p++;
    }
}

/**
  * @brief EBI function
  * @param  None
  * @retval None
  */
void mcu_ebi_init(void)
{
    ebi_pin_init();

    s_lcd.instance       = EBI_NOR_SRAM_DEVICE;
    s_lcd.ext            = EBI_NOR_SRAM_EXTENDED_DEVICE;
    /* 限制为LCD的写。 读暂不考虑*/
    s_ebi_timing.data_setup     = (((ald_cmu_get_hclk2_clock()) * 3) / 100000000);   /*数据保持时间 = ((30 * (ald_cmu_get_hclk2_clock())) / 1000000000)*/
    s_ebi_timing.data_setup     = (s_ebi_timing.data_setup > 0x01) ? s_ebi_timing.data_setup : 0x01;
    s_ebi_timing.addr_setup     = (s_ebi_timing.data_setup > 0x01) ? (s_ebi_timing.data_setup - 2) : 0x00;
    s_ebi_timing.addr_hold      = 1;   /*不复用*/
    s_ebi_timing.bus_dur        = 2;   /*异步写和读之间需要2个EBI时钟恢复*/
    s_ebi_timing.div            = 2;   /*在异步NOR Flash,SRAM,PSRAM中，无效*/
    s_ebi_timing.latency        = 5;   /*在异步NOR Flash,SRAM,PSRAM中，无效*/
    /* s_timing.mode           = EBI_ACCESS_MODE_A; */
    s_lcd.init.bank      = EBI_NORSRAM_BANK3;
    s_lcd.init.mux       = EBI_DATA_ADDRESS_MUX_DISABLE;
    s_lcd.init.type      = EBI_MEMORY_TYPE_SRAM;
    s_lcd.init.width     = EBI_NORSRAM_MEM_BUS_WIDTH_16;
    s_lcd.init.acc_mode  = EBI_BURST_ACCESS_MODE_DISABLE;
    s_lcd.init.polarity  = EBI_WAIT_SIGNAL_POLARITY_LOW;
    s_lcd.init.wrap_mode = EBI_WRAP_MODE_DISABLE;
    s_lcd.init.active    = EBI_WAIT_TIMING_BEFORE_WS;
    s_lcd.init.write     = EBI_WRITE_OPERATION_ENABLE;
    s_lcd.init.signal    = EBI_WAIT_SIGNAL_DISABLE;
    s_lcd.init.ext_mode  = EBI_EXTENDED_MODE_DISABLE;
    s_lcd.init.wait      = EBI_ASYNCHRONOUS_WAIT_DISABLE;
    s_lcd.init.burst     = EBI_WRITE_BURST_DISABLE;

    ald_nor_init(&s_lcd, &s_ebi_timing, &s_ebi_timing);
}

/**
  * @brief  Initializate ebi pin.
  * @param  None.
  * @retval None.
  */
static void ebi_pin_init(void)
{
    gpio_init_t gpio_config;

    gpio_config.mode  = GPIO_MODE_OUTPUT;
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_6;
    gpio_config.podrv = GPIO_OUT_DRIVE_6;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_7;

    /*初始化EBI_A0管脚*/
    ald_gpio_init(GPIOC, GPIO_PIN_0, &gpio_config);

    /*初始化EBI数据管脚(模式与EBI地址管脚配置相同)*/
    ald_gpio_init(GPIOD, GPIO_PIN_14 | GPIO_PIN_15 | GPIO_PIN_0 | GPIO_PIN_1
                  | GPIO_PIN_8 | GPIO_PIN_9 | GPIO_PIN_10, &gpio_config);
    ald_gpio_init(GPIOE, GPIO_PIN_7 | GPIO_PIN_8 | GPIO_PIN_9 | GPIO_PIN_10
                  | GPIO_PIN_11 | GPIO_PIN_12 | GPIO_PIN_13 | GPIO_PIN_14 | GPIO_PIN_15, &gpio_config);

    /*初始化EBI控制管脚(模式与EBI地址管脚配置相同)*/
    ald_gpio_init(GPIOC, GPIO_PIN_11, &gpio_config);  /*EBI_NE3*/
    ald_gpio_init(GPIOD, GPIO_PIN_5, &gpio_config);  /*EBI_NWE*/
    ald_gpio_init(GPIOD, GPIO_PIN_4, &gpio_config);  /*EBI_NOE*/

    /* 点亮背光 */
    gpio_config.func  = GPIO_FUNC_1;
    ald_gpio_init(GPIOB, GPIO_PIN_12, &gpio_config);
    ald_gpio_write_pin(GPIOB, GPIO_PIN_12, 1);
}

/**
  * @brief LCD init
  * @param  None.
  * @retval None.
  */
void lcd_init(void)
{
    lcd_write_reg(0xF000, 0x55);
    lcd_write_reg(0xF001, 0xAA);
    lcd_write_reg(0xF002, 0x52);
    lcd_write_reg(0xF003, 0x08);
    lcd_write_reg(0xF004, 0x01);
    /* AVDD Set AVDD 5.2V */
    lcd_write_reg(0xB000, 0x0D);
    lcd_write_reg(0xB001, 0x0D);
    lcd_write_reg(0xB002, 0x0D);
    /* AVDD ratio */
    lcd_write_reg(0xB600, 0x34);
    lcd_write_reg(0xB601, 0x34);
    lcd_write_reg(0xB602, 0x34);
    /* AVEE -5.2V */
    lcd_write_reg(0xB100, 0x0D);
    lcd_write_reg(0xB101, 0x0D);
    lcd_write_reg(0xB102, 0x0D);
    /* AVEE ratio */
    lcd_write_reg(0xB700, 0x34);
    lcd_write_reg(0xB701, 0x34);
    lcd_write_reg(0xB702, 0x34);
    /* VCL -2.5V */
    lcd_write_reg(0xB200, 0x00);
    lcd_write_reg(0xB201, 0x00);
    lcd_write_reg(0xB202, 0x00);
    /* VCL ratio */
    lcd_write_reg(0xB800, 0x24);
    lcd_write_reg(0xB801, 0x24);
    lcd_write_reg(0xB802, 0x24);
    /* VGH 15V (Free pump) */
    lcd_write_reg(0xBF00, 0x01);
    lcd_write_reg(0xB300, 0x0F);
    lcd_write_reg(0xB301, 0x0F);
    lcd_write_reg(0xB302, 0x0F);
    /* VGH ratio */
    lcd_write_reg(0xB900, 0x34);
    lcd_write_reg(0xB901, 0x34);
    lcd_write_reg(0xB902, 0x34);
    /* VGL_REG -10V */
    lcd_write_reg(0xB500, 0x08);
    lcd_write_reg(0xB501, 0x08);
    lcd_write_reg(0xB502, 0x08);
    lcd_write_reg(0xC200, 0x03);
    /* VGLX ratio */
    lcd_write_reg(0xBA00, 0x24);
    lcd_write_reg(0xBA01, 0x24);
    lcd_write_reg(0xBA02, 0x24);
    /* VGMP/VGSP 4.5V/0V */
    lcd_write_reg(0xBC00, 0x00);
    lcd_write_reg(0xBC01, 0x78);
    lcd_write_reg(0xBC02, 0x00);
    /* VGMN/VGSN -4.5V/0V */
    lcd_write_reg(0xBD00, 0x00);
    lcd_write_reg(0xBD01, 0x78);
    lcd_write_reg(0xBD02, 0x00);
    /* VCOM */
    lcd_write_reg(0xBE00, 0x00);
    lcd_write_reg(0xBE01, 0x64);
    /* Gamma Setting */
    lcd_write_reg(0xD100, 0x00);
    lcd_write_reg(0xD101, 0x33);
    lcd_write_reg(0xD102, 0x00);
    lcd_write_reg(0xD103, 0x34);
    lcd_write_reg(0xD104, 0x00);
    lcd_write_reg(0xD105, 0x3A);
    lcd_write_reg(0xD106, 0x00);
    lcd_write_reg(0xD107, 0x4A);
    lcd_write_reg(0xD108, 0x00);
    lcd_write_reg(0xD109, 0x5C);
    lcd_write_reg(0xD10A, 0x00);
    lcd_write_reg(0xD10B, 0x81);
    lcd_write_reg(0xD10C, 0x00);
    lcd_write_reg(0xD10D, 0xA6);
    lcd_write_reg(0xD10E, 0x00);
    lcd_write_reg(0xD10F, 0xE5);
    lcd_write_reg(0xD110, 0x01);
    lcd_write_reg(0xD111, 0x13);
    lcd_write_reg(0xD112, 0x01);
    lcd_write_reg(0xD113, 0x54);
    lcd_write_reg(0xD114, 0x01);
    lcd_write_reg(0xD115, 0x82);
    lcd_write_reg(0xD116, 0x01);
    lcd_write_reg(0xD117, 0xCA);
    lcd_write_reg(0xD118, 0x02);
    lcd_write_reg(0xD119, 0x00);
    lcd_write_reg(0xD11A, 0x02);
    lcd_write_reg(0xD11B, 0x01);
    lcd_write_reg(0xD11C, 0x02);
    lcd_write_reg(0xD11D, 0x34);
    lcd_write_reg(0xD11E, 0x02);
    lcd_write_reg(0xD11F, 0x67);
    lcd_write_reg(0xD120, 0x02);
    lcd_write_reg(0xD121, 0x84);
    lcd_write_reg(0xD122, 0x02);
    lcd_write_reg(0xD123, 0xA4);
    lcd_write_reg(0xD124, 0x02);
    lcd_write_reg(0xD125, 0xB7);
    lcd_write_reg(0xD126, 0x02);
    lcd_write_reg(0xD127, 0xCF);
    lcd_write_reg(0xD128, 0x02);
    lcd_write_reg(0xD129, 0xDE);
    lcd_write_reg(0xD12A, 0x02);
    lcd_write_reg(0xD12B, 0xF2);
    lcd_write_reg(0xD12C, 0x02);
    lcd_write_reg(0xD12D, 0xFE);
    lcd_write_reg(0xD12E, 0x03);
    lcd_write_reg(0xD12F, 0x10);
    lcd_write_reg(0xD130, 0x03);
    lcd_write_reg(0xD131, 0x33);
    lcd_write_reg(0xD132, 0x03);
    lcd_write_reg(0xD133, 0x6D);
    lcd_write_reg(0xD200, 0x00);
    lcd_write_reg(0xD201, 0x33);
    lcd_write_reg(0xD202, 0x00);
    lcd_write_reg(0xD203, 0x34);
    lcd_write_reg(0xD204, 0x00);
    lcd_write_reg(0xD205, 0x3A);
    lcd_write_reg(0xD206, 0x00);
    lcd_write_reg(0xD207, 0x4A);
    lcd_write_reg(0xD208, 0x00);
    lcd_write_reg(0xD209, 0x5C);
    lcd_write_reg(0xD20A, 0x00);

    lcd_write_reg(0xD20B, 0x81);
    lcd_write_reg(0xD20C, 0x00);
    lcd_write_reg(0xD20D, 0xA6);
    lcd_write_reg(0xD20E, 0x00);
    lcd_write_reg(0xD20F, 0xE5);
    lcd_write_reg(0xD210, 0x01);
    lcd_write_reg(0xD211, 0x13);
    lcd_write_reg(0xD212, 0x01);
    lcd_write_reg(0xD213, 0x54);
    lcd_write_reg(0xD214, 0x01);
    lcd_write_reg(0xD215, 0x82);
    lcd_write_reg(0xD216, 0x01);
    lcd_write_reg(0xD217, 0xCA);
    lcd_write_reg(0xD218, 0x02);
    lcd_write_reg(0xD219, 0x00);
    lcd_write_reg(0xD21A, 0x02);
    lcd_write_reg(0xD21B, 0x01);
    lcd_write_reg(0xD21C, 0x02);
    lcd_write_reg(0xD21D, 0x34);
    lcd_write_reg(0xD21E, 0x02);
    lcd_write_reg(0xD21F, 0x67);
    lcd_write_reg(0xD220, 0x02);
    lcd_write_reg(0xD221, 0x84);
    lcd_write_reg(0xD222, 0x02);
    lcd_write_reg(0xD223, 0xA4);
    lcd_write_reg(0xD224, 0x02);
    lcd_write_reg(0xD225, 0xB7);
    lcd_write_reg(0xD226, 0x02);
    lcd_write_reg(0xD227, 0xCF);
    lcd_write_reg(0xD228, 0x02);
    lcd_write_reg(0xD229, 0xDE);
    lcd_write_reg(0xD22A, 0x02);
    lcd_write_reg(0xD22B, 0xF2);
    lcd_write_reg(0xD22C, 0x02);
    lcd_write_reg(0xD22D, 0xFE);
    lcd_write_reg(0xD22E, 0x03);
    lcd_write_reg(0xD22F, 0x10);
    lcd_write_reg(0xD230, 0x03);
    lcd_write_reg(0xD231, 0x33);
    lcd_write_reg(0xD232, 0x03);
    lcd_write_reg(0xD233, 0x6D);
    lcd_write_reg(0xD300, 0x00);
    lcd_write_reg(0xD301, 0x33);
    lcd_write_reg(0xD302, 0x00);
    lcd_write_reg(0xD303, 0x34);
    lcd_write_reg(0xD304, 0x00);
    lcd_write_reg(0xD305, 0x3A);
    lcd_write_reg(0xD306, 0x00);
    lcd_write_reg(0xD307, 0x4A);
    lcd_write_reg(0xD308, 0x00);
    lcd_write_reg(0xD309, 0x5C);
    lcd_write_reg(0xD30A, 0x00);

    lcd_write_reg(0xD30B, 0x81);
    lcd_write_reg(0xD30C, 0x00);
    lcd_write_reg(0xD30D, 0xA6);
    lcd_write_reg(0xD30E, 0x00);
    lcd_write_reg(0xD30F, 0xE5);
    lcd_write_reg(0xD310, 0x01);
    lcd_write_reg(0xD311, 0x13);
    lcd_write_reg(0xD312, 0x01);
    lcd_write_reg(0xD313, 0x54);
    lcd_write_reg(0xD314, 0x01);
    lcd_write_reg(0xD315, 0x82);
    lcd_write_reg(0xD316, 0x01);
    lcd_write_reg(0xD317, 0xCA);
    lcd_write_reg(0xD318, 0x02);
    lcd_write_reg(0xD319, 0x00);
    lcd_write_reg(0xD31A, 0x02);
    lcd_write_reg(0xD31B, 0x01);
    lcd_write_reg(0xD31C, 0x02);
    lcd_write_reg(0xD31D, 0x34);
    lcd_write_reg(0xD31E, 0x02);
    lcd_write_reg(0xD31F, 0x67);
    lcd_write_reg(0xD320, 0x02);
    lcd_write_reg(0xD321, 0x84);
    lcd_write_reg(0xD322, 0x02);
    lcd_write_reg(0xD323, 0xA4);
    lcd_write_reg(0xD324, 0x02);
    lcd_write_reg(0xD325, 0xB7);
    lcd_write_reg(0xD326, 0x02);
    lcd_write_reg(0xD327, 0xCF);
    lcd_write_reg(0xD328, 0x02);
    lcd_write_reg(0xD329, 0xDE);
    lcd_write_reg(0xD32A, 0x02);
    lcd_write_reg(0xD32B, 0xF2);
    lcd_write_reg(0xD32C, 0x02);
    lcd_write_reg(0xD32D, 0xFE);
    lcd_write_reg(0xD32E, 0x03);
    lcd_write_reg(0xD32F, 0x10);
    lcd_write_reg(0xD330, 0x03);
    lcd_write_reg(0xD331, 0x33);
    lcd_write_reg(0xD332, 0x03);
    lcd_write_reg(0xD333, 0x6D);
    lcd_write_reg(0xD400, 0x00);
    lcd_write_reg(0xD401, 0x33);
    lcd_write_reg(0xD402, 0x00);
    lcd_write_reg(0xD403, 0x34);
    lcd_write_reg(0xD404, 0x00);
    lcd_write_reg(0xD405, 0x3A);
    lcd_write_reg(0xD406, 0x00);
    lcd_write_reg(0xD407, 0x4A);
    lcd_write_reg(0xD408, 0x00);
    lcd_write_reg(0xD409, 0x5C);
    lcd_write_reg(0xD40A, 0x00);
    lcd_write_reg(0xD40B, 0x81);

    lcd_write_reg(0xD40C, 0x00);
    lcd_write_reg(0xD40D, 0xA6);
    lcd_write_reg(0xD40E, 0x00);
    lcd_write_reg(0xD40F, 0xE5);
    lcd_write_reg(0xD410, 0x01);
    lcd_write_reg(0xD411, 0x13);
    lcd_write_reg(0xD412, 0x01);
    lcd_write_reg(0xD413, 0x54);
    lcd_write_reg(0xD414, 0x01);
    lcd_write_reg(0xD415, 0x82);
    lcd_write_reg(0xD416, 0x01);
    lcd_write_reg(0xD417, 0xCA);
    lcd_write_reg(0xD418, 0x02);
    lcd_write_reg(0xD419, 0x00);
    lcd_write_reg(0xD41A, 0x02);
    lcd_write_reg(0xD41B, 0x01);
    lcd_write_reg(0xD41C, 0x02);
    lcd_write_reg(0xD41D, 0x34);
    lcd_write_reg(0xD41E, 0x02);
    lcd_write_reg(0xD41F, 0x67);
    lcd_write_reg(0xD420, 0x02);
    lcd_write_reg(0xD421, 0x84);
    lcd_write_reg(0xD422, 0x02);
    lcd_write_reg(0xD423, 0xA4);
    lcd_write_reg(0xD424, 0x02);
    lcd_write_reg(0xD425, 0xB7);
    lcd_write_reg(0xD426, 0x02);
    lcd_write_reg(0xD427, 0xCF);
    lcd_write_reg(0xD428, 0x02);
    lcd_write_reg(0xD429, 0xDE);
    lcd_write_reg(0xD42A, 0x02);
    lcd_write_reg(0xD42B, 0xF2);
    lcd_write_reg(0xD42C, 0x02);
    lcd_write_reg(0xD42D, 0xFE);
    lcd_write_reg(0xD42E, 0x03);
    lcd_write_reg(0xD42F, 0x10);
    lcd_write_reg(0xD430, 0x03);
    lcd_write_reg(0xD431, 0x33);
    lcd_write_reg(0xD432, 0x03);
    lcd_write_reg(0xD433, 0x6D);
    lcd_write_reg(0xD500, 0x00);
    lcd_write_reg(0xD501, 0x33);
    lcd_write_reg(0xD502, 0x00);
    lcd_write_reg(0xD503, 0x34);
    lcd_write_reg(0xD504, 0x00);
    lcd_write_reg(0xD505, 0x3A);
    lcd_write_reg(0xD506, 0x00);
    lcd_write_reg(0xD507, 0x4A);
    lcd_write_reg(0xD508, 0x00);
    lcd_write_reg(0xD509, 0x5C);
    lcd_write_reg(0xD50A, 0x00);
    lcd_write_reg(0xD50B, 0x81);

    lcd_write_reg(0xD50C, 0x00);
    lcd_write_reg(0xD50D, 0xA6);
    lcd_write_reg(0xD50E, 0x00);
    lcd_write_reg(0xD50F, 0xE5);
    lcd_write_reg(0xD510, 0x01);
    lcd_write_reg(0xD511, 0x13);
    lcd_write_reg(0xD512, 0x01);
    lcd_write_reg(0xD513, 0x54);
    lcd_write_reg(0xD514, 0x01);
    lcd_write_reg(0xD515, 0x82);
    lcd_write_reg(0xD516, 0x01);
    lcd_write_reg(0xD517, 0xCA);
    lcd_write_reg(0xD518, 0x02);
    lcd_write_reg(0xD519, 0x00);
    lcd_write_reg(0xD51A, 0x02);
    lcd_write_reg(0xD51B, 0x01);
    lcd_write_reg(0xD51C, 0x02);
    lcd_write_reg(0xD51D, 0x34);
    lcd_write_reg(0xD51E, 0x02);
    lcd_write_reg(0xD51F, 0x67);
    lcd_write_reg(0xD520, 0x02);
    lcd_write_reg(0xD521, 0x84);
    lcd_write_reg(0xD522, 0x02);
    lcd_write_reg(0xD523, 0xA4);
    lcd_write_reg(0xD524, 0x02);
    lcd_write_reg(0xD525, 0xB7);
    lcd_write_reg(0xD526, 0x02);
    lcd_write_reg(0xD527, 0xCF);
    lcd_write_reg(0xD528, 0x02);
    lcd_write_reg(0xD529, 0xDE);
    lcd_write_reg(0xD52A, 0x02);
    lcd_write_reg(0xD52B, 0xF2);
    lcd_write_reg(0xD52C, 0x02);
    lcd_write_reg(0xD52D, 0xFE);
    lcd_write_reg(0xD52E, 0x03);
    lcd_write_reg(0xD52F, 0x10);
    lcd_write_reg(0xD530, 0x03);
    lcd_write_reg(0xD531, 0x33);
    lcd_write_reg(0xD532, 0x03);
    lcd_write_reg(0xD533, 0x6D);
    lcd_write_reg(0xD600, 0x00);
    lcd_write_reg(0xD601, 0x33);
    lcd_write_reg(0xD602, 0x00);
    lcd_write_reg(0xD603, 0x34);
    lcd_write_reg(0xD604, 0x00);
    lcd_write_reg(0xD605, 0x3A);
    lcd_write_reg(0xD606, 0x00);
    lcd_write_reg(0xD607, 0x4A);
    lcd_write_reg(0xD608, 0x00);
    lcd_write_reg(0xD609, 0x5C);
    lcd_write_reg(0xD60A, 0x00);
    lcd_write_reg(0xD60B, 0x81);

    lcd_write_reg(0xD60C, 0x00);
    lcd_write_reg(0xD60D, 0xA6);
    lcd_write_reg(0xD60E, 0x00);
    lcd_write_reg(0xD60F, 0xE5);
    lcd_write_reg(0xD610, 0x01);
    lcd_write_reg(0xD611, 0x13);
    lcd_write_reg(0xD612, 0x01);
    lcd_write_reg(0xD613, 0x54);
    lcd_write_reg(0xD614, 0x01);
    lcd_write_reg(0xD615, 0x82);
    lcd_write_reg(0xD616, 0x01);
    lcd_write_reg(0xD617, 0xCA);
    lcd_write_reg(0xD618, 0x02);
    lcd_write_reg(0xD619, 0x00);
    lcd_write_reg(0xD61A, 0x02);
    lcd_write_reg(0xD61B, 0x01);
    lcd_write_reg(0xD61C, 0x02);
    lcd_write_reg(0xD61D, 0x34);
    lcd_write_reg(0xD61E, 0x02);
    lcd_write_reg(0xD61F, 0x67);
    lcd_write_reg(0xD620, 0x02);
    lcd_write_reg(0xD621, 0x84);
    lcd_write_reg(0xD622, 0x02);
    lcd_write_reg(0xD623, 0xA4);
    lcd_write_reg(0xD624, 0x02);
    lcd_write_reg(0xD625, 0xB7);
    lcd_write_reg(0xD626, 0x02);
    lcd_write_reg(0xD627, 0xCF);
    lcd_write_reg(0xD628, 0x02);
    lcd_write_reg(0xD629, 0xDE);
    lcd_write_reg(0xD62A, 0x02);
    lcd_write_reg(0xD62B, 0xF2);
    lcd_write_reg(0xD62C, 0x02);
    lcd_write_reg(0xD62D, 0xFE);
    lcd_write_reg(0xD62E, 0x03);
    lcd_write_reg(0xD62F, 0x10);
    lcd_write_reg(0xD630, 0x03);
    lcd_write_reg(0xD631, 0x33);
    lcd_write_reg(0xD632, 0x03);
    lcd_write_reg(0xD633, 0x6D);
    /* LV2 Page 0 enable */
    lcd_write_reg(0xF000, 0x55);
    lcd_write_reg(0xF001, 0xAA);
    lcd_write_reg(0xF002, 0x52);
    lcd_write_reg(0xF003, 0x08);
    lcd_write_reg(0xF004, 0x00);
    /* Display control */
    lcd_write_reg(0xB100, 0xCC);
    lcd_write_reg(0xB101, 0x00);
    /* Source hold time */
    lcd_write_reg(0xB600, 0x05);
    /* Gate EQ control */
    lcd_write_reg(0xB700, 0x70);
    lcd_write_reg(0xB701, 0x70);
    /* Source EQ control (Mode 2) */
    lcd_write_reg(0xB800, 0x01);
    lcd_write_reg(0xB801, 0x03);
    lcd_write_reg(0xB802, 0x03);
    lcd_write_reg(0xB803, 0x03);
    /* Inversion mode (2-dot) */
    lcd_write_reg(0xBC00, 0x02);
    lcd_write_reg(0xBC01, 0x00);
    lcd_write_reg(0xBC02, 0x00);
    /* Timing control 4H w/ 4-delay */
    lcd_write_reg(0xC900, 0xD0);
    lcd_write_reg(0xC901, 0x02);
    lcd_write_reg(0xC902, 0x50);
    lcd_write_reg(0xC903, 0x50);
    lcd_write_reg(0xC904, 0x50);
    lcd_write_reg(0x3500, 0x00);
    lcd_write_reg(0x3A00, 0x55); /* 16-bit/pixel */
    LCD_SET_REG(0x1100);
    delay(120);
    LCD_SET_REG(0x2900);    /* 开启显示 */
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
