/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    zhuxf          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define WRITE_READ_ADDRESS    0x8000
#define NAND_PAGE_SIZE        0x0800  /* 2048 bytes per page w/o Spare Area */
#define NAND_BLOCK_SIZE       0x0040  /* 64x2048 bytes pages per block */
#define NAND_PLANE_SIZE       0x0400  /* 1024 Block per plane */
#define NAND_SPARE_AREA_SIZE  0x0040  /* last 64 bytes as spare area */
#define NAND_BLOCK_NBR        0x0400  /* 1 planes of 1024 block */
#define NAND_PLANE_NBR        1       /* 1 device of 1 planes */
#define BUFFER_SIZE           0X800   /* (NAND_PAGE_SIZE * NAND_PLANE_NBR) */

nand_handle_t         g_nand_handle;
ald_ebi_nand_timing_t g_nand_timing; 
nand_id_t             g_nand_id;
nand_address_t        g_nand_addr;

uint8_t g_tx_buf[BUFFER_SIZE];
uint8_t g_rx_buf[BUFFER_SIZE];

static void nand_pin_init(void);

/**
  * @brief  Fills buffer with user predefined data.
  * @param  buf: pointer on the buffer to fill
  * @param  size: size of the buffer to fill
  * @param  offset: first value to fill on the buffer
  * @retval None
  */
static void fill_buffer(uint8_t *buf, uint32_t size, uint32_t offset)
{
    uint16_t i = 0;

    for (i = 0; i < size; ++i)
        buf[i] = i + offset;
}

/**
  * @brief  Translate logical address into a phy one.
  * @param  addr: Address
  * @param  n_addr: NAND addr.
  * @retval Status
  */
static void nand_addr_get(uint32_t addr, nand_address_t *n_addr)
{
    n_addr->page  = (addr % (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE);
    n_addr->block = (addr % (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));
    n_addr->plane = addr / (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);    /*使能倍频，由晶振三分频倍频至72MHz*/
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);    /*选择倍频时钟为系统时钟*/
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);    /*使能所有外设时钟*/

    ald_cmu_div_config(CMU_HCLK_2, CMU_DIV_1);

    nand_pin_init();

    g_nand_handle.instance          = EBI_NAND_DEVICE;
    g_nand_handle.init.bank         = EBI_NAND_BANK2;
    g_nand_handle.init.wait         = EBI_NAND_WAIT_FEATURE_ENABLE;
    g_nand_handle.init.width        = EBI_NAND_MEM_BUS_WIDTH_8;
    g_nand_handle.init.ecc          = EBI_NAND_ECC_ENABLE;
    g_nand_handle.init.size         = EBI_NAND_ECC_PAGE_SIZE_2048BYTE;
    g_nand_handle.init.cle_time     = 0x0;
    g_nand_handle.init.ale_time     = 0x0;

    g_nand_handle.config.page_size  = NAND_PAGE_SIZE;
    g_nand_handle.config.spare_size = NAND_SPARE_AREA_SIZE;
    g_nand_handle.config.block_size = NAND_BLOCK_SIZE;
    g_nand_handle.config.block_nbr  = NAND_BLOCK_NBR;
    g_nand_handle.config.plane_nbr  = NAND_PLANE_NBR;
    g_nand_handle.config.plane_size = NAND_PLANE_SIZE;

    /*限制条件为：nand flash 手册中的相关参数 + 芯片外设限制。
    HCLK2最大频率72M，需求<= 13ns的不考虑*/
    g_nand_timing.hiz_time    = (((ald_cmu_get_hclk2_clock()) * 7) / 50000000); //(hclk2_clk*140)/1000000000;
    g_nand_timing.wait_time   = ((g_nand_timing.hiz_time) * 4) / 3;
    g_nand_timing.hold_time   = g_nand_timing.wait_time / 2;
    g_nand_timing.time        = g_nand_timing.hold_time;

    if ((g_nand_timing.hiz_time - g_nand_timing.time) < 2)
    {
        g_nand_timing.wait_time++;
        g_nand_timing.hold_time++;
    }

    /*将实际时序转为寄存器值+时序寄存器范围限制*/
    g_nand_timing.hiz_time     = (g_nand_timing.hiz_time < 0x01) ?  0x00 : (g_nand_timing.hiz_time - 1); /*MEMHIZTx/ATTHIZTx   reg范围 0x00--0xFE  HCLK周期 = reg+1 */
    g_nand_timing.hiz_time     = (g_nand_timing.hiz_time > 0xFE) ?  0xFE : g_nand_timing.hiz_time;
    g_nand_timing.hold_time    = (g_nand_timing.hold_time < 0x01) ? 0x01 : g_nand_timing.hold_time;      /*MEMHOLDx/ATTHOLDx   reg范围 0x01--0xFE  HCLK周期 = reg   */
    g_nand_timing.hold_time    = (g_nand_timing.hold_time > 0xFE) ? 0xFE : g_nand_timing.hold_time;
    g_nand_timing.wait_time    = (g_nand_timing.wait_time < 0x01) ? 0x00 : (g_nand_timing.wait_time - 1); /*MEMWAITx/ATTWAITx   reg范围 0x01--0xFF  HCLK周期 = reg+1 */
    g_nand_timing.wait_time    = (g_nand_timing.wait_time > 0xFF) ? 0xFF : g_nand_timing.wait_time;
    g_nand_timing.time         = (g_nand_timing.time < 0x01) ?      0x00 : (g_nand_timing.time - 1);     /*MEMSETUPx/ATTSETUPx reg范围 0x00--0xFE  HCLK周期 = reg+1 */
    g_nand_timing.time         = (g_nand_timing.time > 0xFE) ?      0xFE : g_nand_timing.time;

    /* NAND Flash initialization */
    ald_nand_init(&g_nand_handle, &g_nand_timing, &g_nand_timing);

    ald_nand_reset(&g_nand_handle);

    ald_nand_read_id(&g_nand_handle, &g_nand_id);
    nand_addr_get(WRITE_READ_ADDRESS, &g_nand_addr);

    if (ald_nand_erase_block(&g_nand_handle, &g_nand_addr) == OK)
    {
    }

    fill_buffer(g_tx_buf, BUFFER_SIZE, 0x0);

    if (ald_nand_write_page_8b(&g_nand_handle, &g_nand_addr, g_tx_buf, NAND_PLANE_NBR) != OK)
    {
        ALD_PANIC();
    }

    if (ald_nand_read_page_8b(&g_nand_handle, &g_nand_addr, g_rx_buf, NAND_PLANE_NBR) != OK)
    {
        ALD_PANIC();
    }

    while (1)
    {
        ald_delay_ms(1000);
    }
}

/**
  * @brief  Nand pin init.
  * @param  None
  * @retval None
  */
void nand_pin_init(void)
{
    uint8_t i;
    gpio_init_t gpio_config;

    gpio_config.mode  = GPIO_MODE_OUTPUT;
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.podrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_7;

    /* Data pin init */
    for (i = 0; i < EBI_NAND_DATA_PINn; ++i)
    {
        ald_gpio_init(EBI_DATA_PORT[i], EBI_DATA_PIN[i], &gpio_config);
    }

    /* Control pin init */
    gpio_config.mode  = GPIO_MODE_OUTPUT;
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.podrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_7;

    /* Output enable pin init */
    ald_gpio_init(EBI_NOE_PORT, EBI_NOE_PIN, &gpio_config);
    /* Write enable init EBI_NWE*/
    ald_gpio_init(EBI_NWE_PORT, EBI_NWE_PIN, &gpio_config);
    /* Chip select pin init EBI_NCE2 */
    ald_gpio_init(EBI_NCE2_PORT, EBI_NCE2_PIN, &gpio_config);
    /* EBI_CLE init */
    ald_gpio_init(EBI_CLE_PORT, EBI_CLE_PIN, &gpio_config);
    /* Address Latch pin init EBI_ALE */
    ald_gpio_init(EBI_ALE_PORT, EBI_ALE_PIN, &gpio_config);
    /* Ready/Busy pin init */
    gpio_config.mode = GPIO_MODE_INPUT;
    gpio_config.func = GPIO_FUNC_7;
    ald_gpio_init(EBI_NWAIT_PORT, EBI_NWAIT_PIN, &gpio_config);
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
