/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
static md_dma_descriptor_t s_dma0_ctrl_base[32] __attribute__((aligned(512)));
static md_dma_descriptor_t s_scatter_gather_ctrl[5];
static md_dma_config_t s_dma_config;

static char s_src1_buf[] = "task1:hello world!\r\n";
static char s_src2_buf[] = "task2:hello world!\r\n";
static char s_src3_buf[] = "task3:hello world!\r\n";
static char s_src4_buf[] = "task4:hello world!\r\n";
static char s_src5_buf[] = "task5:hello world!\r\n\r\n";

/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
void init_uart(void);

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  DMA initialization function
  * @retval Status.
  */
void init_dma(void)
{
    s_dma_config.channel = 0;
    s_dma_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    s_dma_config.primary = DISABLE;
    s_dma_config.R_power = MD_DMA_R_POWER_1;
    s_dma_config.dst     = (void *)&UART2->TXBUF;
    s_dma_config.src     = s_src1_buf;
    s_dma_config.dst_inc = MD_DMA_DATA_INC_NONE;
    s_dma_config.src_inc = MD_DMA_DATA_INC_BYTE;
    s_dma_config.size    = strlen(s_src1_buf);
    s_dma_config.msel    = MD_DMA_MSEL_UART2;
    s_dma_config.msigsel = MD_DMA_MSIGSEL_UART_TXEMPTY;
    s_dma_config.burst   = ENABLE;
    s_dma_config.interrupt = DISABLE;
    s_dma_config.high_prio = ENABLE;

    md_dma_config_sg_alt_desc(&s_scatter_gather_ctrl[0], &s_dma_config, DISABLE);

    s_dma_config.src     = s_src2_buf;
    s_dma_config.size    = strlen(s_src2_buf);
    md_dma_config_sg_alt_desc(&s_scatter_gather_ctrl[1], &s_dma_config, DISABLE);

    s_dma_config.src     = s_src3_buf;
    s_dma_config.size    = strlen(s_src3_buf);
    md_dma_config_sg_alt_desc(&s_scatter_gather_ctrl[2], &s_dma_config, DISABLE);

    s_dma_config.src     = s_src4_buf;
    s_dma_config.size    = strlen(s_src4_buf);
    md_dma_config_sg_alt_desc(&s_scatter_gather_ctrl[3], &s_dma_config, DISABLE);

    s_dma_config.src     = s_src5_buf;
    s_dma_config.size    = strlen(s_src5_buf);
    md_dma_config_sg_alt_desc(&s_scatter_gather_ctrl[4], &s_dma_config, DISABLE);

    s_dma_config.channel = 0;
    s_dma_config.data_width = MD_DMA_DATA_SIZE_WORD;
    s_dma_config.primary = ENABLE;
    s_dma_config.R_power = MD_DMA_R_POWER_4;
    s_dma_config.dst     = (void *)&s_dma0_ctrl_base[12];
    s_dma_config.src     = s_scatter_gather_ctrl;
    s_dma_config.dst_inc = MD_DMA_DATA_INC_WORD;
    s_dma_config.src_inc = MD_DMA_DATA_INC_WORD;
    s_dma_config.size    = 5 * 4;
    s_dma_config.msel    = MD_DMA_MSEL_UART2;
    s_dma_config.msigsel = MD_DMA_MSIGSEL_UART_TXEMPTY;
    s_dma_config.burst   = ENABLE;
    s_dma_config.interrupt = ENABLE;
    s_dma_config.high_prio = DISABLE;
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_PER_SG_PRIMARY, &s_dma_config);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    /* DMA init */
    memset(s_dma0_ctrl_base, 0x0, sizeof(s_dma0_ctrl_base));
    memset(&s_dma_config, 0x0, sizeof(md_dma_config_t));
    md_dma_enable();
    md_dma_set_ctrlbase((uint32_t)s_dma0_ctrl_base);
    md_dma_config_struct(&s_dma_config);
    md_dma_clear_flag_done(MD_DMA_CH_0);
    init_dma();
    md_dma_enable_channel(MD_DMA_CH_0);

    /* UART init */
    init_uart();
    md_uart_clear_it_tbc(UART2);
    md_uart_enable_txdma(UART2);

    while (1)
    {
        if (md_dma_is_active_flag_done(MD_DMA_CH_0))
        {
            md_dma_clear_flag_done(MD_DMA_CH_0);

            init_dma();
            md_dma_enable_channel(MD_DMA_CH_0);
        }
    }
}

/**
  * @brief  UART initialization function
  * @retval Status.
  */
void init_uart(void)
{
    md_gpio_init_t gpio_init;
    md_uart_init_t uart_init;

    md_gpio_init_struct(&gpio_init);

    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);

    md_uart_init_struct(&uart_init);
    md_uart_init(UART2, &uart_init);

    md_uart_enable_it_tfempty(UART2);

    return;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
