/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
static md_dma_descriptor_t s_dma0_ctrl_base[1 + 16] __attribute__((aligned(512)));
md_dma_config_t g_dma_config;

/* Public Variables ---------------------------------------------------------- */
char g_src1[] = "task1: hello world!\r\n";
char g_src2[] = "task2: hello world!\r\n\r\n";

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

void init_uart(void);

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  DMA initialization function
  * @retval None.
  */
void init_dma(void)
{
    memset(&g_dma_config, 0x0, sizeof(md_dma_config_t));

    md_dma_enable();
    md_dma_set_ctrlbase((uint32_t)s_dma0_ctrl_base);

    md_dma_config_struct(&g_dma_config);

    g_dma_config.channel = 0;
    g_dma_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_config.primary = DISABLE;
    g_dma_config.R_power = MD_DMA_R_POWER_1;
    g_dma_config.dst     = (void *)&UART2->TXBUF;
    g_dma_config.src     = g_src2;
    g_dma_config.dst_inc = MD_DMA_DATA_INC_NONE;
    g_dma_config.src_inc = MD_DMA_DATA_INC_BYTE;
    g_dma_config.size    = strlen(g_src2);
    g_dma_config.msel    = MD_DMA_MSEL_UART2;
    g_dma_config.msigsel = MD_DMA_MSIGSEL_UART_TXEMPTY;
    g_dma_config.burst   = ENABLE;
    g_dma_config.interrupt = ENABLE;

    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_PINGPONG, &g_dma_config);

    g_dma_config.channel = 0;
    g_dma_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_config.primary = ENABLE;
    g_dma_config.R_power = MD_DMA_R_POWER_1;
    g_dma_config.dst     = (void *)&UART2->TXBUF;
    g_dma_config.src     = g_src1;
    g_dma_config.dst_inc = MD_DMA_DATA_INC_NONE;
    g_dma_config.src_inc = MD_DMA_DATA_INC_BYTE;
    g_dma_config.size    = strlen(g_src1);
    g_dma_config.msel    = MD_DMA_MSEL_UART2;
    g_dma_config.msigsel = MD_DMA_MSIGSEL_UART_TXEMPTY;
    g_dma_config.burst   = ENABLE;
    g_dma_config.interrupt = ENABLE;

    md_dma_clear_flag_done(MD_DMA_CH_0);
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_PINGPONG, &g_dma_config);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    init_uart();

    memset(s_dma0_ctrl_base, 0x0, sizeof(s_dma0_ctrl_base));
    init_dma();

    __NVIC_EnableIRQ(DMA_IRQn);

    md_uart_clear_it_tfth(UART2);

    md_dma_enable_channel(MD_DMA_CH_0);

    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @brief  UART initialization function
  * @retval Status.
  */
void init_uart(void)
{
    md_gpio_init_t gpio_init;
    md_uart_init_t uart_init;

    md_gpio_init_struct(&gpio_init);

    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);

    md_uart_init_struct(&uart_init);
    md_uart_init(UART2, &uart_init);
    md_uart_enable_txdma(UART2);
}

/**
  * @brief  DAM pingpong reconfiguration function
  * @retval Status.
  */
void dma_uart_pingpong_config_struct(uint8_t channel, void *src, uint16_t length)
{
    md_dma_descriptor_t *desc;

    desc = (md_dma_descriptor_t *)(DMA0->CTRLBASE) + channel;

    if (desc->ctrl.cycle_ctrl != 0)
        desc = (md_dma_descriptor_t *)(DMA0->ALTCTRLBASE) + channel;

    desc->ctrl.cycle_ctrl = MD_DMA_CYCLE_CTRL_PINGPONG;

    desc->src = (void *)((uint32_t)src + ((length - 1) << desc->ctrl.src_size));
    desc->ctrl.n_minus_1 = length - 1;
}

/**
  * @brief  DAM handler
  * @retval Status.
  */
void DMA_Handler(void)
{
    if (md_dma_is_active_flag_done(MD_DMA_CH_0) && md_dma_is_enabled_it_done(MD_DMA_CH_0))
    {
        if (md_dma_is_set_use_alternate(MD_DMA_CH_0))
            dma_uart_pingpong_config_struct(0, g_src1, strlen(g_src1));
        else
            dma_uart_pingpong_config_struct(0, g_src2, strlen(g_src2));

        md_dma_clear_flag_done(MD_DMA_CH_0);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
