/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include <stdio.h>
#include "main.h"

/* Private Macros ----------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */
static uint8_t s_src1_buf[] = "task1: hello world!\r\n";
static uint8_t s_src2_buf[] = "task2: hello world!\r\n\r\n";

static uint8_t  s_complete_flag = 0;
static uint8_t  s_index = 0;

/* Public Variables -------------------------------------------------------- */
dma_config_t g_dma_config;
uart_handle_t g_uart_init;

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */
/**
  * @brief  DMA pingpong struct config.
  * @param  arg: Pointer.
  * @retval None.
  */
void dma_i2s_pingpong_config_struct(uint8_t channel, void *src, uint16_t length)
{
    dma_descriptor_t *desc;

    desc = (dma_descriptor_t *)(DMA0->CTRLBASE) + channel;

    if (desc->ctrl.cycle_ctrl != 0)
        desc = (dma_descriptor_t *)(DMA0->ALTCTRLBASE) + channel;

    desc->ctrl.cycle_ctrl = DMA_CYCLE_CTRL_PINGPONG;

    desc->src = (void *)((uint32_t)src + ((length - 1) << desc->ctrl.src_size));
    desc->ctrl.n_minus_1 = length - 1;

    return;
}

/**
  * @brief  DMA send complete.
  * @param  arg: Pointer.
  * @retval None.
  */
void dma_complete(void *arg)
{
    s_complete_flag++;

    return;
}

/**
  * @brief  UART pin initialization function
  * @retval Status.
  */
void init_uart_pin(void)
{
    gpio_init_t gpio_init;

    /* Initialize tx pin */
    gpio_init.mode = GPIO_MODE_OUTPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_3;
    ald_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    /* Initialize rx pin */
    gpio_init.mode = GPIO_MODE_INPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_3;
    ald_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);

    return;
}

/**
  * @brief  UART initialization function
  * @retval Status.
  */
void init_uart(void)
{
    init_uart_pin();

    g_uart_init.perh             = UART2;
    g_uart_init.init.baud        = 115200;
    g_uart_init.init.word_length = UART_WORD_LENGTH_8B;
    g_uart_init.init.stop_bits   = UART_STOP_BITS_1;
    g_uart_init.init.parity      = UART_PARITY_NONE;
    g_uart_init.init.mode        = UART_MODE_UART;
    g_uart_init.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    g_uart_init.tx_cplt_cbk      = NULL;
    g_uart_init.rx_cplt_cbk      = NULL;
    g_uart_init.error_cbk        = NULL;
    ald_uart_init(&g_uart_init);

    return;
}

/**
  * @brief  DMA initialization function
  * @retval None.
  */
void init_dma(void)
{
    g_dma_config.channel = 0;
    g_dma_config.primary = DISABLE;
    g_dma_config.R_power = DMA_R_POWER_1;
    g_dma_config.data_width = DMA_DATA_SIZE_BYTE;
    g_dma_config.dst     = (void *)(&UART2->TXBUF);
    g_dma_config.dst_inc = DMA_DATA_INC_NONE;
    g_dma_config.src     = s_src2_buf;
    g_dma_config.src_inc = DMA_DATA_INC_BYTE;
    g_dma_config.size    = sizeof(s_src2_buf);
    g_dma_config.msel    = DMA_MSEL_UART2;
    g_dma_config.msigsel = DMA_MSIGSEL_UART_TXEMPTY;
    g_dma_config.burst   = ENABLE;
    g_dma_config.interrupt = ENABLE;
    g_dma_config.high_prio = DISABLE;
    ald_dma_config_ping_pong(DMA0, &g_dma_config, 0, dma_complete);

    g_dma_config.channel = 0;
    g_dma_config.primary = ENABLE;
    g_dma_config.R_power = DMA_R_POWER_1;
    g_dma_config.data_width = DMA_DATA_SIZE_BYTE;
    g_dma_config.dst     = (void *)(&UART2->TXBUF);
    g_dma_config.dst_inc = DMA_DATA_INC_NONE;
    g_dma_config.src     = s_src1_buf;
    g_dma_config.src_inc = DMA_DATA_INC_BYTE;
    g_dma_config.size    = sizeof(s_src1_buf);
    g_dma_config.msel    = DMA_MSEL_UART2;
    g_dma_config.msigsel = DMA_MSIGSEL_UART_TXEMPTY;
    g_dma_config.burst   = ENABLE;
    g_dma_config.interrupt = ENABLE;
    g_dma_config.high_prio = DISABLE;
    ald_dma_config_ping_pong(DMA0, &g_dma_config, 1, dma_complete);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD CMU_PLL1_INPUT_HOSC_3*/
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    init_uart();

    init_dma();

    ald_uart_dma_req_config(&g_uart_init, UART_DMA_REQ_TX, ENABLE);

    while (1)
    {
        if (s_complete_flag)
        {
            s_complete_flag = 0;

            if (s_index)
                dma_i2s_pingpong_config_struct(0, s_src1_buf, sizeof(s_src1_buf));
            else
                dma_i2s_pingpong_config_struct(0, s_src2_buf, sizeof(s_src2_buf));

            s_index++;
            s_index &= 0x01;

            ald_dma_channel_config(DMA0, 0, ENABLE);
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
