/**********************************************************************************
 *
 * @file    main.c
 * @brief   ADC_DMA C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          24 Jan 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
adc_handle_t g_adc_config;
uint16_t g_adc_result[4] = {0U};
uint32_t g_adc_conv_result[4] = {0U};

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  DMA pingpong struct config.
  * @param  arg: Pointer.
  * @retval None.
  */
void dma_i2s_pingpong_config_struct(uint8_t channel, adc_handle_t *h_adc, uint16_t length)
{
    dma_descriptor_t *desc;

    desc = (dma_descriptor_t *)(DMA0->CTRLBASE) + channel;

    if (desc->ctrl.cycle_ctrl != 0)
        desc = (dma_descriptor_t *)(DMA0->ALTCTRLBASE) + channel;

    desc->ctrl.cycle_ctrl = DMA_CYCLE_CTRL_PINGPONG;

    desc->src = (void *)&h_adc->perh->NCHDR;
    desc->ctrl.n_minus_1 = length - 1;

    return;
}

/**
  * @brief  DMA send complete.
  * @param  arg: Pointer.
  * @retval None.
  */
void dma_complete(void *arg)
{
    uint8_t i;

    for (i = 0; i < 4; i++)
        g_adc_conv_result[i] = g_adc_result[i] * 3320 / 4096;

    dma_i2s_pingpong_config_struct(0, &g_adc_config, sizeof(g_adc_result) / 2);

    ald_dma_channel_config(DMA0, 0, ENABLE);

    return;
}

/**
  * @brief  DMA initialization.
  * @param  hperh: pointer of adc handler.
  * @retval None.
  */
void init_dma(adc_handle_t *hperh)
{
    dma_config_t g_dma_config;
    memset(&g_dma_config, 0x00, sizeof(g_dma_config));
    g_dma_config.channel    = 0;
    g_dma_config.primary    = DISABLE;
    g_dma_config.R_power    = DMA_R_POWER_1;
    g_dma_config.data_width = DMA_DATA_SIZE_HALFWORD;
    g_dma_config.dst        = g_adc_result;
    g_dma_config.dst_inc    = DMA_DATA_INC_HALFWORD;
    g_dma_config.src        = (void *)&hperh->perh->NCHDR;
    g_dma_config.src_inc    = DMA_DATA_INC_NONE;
    g_dma_config.size       = 4;  /* sizeof(g_adc_result) / 2; */
    g_dma_config.msel       = hperh->perh == ADC0 ? DMA_MSEL_ADC0 : DMA_MSEL_ADC1;
    g_dma_config.msigsel    = DMA_MSIGSEL_ADC;
    g_dma_config.burst      = ENABLE;
    g_dma_config.interrupt  = ENABLE;
    g_dma_config.high_prio  = DISABLE;
    ald_dma_config_ping_pong(DMA0, &g_dma_config, 0, dma_complete);

    g_dma_config.channel    = 0;
    g_dma_config.primary    = ENABLE;
    g_dma_config.R_power    = DMA_R_POWER_1;
    g_dma_config.data_width = DMA_DATA_SIZE_HALFWORD;
    g_dma_config.dst        = g_adc_result;
    g_dma_config.dst_inc    = DMA_DATA_INC_HALFWORD;
    g_dma_config.src        = (void *)&hperh->perh->NCHDR;
    g_dma_config.src_inc    = DMA_DATA_INC_NONE;
    g_dma_config.size       = 4;
    g_dma_config.msel       = hperh->perh == ADC0 ? DMA_MSEL_ADC0 : DMA_MSEL_ADC1;
    g_dma_config.msigsel    = DMA_MSIGSEL_ADC;
    g_dma_config.burst      = ENABLE;
    g_dma_config.interrupt  = ENABLE;
    g_dma_config.high_prio  = DISABLE;
    ald_dma_config_ping_pong(DMA0, &g_dma_config, 1, dma_complete);

    SET_BIT(hperh->perh->CON1, ADC_CON1_DMA_MSK);

    return;
}

/**
  * @brief  Initializate pin of adc module.
  * @retval None.
  */
void init_adc_pin(void)
{
    /* Initialize ADC pin */
    gpio_init_t gpio_config;

    memset(&gpio_config, 0, sizeof(gpio_init_t));

    gpio_config.mode = GPIO_MODE_CLOSE;
    gpio_config.pupd = GPIO_FLOATING;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.func = GPIO_FUNC_0;
    ald_gpio_init(ADC_IN_PORT, ADC_IN0_PIN, &gpio_config);
    ald_gpio_init(ADC_IN_PORT, ADC_IN1_PIN, &gpio_config);
    ald_gpio_init(ADC_IN_PORT, ADC_IN2_PIN, &gpio_config);
    ald_gpio_init(ADC_IN_PORT, ADC_IN3_PIN, &gpio_config);

    return;
}

/**
  * @brief  Initializate adc module.
  * @retval None.
  */
void init_adc(void)
{
    adc_nch_conf_t nch_config;

    memset(&g_adc_config, 0, sizeof(adc_handle_t));
    memset(&nch_config, 0, sizeof(adc_nch_conf_t));

    /* Initialize adc */
    g_adc_config.perh = ADC0;
    g_adc_config.init.align    = ADC_DATAALIGN_RIGHT;
    g_adc_config.init.scan     = ENABLE;
    g_adc_config.init.cont     = ENABLE;
    g_adc_config.init.disc     = ADC_ALL_DISABLE;
    g_adc_config.init.disc_nr  = ADC_DISC_NR_4;
    g_adc_config.init.data_bit = ADC_CONV_BIT_12;
    g_adc_config.init.div      = ADC_CKDIV_4;
    g_adc_config.init.nche_sel = ADC_NCHESEL_MODE_ALL;
    g_adc_config.init.nch_nr   = ADC_NCH_NR_4;
    g_adc_config.init.n_ref    = ADC_NEG_REF_VSS;
    g_adc_config.init.p_ref    = ADC_POS_REF_VDD;
    g_adc_config.normal_cplt_cbk = NULL;
    ald_adc_init(&g_adc_config);

    /* Initialize normal convert channel */
    nch_config.ch   = ADC_CHANNEL_0;
    nch_config.idx  = ADC_NCH_IDX_1;
    nch_config.samp = ADC_SAMPLETIME_15;
    ald_adc_normal_channel_config(&g_adc_config, &nch_config);

    nch_config.ch   = ADC_CHANNEL_1;
    nch_config.idx  = ADC_NCH_IDX_2;
    nch_config.samp = ADC_SAMPLETIME_15;
    ald_adc_normal_channel_config(&g_adc_config, &nch_config);

    nch_config.ch   = ADC_CHANNEL_2;
    nch_config.idx  = ADC_NCH_IDX_3;
    nch_config.samp = ADC_SAMPLETIME_15;
    ald_adc_normal_channel_config(&g_adc_config, &nch_config);

    nch_config.ch   = ADC_CHANNEL_3;
    nch_config.idx  = ADC_NCH_IDX_4;
    nch_config.samp = ADC_SAMPLETIME_15;
    ald_adc_normal_channel_config(&g_adc_config, &nch_config);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

    /* Enable mode clock */
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Enable DMA0 interupt */
    ald_mcu_irq_config(DMA_IRQn, 1, 1, ENABLE);

    /*  Initialize  ADC */
    init_adc_pin();
    init_adc();

    /* Initialize DMA */
    init_dma(&g_adc_config);

    /* start convert */
    ADC_ENABLE(&g_adc_config);
    ADC_NH_TRIG_BY_SOFT(&g_adc_config);

    while (1)
    {

    }
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
