/*
 * Copyright (c) 2002, Adam Dunkels.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This file is part of the Contiki OS
 *
 */

/**
 * \ingroup platform
 *
 * \defgroup native_platform Native platform
 *
 * Platform running in the host (Windows or Linux) environment.
 *
 * Used mainly for development and debugging.
 * @{
 */

#include <stdarg.h>
#include "ald_conf.h"
#include "contiki.h"
#include "net/netstack.h"

#include "dev/serial-line.h"
#include "dev/button-hal.h"
#include "dev/gpio-hal.h"
#include "dev/leds.h"

#include "net/ipv6/uip.h"
#include "net/ipv6/uip-debug.h"
#include "net/queuebuf.h"
#include "lib/random.h"
#if NETSTACK_CONF_WITH_IPV6
    #include "net/ipv6/uip-ds6.h"
#endif /* NETSTACK_CONF_WITH_IPV6 */
/* Log configuration */
#include "sys/log.h"
#define LOG_MODULE "Native"
#define LOG_LEVEL LOG_LEVEL_MAIN

const uint32_t *CHIP_UID=(uint32_t*)(0x00080000+0x03E0);
const uint32_t *CHIP_ID=(uint32_t*)(0x00080000+0x03F8);
#ifdef PLATFORM_CONF_MAC_ADDR
static uint8_t mac_addr[] = PLATFORM_CONF_MAC_ADDR;
#else /* PLATFORM_CONF_MAC_ADDR */
static uint8_t mac_addr[] = { 0xe1, 0xa2, 0xa3, 0xa4, 0xb5, 0xb6, 0xb7, 0xb8 };
#endif /* PLATFORM_CONF_MAC_ADDR */

static void
set_lladdr(void)
{
    //printf("CHIP_ID =%08x\r\n",CHIP_ID[0]);
    //printf("CHIP_UID=%08x %08x %08x\r\n",CHIP_UID[0],CHIP_UID[2],CHIP_UID[4]);
    linkaddr_t addr;
    *((uint32_t*)mac_addr+1)+=CHIP_UID[0];
    //*((uint32_t*)mac_addr+1)=CHIP_UID[2];
    memset(&addr, 0, sizeof(linkaddr_t));
#if NETSTACK_CONF_WITH_IPV6
    memcpy(addr.u8, mac_addr, sizeof(addr.u8));
#else
    int i;
    for (i = 0; i < sizeof(linkaddr_t); ++i) {
        addr.u8[i] = mac_addr[7 - i];
    }
#endif
    linkaddr_set_node_addr(&addr);
}
/*---------------------------------------------------------------------------*/
#if NETSTACK_CONF_WITH_IPV6
static void
set_global_address(void)
{
    uip_ipaddr_t ipaddr;
    const uip_ipaddr_t *default_prefix = uip_ds6_default_prefix();

    /* Assign a unique local address (RFC4193,
       http://tools.ietf.org/html/rfc4193). */
    uip_ip6addr_copy(&ipaddr, default_prefix);

    /* Assumes that the uip_lladdr is set */
    uip_ds6_set_addr_iid(&ipaddr, &uip_lladdr);
    uip_ds6_addr_add(&ipaddr, 0, ADDR_AUTOCONF);

    LOG_INFO("Added global IPv6 address ");
    LOG_INFO_6ADDR(&ipaddr);
    LOG_INFO_("\n");

    /* set the PREFIX::1 address to the IF */
    uip_ip6addr_copy(&ipaddr, default_prefix);
    ipaddr.u8[15] = 1;
    uip_ds6_defrt_add(&ipaddr, 0);
}
#endif

extern void es32_init(void);
/*---------------------------------------------------------------------------*/
void
platform_init_stage_one()
{
    es32_init();
    //gpio_hal_init();
    //button_hal_init();
    //leds_init();
    return;
}
/*---------------------------------------------------------------------------*/
void
platform_init_stage_two()
{
    set_lladdr();
}
/*---------------------------------------------------------------------------*/
void
platform_init_stage_three()
{
    
#if NETSTACK_CONF_WITH_IPV6
    set_global_address();
#endif /* NETSTACK_CONF_WITH_IPV6 */
    random_init(CHIP_UID[0]);
}
/*---------------------------------------------------------------------------*/
void platform_idle()
{
}
/*---------------------------------------------------------------------------*/
extern void uart_send_str(const char *s);
int log_output(const char *fmt, ...)
{
    char buf[256] = {0};

    if (fmt[0] == '[') {
        sprintf(buf, "[%4d]", ald_get_tick());
        uart_send_str(buf);
    }

    va_list args;
    va_start(args, fmt);
    vsnprintf(buf, 256, fmt, args);
    va_end(args);
    uart_send_str(buf);
    return 0;
}
void
log_message(char *m1, char *m2)
{
    printf("%s%s\r\n", m1, m2);
}
/*---------------------------------------------------------------------------*/
void
uip_log(char *m)
{
    printf("%s\r\n", m);
}
/*---------------------------------------------------------------------------*/
/** @} */
