/*
 * Copyright (c) 2007, Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * This file is part of the Contiki operating system.
 *
 */

/**
 * \file
 *         Native (non-specific) code for the Contiki real-time module rt
 * \author
 *         Adam Dunkels <adam@sics.se>
 */

#include "sys/rtimer.h"
#include "sys/clock.h"
#include "ald_conf.h"
#include <string.h>
#define DEBUG 0
#if DEBUG
#include <stdio.h>
#define PRINTF(...) printf(__VA_ARGS__)
#else
#define PRINTF(...)
#endif
             
timer_handle_t g_timer2;
timer_handle_t g_timer1;

__STATIC_INLINE uint32_t SYSTICK_IsActiveCounterFlag(void)
{
    return ((SysTick->CTRL & SysTick_CTRL_COUNTFLAG_Msk) == (SysTick_CTRL_COUNTFLAG_Msk));
}
uint32_t getCurrentMicros(void)
{
    /* Ensure COUNTFLAG is reset by reading SysTick control and status register */
    SYSTICK_IsActiveCounterFlag(); 
    const uint32_t tms = SysTick->LOAD + 1;
    uint32_t m = ald_get_tick();
    __IO uint32_t u = tms - SysTick->VAL;
    if (SYSTICK_IsActiveCounterFlag()) {
        m = ald_get_tick();
        u = tms - SysTick->VAL;
    }
    return (m * 1000 + (u  / (tms/1000)));
}
void ald_timer_period_elapsed_callback(struct timer_handle_s *arg)
{
    TIMER_DISABLE(&g_timer1);
    rtimer_run_next();
}
/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    timer_clock_config_t timer_config;
    /* Initialize AD16C4T1 */
    memset(&g_timer1, 0x0, sizeof(g_timer1));  /* initialize the g_ad16c4t_init */
    g_timer1.perh = AD16C4T1;
    g_timer1.init.prescaler    = ald_cmu_get_sys_clock()/RTIMER_ARCH_SECOND - 1;             /* clk_count: RTIMER */
    g_timer1.init.mode         = TIMER_CNT_MODE_UP;  /* count up */
    g_timer1.init.period       = 1;           /* period is 1000 count clock */
    g_timer1.init.clk_div      = TIMER_CLOCK_DIV1;   /* working clock of dead time and filter */
    g_timer1.init.re_cnt       = 0;             /* 10 repeat count */
    g_timer1.period_elapse_cbk = ald_timer_period_elapsed_callback;  /* update period callback function */
    ald_timer_base_init(&g_timer1);

    /* Initialize clock source */
    memset(&timer_config,0,sizeof(timer_clock_config_t));
    timer_config.source = TIMER_SRC_INTER;   /**< internal clock sources */
    ald_timer_config_clock_source(&g_timer1, &timer_config);

    ald_mcu_irq_config(AD16C4T1_UP_IRQn, 0, 0, ENABLE);/* Enable AD16C4T1 interrupt */
    //ald_timer_base_start_by_it(&g_timer1);       /* Start UPDATE interrupt by interrupt */
}

uint32_t rtimer_arch_now()
{
    return getCurrentMicros()/(1000000/RTIMER_ARCH_SECOND);
}


void
rtimer_arch_init(void)
{
    init_timer();
}
/*---------------------------------------------------------------------------*/
void
rtimer_arch_schedule(rtimer_clock_t t)
{
    WRITE_REG(g_timer1.perh->AR, t-1);
    ald_timer_base_start_by_it(&g_timer1);
}
/*---------------------------------------------------------------------------*/
