/**
  **************************************************************************************
  * @file    usbdma.c
  * @brief   USB Library DMA handling functions.
  * @data    11/9/2018
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. ALL rights reserved.
  *
  **************************************************************************************
  */

#include <stdbool.h>
#include <stdint.h>
#include "string.h"
#include "ald_dma.h"
#include "usblib/drivers/usb_lowlayer_api.h"
#include "usblib/drivers/debug.h"
#include "usblib/drivers/type.h"
#include "usblib/usblib.h"
#include "usblib/usblibpriv.h"

/** @addtogroup usblib_dma_api
  * @{
  */

//*****************************************************************************
//
// DMA controller.
//
//*****************************************************************************
static tUSBDMAInstance g_psUSBDMAInst[1];

//*****************************************************************************
//
// DMA callback.
//
//*****************************************************************************
static void (*DMACallback)(void) = NULL;

/**
  * /@brief USBLibDMAChannelStatus() for USB controllers that use the DMA.
  */
static uint32_t
dmausb_channel_status(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel)
{
  uint32_t ui32Status;

  //
  // Initialize the current status to no events.
  //
  ui32Status = USBLIBSTATUS_DMA_IDLE;

  //
  // Check if there is a pending DMA transfer.
  //
  if (psUSBDMAInst->ui32Complete & (1 << ui32Channel))
  {
    //
    // Return that the DMA transfer has completed and clear the
    // DMA pending flag.
    //
    ui32Status = USBLIBSTATUS_DMA_COMPLETE;
  }
  else if (psUSBDMAInst->ui32Pending & (1 << ui32Channel))
  {
    //
    // DMA transfer is still pending.
    //
    ui32Status = USBLIBSTATUS_DMA_PENDING;
  }
  else
  {
    //
    // DMA transfer is still pending.
    //
    ui32Status = USBLIBSTATUS_DMA_IDLE;
  }

  return (ui32Status);
}

/**
  * /@brief USBLibDMAIntStatus() for USB controllers.
  */
static uint32_t
dmausb_int_status(tUSBDMAInstance *psUSBDMAInst)
{
  uint32_t ui32Status, ui32Pending;
  int32_t i32Channel;

  //
  // No pending interrupts by default.
  //
  ui32Status = 0;

  //
  // Save the pending channels.
  //
  ui32Pending = psUSBDMAInst->ui32Pending;

  //
  // Loop through channels to find out if any pending DMA transfers have
  // completed.
  //
  for (i32Channel = 0; i32Channel < USB_MAX_DMA_CHANNELS; i32Channel++)
  {
    //
    // If pending and stopped then the DMA completed.
    //
#if defined ES32F0271
    if ((ui32Pending & 1) && (md_usb_dma_channel_isenabled(i32Channel) == 0))
#else
    //
    // Add new logic.
    //
#endif
    {
      ui32Status |= (1 << i32Channel);
    }

    ui32Pending >>= 1;

    //
    // Done if this is zero.
    //
    if (ui32Pending == 0)
    {
      break;
    }
  }

  return (ui32Status);
}

/**
  * /@brief USBLibDMAIntStatusClear() for USB controllers that use DMA for DMA or have
  *    an integrated DMA controller.
  */
static void
dmausb_int_statusClear(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Status)
{
  //
  // Clear out the requested interrupts.  Since the USB interface does not
  // have a true interrupt clear, this clears the current completed
  // status for the requested channels.
  //
  psUSBDMAInst->ui32Complete &= ~ui32Status;

  return;
}

/**
  * /@brief USBLibDMAIntHandler() for USB controllers that use DMA for DMA or have an
  *      integrated DMA controller.
  */
static void
dmausb_int_handler(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32DMAIntStatus)
{
  uint32_t ui32Channel;

  if (ui32DMAIntStatus == 0)
  {
    return;
  }

  //
  // Determine if the DMA is used or the USB DMA controller.
  //
  for (ui32Channel = 0; ui32Channel < USB_MAX_DMA_CHANNELS; ui32Channel++)
  {
    //
    // Mark any pending interrupts as completed.
    //
    if (ui32DMAIntStatus & 1)
    {
      psUSBDMAInst->ui32Pending &= ~(1 << ui32Channel);
      psUSBDMAInst->ui32Complete |= (1 << ui32Channel);
    }

    //
    // Check the next channel.
    //
    ui32DMAIntStatus >>= 1;

    //
    // Break if there are no more pending DMA interrupts.
    //
    if (ui32DMAIntStatus == 0)
    {
      break;
    }
  }
}

#ifdef ES32F0271
/**
  * /@brief dma_out_usb_int_handler() for USB controllers that use DMA for DMA or have an
  *        external DMA controller.
  */
static void
dma_out_usb_int_handler(void *psUSBDMAInst)
{
  //
  // Unused param;
  //
  UNUSED(psUSBDMAInst);

  //
  // This function will callback the Device or Host USB controller handler.
  //
  if (NULL != DMACallback)
    DMACallback();
}
#else
/**
  * /@brief USBLibDMAIntHandler() for USB controllers that use DMA for DMA or have an
  *        integrated DMA controller.
  */
static void
dma_out_usb_int_handler(void *psUSBDMAInst)
{
  //
  // We do nothing in this function.
  //
  UNUSED(psUSBDMAInst);
  UNUSED(dma_out_usb_int_handler);
  if (DMACallback)
    ;
}
#endif

/**
  * /@brief USBLibDMAChannelEnable() for USB controllers that use DMA.
  */
static void
dmausb_channel_enable(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel)
{
  //
  // Disable the USB interrupt if it was enabled.
  //
  md_usb_system_int_disable();

  //
  // Mark this channel as pending and not complete.
  //
  psUSBDMAInst->ui32Pending |= (1 << ui32Channel);
  psUSBDMAInst->ui32Complete &= ~(1 << ui32Channel);

  //
  // Enable DMA for the endpoint.
  //
#ifdef ES32F0271

  if (psUSBDMAInst->pui32Config[ui32Channel].config.src_inc == ALD_DMA_DATA_SINC_ENABLE)
  {
    md_usb_dma_endpoint_enable(psUSBDMAInst->pui8Endpoint[ui32Channel],
                               USB_EP_DEV_OUT | USB_EP_HOST_IN);
  }
  else
  {
    md_usb_dma_endpoint_enable(psUSBDMAInst->pui8Endpoint[ui32Channel],
                               USB_EP_DEV_IN | USB_EP_HOST_OUT);
  }

#else

#endif

  //
  // Enable the interrupt if it was enabled before.
  //
  md_usb_system_int_enable();

  //
  // Enable the DMA channel.
  //
  md_usb_dma_channel_enable(ui32Channel);
}

/**
  * /@brief USBLibDMAChannelDisable() for USB controllers.
  */
static void
dmausb_channel_disable(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel)
{
  //
  // Disable DMA for the endpoint.
  //
#ifdef ES32F0271
  if (psUSBDMAInst->pui32Config[ui32Channel].config.src_inc == ALD_DMA_DATA_SINC_ENABLE)
  {
    md_usb_dma_endpoint_disable(psUSBDMAInst->pui8Endpoint[ui32Channel],
                                USB_EP_DEV_OUT);
  }
  else
  {
    md_usb_dma_endpoint_disable(psUSBDMAInst->pui8Endpoint[ui32Channel],
                                USB_EP_DEV_IN);
  }

#else
#endif

  //
  // Disable the DMA channel.
  //
  md_usb_dma_channel_disable(ui32Channel);

  //
  // Clear out any pending or complete flag set for this DMA channel.
  //
  psUSBDMAInst->ui32Pending &= ~(1 << ui32Channel);
  psUSBDMAInst->ui32Complete &= ~(1 << ui32Channel);
}

/**
  * /@brief USBLibDMAChannelIntEnable() for USB controllers that use DMA.
  */
static void
dmausb_channel_int_enable(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel)
{
#ifdef ES32F0271
  //
  // There is no way to Enable channel interrupts when using DMA.
  //
#else
  //
  // Other DMA controller.
  //
#endif
}

/**
  * /@brief USBLibDMAChannelIntDisable() for USB controllers that use DMA.
  */
static void
dmausb_channel_int_disable(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel)
{
#ifdef ES32F0271
  //
  // There is no way to disable channel interrupts when using DMA.
  //
#else
  //
  // Other DMA controller.
  //
#endif
}

/**
  * /@brief USBLibDMATransfer() for USB controllers that use the DMA controller.
  */
static uint32_t
dmausb_transfer(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel,
                void *pvBuffer, uint32_t ui32Size)
{
  void *pvFIFO;
  uint32_t ui32PacketCount;
  uint32_t ui32TransferCount;

  if ((ui32Size < 64) || ((uint32_t)pvBuffer & 0x3))
  {
    return (0);
  }

  //
  // Mark this channel as pending and not complete.
  //
  psUSBDMAInst->ui32Pending |= (1 << ui32Channel);
  psUSBDMAInst->ui32Complete &= ~(1 << ui32Channel);

  //
  // Save the pointer to the data and the byte count.
  //
  psUSBDMAInst->ppui32Data[ui32Channel] = pvBuffer;
  psUSBDMAInst->pui32Count[ui32Channel] = ui32Size;

  //
  // Need the address of the FIFO.
  //
  pvFIFO = (void *)md_usb_hosdev_fifo_addr_get(psUSBDMAInst->pui8Endpoint[ui32Channel]);

#ifdef ES32F0271
  ui32TransferCount = ui32Size;

  //
  // If source increment is none this is an RX transfer.
  //
  if (psUSBDMAInst->pui32Config[ui32Channel].config.src_inc == ALD_DMA_DATA_SINC_DISABLE)
  {
    psUSBDMAInst->pui32Config[ui32Channel].config.src = pvFIFO;
    psUSBDMAInst->pui32Config[ui32Channel].config.dst = pvBuffer;
    psUSBDMAInst->pui32Config[ui32Channel].config.size = ui32TransferCount;
    ald_dma_config_basic(&psUSBDMAInst->pui32Config[ui32Channel]);
  }
  else
  {
    psUSBDMAInst->pui32Config[ui32Channel].config.src = pvBuffer;
    psUSBDMAInst->pui32Config[ui32Channel].config.dst = pvFIFO;
    psUSBDMAInst->pui32Config[ui32Channel].config.size = ui32TransferCount;
    ald_dma_config_basic(&psUSBDMAInst->pui32Config[ui32Channel]);
  }

#else
  UNUSED(pvFIFO);
  UNUSED(ui32TransferCount);

#endif

  //
  // Set the mode based on the size of the transfer.  More than one
  // packet requires mode 1.
  //
  if (ui32Size > psUSBDMAInst->pui32MaxPacketSize[ui32Channel])
  {
    //
    // Calculate the number of packets required for this transfer.
    //
    ui32PacketCount = ((ui32Size /
                        psUSBDMAInst->pui32MaxPacketSize[ui32Channel]));

    //
    // Set the packet count so that the last packet does not generate
    // another IN request.
    //
    md_usb_hosdev_endpoint_pkcount_set(psUSBDMAInst->pui8Endpoint[ui32Channel],
                                       ui32PacketCount);

    //
    // Configure the USB endpoint in mode 1 for this DMA transfer.
    //
    md_usb_dma_endpoint_config(psUSBDMAInst->pui8Endpoint[ui32Channel],
                               psUSBDMAInst->pui32EPDMAMode1[ui32Channel]);
  }
  else
  {
    //
    // Configure the USB endpoint in mode 0 for this DMA transfer.
    //
    md_usb_dma_endpoint_config(psUSBDMAInst->pui8Endpoint[ui32Channel],
                               psUSBDMAInst->pui32EPDMAMode0[ui32Channel]);
  }

  //
  // Enable the DMA channel to start the transfer
  //
  dmausb_channel_enable(psUSBDMAInst, ui32Channel);

  return (ui32Size);
}

/**
  * /@brief USBLibDMAChannelAllocate() for USB controllers.
  */
static uint32_t
dmausb_channel_allocate(tUSBDMAInstance *psUSBDMAInst, uint8_t ui8Endpoint,
                        uint32_t ui32MaxPacketSize, uint32_t ui32Config)
{
  uint32_t ui32Channel;

  //
  // The DMA channels are organized in pairs on this controller and the
  // transmit channels are 1, 3, and 5 while receive are 0, 2, and 4.
  //
  if (ui32Config & USB_DMA_EP_RX)
  {
    ui32Channel = 0;
  }
  else
  {
    ui32Channel = 1;
  }

  //
  // Search for an available DMA channel to use.
  //
  for (; ui32Channel < USB_MAX_DMA_CHANNELS_0; ui32Channel += 2)
  {
    //
    // If the current endpoint value is zero then this channel is
    // available.
    //
    if ((psUSBDMAInst->pui8Endpoint[ui32Channel] == 0) && (ald_dma_channel_avaiable(ui32Channel) == 1))
    {
      //
      // Save the endpoint for this DMA channel.
      //
      psUSBDMAInst->pui8Endpoint[ui32Channel] = ui8Endpoint;

      //
      // Save the maximum packet size for the endpoint.
      //
      psUSBDMAInst->pui32MaxPacketSize[ui32Channel] = ui32MaxPacketSize;

      //
      // Set the channel configuration based on the direction.
      //
      if (ui32Config & USB_DMA_EP_RX)
      {
#ifdef ES32F0271
        psUSBDMAInst->pui32Config[ui32Channel].perh             = DMA1;
        psUSBDMAInst->pui32Config[ui32Channel].config.channel   = ui32Channel;
        psUSBDMAInst->pui32Config[ui32Channel].config.cyclemode = ALD_DMA_CYCLEMODE_DISABLE;
        psUSBDMAInst->pui32Config[ui32Channel].config.dirmode   = ALD_DMA_DIRMODE_DISABLE;
        psUSBDMAInst->pui32Config[ui32Channel].config.interrupt = ENABLE;
        psUSBDMAInst->pui32Config[ui32Channel].config.mode      = ALD_DMA_MODE_MEM_TO_MEM;
        psUSBDMAInst->pui32Config[ui32Channel].config.priority  = ALD_DMA_PRIORITY_LEVEL1;
        psUSBDMAInst->pui32Config[ui32Channel].config.signal    = ALD_DMA_SIGNAL_NONE;
        psUSBDMAInst->pui32Config[ui32Channel].config.srcdata_width = ALD_DMA_SRCDATA_SIZE_BYTE;
        psUSBDMAInst->pui32Config[ui32Channel].config.src_inc   = ALD_DMA_DATA_SINC_DISABLE;
        psUSBDMAInst->pui32Config[ui32Channel].config.dst_inc   = ALD_DMA_DATA_DINC_ENABLE;
        psUSBDMAInst->pui32Config[ui32Channel].cplt_arg         = NULL;
        psUSBDMAInst->pui32Config[ui32Channel].cplt_cbk         = dma_out_usb_int_handler;
        psUSBDMAInst->pui32Config[ui32Channel].err_arg          = NULL;
        psUSBDMAInst->pui32Config[ui32Channel].err_cbk          = NULL;
#else
#endif

        //
        // If in device mode and Isochronous.
        //
        if (((ui32Config & USB_DMA_EP_HOST) == 0) &&
            ((ui32Config & USB_DMA_EP_TYPE_M) == USB_DMA_EP_TYPE_ISOC))
        {
          //
          // USB_EP_AUTO_REQUEST is required for device
          // Isochronous endpoints.
          //
          psUSBDMAInst->pui32EPDMAMode0[ui32Channel] =
            USB_EP_DMA_MODE_0 |
            USB_EP_AUTO_REQUEST |
            USB_EP_HOST_IN;
        }
        else
        {
          psUSBDMAInst->pui32EPDMAMode0[ui32Channel] =
            USB_EP_DMA_MODE_0 |
            USB_EP_AUTO_CLEAR |
            USB_EP_HOST_IN;
        }

        //
        // Do not set auto request in device mode unless it is an
        // isochronous endpoint.
        //
        if (((ui32Config & USB_DMA_EP_HOST) == 0) &&
            ((ui32Config & USB_DMA_EP_TYPE_M) != USB_DMA_EP_TYPE_ISOC))
        {
          psUSBDMAInst->pui32EPDMAMode1[ui32Channel] =
            USB_EP_DMA_MODE_1 |
            USB_EP_HOST_IN |
            USB_EP_AUTO_CLEAR;
        }
        else
        {
          psUSBDMAInst->pui32EPDMAMode1[ui32Channel] =
            USB_EP_DMA_MODE_1 |
            USB_EP_HOST_IN |
            USB_EP_AUTO_REQUEST |
            USB_EP_AUTO_CLEAR;
        }
      }
      else
      {
#ifdef ES32F0271
        psUSBDMAInst->pui32Config[ui32Channel].perh             = DMA1;
        psUSBDMAInst->pui32Config[ui32Channel].config.channel   = ui32Channel;
        psUSBDMAInst->pui32Config[ui32Channel].config.cyclemode = ALD_DMA_CYCLEMODE_DISABLE;
        psUSBDMAInst->pui32Config[ui32Channel].config.dirmode   = ALD_DMA_DIRMODE_DISABLE;
        psUSBDMAInst->pui32Config[ui32Channel].config.interrupt = ENABLE;
        psUSBDMAInst->pui32Config[ui32Channel].config.mode      = ALD_DMA_MODE_MEM_TO_MEM;
        psUSBDMAInst->pui32Config[ui32Channel].config.priority  = ALD_DMA_PRIORITY_LEVEL1;
        psUSBDMAInst->pui32Config[ui32Channel].config.signal    = ALD_DMA_SIGNAL_NONE;
        psUSBDMAInst->pui32Config[ui32Channel].config.dstdata_width = ALD_DMA_DSTDATA_SIZE_BYTE;
        psUSBDMAInst->pui32Config[ui32Channel].config.src_inc   = ALD_DMA_DATA_SINC_ENABLE;
        psUSBDMAInst->pui32Config[ui32Channel].config.dst_inc   = ALD_DMA_DATA_DINC_DISABLE;
        psUSBDMAInst->pui32Config[ui32Channel].cplt_arg         = NULL;
        psUSBDMAInst->pui32Config[ui32Channel].cplt_cbk         = dma_out_usb_int_handler;
        psUSBDMAInst->pui32Config[ui32Channel].err_arg          = NULL;
        psUSBDMAInst->pui32Config[ui32Channel].err_cbk          = NULL;

#else
#endif

        psUSBDMAInst->pui32EPDMAMode0[ui32Channel] = USB_EP_DMA_MODE_0 |
            USB_EP_HOST_OUT;
        psUSBDMAInst->pui32EPDMAMode1[ui32Channel] = USB_EP_DMA_MODE_1 |
            USB_EP_HOST_OUT |
            USB_EP_AUTO_SET;
      }

      //
      // Map the DMA channel to the given endpoint.
      //
      md_usb_dma_endpoint_channel_set(ui8Endpoint, ui32Channel);

      if (ui32Config & USB_DMA_EP_RX)
      {
        md_usb_dma_endpoint_disable(ui8Endpoint, USB_EP_DEV_OUT);
      }
      else
      {
        md_usb_dma_endpoint_disable(ui8Endpoint, USB_EP_DEV_IN);
      }

      //
      // Return DMA channel.
      //
      return (ui32Channel);
    }
  }

  return (0);
}

/**
  * /@brief USBLibDMAChannelRelease() for USB controllers that use DMA for DMA.
  */
static void
dmausb_channel_release(tUSBDMAInstance *psUSBDMAInst, uint8_t ui32Channel)
{
  ASSERT(ui32Channel < USB_MAX_DMA_CHANNELS_0);

  //
  // Clear out the state for this endpoint.
  //
  psUSBDMAInst->pui8Endpoint[ui32Channel] = 0;
  memset(&g_psUSBDMAInst[0].pui32Config[ui32Channel], 0x00, sizeof(ald_dma_handle_t));
  psUSBDMAInst->ui32Pending &= ~(1 << (ui32Channel));
  psUSBDMAInst->ui32Complete &= ~(1 << (ui32Channel));
  psUSBDMAInst->pui32MaxPacketSize[ui32Channel] = 0;
}

/**
  * /@brief USBLibDMAUnitSizeSet() for USB controllers that use DMA for DMA.
  */
static void
dmausb_unit_size_set(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel,
                     uint32_t ui32BitSize)
{
#ifdef ES32F0271

  //
  // If source increment is none this is an RX transfer.
  //
  if (psUSBDMAInst->pui32Config[ui32Channel].config.src_inc == ALD_DMA_DATA_SINC_DISABLE)
  {
    //
    // Set the data width and increment.
    //
    if (ui32BitSize == 8)
    {
      psUSBDMAInst->pui32Config[ui32Channel].config.dstdata_width = ALD_DMA_DSTDATA_SIZE_BYTE;
    }
    else if (ui32BitSize == 16)
    {
      psUSBDMAInst->pui32Config[ui32Channel].config.dstdata_width = ALD_DMA_DSTDATA_SIZE_HALFWORD;
    }
    else if (ui32BitSize == 32)
    {
      psUSBDMAInst->pui32Config[ui32Channel].config.dstdata_width = ALD_DMA_DSTDATA_SIZE_WORD;
    }
    else
    {
      //
      // The default data width.
      //
      psUSBDMAInst->pui32Config[ui32Channel].config.dstdata_width = ALD_DMA_DSTDATA_SIZE_BYTE;
    }

    //
    // Config source datawidth.
    //
    psUSBDMAInst->pui32Config[ui32Channel].config.srcdata_width = ALD_DMA_SRCDATA_SIZE_BYTE;
  }
  else
  {
    //
    // Set the data width and increment.
    //
    if (ui32BitSize == 8)
    {
      psUSBDMAInst->pui32Config[ui32Channel].config.srcdata_width = ALD_DMA_SRCDATA_SIZE_BYTE;
    }
    else if (ui32BitSize == 16)
    {
      psUSBDMAInst->pui32Config[ui32Channel].config.srcdata_width = ALD_DMA_SRCDATA_SIZE_HALFWORD;
    }
    else if (ui32BitSize == 32)
    {
      psUSBDMAInst->pui32Config[ui32Channel].config.srcdata_width = ALD_DMA_SRCDATA_SIZE_WORD;
    }
    else
    {
      //
      // The default data width.
      //
      psUSBDMAInst->pui32Config[ui32Channel].config.srcdata_width = ALD_DMA_SRCDATA_SIZE_BYTE;
    }

    psUSBDMAInst->pui32Config[ui32Channel].config.dstdata_width = ALD_DMA_DSTDATA_SIZE_BYTE;
  }

#else
  //
  // Other DMA controller.
  //
#endif
}

/**
  * /@brief USBLibDMAArbSizeSet() for USB controllers.
  */
static void
dmausb_arb_size_set(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel,
                    uint32_t ui32ArbSize)
{
#ifdef ES32F0271

  //
  // Set the data dbusel.
  //
  if (ui32ArbSize == 4)
  {
    //
    // 4 times transfer.
    //
    psUSBDMAInst->pui32Config[ui32Channel].config.dst_dbusel = ALD_DMA_DSTDBUSEL_INCR4;
    psUSBDMAInst->pui32Config[ui32Channel].config.src_dbusel = ALD_DMA_SRCDBUSEL_INCR4;
  }
  else if (ui32ArbSize == 8)
  {
    //
    // 8 times transfer.
    //
    psUSBDMAInst->pui32Config[ui32Channel].config.dst_dbusel = ALD_DMA_DSTDBUSEL_INCR8;
    psUSBDMAInst->pui32Config[ui32Channel].config.src_dbusel = ALD_DMA_SRCDBUSEL_INCR8;
  }
  else if (ui32ArbSize == 16)
  {
    //
    // 16 times transfer.
    //
    psUSBDMAInst->pui32Config[ui32Channel].config.dst_dbusel = ALD_DMA_DSTDBUSEL_INCR16;
    psUSBDMAInst->pui32Config[ui32Channel].config.src_dbusel = ALD_DMA_SRCDBUSEL_INCR16;
  }
  else
  {
    //
    // The default dbusel mode.
    //
    psUSBDMAInst->pui32Config[ui32Channel].config.dst_dbusel = ALD_DMA_DSTDBUSEL_SINGLE;
    psUSBDMAInst->pui32Config[ui32Channel].config.src_dbusel = ALD_DMA_SRCDBUSEL_SINGLE;
  }

#else
  //
  // Other.
  //
#endif
}

/**
  * @brief This function is used to return the current DMA pointer for a given
  *        DMA channel.
  *
  *        This function returns the address that is in use by the DMA channel passed
  *        in via the \e ui32Channel parameter.  This is not the real-time pointer,
  *        but the starting address of the DMA transfer for this DMA channel.
  *
  * @param psUSBDMAInst is a generic instance pointer that can be used to
  *        distinguish between different hardware instances.
  * @param ui32Channel is the DMA channel number for this request.
  * @retval The current DMA address for the given DMA channel.
  */
void *
usb_lib_dma_addr_get(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel)
{
  return (psUSBDMAInst->ppui32Data[ui32Channel]);
}

/**
  * @brief This function is used to return the current DMA transfer size for a given
  *        DMA channel.
  *
  *        This function returns the DMA transfer size that is in use by the DMA
  *        channel passed in via the \e ui32Channel parameter.
  *
  * @param psUSBDMAInst is a generic instance pointer that can be used to
  *        distinguish between different hardware instances.
  * @param ui32Channel is the DMA channel number for this request.
  * @retval The current DMA transfer size for the given DMA channel.
  */
uint32_t
usb_lib_dma_size_get(tUSBDMAInstance *psUSBDMAInst, uint32_t ui32Channel)
{
  return (psUSBDMAInst->pui32Count[ui32Channel]);
}

/**
  * @brief This function is used to set the DMA callback function.
  * @param CallBack function pointer.
  * @retval None.
  */
void
usb_lib_dma_call_back_set(void (*CallBack)(void))
{
  DMACallback = CallBack;
}

/**
  * @brief This function is used to initialize the DMA interface for a USB instance.
  *
  *        This function performs any initialization and configuration of the DMA
  *        portions of the USB controller.  This function returns a pointer that
  *        is used with the remaining USBLibDMA APIs or the function returns zero
  *        if the requested controller cannot support DMA.  If this function is called
  *        when already initialized it will not reinitialize the DMA controller and
  *        will instead return the previously initialized DMA instance.
  *
  * @param ui32Index is the index of the USB controller for this instance.
  * @retval A pointer to use with USBLibDMA APIs.
  */
tUSBDMAInstance *
usb_lib_dma_init(uint32_t ui32Index)
{
  uint32_t ui32Channel;

  //
  // Initialize the function pointers.
  //
  g_psUSBDMAInst[0].pfnArbSizeSet            = dmausb_arb_size_set;
  g_psUSBDMAInst[0].pfnChannelAllocate       = dmausb_channel_allocate;
  g_psUSBDMAInst[0].pfnChannelDisable        = dmausb_channel_disable;
  g_psUSBDMAInst[0].pfnChannelEnable         = dmausb_channel_enable;
  g_psUSBDMAInst[0].pfnChannelIntEnable      = dmausb_channel_int_enable;
  g_psUSBDMAInst[0].pfnChannelIntDisable     = dmausb_channel_int_disable;
  g_psUSBDMAInst[0].pfnChannelRelease        = dmausb_channel_release;
  g_psUSBDMAInst[0].pfnChannelStatus         = dmausb_channel_status;
  g_psUSBDMAInst[0].pfnIntHandler            = dmausb_int_handler;
  g_psUSBDMAInst[0].pfnIntStatus             = dmausb_int_status;
  g_psUSBDMAInst[0].pfnIntStatusClear        = dmausb_int_statusClear;
  g_psUSBDMAInst[0].pfnTransfer              = dmausb_transfer;
  g_psUSBDMAInst[0].pfnUnitSizeSet           = dmausb_unit_size_set;

  //
  // Clear out the endpoint and the current configuration.
  //
  for (ui32Channel = 0; ui32Channel < USB_MAX_DMA_CHANNELS; ui32Channel++)
  {
    g_psUSBDMAInst[0].pui8Endpoint[ui32Channel] = 0;
    memset(g_psUSBDMAInst[0].pui32Config, 0x00, sizeof(g_psUSBDMAInst[0].pui32Config));
    g_psUSBDMAInst[0].ui32Pending = 0;
    g_psUSBDMAInst[0].ui32Complete = 0;
  }

  return (&g_psUSBDMAInst[0]);
}

/**
  * @} usblib_dma_api
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics **** END OF FILE ****/
