/**
  **************************************************************************************
  * @file    usbhhid.c
  * @brief   This file contains the host HID driver.
  * @data    11/9/2018
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. ALL rights reserved.
  *
  **************************************************************************************
  */

#include <stdbool.h>
#include <stdint.h>
#include "usblib/drivers/usb_lowlayer_api.h"
#include "usblib/drivers/debug.h"
#include "usblib/drivers/type.h"
#include "usblib/usblib.h"
#include "usblib/usblibpriv.h"
#include "usblib/usbhid.h"
#include "usblib/host/usbhost.h"
#include "usblib/host/usbhostpriv.h"
#include "usblib/host/usbhhid.h"

static void *hid_driver_open(tUSBHostDevice *psDevice);
static void hid_driver_close(void *pvInstance);

/** @addtogroup usblib_host_class
  * @{
  */

//*****************************************************************************
//
// If the user has not explicitly stated the maximum number of HID devices to
// support, we assume that we need to support up to the maximum number of USB
// devices that the build is configured for.
//
//*****************************************************************************
#ifndef MAX_HID_DEVICES
  #define MAX_HID_DEVICES MAX_USB_DEVICES
#endif

//*****************************************************************************
//
// This is the structure that holds all of the data for a given instance of
// a HID device.
//
//*****************************************************************************
struct tHIDInstance
{
  //
  // Save the device instance.
  //
  tUSBHostDevice *psDevice;

  //
  // Used to save the callback.
  //
  tUSBCallback pfnCallback;

  //
  // Callback data provided by caller.
  //
  void *pvCBData;

  //
  // Used to remember what type of device was registered.
  //
  tHIDSubClassProtocol iDeviceType;

  //
  // Interrupt IN pipe.
  //
  uint32_t ui32IntInPipe;
};

//*****************************************************************************
//
// The instance data storage for attached hid devices.
//
//*****************************************************************************
tHIDInstance g_psHIDDevice[MAX_HID_DEVICES];

//*****************************************************************************
//
//! This constant global structure defines the HID Class Driver that is
//! provided with the USB library.
//
//*****************************************************************************
const tUSBHostClassDriver g_sUSBHIDClassDriver =
{
  USB_CLASS_HID,
  hid_driver_open,
  hid_driver_close,
  0
};

/**
  * @brief  This function is used to open an instance of a HID device.
  *         This function creates an instance of an specific type of HID device.  The
  *         \e iDeviceType parameter is one subclass/protocol values of the types
  *         specified in enumerated types tHIDSubClassProtocol.  Only devices that
  *         enumerate with this type will be called back via the \e pfnCallback
  *         function.  The \e pfnCallback parameter is the callback function for any
  *         events that occur for this device type.  The \e pfnCallback function must
  *         point to a valid function of type \e tUSBCallback for this call to complete
  *         successfully.  To release this device instance the caller of usbhhid_open()
  *         should call usbhhid_close() and pass in the value returned from the
  *         usbhhid_open() call.
  *
  * @param  iDeviceType is the type of device that should be loaded for this
  *         instance of the HID device.
  * @param  pfnCallback is the function that will be called whenever changes
  *         are detected for this device.
  * @param  pvCBData is the data that will be returned in when the
  *         \e pfnCallback function is called.
  * @retval This function returns and instance value that should be used with
  *         any other APIs that require an instance value.  If a value of 0 is returned
  *         then the device instance could not be created.
  */
tHIDInstance *
usbhhid_open(tHIDSubClassProtocol iDeviceType, tUSBCallback pfnCallback,
             void *pvCBData)
{
  uint32_t ui32Loop;

  //
  // Find a free device instance structure.
  //
  for (ui32Loop = 0; ui32Loop < MAX_HID_DEVICES; ui32Loop++)
  {
    if (g_psHIDDevice[ui32Loop].iDeviceType == eUSBHHIDClassNone)
    {
      //
      // Save the instance data for this device.
      //
      g_psHIDDevice[ui32Loop].pfnCallback = pfnCallback;
      g_psHIDDevice[ui32Loop].iDeviceType = iDeviceType;
      g_psHIDDevice[ui32Loop].pvCBData = pvCBData;

      //
      // Return the device instance pointer.
      //
      return (tHIDInstance *)(&g_psHIDDevice[ui32Loop]);
    }
  }

  //
  // If we get here, there are no space device slots so return NULL to
  // indicate a problem.
  //
  return (0);
}

/**
  * @brief  This function is used to release an instance of a HID device.
  *         This function releases an instance of a HID device that was created by a
  *         call to usbhhid_open().  This call is required to allow other HID devices
  *         to be enumerated after another HID device has been disconnected.  The
  *         \e psHIDInstance parameter should hold the value that was returned from
  *         the previous call to usbhhid_open().
  *
  * @param  psHIDInstance is the instance value for a HID device to release.
  * @retval None.
  */
void
usbhhid_close(tHIDInstance *psHIDInstance)
{
  //
  // Disable any more notifications from the HID layer.
  //
  psHIDInstance->pfnCallback = 0;

  //
  // Mark this device slot as free.
  //
  psHIDInstance->iDeviceType = eUSBHHIDClassNone;
}

//*****************************************************************************
//
// This function handles callbacks for the interrupt IN endpoint.
//
//*****************************************************************************
static void
hid_int_in_callback(uint32_t ui32Pipe, uint32_t ui32Event)
{
  int32_t i32Dev;

  switch (ui32Event)
  {
    //
    // Handles a request to schedule a new request on the interrupt IN
    // pipe.
    //
    case USB_EVENT_SCHEDULER:
    {
      usbhcd_pipe_schedule(ui32Pipe, 0, 1);
      break;
    }

    //
    // Called when new data is available on the interrupt IN pipe.
    //
    case USB_EVENT_RX_AVAILABLE:
    {
      //
      // Determine which device this notification is intended for.
      //
      for (i32Dev = 0; i32Dev < MAX_HID_DEVICES; i32Dev++)
      {
        //
        // Does this device own the pipe we have been passed?
        //
        if (g_psHIDDevice[i32Dev].ui32IntInPipe == ui32Pipe)
        {
          //
          // Yes - send the report data to the USB host HID device
          // class driver.
          //
          g_psHIDDevice[i32Dev].pfnCallback(
            g_psHIDDevice[i32Dev].pvCBData,
            USB_EVENT_RX_AVAILABLE, ui32Pipe, 0);
        }
      }

      break;
    }
  }
}

/**
  * @brief  This function is used to open an instance of the HID driver.
  *         This function will attempt to open an instance of the HID driver based on
  *         the information contained in the psDevice structure.  This call can fail if
  *         there are not sufficient resources to open the device.  The function will
  *         return a value that should be passed back into USBHIDClose() when the
  *         driver is no longer needed.
  *
  * @param  psDevice is an instance pointer that needs to be opened.
  * @retval The function will return a pointer to a HID driver instance.
  */
static void *
hid_driver_open(tUSBHostDevice *psDevice)
{
  int32_t i32Idx, i32Dev;
  tEndpointDescriptor *psEndpointDescriptor;
  tInterfaceDescriptor *psInterface;

  //
  // Get the interface descriptor.
  //
  psInterface = usb_desc_get_interface(psDevice->psConfigDescriptor, 0, 0);

  //
  // Search the currently open instances for one that supports the protocol
  // of this device.
  //
  for (i32Dev = 0; i32Dev < MAX_HID_DEVICES; i32Dev++)
  {
    if (g_psHIDDevice[i32Dev].iDeviceType ==
        psInterface->bInterfaceProtocol)
    {
      //
      // Save the device pointer.
      //
      g_psHIDDevice[i32Dev].psDevice = psDevice;

      for (i32Idx = 0; i32Idx < 3; i32Idx++)
      {
        //
        // Get the first endpoint descriptor.
        //
        psEndpointDescriptor = usb_desc_get_interface_endpoint(psInterface,
                               i32Idx,
                               256);

        //
        // If no more endpoints then break out.
        //
        if (psEndpointDescriptor == 0)
        {
          break;
        }

        //
        // Interrupt
        //
        if ((psEndpointDescriptor->bmAttributes & USB_EP_ATTR_TYPE_M) ==
            USB_EP_ATTR_INT)
        {
          //
          // Interrupt IN.
          //
          if (psEndpointDescriptor->bEndpointAddress & USB_EP_DESC_IN)
          {
            g_psHIDDevice[i32Dev].ui32IntInPipe =
              usbhcd_pipe_alloc(0, USBHCD_PIPE_INTR_IN,
                                psDevice, hid_int_in_callback);
            usbhcd_pipe_config(g_psHIDDevice[i32Dev].ui32IntInPipe,
                               psEndpointDescriptor->wMaxPacketSize,
                               psEndpointDescriptor->bInterval,
                               (psEndpointDescriptor->bEndpointAddress &
                                USB_EP_DESC_NUM_M));
          }
        }
      }

      //
      // If there is a callback function call it to inform the application that
      // the device has been enumerated.
      //
      if (g_psHIDDevice[i32Dev].pfnCallback != 0)
      {
        g_psHIDDevice[i32Dev].pfnCallback(
          g_psHIDDevice[i32Dev].pvCBData,
          USB_EVENT_CONNECTED,
          (uint32_t)&g_psHIDDevice[i32Dev], 0);
      }

      //
      // Save the device pointer.
      //
      g_psHIDDevice[i32Dev].psDevice = psDevice;

      return (&g_psHIDDevice[i32Dev]);
    }
  }

  //
  // If we get here, no user has registered an interest in this particular
  // HID device so we return an error.
  //
  return (0);
}

/**
  * @brief  This function is used to release an instance of the HID driver.
  *         This function will free up any resources in use by the HID driver instance
  *         that is passed in.  The \e pvInstance pointer should be a valid value that
  *         was returned from a call to USBHIDOpen().
  *
  * @param  pvInstance is an instance pointer that needs to be released.
  * @retval None.
  */
static void
hid_driver_close(void *pvInstance)
{
  tHIDInstance *psInst;

  //
  // Get our instance pointer.
  //
  psInst = (tHIDInstance *)pvInstance;

  //
  // Reset the device pointer.
  //
  psInst->psDevice = 0;

  //
  // Free the Interrupt IN pipe.
  //
  if (psInst->ui32IntInPipe != 0)
  {
    usbhcd_pipe_free(psInst->ui32IntInPipe);
  }

  //
  // If the callback exists, call it with a DISCONNECTED event.
  //
  if (psInst->pfnCallback != 0)
  {
    psInst->pfnCallback(psInst->pvCBData, USB_EVENT_DISCONNECTED,
                        (uint32_t)pvInstance, 0);
  }
}

/**
  * @brief  This function is used to set the idle timeout for a HID device.
  *         This function will send the Set Idle command to a HID device to set the
  *         idle timeout for a given report.  The length of the timeout is specified
  *         by the \e ui8Duration parameter and the report the timeout for is in the
  *         \e ui8ReportID value.
  *
  * @param  psHIDInstance is the value that was returned from the call to usbhhid_open().
  * @param  ui8Duration is the duration of the timeout in milliseconds.
  * @param  ui8ReportID is the report identifier to set the timeout on.
  * @retval Always returns 0.
  */
uint32_t
usbhhid_set_idle(tHIDInstance *psHIDInstance, uint8_t ui8Duration,
                 uint8_t ui8ReportID)
{
  tUSBRequest sSetupPacket;

  //
  // This is a Class specific interface OUT request.
  //
  sSetupPacket.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS |
                               USB_RTYPE_INTERFACE;

  //
  // Request a Device Descriptor.
  //
  sSetupPacket.bRequest = USBREQ_SET_IDLE;
  sSetupPacket.wValue = (ui8Duration << 8) | ui8ReportID;

  //
  // Set this on interface 1.
  //
  sSetupPacket.wIndex = 0;

  //
  // This is always 0 for this request.
  //
  sSetupPacket.wLength = 0;

  //
  // Put the setup packet in the buffer.
  //
  return (usbhcd_control_transfer(0, &sSetupPacket, psHIDInstance->psDevice,
                                  0, 0, MAX_PACKET_SIZE_EP0));
}

/**
  * @brief  This function can be used to retrieve the report descriptor for a given
  *         device instance.
  *         This function is used to return a report descriptor from a HID device
  *         instance so that it can determine how to interpret reports that are
  *         returned from the device indicated by the \e psHIDInstance parameter.
  *         This call is blocking and will return the number of bytes read into the
  *         \e pui8Buffer.
  *
  * @param  psHIDInstance is the value that was returned from the call to usbhhid_open().
  * @param  pui8Buffer is the memory buffer to use to store the report descriptor.
  * @param  ui32Size is the size in bytes of the buffer pointed to by \e pui8Buffer.
  * @retval Returns the number of bytes read into the \e pui8Buffer.
  */
uint32_t
usbhhid_get_report_descriptor(tHIDInstance *psHIDInstance, uint8_t *pui8Buffer,
                              uint32_t ui32Size)
{
  tUSBRequest sSetupPacket;
  uint32_t ui32Bytes;

  //
  // This is a Standard Device IN request.
  //
  sSetupPacket.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_STANDARD |
                               USB_RTYPE_INTERFACE;

  //
  // Request a Report Descriptor.
  //
  sSetupPacket.bRequest = USBREQ_GET_DESCRIPTOR;
  sSetupPacket.wValue = USB_HID_DTYPE_REPORT << 8;

  //
  // Index is always 0 for device requests.
  //
  sSetupPacket.wIndex = 0;

  //
  // All devices must have at least an 8 byte max packet size so just ask
  // for 8 bytes to start with.
  //
  sSetupPacket.wLength = ui32Size;

  //
  // Now get the full descriptor now that the actual maximum packet size
  // is known.
  //
  ui32Bytes = usbhcd_control_transfer(0, &sSetupPacket,
                                      psHIDInstance->psDevice, pui8Buffer, ui32Size,
                                      psHIDInstance->psDevice->sDeviceDescriptor.bMaxPacketSize0);

  return (ui32Bytes);
}

/**
  * @brief  This function is used to set or clear the boot protocol state of a device.
  *         A USB host device can use this function to set the protocol for a connected
  *         HID device.  This is commonly used to set keyboards and mice into their
  *         simplified boot protocol modes to fix the report structure to a know
  *         state.
  *
  * @param  psHIDInstance is the value that was returned from the call to usbhhid_open().
  * @param  ui32BootProtocol is either zero or non-zero to indicate which
  *         protocol to use for the device.
  * @retval This function returns 0.
  */
uint32_t
usbhhid_set_protocol(tHIDInstance *psHIDInstance, uint32_t ui32BootProtocol)
{
  tUSBRequest sSetupPacket;

  //
  // This is a Standard Device IN request.
  //
  sSetupPacket.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS |
                               USB_RTYPE_INTERFACE;

  //
  // Request a Report Descriptor.
  //
  sSetupPacket.bRequest = USBREQ_SET_PROTOCOL;

  if (ui32BootProtocol)
  {
    //
    // Boot Protocol.
    //
    sSetupPacket.wValue = 0;
  }
  else
  {
    //
    // Report Protocol.
    //
    sSetupPacket.wValue = 1;
  }

  //
  // Index is always 0 for device requests.
  //
  sSetupPacket.wIndex = 0;

  //
  // Always 0.
  //
  sSetupPacket.wLength = 0;

  //
  // Now get the full descriptor now that the actual maximum packet size
  // is known.
  //
  usbhcd_control_transfer(0, &sSetupPacket, psHIDInstance->psDevice, 0, 0,
                          psHIDInstance->psDevice->sDeviceDescriptor.bMaxPacketSize0);

  return (0);
}

/**
  * @brief  This function is used to retrieve a report from a HID device.
  *         This function is used to retrieve a report from a USB pipe.  It is usually
  *         called when the USB HID layer has detected a new data available in a USB
  *         pipe.  The USB HID host device code will receive a
  *         \b USB_EVENT_RX_AVAILABLE event when data is available, allowing the
  *         callback function to retrieve the data.
  *
  * @param  psHIDInstance is the value that was returned from the call to usbhhid_open().
  * @param  ui32Interface is the interface to retrieve the report from.
  * @param  pui8Data is the memory buffer to use to store the report.
  * @param  ui32Size is the size in bytes of the buffer pointed to by \e pui8Buffer.
  * @retval Returns the number of bytes read from report.
  */
uint32_t
usbhhid_get_report(tHIDInstance *psHIDInstance, uint32_t ui32Interface,
                   uint8_t *pui8Data, uint32_t ui32Size)
{
  //
  // Read the Data out.
  //
  ui32Size = usbhcd_pipe_read_non_blocking(psHIDInstance->ui32IntInPipe,
             pui8Data, ui32Size);

  //
  // Return the number of bytes read from the interrupt in pipe.
  //
  return (ui32Size);
}

/**
  * @brief  This function is used to send a report to a HID device.
  *         This function is used to send a report to a USB HID device.  It can be
  *         only be called from outside the callback context as this function will not
  *         return from the call until the data has been sent successfully.
  *
  * @param  psHIDInstance is the value that was returned from the call to usbhhid_open().
  * @param  ui32Interface is the interface to retrieve the report from.
  * @param  pui8Data is the memory buffer to use to store the report.
  * @param  ui32Size is the size in bytes of the buffer pointed to by \e pui8Buffer.
  * @retval Returns the number of bytes sent to the device.
  */
uint32_t
usbhhid_set_report(tHIDInstance *psHIDInstance, uint32_t ui32Interface,
                   uint8_t *pui8Data, uint32_t ui32Size)
{
  tUSBRequest sSetupPacket;

  //
  // This is a class specific OUT request.
  //
  sSetupPacket.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS |
                               USB_RTYPE_INTERFACE;

  //
  // Request a Report Descriptor.
  //
  sSetupPacket.bRequest = USBREQ_SET_REPORT;
  sSetupPacket.wValue = USB_HID_REPORT_OUTPUT << 8;

  //
  // Index is always 0 for device requests.
  //
  sSetupPacket.wIndex = (uint16_t)ui32Interface;

  //
  // Always 0.
  //
  sSetupPacket.wLength = ui32Size;

  //
  // Now get the full descriptor now that the actual maximum packet size
  // is known.
  //
  usbhcd_control_transfer(0, &sSetupPacket, psHIDInstance->psDevice,
                          pui8Data, ui32Size,
                          psHIDInstance->psDevice->sDeviceDescriptor.bMaxPacketSize0);

  return (ui32Size);
}

/**
  * @brief  This function forwards an LPM request for a device to enter L1 sleep state.
  *         This function forwards a request from a HID device class to the host
  *         controller to request that a device enter the LPM L1 sleep state.  The
  *         caller must check the return value to see if the request can be
  *         attempted at this time.  If another LPM transaction is busy on this or
  *         another device, then this function returns \b USBHCD_LPM_PENDING.  If
  *         the LPM request was scheduled to be sent the function returns
  *         \b USBHCD_LPM_AVAIL.  The caller should check the usbhcdlpm_status()
  *         function to determine if the request completed successfully or if there
  *         was an error.
  *
  * @param  psHIDInstance is the value that was returned from the call to usbhhid_open().
  * @retval This function returns the following values:
  *         - USBHCD_LPM_AVAIL - The transition to L1 state is scheduled to be sent.
  *         - USBHCD_LPM_PENDING - There is already an LPM request pending.
  */
uint32_t
usbhhidlpm_sleep(tHIDInstance *psHIDInstance)
{
  //
  // Forward the request to the control endpoint of the device.
  //
  return (usbhcdlpm_sleep(psHIDInstance->psDevice));
}

/**
  * @brief  This function forwards an LPM request for a device to enter L1 sleep state.
  *         This function returns the current status of LPM requests for a given
  *         device.  This is called to determine if a previous request completed
  *         successfully or if there was an error.
  *
  * @param  psHIDInstance is the value that was returned from the call to usbhhid_open().
  * @retval This function returns the following values:
  *         - \b USBHCD_LPM_AVAIL - There are no pending LPM requests on this specific
  *           device or the last request completed successfully.
  *         - \b USBHCD_LPM_ERROR - The last LPM request for this device did not
  *           complete successfully.
  *         - \b USBHCD_LPM_PENDING - The last LPM request has not completed.
  */
uint32_t
usbhhidlpm_status(tHIDInstance *psHIDInstance)
{
  //
  // Call the host controller function to get the current LPM status.
  //
  return (usbhcdlpm_status(psHIDInstance->psDevice));
}

/**
  * @} usblib_host_class
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics Co., Ltd. *** END OF FILE ****/
