/**
  **************************************************************************************
  * @file    usbhaudio.h
  * @brief   USB host audio class driver.
  * @data    11/9/2018
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. ALL rights reserved.
  *
  **************************************************************************************
  */

#ifndef __USBHAUDIO_H__
#define __USBHAUDIO_H__

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif

/** @addtogroup usblib_host_class
  * @{
  */

//*****************************************************************************
//
// USB host audio specific events
//
//*****************************************************************************

//*****************************************************************************
//
//! This USB host audio event indicates that the device is connected and
//! ready to send or receive buffers.  The \e pvBuffer and \e ui32Param
//! values are not used in this event.
//
//*****************************************************************************
#define USBH_AUDIO_EVENT_OPEN   (USBH_AUDIO_EVENT_BASE + 0)

//*****************************************************************************
//
//! This USB host audio event indicates that the previously connected device
//! has been disconnected. The \e pvBuffer and \e ui32Param values are not used
//! in this event.
//
//*****************************************************************************
#define USBH_AUDIO_EVENT_CLOSE  (USBH_AUDIO_EVENT_BASE + 1)

//*****************************************************************************
//
// This definition is used with the usb_host_audio_format_get() and
// usb_host_audio_format_set() API's to determine if the audio input is being
// accesses(USBH_AUDIO_FORMAT_IN set) or audio output(USBH_AUDIO_FORMAT clear).
//
//*****************************************************************************
#define USBH_AUDIO_FORMAT_IN    0x00000001
#define USBH_AUDIO_FORMAT_OUT   0x00000000

typedef struct
{
  uint8_t ui8Channels;
  uint8_t ui8Bits;
  uint32_t ui32SampleRate;
}
tUSBAudioFormat;

typedef struct tUSBHostAudioInstance tUSBHostAudioInstance;

//*****************************************************************************
//
// The prototype for the host USB Audio driver callback function.
//
//*****************************************************************************
typedef void (*tUSBHostAudioCallback)(tUSBHostAudioInstance *psAudioInstance,
                                      uint32_t ui32Event,
                                      uint32_t ui32MsgParam,
                                      void *pvMsgData);

//*****************************************************************************
//
// API Function Prototypes
//
//*****************************************************************************
extern tUSBHostAudioInstance *usb_host_audio_open(uint32_t ui32Index,
    tUSBHostAudioCallback pfnCallback);
extern void usb_host_audio_close(tUSBHostAudioInstance *psAudioInstance);
extern int32_t usb_host_audio_play(tUSBHostAudioInstance *psAudioInstance,
                                   void *pvBuffer, uint32_t ui32Size,
                                   tUSBHostAudioCallback pfnCallback);

extern uint32_t usb_host_audio_format_get(tUSBHostAudioInstance *psAudioInstance,
    uint32_t ui32SampleRate,
    uint32_t ui32Bits, uint32_t ui32Channels,
    uint32_t ui32Flags);
extern uint32_t usb_host_audio_format_set(tUSBHostAudioInstance *psAudioInstance,
    uint32_t ui32SampleRate,
    uint32_t ui32Bits, uint32_t ui32Channels,
    uint32_t ui32Flags);

extern int32_t usb_host_audio_record(tUSBHostAudioInstance *psAudioInstance,
                                     void *pvBuffer, uint32_t ui32Size,
                                     tUSBHostAudioCallback pfnAudioCallback);

extern uint32_t usb_host_audio_volume_get(tUSBHostAudioInstance *psAudioInstance,
    uint32_t ui32Interface,
    uint32_t ui32Channel);

extern void usb_host_audio_volume_set(tUSBHostAudioInstance *psAudioInstance,
                                      uint32_t ui32Interface, uint32_t ui32Channel,
                                      uint32_t ui32Value);

extern uint32_t usb_host_audio_volume_max_get(tUSBHostAudioInstance *psAudioInstance,
    uint32_t ui32Interface,
    uint32_t ui32Channel);

extern uint32_t usb_host_audio_volume_min_get(tUSBHostAudioInstance *psAudioInstance,
    uint32_t ui32Interface,
    uint32_t ui32Channel);

extern uint32_t usb_host_audio_volume_res_get(tUSBHostAudioInstance *psAudioInstance,
    uint32_t ui32Interface,
    uint32_t ui32Channel);
extern uint32_t usb_host_audio_lpm_sleep(tUSBHostAudioInstance *psAudioInstance);
extern uint32_t usb_host_audio_lpm_status(tUSBHostAudioInstance *psAudioInstance);

/**
  * @} usblib_host_class
  */

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics Co., Ltd. *** END OF FILE ****/
