/**
  **************************************************************************************
  * @file    usbdcomp.c
  * @brief   USB composite device class driver.
  * @data    11/9/2018
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. ALL rights reserved.
  *
  **************************************************************************************
  */

#include <stdbool.h>
#include <stdint.h>
#include "usblib/drivers/usb_lowlayer_api.h"
#include "usblib/drivers/debug.h"
#include "usblib/drivers/type.h"
#include "usblib/usblib.h"
#include "usblib/usblibpriv.h"
#include "usblib/usb-ids.h"
#include "usblib/usbcdc.h"
#include "usblib/device/usbdevice.h"
#include "usblib/device/usbdcdc.h"
#include "usblib/device/usbdcomp.h"

/** @addtogroup composite_device_class_api
  * @{
  */

//#if (USBLIB_DEV_COMP_DRIVER_SUPPORTED == 1)&&(USBLIB_DEV_DRIVER_SUPPORTED == 1)
//#if (USBLIB_SOF_TICK_TIMER_SUPPORTED == 0)
//    #error USBLIB_SOF_TICK_TIMER_SUPPORTED should be set to 1
//#endif

/**
  * @brief Device Descriptor.  This is stored in RAM to allow several fields to be
  * changed at runtime based on the client's requirements.
  */
static uint8_t comp_device_descriptor[] =
{
    18,                     // Size of this structure.
    USB_DTYPE_DEVICE,       // Type of this structure.
    USBShort(0x110),        // USB version 1.1 (if we say 2.0, hosts assume
                            // high-speed - see USB 2.0 spec 9.2.6.6)
    USB_CLASS_MISC,         // USB Device Class (spec 5.1.1)
    USB_MISC_SUBCLASS_COMMON, // USB Device Sub-class (spec 5.1.1)
    USB_MISC_PROTOCOL_IAD,  // USB Device protocol (spec 5.1.1)
    64,                     // Maximum packet size for default pipe.
    USBShort(0),            // Vendor ID (filled in during usbd_composite_init).
    USBShort(0),            // Product ID (filled in during usbd_composite_init).
    USBShort(0x100),        // Device Version BCD.
    1,                      // Manufacturer string identifier.
    2,                      // Product string identifier.
    3,                      // Product serial number.
    1                       // Number of configurations.
};

/**
  *
  * @brief Composite class device configuration descriptor.
  *
  * It is vital that the configuration descriptor bConfigurationValue field
  * (byte 6) is 1 for the first configuration and increments by 1 for each
  * additional configuration defined here.  This relationship is assumed in the
  * device stack for simplicity even though the USB 2.0 specification imposes
  * no such restriction on the bConfigurationValue values.
  *
  * Note that this structure is deliberately located in RAM since we need to
  * be able to patch some values in it based on client requirements.
  *
  */
static const uint8_t comp_config_descriptor[] =
{
    /* Configuration descriptor header. */
    9,                          // Size of the configuration descriptor.
    USB_DTYPE_CONFIGURATION,    // Type of this descriptor.
    USBShort(0),                // The total size of this full structure.
    0,                          // The number of interfaces in this
                                // configuration, this will be filled by
                                // the class as it discovers all classes
                                // supported.
    1,                          // The unique value for this configuration.
    0,                          // The string identifier that describes this
                                // configuration.
    USB_CONF_ATTR_BUS_PWR,      // .
    250,                        // The maximum power in 2mA increments.
};

/**
  *
  * @brief Byte offsets used to access various fields in our index/interface/endpoint
  * lookup table (tUSBDCompositeDevice.pui32DeviceWorkspace).  This workspace
  * contains one 4 byte entry per device. The LSB is the device index, next byte
  * is the number of the first interface not within this device, next byte is
  * the number of the first IN endpoint not within this device and the final
  * byte is the number of the first OUT endpoint not within this device.  Using
  * this simple table we can reasonably quickly cross-reference index with
  * interface and endpoint numbers.
  *
  */
#define LOOKUP_INDEX_BYTE       0
#define LOOKUP_INTERFACE_BYTE   1
#define LOOKUP_IN_END_BYTE      2
#define LOOKUP_OUT_END_BYTE     3

/* A marker used to indicate an invalid index into the device table. */
#define INVALID_DEVICE_INDEX    0xFFFFFFFF

/* Various internal handlers needed by this class. */
static void handle_disconnect       (void *composite_instance);
static void interface_change        (void *composite_instance,
                                        uint8_t ui8InterfaceNum,
                                        uint8_t ui8AlternateSetting);
static void config_change_handler   (void *composite_instance,
                                        uint32_t ui32Value);
static void data_sent               (void *composite_instance,
                                        uint32_t ui32Info);
static void data_received           (void *composite_instance,
                                        uint32_t ui32Info);
static void handle_endpoints        (void *composite_instance,
                                        uint32_t ui32Status);
static void handle_requests         (void *composite_instance,
                                        tUSBRequest *usb_request);
static void suspend_handler         (void *composite_instance);
static void resume_handler          (void *composite_instance);
static void reset_handler           (void *composite_instance);
static void handle_device           (void *composite_instance,
                                        uint32_t ui32Request,
                                        void *pvRequestData);
static void get_descriptor          (void *composite_instance, 
                                        tUSBRequest *usb_request);

/* Configuration Descriptor. */
tConfigHeader *g_ppCompConfigDescriptors[1];

/* The device information structure for the USB Composite device. */
const tCustomHandlers comp_handlers_table =
{
    /* get_descriptor */
    .pfnGetDescriptor       = get_descriptor,

    /* RequestHandler */
    .pfnRequestHandler      = handle_requests,

    /* InterfaceChange */
    .pfnInterfaceChange     = interface_change,

    /* ConfigChange */
    .pfnConfigChange        = config_change_handler,

    /* DataReceived */
    .pfnDataReceived        = data_received,

    /* DataSent */
    .pfnDataSent            = data_sent,

    /* ResetHandler */
    .pfnResetHandler        = reset_handler,

    /* SuspendHandler */
    .pfnSuspendHandler      = suspend_handler,

    /* ResumeHandler */
    .pfnResumeHandler       = resume_handler,

    /* DisconnectHandler */
    .pfnDisconnectHandler   = handle_disconnect,

    /* EndpointHandler */
    .pfnEndpointHandler     = handle_endpoints,

    /* DeviceHandler */
    .pfnDeviceHandler       = handle_device,
};

/**
  *
  * @brief Use the lookup table from the field pui32DeviceWorkspace in the
  * tUSBDCompositeDevice structure to determine which device to call given a
  * particular composite device interface number.
  *
  * The returned value is the index into psDevice->tCompositeEntry indicating
  * the device which contains this interface or INVALID_DEVICE_INDEX if no
  * device contains the passed interface number.
  *
  */
static uint32_t interface_to_index(tUSBDCompositeDevice *psDevice,
                                                    uint32_t ui32Interface)
{
    uint32_t loop;
    uint32_t lookup;

    /* Check each lookup entry in turn. */
    for (loop = 0; loop < psDevice->ui32NumDevices; loop++)
    {
        /* Get the look up value from the device. */
        lookup = psDevice->psDevices[loop].ui32DeviceWorkspace;
        lookup = (lookup >> (8 * LOOKUP_INTERFACE_BYTE)) & 0xff;

        /*
         * If the desired interface number is lower than the value in the
         * current lookup table entry, we have found the desired device so
         * return its index.
         */
        if (ui32Interface < lookup)
        {
            return (loop);
        }
    }

    /*
     * If we get here, an invalid interface number was passed so return a
     * marker to indicate this.
     */
    return (INVALID_DEVICE_INDEX);
}

/**
  *
  * @brief Use the lookup table from the field pui32DeviceWorkspace in the
  * tUSBDCompositeDevice structure to determine which device to call given a
  * particular composite device endpoint number.
  *
  * The returned value is the index into psDevice->tCompositeEntry indicating
  * the device which contains this endpoint or INVALID_DEVICE_INDEX if no
  * device contains the passed endpoint number.
  *
  */
static uint32_t endpoint_to_index(tUSBDCompositeDevice *psDevice,
                                    uint32_t ui32Endpoint, bool bInEndpoint)
{
    uint32_t loop, ui32EndpointByte, lookup;

    /* Are we considering an IN or OUT endpoint? */
    ui32EndpointByte = bInEndpoint ? LOOKUP_IN_END_BYTE : LOOKUP_OUT_END_BYTE;

    /* Check each lookup entry in turn. */
    for (loop = 0; loop < psDevice->ui32NumDevices; loop++)
    {
        /* Get the look up byte from the device. */
        lookup = psDevice->psDevices[loop].ui32DeviceWorkspace;
        lookup = (lookup >> (ui32EndpointByte * 8)) & 0xff;

        /*
         * If the desired endpoint number is lower than the value in the
         * current lookup table entry, we have found the desired device so
         * return its index.
         */
        if (ui32Endpoint < lookup)
        {
            return (loop);
        }
    }

    /*
     * If we get here, an invalid endpoint number was passed so return a
     * marker to indicate this.
     */
    return (INVALID_DEVICE_INDEX);
}

/**
  *
  * @brief This function will check if any device classes need a get descriptor
  * handler called.
  *
  */
static void get_descriptor(void *composite_instance, tUSBRequest *usb_request)
{
    uint32_t index;
    const tDeviceInfo *dev_info;
    tUSBDCompositeDevice *comp_device;

    /* Create the composite device pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    /*
     * Determine which device this request is intended for.  We have to be
     * careful here to send this to the callback for the correct device
     * depending upon whether it is a request sent to the device, the interface
     * or the endpoint.
     */
    switch (usb_request->bmRequestType & USB_RTYPE_RECIPIENT_M)
    {
        case USB_RTYPE_INTERFACE:
        {
            index = interface_to_index(comp_device, (usb_request->wIndex & 0xFF));
            break;
        }

        case USB_RTYPE_ENDPOINT:
        {
            index = endpoint_to_index(comp_device,
                                  (usb_request->wIndex & 0x0F),
                                  (usb_request->wIndex & 0x80) ? true : false);
            break;
        }

        /*
         * Requests sent to the device or any other recipient can't be
         * handled here since we have no way of telling where they are
         * supposed to be handled.  As a result, we just stall them.
         *
         * If your composite device has some device-specific descriptors,
         * you should add code here to handle them.
         */
        case USB_RTYPE_DEVICE:
        case USB_RTYPE_OTHER:
        default:
        {
            index = INVALID_DEVICE_INDEX;
            break;
        }
    }

    /* Did we find a device class to pass the request to? */
    if (index != INVALID_DEVICE_INDEX)
    {
        /* Get a pointer to the individual device instance. */
        dev_info = comp_device->psDevices[index].psDevInfo;

        /* Does this device have a get_descriptor callback? */
        if (dev_info->psCallbacks->pfnGetDescriptor)
        {
            /*
             * Remember this device index so that we can correctly route any
             * data notification callbacks to it.
             */
            comp_device->sPrivateData.ui32EP0Owner = index;

            /* Call the device to retrieve the descriptor. */
            dev_info->psCallbacks->pfnGetDescriptor(
                    comp_device->psDevices[index].pvInstance, usb_request);
        }
        else
        {
            /**
              * Oops - we can't satisfy the request so stall EP0 to indicate
              * an error.
              */
            usbdcd_stall_ep0(USBBaseToIndex(
                            comp_device->sPrivateData.ui32USBBase));
        }
    }
    else
    {
        /*
         * We are unable to satisfy the descriptor request so stall EP0 to
         * indicate an error.
         */
        usbdcd_stall_ep0(USBBaseToIndex(
                            comp_device->sPrivateData.ui32USBBase));
    }
}

/**
  * @brief This function will check if any device classes need an suspend handler
  * called.
  *
  */
static void suspend_handler(void *composite_instance)
{
    uint32_t index;
    tUSBDCompositeDevice *comp_device;
    const tDeviceInfo *dev_info;
    void *dev_inst;

    ASSERT(composite_instance != 0);

    /* Create the device instance pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    /* Inform the application that the device has resumed. */
    if (comp_device->pfnCallback)
    {
        comp_device->pfnCallback(composite_instance, USB_EVENT_SUSPEND, 0, 0);
    }

    for (index = 0; index < comp_device->ui32NumDevices; index++)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;
        dev_inst = comp_device->psDevices[index].pvInstance;

        if (dev_info->psCallbacks->pfnSuspendHandler)
        {
            dev_info->psCallbacks->pfnSuspendHandler(dev_inst);
        }
    }
}

/**
  * @brief This function will check if any device classes need an resume handler
  * called.
  *
  */
static void resume_handler(void *composite_instance)
{
    uint32_t index;
    tUSBDCompositeDevice *comp_device;
    const tDeviceInfo *dev_info;
    void *dev_inst;

    ASSERT(composite_instance != 0);

    /* Create the device instance pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    /* Inform the application that the device has resumed. */
    if (comp_device->pfnCallback)
    {
        comp_device->pfnCallback(composite_instance, USB_EVENT_RESUME, 0, 0);
    }

    for (index = 0; index < comp_device->ui32NumDevices; index++)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;
        dev_inst = comp_device->psDevices[index].pvInstance;

        if (dev_info->psCallbacks->pfnResumeHandler)
        {
            dev_info->psCallbacks->pfnResumeHandler(dev_inst);
        }
    }
}

/**
  * @brief This function will check if any device classes need an reset handler
  * called.
  *
  */
static void reset_handler(void *composite_instance)
{
    uint32_t index;
    tUSBDCompositeDevice *comp_device;
    const tDeviceInfo *dev_info;
    void *dev_inst;

    ASSERT(composite_instance != 0);

    /* Create the device instance pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    /* Inform the application that the device has been connected. */
    if (comp_device->pfnCallback)
    {
        comp_device->pfnCallback(composite_instance, USB_EVENT_CONNECTED, 0, 0);
    }

    for (index = 0; index < comp_device->ui32NumDevices; index++)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;
        dev_inst = comp_device->psDevices[index].pvInstance;

        if (dev_info->psCallbacks->pfnResetHandler)
        {
            dev_info->psCallbacks->pfnResetHandler(dev_inst);
        }
    }
}

/**
  * @brief This function is called to handle data being set to the host so that the
  * application callback can be called when the data has been transferred.
  *
  */
static void data_sent(void *composite_instance, uint32_t info)
{
    uint32_t index;
    const tDeviceInfo *dev_info;
    tUSBDCompositeDevice *comp_device;

    // Create the device instance pointer.
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    /*
     * Pass this notification on to the device which last handled a
     * transaction on endpoint 0 (assuming we know who that was).
     */
    index = comp_device->sPrivateData.ui32EP0Owner;

    if (index != INVALID_DEVICE_INDEX)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;

        if (dev_info->psCallbacks->pfnDataSent)
        {
            dev_info->psCallbacks->pfnDataSent(
                comp_device->psDevices[index].pvInstance, info);
        }
    }
}

/**
  * @brief This function is called to handle data being received back from the host so
  * that the application callback can be called when the new data is ready.
  *
  */
static void data_received(void *composite_instance, uint32_t info)
{
    uint32_t index;
    const tDeviceInfo *dev_info;
    tUSBDCompositeDevice *comp_device;

    /* Create the device instance pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    /*
     * Pass this notification on to the device which last handled a
     * transaction on endpoint 0 (assuming we know who that was).
     */
    index = comp_device->sPrivateData.ui32EP0Owner;

    if (index != INVALID_DEVICE_INDEX)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;

        if (dev_info->psCallbacks->pfnDataReceived)
        {
            dev_info->psCallbacks->pfnDataReceived(
                comp_device->psDevices[index].pvInstance, info);
        }
    }
}

/**
  * @brief This function will check if any device classes need an endpoint handler
  * called.
  *
  */
static void handle_endpoints(void *composite_instance, uint32_t status)
{
    uint32_t index;
    const tDeviceInfo *dev_info;
    tUSBDCompositeDevice *comp_device;

    ASSERT(composite_instance != 0);

    /* Create the device instance pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    /*
     * Call each of the endpoint handlers.  This may seem odd since we should
     * only call the handler whose endpoint needs service.  Unfortunately, if
     * the device class driver is using DMA, we have no way of knowing which
     * handler to call (since ui32Status will be 0).  Since the handlers are
     * set up to ignore any callback that is not for them, this is safe.
     */
    for (index = 0; index < comp_device->ui32NumDevices; index++)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;

        if (dev_info->psCallbacks->pfnEndpointHandler)
        {
            dev_info->psCallbacks->pfnEndpointHandler(
                comp_device->psDevices[index].pvInstance, status);
        }
    }
}

/**
  * @brief Device instance specific handler.
  *
  */
static void handle_device(void *composite_instance, uint32_t req_type,
              void *req_data)
{
    uint32_t index;
    tUSBDCompositeDevice *comp_device;
    const tDeviceInfo *dev_info;

    ASSERT(composite_instance != 0);

    /* Create the device instance pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    for (index = 0; index < comp_device->ui32NumDevices; index++)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;

        if (dev_info->psCallbacks->pfnDeviceHandler)
        {
            dev_info->psCallbacks->pfnDeviceHandler(
                comp_device->psDevices[index].pvInstance, req_type, req_data);
        }
    }

    if (comp_device->pfnCallback)
    {
        switch (req_type)
        {
#if (USBLIB_LPM_SUPPORTED == 1)
            case USB_EVENT_LPM_RESUME:
            {
                /* Pass the LPM resume event to the client. */
                comp_device->pfnCallback(0, USB_EVENT_LPM_RESUME, 0, (void *)0);
                break;
            }

            case USB_EVENT_LPM_SLEEP:
            {
                /* Pass the LPM sleep event to the client. */
                comp_device->pfnCallback(0, USB_EVENT_LPM_SLEEP, 0, (void *)0);
                break;
            }

            case USB_EVENT_LPM_ERROR:
            {
                /* Pass the LPM error event to the client. */
                comp_device->pfnCallback(0, USB_EVENT_LPM_ERROR, 0, (void *)0);
                break;
            }
#endif

            default:
            {
                break;
            }
        }
    }
}

/**
  * @brief This function is called by the USB device stack whenever the device is
  * disconnected from the host.
  */
static void handle_disconnect(void *composite_instance)
{
    uint32_t index;
    const tDeviceInfo *dev_info;
    tUSBDCompositeDevice *comp_device;

    ASSERT(composite_instance != 0);

    // Create the device instance pointer.
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    // Inform the application that the device has been disconnected.
    if (comp_device->pfnCallback)
    {
        comp_device->pfnCallback(composite_instance,
                              USB_EVENT_DISCONNECTED, 0, 0);
    }

    for (index = 0; index < comp_device->ui32NumDevices; index++)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;

        if (dev_info->psCallbacks->pfnDisconnectHandler)
        {
            dev_info->psCallbacks->pfnDisconnectHandler(
                comp_device->psDevices[index].pvInstance);
        }
    }
}

/**
  * @brief This function is called by the USB device stack whenever the device
  * interface changes.  It will be passed on to the device classes if they have
  * a handler for this function.
  */
static void interface_change(void *composite_instance, uint8_t ui8InterfaceNum,
                 uint8_t ui8AlternateSetting)
{
    uint32_t index;
    const tDeviceInfo *dev_info;
    tUSBDCompositeDevice *comp_device;

    ASSERT(composite_instance != 0);

    /* Create the device instance pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    for (index = 0; index < comp_device->ui32NumDevices; index++)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;

        if (dev_info->psCallbacks->pfnInterfaceChange)
        {
            dev_info->psCallbacks->pfnInterfaceChange(
                    comp_device->psDevices[index].pvInstance,
                    ui8InterfaceNum, ui8AlternateSetting);
        }
    }
}

/**
  * @brief This function is called by the USB device stack whenever the device
  * configuration changes. It will be passed on to the device classes if they
  * have a handler for this function.
  */
static void config_change_handler(void *composite_instance, uint32_t value)
{
    uint32_t index;
    const tDeviceInfo *dev_info;
    tUSBDCompositeDevice *comp_device;

    ASSERT(composite_instance != 0);

    /* Create the device instance pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    /* Inform the application that the device configuration has changed. */
    if (comp_device->pfnCallback)
    {
        comp_device->pfnCallback(composite_instance, USB_EVENT_CONFIG_CHANGE,
                              value, 0);
    }

    for (index = 0; index < comp_device->ui32NumDevices; index++)
    {
        dev_info = comp_device->psDevices[index].psDevInfo;

        if (dev_info->psCallbacks->pfnConfigChange)
        {
            dev_info->psCallbacks->pfnConfigChange(
                comp_device->psDevices[index].pvInstance, value);
        }
    }
}

/**
  *
  * This function is called by the USB device stack whenever a non-standard
  * request is received.
  *
  * @param composite_instance
  * @param usb_request points to the request received.
  *
  * This call  will be passed on to the device classes if they have a handler
  * for this function.
  *
  * @return None.
  *
  */
static void handle_requests(void *composite_instance, tUSBRequest *usb_request)
{
    uint32_t index;
    const tDeviceInfo *dev_info;
    tUSBDCompositeDevice *comp_device;

    /* Create the device instance pointer. */
    comp_device = (tUSBDCompositeDevice *)composite_instance;

    /*
     * Determine which device this request is intended for.  We have to be
     * careful here to send this to the callback for the correct device
     * depending upon whether it is a request sent to the device, the interface
     * or the endpoint.
     */
    switch (usb_request->bmRequestType & USB_RTYPE_RECIPIENT_M)
    {
        case USB_RTYPE_INTERFACE:
        {
            index = interface_to_index(comp_device,
                                   (usb_request->wIndex & 0xFF));
            break;
        }

        case USB_RTYPE_ENDPOINT:
        {
            index = endpoint_to_index(comp_device,
                                  (usb_request->wIndex & 0x0F),
                                  (usb_request->wIndex & 0x80) ? true : false);
            break;
        }

        /*
         * Requests sent to the device or any other recipient can't be
         * handled here since we have no way of telling where they are
         * supposed to be handled.  As a result, we just stall them.
         *
         * If your composite device has some device-specific requests that need
         * to be handled at the device (rather than interface or endpoint)
         * level, you should add code here to handle them.
         */
        case USB_RTYPE_DEVICE:
        case USB_RTYPE_OTHER:
        default:
        {
            index = INVALID_DEVICE_INDEX;
            break;
        }
    }

    /* Did we find a device class to pass the request to? */
    if (index != INVALID_DEVICE_INDEX)
    {
        /* Get a pointer to the individual device instance. */
        dev_info = comp_device->psDevices[index].psDevInfo;

        /* Does this device have a RequestHandler callback? */
        if (dev_info->psCallbacks->pfnRequestHandler)
        {
            /*
             * Remember this device index so that we can correctly route any
             * data notification callbacks to it.
             */
            comp_device->sPrivateData.ui32EP0Owner = index;

            /* Yes - call the device to retrieve the descriptor. */
            dev_info->psCallbacks->pfnRequestHandler(
                    comp_device->psDevices[index].pvInstance, usb_request);
        }
        else
        {
            /*
             * Oops - we can't satisfy the request so stall EP0 to indicate
             * an error.
             */
            usbdcd_stall_ep0(USBBaseToIndex(comp_device->sPrivateData.ui32USBBase));
        }
    }
    else
    {
        /*
         * We are unable to satisfy the descriptor request so stall EP0 to
         * indicate an error.
         */
        usbdcd_stall_ep0(USBBaseToIndex(comp_device->sPrivateData.ui32USBBase));
    }
}

/**
  * @brief This function handles sending interface number changes to device instances.
  */
static void composite_interface_change(tCompositeEntry *comp_device,
                                  uint8_t old_interface, uint8_t new_interface)
{
    uint8_t pui8Interfaces[2];

    if (comp_device->psDevInfo->psCallbacks->pfnDeviceHandler)
    {
        // Create the data to pass to the device handler.
        pui8Interfaces[0] = old_interface;
        pui8Interfaces[1] = new_interface;

        /*
         * Call the device handler to inform the class of the interface number
         * change.
         */
        comp_device->psDevInfo->psCallbacks->pfnDeviceHandler(
            comp_device->pvInstance, USB_EVENT_COMP_IFACE_CHANGE,
                                                (void *)pui8Interfaces);
    }
}

/**
  * @brief This function handles sending endpoint number changes to device instances.
  */
static void composite_ep_change(tCompositeEntry *comp_device,
                                            uint8_t old_ep, uint8_t new_ep)
{
    uint8_t pui8Interfaces[2];

    if (comp_device->psDevInfo->psCallbacks->pfnDeviceHandler)
    {
        /* Create the data to pass to the device handler. */
        pui8Interfaces[0] = old_ep;
        pui8Interfaces[1] = new_ep;

        new_ep--;

        /*
         * Call the device handler to inform the class of the interface number
         * change.
         */
        comp_device->psDevInfo->psCallbacks->pfnDeviceHandler(
            comp_device->pvInstance, USB_EVENT_COMP_EP_CHANGE,
                                                    (void *)pui8Interfaces);
    }
}

/**
  * @brief This function merges the configuration descriptors into a single multiple
  *  instance device.
  */
uint32_t build_composite_descriptor(tUSBDCompositeDevice *comp_device)
{
#if (defined USBLIB_DEBUG_EN)&&(USBLIB_DEBUG_EN == 1)
    uint32_t dbg_des_idx;
#endif
    uint32_t index, ui32Offset, ui32CPIdx, ui32FixINT,ui32Dev;
    uint16_t ui16TotalLength, ui16Bytes;
    uint8_t ui8Interface, ui8INEndpoint, ui8OUTEndpoint;
    uint8_t *pui8Data, *pui8Config;
    const tConfigHeader *psConfigHeader;
    tDescriptorHeader *psHeader;
    const uint8_t *pui8Descriptor;
    tInterfaceDescriptor *psInterface;
    tEndpointDescriptor *psEndpoint;
    const tDeviceInfo *psDevice;
	bool bINEndpointSet = false;
	bool bOUTEndpointSet = false;
    
    /* Save the number of devices to look through. */
    ui32Dev = 0;
    index = 0;
    ui8Interface = 0;
    ui8INEndpoint = 1;
    ui8OUTEndpoint = 1;
    ui32Offset = 0;
    ui32FixINT = 0;
    bINEndpointSet = false;
	bOUTEndpointSet = false;

    /*
     * This puts the first section pointer in the first entry in the list
     * of sections.
     */
    comp_device->sPrivateData.ppsCompSections[0] =
                            &comp_device->sPrivateData.psCompSections[0];

    /*
     * Put the pointer to this instances configuration descriptor into the
     * front of the list.
     */
    comp_device->sPrivateData.ppsCompSections[0]->pui8Data =
                    (uint8_t *)&comp_device->sPrivateData.sConfigDescriptor;

    comp_device->sPrivateData.ppsCompSections[0]->ui16Size =
                        comp_device->sPrivateData.sConfigDescriptor.bLength;

    /*
     * The configuration descriptor is 9 bytes so initialize the total length
     * to 9 bytes.
     */
    ui16TotalLength = 9;

    /*
     * Copy the section pointer into the section array for the composite
     * device.  This is awkward but is required given the definition
     * of the structures.
     */
    comp_device->sPrivateData.ppsCompSections[1] =
                                &comp_device->sPrivateData.psCompSections[1];

    /*
     * Copy the pointer to the application supplied space into the section
     * list.
     */
    comp_device->sPrivateData.ppsCompSections[1]->ui16Size = 0;
    comp_device->sPrivateData.ppsCompSections[1]->pui8Data =
                                            comp_device->sPrivateData.pui8Data;

    /*
     * Create a local pointer to the data that is used to copy data from
     * the other devices into the composite descriptor.
     */
    pui8Data = comp_device->sPrivateData.pui8Data;

    /*
     * Consider each device in turn.
     */
    while (ui32Dev < comp_device->ui32NumDevices)
    {
#if (defined USBLIB_DEBUG_EN)&&(USBLIB_DEBUG_EN == 1)
        /* Debug */
        USBLIB_DEBUG_PRINTF(":[dev %d] \r\n", ui32Dev);
#endif
        /* Save the current starting address of this descriptor. */
        pui8Config = pui8Data + ui32Offset;

        /* Create a local pointer to the configuration header. */
        psDevice = comp_device->psDevices[ui32Dev].psDevInfo;
        psConfigHeader = psDevice->ppsConfigDescriptors[0];

        /*
         * Loop through each of the sections in this device's configuration
         * descriptor.
         */
        for (index = 0; index < psConfigHeader->ui8NumSections; index++)
        {
            /*
             * Initialize the local offset in this descriptor.  We include
             * a special case here to ignore the initial 9 byte configuration
             * descriptor since this has already been handled.
             */
            if (index)
            {
                /*
                 * This is not the first section so we handle everything in
                 * it.
                 */
                ui16Bytes = 0;
            }
            else
            {
                /*
                 * This is the first section for this device so skip the 9
                 * byte configuration descriptor since we've already handled
                 * this.
                 */
                ui16Bytes = 9;

                /*
                 * If this section includes only the configuration descriptor,
                 * skip it entirely.
                 */
                if (psConfigHeader->psSections[index]->ui16Size <= ui16Bytes)
                {
                    continue;
                }
            }

            /* Get a pointer to the configuration descriptor. */
            pui8Descriptor = psConfigHeader->psSections[index]->pui8Data;

#if (defined USBLIB_DEBUG_EN)&&(USBLIB_DEBUG_EN == 1)
            /* Debug */
            USBLIB_DEBUG_PRINTF(":configuration description %d of dev %d: \r\n",
                                                                index, ui32Dev);
            /* Printf the configuration description */
            if (((tDescriptorHeader *)pui8Descriptor)->bDescriptorType
                                                    == USB_DTYPE_CONFIGURATION)
            {
                for (dbg_des_idx=0; dbg_des_idx<9; dbg_des_idx++)
                {
                    USBLIB_DEBUG_PRINTF("%02X ", pui8Descriptor[dbg_des_idx]);
                }
                
                USBLIB_DEBUG_PRINTF("\r\n");
            }
#endif

            /*
             * Bounds check the allocated space and return if there is not
             * enough space.
             */
            if (ui32Offset > comp_device->sPrivateData.ui32DataSize)
            {
                return (1);
            }

            /* Copy the descriptor from the device into the descriptor list. */
            for (ui32CPIdx = 0;
                ui32CPIdx < psConfigHeader->psSections[index]->ui16Size;
                ui32CPIdx++)
            {
                pui8Data[ui32CPIdx + ui32Offset] = pui8Descriptor[ui32CPIdx];
            }

            bINEndpointSet = false;
            bOUTEndpointSet = false;
            
            /* Read out the descriptors in this section. */      
            while(ui16Bytes < psConfigHeader->psSections[index]->ui16Size)
            {
                //
                // Create a descriptor header pointer.
                //
                psHeader = (tDescriptorHeader *)&pui8Data[ui32Offset +
                                                          ui16Bytes];

                //
                // Check for interface descriptors and modify the numbering to
                // match the composite device.
                //
                if(psHeader->bDescriptorType == USB_DTYPE_INTERFACE)
                {
                    psInterface = (tInterfaceDescriptor *)psHeader;

                    //
                    // See if this is an alternate setting or the initial
                    // setting.
                    //
                    if(psInterface->bAlternateSetting != 0)
                    {
                        //
                        // If this is an alternate setting then use the
                        // previous interface number because the current one
                        // has already been incremented.
                        //
                        psInterface->bInterfaceNumber = ui8Interface - 1;
                    }
                    else
                    {
                        //
                        // Notify the class that it's interface number has
                        // changed.
                        //
                        composite_interface_change(
                                &comp_device->psDevices[ui32Dev],
                                psInterface->bInterfaceNumber,
                                ui8Interface);
                        //
                        // This was the non-alternate setting so save the
                        // value and move to the next interface number.
                        //
                        psInterface->bInterfaceNumber = ui8Interface;

                        //
                        // No strings allowed on interface descriptors for
                        // composite devices.
                        //
                        psInterface->iInterface = 0;

                        ui8Interface++;
                    }
                }
                //
                // Check for endpoint descriptors and modify the numbering to
                // match the composite device.
                //
                else if(psHeader->bDescriptorType == USB_DTYPE_ENDPOINT)
                {
                    psEndpoint = (tEndpointDescriptor *)psHeader;

                    //
                    // Check if this is an IN or OUT endpoint.
                    //
                    if(psEndpoint->bEndpointAddress & USB_RTYPE_DIR_IN)
                    {
                        //
                    	// TivaWare Update 2.2.x - Removed check for Fixed Interrupt
                    	// class which caused multiple Windows 10 issues with USB
                    	// composite device enumeration.
                        //

                    	// Check if an OUT Endpoint was set during this loop
                    	if (bOUTEndpointSet)
                            {
                    		// Set IN Endpoint Number to be one less than the OUT endpoint
                    		// since it was incremented when setting the IN Endpoint.

                    		ui8INEndpoint = ui8OUTEndpoint-1;
                        }

                            //
                            // Notify the class that it's interface number has
                            // changed.
                            //
                            composite_ep_change(
                                        &comp_device->psDevices[ui32Dev],
                                        psEndpoint->bEndpointAddress,
                                        ui8INEndpoint);

                            psEndpoint->bEndpointAddress = ui8INEndpoint++ |
                                                           USB_RTYPE_DIR_IN;
                    	// Check if a BULK Endpoint for Data transfer is being set.
                    	if(((psEndpoint->bmAttributes & USB_EP_ATTR_TYPE_M) ==
                    			USB_EP_ATTR_BULK) &&
                    			(comp_device->ui16PID == USB_PID_COMP_SERIAL))
                    	{
                    		// Set the boolean that an IN Endpoint for Data transfer is set.
                    		bINEndpointSet = true;
                        }
                    }
                    else
                    {
                    	// Check if an IN Endpoint was set during this loop
                    	if (bINEndpointSet)
                    	{
                    		// Set OUT Endpoint Number to be one less than the IN endpoint
                    		// since it was incremented when setting the IN Endpoint.
                    		ui8OUTEndpoint = ui8INEndpoint-1;
                    	}
                        //
                        // Notify the class that it's interface number has
                        // changed.
                        //
                        composite_ep_change(&comp_device->psDevices[ui32Dev],
                                          psEndpoint->bEndpointAddress,
                                          ui8OUTEndpoint);
                        psEndpoint->bEndpointAddress = ui8OUTEndpoint++;
                        // Check if a BULK Endpoint for Data transfer is being set.
                        if(((psEndpoint->bmAttributes & USB_EP_ATTR_TYPE_M) ==
                        		USB_EP_ATTR_BULK) &&
                        		(comp_device->ui16PID == USB_PID_COMP_SERIAL))
                        {
                        	// Set the boolean that an OUT Endpoint for Data transfer is set.
                        	bOUTEndpointSet = true;
                        }
                    }
                }

                //
                // Move on to the next descriptor.
                //
                ui16Bytes += psHeader->bLength;
            }
            ui32Offset += psConfigHeader->psSections[index]->ui16Size;

            ui16TotalLength += ui16Bytes;
        }

        /*
         * Allow the device class to make adjustments to the configuration
         * descriptor.
         */
        comp_device->psDevices[ui32Dev].psDevInfo->psCallbacks->pfnDeviceHandler(
                                comp_device->psDevices[ui32Dev].pvInstance,
                                USB_EVENT_COMP_CONFIG, (void *)pui8Config);

        /*
         * Add an entry into the device workspace array to allow us to quickly
         * map interface and endpoint numbers to device instances later.
         */
        comp_device->psDevices[ui32Dev].ui32DeviceWorkspace =
                                (ui32Dev << (LOOKUP_INDEX_BYTE * 8)) |
                                (ui8Interface << (LOOKUP_INTERFACE_BYTE * 8)) |
                                (ui8OUTEndpoint << (LOOKUP_OUT_END_BYTE * 8)) |
                                (ui8INEndpoint << (LOOKUP_IN_END_BYTE * 8));

        /* Move on to the next device. */
        ui32Dev++;
    }

    /*
     * Modify the configuration descriptor to match the number of interfaces
     * and the new total size.
     */
    comp_device->sPrivateData.sCompConfigHeader.ui8NumSections = 2;
    comp_device->sPrivateData.ppsCompSections[1]->ui16Size      = ui32Offset;
    comp_device->sPrivateData.sConfigDescriptor.bNumInterfaces  = ui8Interface;
    comp_device->sPrivateData.sConfigDescriptor.wTotalLength    = ui16TotalLength;

    /* Debug */
#if (defined USBLIB_DEBUG_EN)&&(USBLIB_DEBUG_EN == 1)
    USBLIB_DEBUG_PRINTF(":The total length of config is:%d", ui16TotalLength);
#endif

  return (0);
}

/**
  * @brief This function should be called once for the composite class device to
  *        initialize basic operation and prepare for enumeration.
  *
  *        In order for an application to initialize the USB composite device class,
  *        it must first call this function with the a valid composite device class
  *        structure in the \e psDevice parameter.  This allows this function to
  *        initialize the USB controller and device code to be prepared to enumerate
  *        and function as a USB composite device.  The \e ui32Size and \e pui8Data
  *        parameters should be large enough to hold all of the class instances
  *        passed in via the \e psDevice structure.  This is typically the full size
  *        of the configuration descriptor for a device minus its configuration
  *        header(9 bytes).
  *
  *        This function returns a void pointer that must be passed in to all other
  *        APIs used by the composite class.
  *
  *        See the documentation on the tUSBDCompositeDevice structure for more
  *        information on how to properly fill the structure members.
  *
  * @param ui32Index is the index of the USB controller to initialize for
  *        composite device operation.
  * @param psDevice points to a structure containing parameters customizing
  *        the operation of the composite device.
  * @param ui32Size is the size in bytes of the data pointed to by the
  *        \e pui8Data parameter.
  * @param pui8Data is the data area that the composite class can use to build
  *        up descriptors.
  * @retval This function returns 0 on failure or a non-zero void pointer on
  *        success.
  */
void * usbd_composite_init(uint32_t ui32Index, tUSBDCompositeDevice *psDevice,
                    uint32_t ui32Size, uint8_t *pui8Data)
{
    tCompositeInstance *psInst;
    int32_t i32Idx;
    uint8_t *pui8Temp;

    /* Check parameter validity. */
    ASSERT(ui32Index == 0);
    ASSERT(psDevice);
    ASSERT(psDevice->ppui8StringDescriptors);

    /* Initialize the work space in the passed instance structure. */
    psInst = &psDevice->sPrivateData;
    psInst->ui32DataSize = ui32Size;
    psInst->pui8Data = pui8Data;

    /* No device is currently transferring data on EP0. */
    psInst->ui32EP0Owner = INVALID_DEVICE_INDEX;

    /* Initialize the device information structure. */
    psInst->sDevInfo.psCallbacks = &comp_handlers_table;
    psInst->sDevInfo.pui8DeviceDescriptor = comp_device_descriptor;
    psInst->sDevInfo.ppsConfigDescriptors =
                    (const tConfigHeader * const *)g_ppCompConfigDescriptors;
    psInst->sDevInfo.ppui8StringDescriptors = 0;
    psInst->sDevInfo.ui32NumStringDescriptors = 0;

    /* Initialize the device info structure for the composite device. */
    usbdcd_device_info_init(0, &psInst->sDevInfo);

    g_ppCompConfigDescriptors[0] = &psInst->sCompConfigHeader;
    g_ppCompConfigDescriptors[0]->ui8NumSections = 0;
    g_ppCompConfigDescriptors[0]->psSections =
            (const tConfigSection * const *)psDevice->sPrivateData.ppsCompSections;

    /* Create a byte pointer to use with the copy. */
    pui8Temp = (uint8_t *)&psInst->sConfigDescriptor;

    /* Copy the default configuration descriptor into the instance data. */
    for (i32Idx = 0; i32Idx < comp_config_descriptor[0]; i32Idx++)
    {
        pui8Temp[i32Idx] = comp_config_descriptor[i32Idx];
    }

    /* Create a byte pointer to use with the copy. */
    pui8Temp = (uint8_t *)&psInst->sDeviceDescriptor;

    /* Copy the default configuration descriptor into the instance data. */
    for (i32Idx = 0; i32Idx < comp_device_descriptor[0]; i32Idx++)
    {
        pui8Temp[i32Idx] = comp_device_descriptor[i32Idx];
    }

    /* Fix up the device descriptor with the client-supplied values. */
    psInst->sDeviceDescriptor.idVendor = psDevice->ui16VID;
    psInst->sDeviceDescriptor.idProduct = psDevice->ui16PID;

    /* Fix up the configuration descriptor with client-supplied values. */
    psInst->sConfigDescriptor.bmAttributes = psDevice->ui8PwrAttributes;
    psInst->sConfigDescriptor.bMaxPower =
                                (uint8_t)(psDevice->ui16MaxPowermA >> 1);

    psInst->sDevInfo.pui8DeviceDescriptor =
                                (const uint8_t *)&psInst->sDeviceDescriptor;

    /*
     * Plug in the client's string table to the device information
     * structure.
     */
    psInst->sDevInfo.ppui8StringDescriptors =
                                psDevice->ppui8StringDescriptors;
    psInst->sDevInfo.ui32NumStringDescriptors =
                                psDevice->ui32NumStringDescriptors;

    /*
     * Enable Clocking to the USB controller so that changes to the USB
     * controller can be made in the build_composite_descriptor() function.
     */
    md_usb_controller_enable();

    /*
     * Create the combined descriptors.
     */
    if (build_composite_descriptor(psDevice))
    {
        return (0);
    }

    /*
     * All is well so now pass the descriptors to the lower layer and put
     * the bulk device on the bus.
     */
    usbdcd_init(ui32Index, &psInst->sDevInfo, (void *)psDevice);

    /*
     * Return the pointer to the instance indicating that everything went
     * well.
     */
    return ((void *)psDevice);
}

/**
  * @brief Shuts down the composite device.
  *
  *        This function terminates composite device interface for the instance
  *        not me supplied. Following this call, the \e composite_instance instance
  *        should not be used in any other calls.
  *
  * @param composite_instance is the pointer to the device instance structure
  *        as returned by usbd_composite_init().
  * @retval None.
  */
void
usbd_composite_term(void *composite_instance)
{
    ASSERT(composite_instance != 0);

    usbdcd_term(0);
}

//#endif

/**
  * @} composite_device_class_api
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics Co., Ltd. *** END OF FILE ****/

