#include "es32f0271.h"
#include "md_tick.h"
#include "md_usb.h"
#include "usbcore.h"
#include "usbenum.h"
#include "usbenum_host.h"
#include <stdio.h>

#define	USBMSG	1

#pragma pack(1)

#define	HUBADDR	1
#define	DEVADDR	2

#define	MAXEPNUM	6

#define	pUSBEPFIFO(x)	((uint8_t *)&USBH->EP0FIFO+(x*4))
#define	USB_INTR_EPxTX(x)	((1<<x)<<8)
#define	USB_INTR_EPxRX(x)	((1<<x)<<0)

extern	_USB_STANDARD_REQUEST	USBRequest;
extern	uint32_t	u32IntrStatus;

static	uint8_t		u8EP0MaxSz;

//*****************************************************************************
//!\Brief		USB Endpoint In
//!\Param		u8EPx		:	Endpoint Index
//!\Param		u16BCnt	:	Byte Count to Read
//!\Param		*pu8Buf	:	Data Buffer to Read
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_issue_epx_in(uint8_t u8EPx, uint16_t u16BCnt, uint8_t *pu8Buf)
{
uint16_t	u16EPxMaxPkt;
	
#if	USBMSG
//	printf("EP%dIn-",EPx);
#endif
	md_usb_set_index_eptidx(u8EPx);			//Endpoint x
	u16EPxMaxPkt=(md_usb_get_count0_rx1()*8);
	while (u16BCnt)
	{
		while(1)
		{
			md_usb_set_index_eptidx(u8EPx);			//Endpoint x
			md_usb_enable_rxcsrl_reqpkt();			//REQPKT
			while (!(u32IntrStatus&USB_INTR_EPxRX(u8EPx)));		//Wait for EPx Interrupt
			u32IntrStatus&=~USB_INTR_EPxRX(u8EPx);
			if (md_usb_is_active_rxcsrl_rxrdy())		//RXPKTRDY
				break;
			if (md_usb_is_active_rxcsrl_stalled()||md_usb_is_active_rxcsrl_error())	//Error
				return(1);
//			if (md_usb_is_active_rxcsrl_nakto())		//Nak Timeout	
//				md_usb_clear_rxcsrl_nakto();
		}
		if ((md_usb_get_count0_rx1()<u16EPxMaxPkt)||(md_usb_get_count0_rx1()>u16BCnt))
			u16BCnt=0;
		else 
			u16BCnt-=md_usb_get_count0_rx1();
		while (md_usb_get_count0_rx1())
			*pu8Buf++=*pUSBEPFIFO(u8EPx);
		md_usb_clear_rxcsrl_rxrdy();		//Clear REQPKT
	}
	return(0);
#if	USBMSG
//	printf("Done\r\n");
#endif
}
//*****************************************************************************
//!\Brief		USB Endpoint Out
//!\Param		u8EPx		:	Endpoint Index
//!\Param		u16BCnt	:	Byte Count to Write
//!\Param		*pu8Buf	:	Data Buffer to Read
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_issue_epx_out(uint8_t u8EPx, uint16_t u16BCnt, uint8_t *pu8Buf)
{
uint16_t	u16EPxMaxPkt;
uint16_t	u16Count;

#if	USBMSG
//	printf("EP%dOut-",epx);
#endif
	md_usb_set_index_eptidx(u8EPx);			//Endpoint x
	u16EPxMaxPkt=(md_usb_get_txmaxp_maxsize()*8);
	while (u16BCnt)
	{
		if (u16BCnt>=u16EPxMaxPkt)
			u16Count=u16EPxMaxPkt;
		else
			u16Count=u16BCnt;
		u16BCnt-=u16Count;
		while (u16Count--)
			*pUSBEPFIFO(u8EPx)=*pu8Buf++;
		while(1)
		{
			md_usb_set_index_eptidx(u8EPx);			//Endpoint x
			md_usb_enable_txcsrl_txrdy();		//TXPKTRDY
			while (!(u32IntrStatus&USB_INTR_EPxTX(u8EPx)));			//Wait for EPx Interrupt
			u32IntrStatus&=~USB_INTR_EPxTX(u8EPx);
			if (!md_usb_is_enabled_txcsrl_txrdy())	//Data Sent
				break;
			if (md_usb_is_active_txcsrl_stalled()||md_usb_is_active_txcsrl_error())	//Error
				return(1);
//			if (md_usb_is_active_txcsrl_nakto())			//Nak Timeout
//				md_usb_clear_txcsrl_nakto();
		}
	}
	return(0);
#if	USBMSG
//	printf("Done\r\n");
#endif
}
//*****************************************************************************
//!\Brief		Issue USB Setup
//!\Param		*pu8RequestData	:	USB Request Data
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_issue_ep0_setup(uint8_t *pu8RequestData)
{
uint8_t	ii;

#if	USBMSG
//	printf("Setup...");
#endif
	for (ii=0;ii<sizeof(_USB_STANDARD_REQUEST);ii++)
		md_usb_set_ep0fifo(*pu8RequestData++);

	md_usb_set_index_eptidx(USBEP0);		//Endpoint 0
	md_usb_set_csr0l_txcsrl(USB_CSR0L_SETUPPKT_MSK|USB_CSR0L_TXRDY_MSK);	//SETUPPKT+TXPKTRDY
	while (!(u32IntrStatus&USB_INTR_EP0));		//Wait for EP0 Interrupt
	u32IntrStatus&=~USB_INTR_EP0;
	if (md_usb_is_active_csr0l_stalled()||md_usb_is_active_csr0l_error())	//Error
	{
		printf("Error\r\n");
		return(1);
	}
#if	USBMSG
//	printf("Done\r\n");
#endif
	return(0);
}
//*****************************************************************************
//!\Brief		Issue USB Control In
//!\Param		u16BCnt	:	Byte Count to Write
//!\Param		*pu8Buf	:	Data Buffer to Read
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_issue_ep0_data_in(uint16_t u16BCnt, uint8_t *pu8Buf)
{
#if	USBMSG
//	printf("CtrlIn...");
#endif
	while (u16BCnt)
	{
		while(1)
		{
			md_usb_set_index_eptidx(USBEP0);		//Endpoint 0
			md_usb_enable_csr0l_reqpkt();		//REQPKT
			while (!(u32IntrStatus&USB_INTR_EP0));		//Wait for EP0 Interrupt
			u32IntrStatus&=~USB_INTR_EP0;
			if (md_usb_is_active_csr0l_rxrdy())	//RxPktRdy
				break;
			if (md_usb_is_active_csr0l_stalled()||md_usb_is_active_csr0l_error())	//Error
			{
				printf("Error\r\n");
				return(1);
			}
//			if (md_usb_is_active_csr0l_nakto())			//Nak Timeout
//				md_usb_clear_csr0l_nakto();
		}
		if (md_usb_get_count0_count()==0)			//Zero Packet Length
		{
			md_usb_clear_csr0l_rxrdy();	//Clear RXPKTRDY
			break;
		}
		if ((md_usb_get_count0_count()<u8EP0MaxSz)||(md_usb_get_count0_count()>u16BCnt))
			u16BCnt=0;
		else
		{
			if (u16BCnt!=u8EP0MaxSz)
				u16BCnt-=md_usb_get_count0_count();
		}
		while (md_usb_get_count0_count())
			*pu8Buf++=md_usb_get_ep0fifo();;
		md_usb_clear_csr0l_rxrdy();	//Clear RXPKTRDY
	}
#if	USBMSG
//	printf("Done\r\n");
#endif
	return(0);
}
//*****************************************************************************
//!\Brief		Issue USB Control Out
//!\Param		u16BCnt	:	Byte Count to Write
//!\Param		*pu8Buf	:	Data Buffer to Write
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_issue_ep0_data_out(uint16_t u16BCnt, uint8_t *pu8Buf)
{
uint8_t	len;

#if	USBMSG
//	printf("CtrlOut...");
#endif
	while (u16BCnt)
	{
		if (u16BCnt>=u8EP0MaxSz)
			len=u8EP0MaxSz;
		else
			len=u16BCnt;
		u16BCnt-=len;
		while (len--)
			md_usb_set_ep0fifo(*pu8Buf++);
		while(1)
		{
			md_usb_set_index_eptidx(USBEP0);		//Endpoint 0
			md_usb_enable_csr0l_txrdy();			//TXPKTRDY
			while (!(u32IntrStatus&USB_INTR_EP0));	//Wait for EP0 Interrupt
			u32IntrStatus&=~USB_INTR_EP0;
			if (!md_usb_is_enabled_csr0l_txrdy())	//Data Sent
				break;
			if (md_usb_is_active_csr0l_stalled()||md_usb_is_active_csr0l_error())	//Error
			{
				printf("Error\r\n");
				return(1);
			}
//			if (md_usb_is_active_csr0l_nakto())			//Nak Timeout
//				md_usb_clear_csr0l_nakto();
		}
	}
#if	USBMSG
//	printf("Done\r\n");
#endif
	return(0);
}
//*****************************************************************************
//!\Brief		Issue USB Status In
//!\Param		None
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_issue_ep0_status_in()
{
#if	USBMSG
//	printf("StsIn...");
#endif
	md_usb_set_index_eptidx(USBEP0);		//Endpoint 0
	md_usb_set_csr0l_txcsrl(USB_CSR0L_STATUSPKT_MSK|USB_CSR0L_REQPKT_MSK);	//STATUSPKT+REQPKT
	while (!(u32IntrStatus&USB_INTR_EP0));		//Wait for EP0 Interrupt
	u32IntrStatus&=~USB_INTR_EP0;
	while (!md_usb_is_active_csr0l_rxrdy());	//Rx Packet Ready
	md_usb_clear_csr0l_rxrdy();			//Clear RXPKTRDY	
#if	USBMSG
//	printf("Done\r\n");
#endif
	return(0);
}
//*****************************************************************************
//!\Brief		Issue USB Status Out
//!\Param		None
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_issue_ep0_status_out()
{
#if	USBMSG
//	printf("StsOut...");
#endif
	md_usb_set_index_eptidx(USBEP0);		//Endpoint 0
	md_usb_set_csr0l_txcsrl(USB_CSR0L_STATUSPKT_MSK|USB_CSR0L_TXRDY_MSK);	//STATUSPKT+TXPKTRDY
	while (!(u32IntrStatus&USB_INTR_EP0));		//Wait for EP0 Interrupt
	u32IntrStatus&=~USB_INTR_EP0;
#if	USBMSG
//	printf("Done\r\n");
#endif
	return(0);
}
//*****************************************************************************
//!\Brief		USB Standard Request
//!\				Issue Setup Phase, Data In/Out Phase (if There is Data Phase) and Ststua In/Out
//!\Param		*pu8RequestData	:	USB Request Data
//!\Param		*pu8Buf		:	Data Buffer for Data In/Out Phase
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_request_transaction(uint8_t * pu8RequestData, uint8_t *pu8Buf)
{
_pUSB_STANDARD_REQUEST	pUSBRequest;
	
	pUSBRequest=(_pUSB_STANDARD_REQUEST)pu8RequestData;
#if	USBMSG
	printf("USB Request : %d\r\n",pUSBRequest->bRequest);
#endif
	if (usbh_issue_ep0_setup(pu8RequestData))
		return(1);
	if (pUSBRequest->wLength==0)		//No Data Phase
	{
		if (usbh_issue_ep0_status_in())
			return(1);
		return(0);
	}
	if (pUSBRequest->bmRequestType&0x80)	//Data In Pahse
	{
		if (usbh_issue_ep0_data_in(pUSBRequest->wLength,pu8Buf))
			return(1);
		if (usbh_issue_ep0_status_out())
			return(1);
	}
	else					//Data Out Pahse
	{
		if (usbh_issue_ep0_data_out(pUSBRequest->wLength,pu8Buf))
			return(1);
		if (usbh_issue_ep0_status_in())
			return(1);
	}
	return(0);
}
//*****************************************************************************
//!\Brief		Get Descriptor on Addressed Phase
//!\Param		u8DescType	: Descriptor Type
//!\Param		u8DescIndex	: Descriptor Index
//!\Param		u16BCnt			: Byte Count to Get
//!\Param		*pu8_desc_buf	:	Data Buffer for Descriptor Data In
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_get_descriptor(uint8_t u8DescType, uint8_t u8DescIndex, uint16_t u16BCnt, uint8_t *pu8DescBuf)
{
#if	USBMSG
	printf("GetDescsciptor : %d\r\n",u8DescType);
#endif

	USBRequest.bmRequestType=0x80;
	USBRequest.bRequest=USB_REQ_GET_DESCRIPTOR;	//GETDESCRIPTOR
	USBRequest.wValue=((u8DescType<<8)|u8DescIndex);	//Type/Index
	USBRequest.wIndex=0;							//Index
	USBRequest.wLength=u16BCnt;					//Length
	
	if (usbh_request_transaction((uint8_t *)&USBRequest,pu8DescBuf))
		return(1);
	return(0);
}
//*****************************************************************************
//!\Brief		Set Address
//!\Param		u8Addr	:	Device Address to Set
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_set_address(uint8_t u8Addr)
{
#if	USBMSG
	printf("SetAddress	: %d\r\n",u8Addr);
#endif

	USBRequest.bmRequestType=0x0;
	USBRequest.bRequest=USB_REQ_SET_ADDRESS;	//SETADDRESS
	USBRequest.wValue=u8Addr;				//Address
	USBRequest.wIndex=0;					//Index
	USBRequest.wLength=0;					//Length
	
	if (usbh_request_transaction((uint8_t *)&USBRequest,NULL))
		return(1);
	return(0);
}
//*****************************************************************************
//!\Brief		Set Configuration
//!\Param		u8Config	:	Config Value to Set
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_set_configuration(uint8_t u8Config)
{
#if	USBMSG
	printf("SetConfiguration : %d\r\n",u8Config);
#endif

	USBRequest.bmRequestType=0x0;
	USBRequest.bRequest=USB_REQ_SET_CONFIGURATION;	//SETCONFIGURATION
	USBRequest.wValue=u8Config;		//Config Value
	USBRequest.wIndex=0;					//Index
	USBRequest.wLength=0;					//Length
	
	if (usbh_request_transaction((uint8_t *)&USBRequest,NULL))
		return(1);
	return(0);
}
//*****************************************************************************
//!\Brief		Set Alternate Interface
//!\Param		u8Index		:	Interface Index
//!\Param		u8Alternate	:	Alternate Value to Set
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_set_interface(uint8_t u8Index, uint8_t u8Alternate)
{
#if	USBMSG
	printf("SetInterface : %d/%d\r\n",u8Index,u8Alternate);
#endif

	USBRequest.bmRequestType=0x0;
	USBRequest.bRequest=USB_REQ_SET_INTERFACE;	//SETINTERFACE
	USBRequest.wValue=u8Alternate;		//Alternate
	USBRequest.wIndex=u8Index;				//Index
	USBRequest.wLength=0;						//Length
	
	if (usbh_request_transaction((uint8_t *)&USBRequest,NULL))
		return(1);
	return(0);
}
//*****************************************************************************
//!\Brief		Clear Feature
//!\Param		u8Selector	:	Feature Selector
//!\Param		u8Index		:	Interface/Endpoint Index
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_clear_feature(uint8_t u8Selector, uint8_t u8Index)		//[00 01 02] 01 XX 00 XX 00 00 00
{
#if	USBMSG
	printf("ClearFeature : %d/%d\r\n",u8Selector,u8Index);
#endif

	if (u8Selector==USB_FEATURE_REMOTE_WAKE)
		USBRequest.bmRequestType=0x0;
	if (u8Selector==USB_FEATURE_EP_HALT)
		USBRequest.bmRequestType=0x2;
	USBRequest.bRequest=USB_REQ_CLEAR_FEATURE;	//CLEARFEATURE
	USBRequest.wValue=u8Selector;			//Selector
	USBRequest.wIndex=u8Index;				//Index
	USBRequest.wLength=0;						//Length
	
	if (usbh_request_transaction((uint8_t *)&USBRequest,NULL))
		return(1);
	return(0);
}
//*****************************************************************************
//!\Brief		Wait for USB Device Plug In
//!\				Issue Reset and Enable Disconnect and Resume Interrupt
//!\Param		None
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_check_connect()
{
//	while (!(u32IntrStatus&USB_INTR_CONN));	//Device Connected
//	u32IntrStatus&=~USB_INTR_CONN;
	if (!md_usb_is_active_rif_conrif())	//Device Not Connected
		return(1);
	md_usb_enable_icr_conicr();

	md_tick_waitms(10,2);		//Reset 20mS
//	md_tick_waitms(10,1);		//Reset 10mS
//	md_tick_waitms(10,1);		//Reset 10mS
//	md_tick_waitms(10,1);		//Suspend 10mS
	md_usb_enable_power_reset();
	md_tick_waitms(10,2);		//Reset 20mS
	md_usb_disable_power_reset();
	md_tick_waitms(10,2);		//Wait 20mS
//	while (!(u32IntrStatus&USB_INTR_SOF));	//Wait for SOF
	md_usb_enable_idr_conid();			//Disable Connect and SOF Interrupt
	md_usb_set_ier(USB_IER_DISCONIE_MSK|USB_IER_RESIE_MSK);	//Interrupt Enable for Disconnect and Resume
	return(0);
}
//*****************************************************************************
//!\Brief		Issue Reset Signal 20mS to Devie
//!\Param		None
//!\RetVal	0x0 : No error, 0x1 : Error
//*****************************************************************************
uint8_t	usbh_reset_device()
{
	md_usb_enable_power_reset();
	md_tick_waitms(10,2);		//Reset 20mS
	md_usb_disable_power_reset();
	md_tick_waitms(10,2);		//Wait 20mS
	return(0);
}
//*****************************************************************************
//!\Brief	Dump Descriptor Data
//!\Param	u16BCnt	:	Byte Count to Dump
//!\Param	*pu8Buf	:	Data Buffer to Dump
//!\RetVal	None
//*****************************************************************************
void	usbh_dump_descriptor(uint16_t u16BCnt, uint8_t *pu8Buf)
{
uint8_t	Length, DescriptorType;
uint8_t	ii;
	
	while (u16BCnt)
	{
		Length=*pu8Buf++;		//Length
		DescriptorType=*pu8Buf++;		//DescriptotType
		printf("Length=0x%02x\r\n",Length);
		printf("Type=0x%02x\r\n",DescriptorType);
		printf("Data=\r\n");
		for (ii=0;ii<(Length-2);ii++)
		{
			printf("0x%02x ",*pu8Buf++);
			if ((ii&0x7)==0x7)
				printf("\r\n");
		}
		printf("\r\n");
		u16BCnt-=Length;
	}
}
//*****************************************************************************
//!\Brief		Enumeration for USB Device
//!\				Retrive Needed Data for USB Host
//!\Param		*pu8EnumBuf	:	Data Buffer for Enumeration
//!\RetVal	0x00=Pass, 0x1=Fail
//*****************************************************************************
uint8_t	usbh_dev_enumeration(uint8_t *pu8EnumBuf)
{
_pUSB_DEVICE_DESC					pDeviceDesc;
_pUSB_CONFIGURATION_DESC	pConfigurationDesc;
uint8_t	ii;

	u8EP0MaxSz=0x40;

	md_usb_set_faddr_addr(0);		//Default Address
//Addressing Phase
	if (usbh_get_descriptor(USB_DTYPE_DEVICE,0,u8EP0MaxSz,pu8EnumBuf))
		return(1);
	pDeviceDesc=(_pUSB_DEVICE_DESC)pu8EnumBuf;
#if	USBMSG
	usbh_dump_descriptor(sizeof(_USB_DEVICE_DESC),pu8EnumBuf);
#endif
	if (pDeviceDesc->bLength!=sizeof(_USB_DEVICE_DESC))	//bLength Mismatch
	{
		printf("Not Legel Device Descriptor Length\r\n");
		return(1);
	}
	if (pDeviceDesc->bDescriptorType!=USB_DTYPE_DEVICE)	//bDescriptorType Mismatch
	{
		printf("Not Legel Device Descriptor Type\r\n");
		return(1);
	}
	u8EP0MaxSz=pDeviceDesc->bMaxPacketSize0;
	if (usbh_set_address(DEVADDR))
		return(1);
	md_tick_waitms(5,1);		//Wait 5mS
	md_usb_set_faddr_addr(DEVADDR);	//Device Address
//Descriptor Parser
	if (usbh_get_descriptor(USB_DTYPE_DEVICE,0,sizeof(_USB_DEVICE_DESC),pu8EnumBuf))
		return(1);
#if	USBMSG
	usbh_dump_descriptor(sizeof(_USB_DEVICE_DESC),pu8EnumBuf);
#endif
	pDeviceDesc=(_pUSB_DEVICE_DESC)pu8EnumBuf;
#if	USBMSG
	printf("VID/PID=0x%04x/0x%04x\r\n",pDeviceDesc->idVendor,pDeviceDesc->idProduct);
#endif
	pu8EnumBuf+=sizeof(_USB_DEVICE_DESC);
	for (ii=0;ii<pDeviceDesc->bNumConfigurations;ii++)
	{
		if (usbh_get_descriptor(USB_DTYPE_CONFIGURATION,ii,sizeof(_USB_CONFIGURATION_DESC),pu8EnumBuf))
			return(1);
#if	USBMSG
		usbh_dump_descriptor(sizeof(_USB_CONFIGURATION_DESC),pu8EnumBuf);
#endif
		pConfigurationDesc=(_pUSB_CONFIGURATION_DESC)pu8EnumBuf;
		if (pConfigurationDesc->bLength!=sizeof(_USB_CONFIGURATION_DESC))	//bLength Mismatch
		{
			printf("Not Legel Configuration Descriptor Length\r\n");
			return(1);
		}
		if (pConfigurationDesc->bDescriptorType!=USB_DTYPE_CONFIGURATION)	//bDescriptorType Mismatch
		{
			printf("Not Legel Configuration Descriptor Type\r\n");
			return(1);
		}
		if (usbh_get_descriptor(USB_DTYPE_CONFIGURATION,ii,pConfigurationDesc->wTotalLength,pu8EnumBuf))
			return(1);
#if	USBMSG
		usbh_dump_descriptor(pConfigurationDesc->wTotalLength,pu8EnumBuf);
#endif
		pu8EnumBuf+=pConfigurationDesc->wTotalLength;
	}
	return(0);
}

