/**
  **************************************************************************************
  * @file    iap_fc.c
  * @brief   iap flash interface.
  * @data    05/17/2019
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. ALL rights reserved.
  *
  **************************************************************************************
  */  
/* Includes-------------------------------------------------------------------*/

#include "md_syscfg.h"
#include "fclib.h"
#include "iap_update.h"

#pragma pack(1)

/* Private Constants----------------------------------------------------------*/
/* Public Structure ----------------------------------------------------------*/
/* Public Macros -------------------------------------------------------------*/
/* Public Constants ----------------------------------------------------------*/

/* Private variables ---------------------------------------------------------*/
/* Private Macros ------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private Functions ---------------------------------------------------------*/

/** @addtogroup Private_Function
  * @{
  */  
  
/**
  * @brief  Check if the APP been programmed and jump to the APP
  * @param  u32EFAddr: flash address
	* @retval 0 : No error, 1 : Error
  */
uint8_t iap_check_and_run(uint32_t u32EFAddr)
{
  /* Test if user code is programmed starting from address "APPLICATION_ADDRESS" */
  if (((*(__IO uint32_t *)u32EFAddr) & 0x2FFFC000) == 0x20000000)  
  {
    __disable_irq();
    
		md_syscfg_set_remap_efbase(SYSCFG,(u32EFAddr/0x1000));	//Remap Base
		md_syscfg_set_remap_memmod(SYSCFG,MD_SYSCFG_MEMMOD_MAIN);	//Remap to Main
		md_syscfg_enable_remap_remap(SYSCFG);			//Start Remap
    NVIC_SystemReset();
		while(1);
  }
  return(0);
}

/**
  * @brief  erase iap flash
  * @param  u32EFAddr: flash address
  * @param  u32EFSize: flash size
	* @retval 0 : No error, 1 : Error
  */
uint8_t iap_flash_erase(uint32_t u32EFAddr, uint32_t u32EFSize)
{
  uint8_t u8Page ;

  if ((u32EFAddr >= (IAP_FLASH_START + IAP_FLASH_SIZE)) || (u32EFAddr < IAP_FLASH_START) )
    return(1);
  if (u32EFSize > IAP_FLASH_SIZE)
    return(1);
	u8Page=((u32EFSize+IAP_FLASH_PAGE_SIZE-1)/IAP_FLASH_PAGE_SIZE);
	while (u8Page--)
	{
    fclib_perase(u32EFAddr, ~u32EFAddr);
		u32EFAddr += IAP_FLASH_PAGE_SIZE;
  }
  return(0);
}
/**
  * @brief  program iap flash
  * @param  u32EFAddr: flash address
  * @param  u32Length: byte length to write
  * @param  pu32Data: data buff to write
	* @retval 0 : No error, 1 : Error
  */
uint8_t iap_flash_program(uint32_t u32EFAddr, uint32_t u32Length, uint32_t *pu32Data)
{
  if ((u32EFAddr >= (IAP_FLASH_START + IAP_FLASH_SIZE)) || (u32EFAddr < IAP_FLASH_START) )
    return(1);
	u32Length=(u32Length+7)/8*8;
  fclib_program(u32EFAddr, ~u32EFAddr, u32Length, pu32Data);
  return(0);
}

/**
  * @brief  read iap flash
  * @param  u32EFAddr: flash address
  * @param  u32Length: byte length to read
  * @param  pu32Data: data buff to read
	* @retval 0 : No error, 1 : Error
  */
uint8_t iap_flash_read(uint32_t u32EFAddr, uint32_t u32Length, uint32_t *pu32Data)
{
  if ((u32EFAddr >= (IAP_FLASH_START + IAP_FLASH_SIZE)) || (u32EFAddr < IAP_FLASH_START) )
    return(1);
	u32Length=(u32Length+4)/4*4;
  fclib_read(u32EFAddr, u32Length, pu32Data);
  return(0);
}

/**
  * @brief  get iap flash info
  * @param  pu32EFAddr: flash start address
  * @param  pu32EFSize: flash size
  * @retval None
  */
void iap_flash_info(uint32_t *pu32EFAddr, uint32_t *pu32EFSize)
{
  if (pu32EFAddr)
    *pu32EFAddr = IAP_FLASH_START;

  if (pu32EFSize)
    *pu32EFSize =  IAP_FLASH_SIZE;
}

