#ifndef	_USBCORE_H
#define	_USBCORE_H

#pragma pack(1)

//Define Descriptor Type
#define	USB_DTYPE_DEVICE					0x1
#define	USB_DTYPE_CONFIGURATION		0x2
#define	USB_DTYPE_STRING					0x3
#define	USB_DTYPE_INTERFACE				0x4
#define	USB_DTYPE_ENDPOINT				0x5
#define USB_DTYPE_QUALIFIER				0x6

//*****************************************************************************
//
// USB Device Class codes used in the tDeviceDescriptor.bDeviceClass field.
// Definitions for the bDeviceSubClass and bDeviceProtocol fields are device
// specific and can be found in the appropriate device class header files.
//
//*****************************************************************************
#define USB_CLASS_AUDIO					0x01
#define USB_CLASS_CDC		  			0x02
#define USB_CLASS_HID		  			0x03
#define USB_CLASS_MONITOR				0x04
#define USB_CLASS_PHYSICAL			0x05
#define USB_CLASS_POWER					0x06
#define USB_CLASS_PRINTER				0x07
#define USB_CLASS_MSD		  			0x08
#define USB_CLASS_HUB		  			0x09
#define USB_CLASS_CDC_DATA			0x0A
#define USB_CLASS_SMART_CARD		0x0B
#define USB_CLASS_SECURITY			0x0D
#define USB_CLASS_VIDEO					0x0E
#define USB_CLASS_HEALTHCARE		0x0F
#define USB_CLASS_DIAG_DEVICE		0xDC
#define USB_CLASS_WIRELESS			0xE0
#define USB_CLASS_MISC					0xEF
#define USB_CLASS_APPLICATION		0xFE
#define USB_CLASS_VENDOR				0xFF

//*****************************************************************************
//
// Generic values for undefined subclass and protocol.
//
//*****************************************************************************
#define USB_SUBCLASS_UNDEFINED  0x00
#define USB_PROTOCOL_UNDEFINED  0x00

//*****************************************************************************
//
// The following are the miscellaneous subclass values.
//
//*****************************************************************************
#define USB_MISC_SUBCLASS_SYNC		0x01
#define USB_MISC_SUBCLASS_COMMON	0x02

//*****************************************************************************
//
// These following are miscellaneous protocol values.
//
//*****************************************************************************
#define USB_MISC_PROTOCOL_IAD		0x01

//*****************************************************************************
//
// These following are hub protocol values.
//
//*****************************************************************************
#define USB_HUB_PROTOCOL_FS     0x00
#define USB_HUB_PROTOCOL_SINGLE 0x01
#define USB_HUB_PROTOCOL_MULTI  0x02

//*****************************************************************************
//
//	Standard USB requests IDs used in the bRequest field of tUSBRequest.
//
//*****************************************************************************
#define	USB_REQ_GET_STATUS				0x0
#define	USB_REQ_CLEAR_FEATURE			0x1
#define	USB_REQ_SET_FEATURE				0x3
#define	USB_REQ_SET_ADDRESS				0x5
#define	USB_REQ_GET_DESCRIPTOR		0x6
#define	USB_REQ_SET_DESCRIPTOR		0x7
#define	USB_REQ_GET_CONFIGURATION	0x8
#define	USB_REQ_SET_CONFIGURATION	0x9
#define	USB_REQ_GET_INTERFACE			0xA
#define	USB_REQ_SET_INTERFACE			0xB
#define	USB_REQ_SYNCH_FRAME				0xC

//*****************************************************************************
//
// The following defines are used with the bmRequestType member of tUSBRequest.
//
// Request types have 3 bit fields:
// 4:0 - Is the recipient type.
// 6:5 - Is the request type.
// 7 - Is the direction of the request.
//
//*****************************************************************************
#define USB_RTYPE_DIR_IN        0x80
#define USB_RTYPE_DIR_OUT       0x00

#define USB_RTYPE_TYPE_M        0x60
#define USB_RTYPE_VENDOR        0x40
#define USB_RTYPE_CLASS         0x20
#define USB_RTYPE_STANDARD      0x00

#define USB_RTYPE_RECIPIENT_M   0x1F
#define USB_RTYPE_OTHER         0x03
#define USB_RTYPE_ENDPOINT      0x02
#define USB_RTYPE_INTERFACE     0x01
#define USB_RTYPE_DEVICE        0x00

//*****************************************************************************
//
// Data returned from a USBREQ_GET_STATUS request to a device.
//
//*****************************************************************************
#define USB_STATUS_SELF_PWR     0x0001  // Currently self powered.
#define USB_STATUS_BUS_PWR      0x0000  // Currently bus-powered.
#define USB_STATUS_PWR_M        0x0001  // Mask for power mode.
#define USB_STATUS_REMOTE_WAKE  0x0002  // Remote wake-up is currently enabled.

//*****************************************************************************
//
// Feature Selectors (tUSBRequest.wValue) passed on USBREQ_CLEAR_FEATURE and
// USBREQ_SET_FEATURE.
//
//*****************************************************************************
#define USB_FEATURE_EP_HALT     0x0000  // Endpoint halt feature.
#define USB_FEATURE_REMOTE_WAKE 0x0001  // Remote wake feature, device only.
#define USB_FEATURE_TEST_MODE   0x0002  // Test mode

//*****************************************************************************
//
// Flags used in constructing the value assigned to the field
// tEndpointDescriptor.bEndpointAddress.
//
//*****************************************************************************
#define USB_EP_DESC_OUT         0x00
#define USB_EP_DESC_IN          0x80
#define USB_EP_DESC_NUM_M       0x0f

//*****************************************************************************
//
// Mask used to extract the maximum packet size (in bytes) from the
// wMaxPacketSize field of the endpoint descriptor.
//
//*****************************************************************************
#define USB_EP_MAX_PACKET_COUNT_M	0x07FF

//*****************************************************************************
//
// Endpoint attributes used in tEndpointDescriptor.bmAttributes.
//
//*****************************************************************************
#define USB_EP_ATTR_CONTROL			0x00
#define USB_EP_ATTR_ISO					0x01
#define USB_EP_ATTR_BULK				0x02
#define USB_EP_ATTR_INT					0x03
#define USB_EP_ATTR_TYPE_M			0x03

#define USB_EP_ATTR_ISOC_M      0x0c
#define USB_EP_ATTR_ISOC_NOSYNC 0x00
#define USB_EP_ATTR_ISOC_ASYNC  0x04
#define USB_EP_ATTR_ISOC_ADAPT  0x08
#define USB_EP_ATTR_ISOC_SYNC   0x0c
#define USB_EP_ATTR_USAGE_M     0x30
#define USB_EP_ATTR_USAGE_DATA  0x00
#define USB_EP_ATTR_USAGE_FEEDBACK		0x10
#define USB_EP_ATTR_USAGE_IMPFEEDBACK	0x20

//*****************************************************************************
//
//! The operating mode required by the USB library client.  This type is used
//! by applications which wish to be able to switch between host and device
//! modes by calling the usb_stack_mode_set() API.
//
//*****************************************************************************
typedef enum
{
  //
  //! A marker indicating that no USB mode has yet been set by the
  //! application.
  //
  eUSBModeNone = 0,

  //
  //! Operate in USB device mode with active monitoring of VBUS and the
  //! ID pin must be pulled to a logic high value.
  //
  eUSBModeDevice,

  //
  //! Operate in USB host mode with active monitoring of VBUS and the ID pin
  //! must be pulled to a logic low value.
  //
  eUSBModeHost,

  //
  //! Operate as an On-The-Go device which requires both VBUS and ID to be
  //! connected directly to the USB controller from the USB connector.
  //
  eUSBModeOTG,

  //
  //! Force host mode so that the VBUS and ID pins are not used or monitored
  //! by the USB controller.
  //
  eUSBModeForceHost,

  //
  //! Forcing device mode so that the VBUS and ID pins are not used or
  //! monitored by the USB controller.
  //
  eUSBModeForceDevice,
}tUSBMode;
//*****************************************************************************
//
//! The standard USB request header as defined in section 9.3 of the USB 2.0
//! specification.
//
//*****************************************************************************
typedef	struct	_USB_STANDARD_REQUEST_
{
  //
  //! Determines the type and direction of the request.
  //
	uint8_t		bmRequestType;

  //
  //! Identifies the specific request being made.
  //
	uint8_t		bRequest;

  //
  //! Word-sized field that varies according to the request; typically used
  //! to pass an index or offset.
  //
	uint16_t	wValue;

  //
  //! The number of bytes to transfer if there is a data stage to the
  //! request.
  //
	uint16_t	wIndex;
  //
  //! The number of bytes to transfer if there is a data stage to the
  //! request.
  //
	uint16_t	wLength;
}_USB_STANDARD_REQUEST, *_pUSB_STANDARD_REQUEST;

//device descriptor format
typedef struct	_USB_DEVICE_DESC_
{
  //
  //! The length of this descriptor in bytes.  All device descriptors are
  //! 18 bytes long.
  //
	uint8_t		bLength;

  //
  //! The type of the descriptor.  For a device descriptor, this will be
  //! USB_DTYPE_DEVICE (1).
  //
	uint8_t		bDescriptorType;

  //
  //! The USB Specification Release Number in BCD format.  For USB 2.0, this
  //! will be 0x0200.
  //	
	uint16_t	bcdUSB;

  //
  //! The device class code.
  //
	uint8_t		bDeviceClass;

  //
  //! The device subclass code.  This value qualifies the value found in the
  //! bDeviceClass field.
  //
	uint8_t		bDeviceSubClass;

  //
  //! The device protocol code.  This value is qualified by the values of
  //! bDeviceClass and bDeviceSubClass.
  //
	uint8_t		bDeviceProtocol;

  //
  //! The maximum packet size for endpoint zero.  Valid values are 8, 16, 32
  //! and 64.
  //
	uint8_t		bMaxPacketSize0;

  //
  //! The device Vendor ID (VID) as assigned by the USB-IF.
  //
	uint16_t	idVendor;

  //
  //! The device Product ID (PID) as assigned by the manufacturer.
  //
  uint16_t	idProduct;

  //
  //! The device release number in BCD format.
  //
  uint16_t	bcdDevice;

  //
  //! The index of a string descriptor describing the manufacturer.
  //
  uint8_t		iManufacturer;

  //
  //! The index of a string descriptor describing the product.
  //
  uint8_t		iProduct;

  //
  //! The index of a string descriptor describing the device's serial
  //! number.
  //
  uint8_t		iSerialNumber;

  //
  //! The number of possible configurations offered by the device.  This
  //! field indicates the number of distinct configuration descriptors that
  //! the device offers.
  //
  uint8_t		bNumConfigurations;
}_USB_DEVICE_DESC, *_pUSB_DEVICE_DESC;

//*****************************************************************************
//
//! This structure describes the USB device qualifier descriptor as defined in
//! the USB 2.0 specification, section 9.6.2.
//
//*****************************************************************************
typedef struct	_USB_QUALIFIER_DESC_
{
  //
  //! The length of this descriptor in bytes.  All device qualifier
  //! descriptors are 10 bytes long.
  //
  uint8_t		bLength;

  //
  //! The type of the descriptor.  For a device descriptor, this will be
  //! USB_DTYPE_DEVICE_QUAL (6).
  //
  uint8_t		bDescriptorType;

  //
  //! The USB Specification Release Number in BCD format.  For USB 2.0, this
  //! will be 0x0200.
  //
  uint16_t	bcdUSB;

  //
  //! The device class code.
  //
  uint8_t		bDeviceClass;

  //
  //! The device subclass code.  This value qualifies the value found in the
  //! bDeviceClass field.
  //
  uint8_t		bDeviceSubClass;

  //
  //! The device protocol code.  This value is qualified by the values of
  //! bDeviceClass and bDeviceSubClass.
  //
  uint8_t		bDeviceProtocol;

  //
  //! The maximum packet size for endpoint zero when operating at a speed
  //! other than high speed.
  //
  uint8_t		bMaxPacketSize0;

  //
  //! The number of other-speed configurations supported.
  //
  uint8_t		bNumConfigurations;

  //
  //! Reserved for future use.  Must be set to zero.
  //
  uint8_t		bReserved;
}_USB_QUALIFIER_DESC, *_pUSB_QUALIFIER_DESC;

//*****************************************************************************
//
//! This structure describes the USB configuration descriptor as defined in
//! USB 2.0 specification section 9.6.3.  This structure also applies to the
//! USB other speed configuration descriptor defined in section 9.6.4.
//
//*****************************************************************************
typedef	struct	_USB_CONFIGURATION_DESC_
{
  //
  //! The length of this descriptor in bytes.  All configuration descriptors
  //! are 9 bytes long.
  //
	uint8_t		bLength;

  //
  //! The type of the descriptor.  For a configuration descriptor, this will
  //! be USB_DTYPE_CONFIGURATION (2).
  //
	uint8_t		bDescriptorType;

  //
  //! The total length of data returned for this configuration.  This
  //! includes the combined length of all descriptors (configuration,
  //! interface, endpoint and class- or vendor-specific) returned for this
  //! configuration.
  //
	uint16_t	wTotalLength;

  //
  //! The number of interface supported by this configuration.
  //
	uint8_t		bNumInterface;

  //
  //! The value used as an argument to the SetConfiguration standard request
  //! to select this configuration.
  //
	uint8_t		bConfigurationValue;

  //
  //! The index of a string descriptor describing this configuration.
  //
	uint8_t		iConfiguration;

  //
  //! Attributes of this configuration.
  //
	uint8_t		bmAttribute;

  //
  //! The maximum power consumption of the USB device from the bus in this
  //! configuration when the device is fully operational.  This is expressed
  //! in units of 2mA so, for example, 100 represents 200mA.
  //
	uint8_t		MaxPower;
}_USB_CONFIGURATION_DESC, *_pUSB_CONFIGURATION_DESC;

//*****************************************************************************
//
//! This structure describes the USB interface descriptor as defined in USB
//! 2.0 specification section 9.6.5.
//
//*****************************************************************************
typedef	struct	_USB_INTERFACE_DESC_
{
  //
  //! The length of this descriptor in bytes.  All interface descriptors
  //! are 9 bytes long.
  //
	uint8_t	bLength;

  //
  //! The type of the descriptor.  For an interface descriptor, this will
  //! be USB_DTYPE_INTERFACE (4).
  //
	uint8_t	bDescriptorType;

  //
  //! The number of this interface.  This is a zero based index into the
  //! array of concurrent interfaces supported by this configuration.
  //
	uint8_t	bInterfaceNumber;

  //
  //! The value used to select this alternate setting for the interface
  //! defined in bInterfaceNumber.
  //
	uint8_t	bALternateSetting;

  //
  //! The number of endpoints used by this interface (excluding endpoint
  //! zero).
  //
	uint8_t	bNumEndpoints;

  //
  //! The interface class code as assigned by the USB-IF.
  //
	uint8_t	bInterfaceClass;

  //
  //! The interface subclass code as assigned by the USB-IF.
  //
	uint8_t	bInterfaceSubClass;

  //
  //! The interface protocol code as assigned by the USB-IF.
  //
	uint8_t	bInterfaceProtocol;

  //
  //! The index of a string descriptor describing this interface.
  //
	uint8_t	iInterface;
}_USB_INTERFACE_DESC, *_pUSB_INTERFACE_DESC;

//*****************************************************************************
//
//! This structure describes the USB endpoint descriptor as defined in USB
//! 2.0 specification section 9.6.6.
//
//*****************************************************************************
typedef	struct	_USB_ENDPOINT_DESC_
{
  //
  //! The length of this descriptor in bytes.  All endpoint descriptors
  //! are 7 bytes long.
  //
	uint8_t	bLength;

  //
  //! The type of the descriptor.  For an endpoint descriptor, this will
  //! be USB_DTYPE_ENDPOINT (5).
  //
	uint8_t	bDescriptorType;

  //
  //! The address of the endpoint.  This field contains the endpoint number
  //! ORed with flag USB_EP_DESC_OUT or USB_EP_DESC_IN to indicate the
  //! endpoint direction.
  //
	uint8_t	bEndpointAddress;

  //
  //! The endpoint transfer type, USB_EP_ATTR_CONTROL, USB_EP_ATTR_ISOC,
  //! USB_EP_ATTR_BULK or USB_EP_ATTR_INT and, if isochronous, additional
  //! flags indicating usage type and synchronization method.
  //
	uint8_t	bmAttribute;

  //
  //! The maximum packet size this endpoint is capable of sending or
  //! receiving when this configuration is selected.  For high speed
  //! isochronous or interrupt endpoints, bits 11 and 12 are used to
  //! pass additional information.
  //
	uint16_t	wMaxPacketSize;

  //
  //! The polling interval for data transfers expressed in frames or
  //! micro frames depending upon the operating speed.
  //
	uint8_t	bInterval;
}_USB_ENDPOINT_DESC, *_pUSB_ENDPOINT_DESC;

//*****************************************************************************
//
//! This structure describes the USB string descriptor for index 0 as defined
//! in USB 2.0 specification section 9.6.7.  Note that the number of language
//! IDs is variable and can be determined by examining bLength.  The number of
//! language IDs present in the descriptor is given by ((bLength - 2) / 2).
//
//*****************************************************************************
typedef struct _USB_STRING_0_DESC_
{
  //
  //! The length of this descriptor in bytes.  This value will vary
  //! depending upon the number of language codes provided in the descriptor.
  //
	uint8_t	bLength;

  //
  //! The type of the descriptor.  For a string descriptor, this will be
  //! USB_DTYPE_STRING (3).
  //
	uint8_t	bDescriptorType;

  //
  //! The language code (LANGID) for the first supported language.  Note that
  //! this descriptor may support multiple languages, in which case, the
  //! number of elements in the wLANGID array will increase and bLength will
  //! be updated accordingly.
  //
	uint16_t	uLanguageID;
}_USB_STRING_0_DESC, *_pUSB_STRING_0_DESC;


//*****************************************************************************
//
//! This structure describes the USB string descriptor for all string indexes
//! other than 0 as defined in USB 2.0 specification section 9.6.7.
//
//*****************************************************************************
typedef struct _USB_STRING_X_DESC_
{
  //
  //! The length of this descriptor in bytes.  This value will be 2 greater
  //! than the number of bytes comprising the UNICODE string that the
  //! descriptor contains.
  //
  uint8_t		bLength;

  //
  //! The type of the descriptor.  For a string descriptor, this will be
  //! USB_DTYPE_STRING (3).
  //
  uint8_t		bDescriptorType;

  //
  //! The first byte of the UNICODE string.  This string is not NULL
  //! terminated.  Its length (in bytes) can be computed by subtracting 2
  //! from the value in the bLength field.
  //
  uint8_t		bString;
}_USB_STRING_X_DESC, *_pUSB_STRING_X_DESC;

void	usb_core_mode_set(tUSBMode);
void	usb_dev_connect(void);
void	usb_dev_disconnect(void);

#endif	//_USBCORE_H
