#include "es32f0283.h"
#include "md_usb.h"
#include "md_tick.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_host_enum.h"
#include <stdio.h>

#define	HOST_ENUM_MSG	1

#pragma pack(1)

#define	USB_INTR_EPxTX(x)	((1 << x) << 8)
#define	USB_INTR_EPxRX(x)	((1 << x) << 0)

/****
//!\Brief	USB Endpoint In
//!\Param	EPx	: Endpoint Index
//!\Param	BCnt	: Byte Count to Read
//!\Param	*pDataBuf	: Data Buffer to Read
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_issue_epx_in(_pUSB_FS_HOST pHost, uint8_t EPx, uint16_t BCnt, uint8_t *pDataBuf)
{
uint16_t	MaxPktSz;
uint16_t	RxCount;
	
#if	HOST_ENUM_MSG
//	printf("EP%dIn-",EPx);
#endif
	md_usb_set_epx_index(USB, EPx);			//Endpoint x
	MaxPktSz = (md_usb_get_rx_max_packet_size(USB) * 8);
#if	HOST_ENUM_MSG
//	printf("MaxPktSz=%d-", MaxPktSz);
#endif
	while (BCnt)
	{
		while(1)
		{
			md_usb_set_epx_index(USB, EPx);			//Endpoint x
			md_usb_enable_epx_rx_request_packet(USB);			//REQPKT
			while (!(USBIntrEvent & USB_INTR_EPxRX(EPx)));	//Wait for EPx Interrupt
			USBIntrEvent &= ~USB_INTR_EPxRX(EPx);
			if (md_usb_is_active_flag_epx_rx_ready(USB))		//RXPKTRDY
				break;
			if (md_usb_is_active_flag_epx_rx_stalled(USB) || md_usb_is_active_flag_epx_rx_error(USB))	//Error
				return(USBH_RESULT_FAIL);
//			if (md_usb_is_active_rxcsrl_nakto())		//Nak Timeout	
//				md_usb_clear_rxcsrl_nakto();
		}
		md_usb_set_epx_index(USB, EPx);			//Endpoint x
		RxCount = md_usb_get_epx_packet_count(USB);
#if	HOST_ENUM_MSG
//		printf("RxCount=%d-", RxCount);
#endif
		if ((RxCount < MaxPktSz) || (RxCount > BCnt))
			BCnt = 0;
		else 
			BCnt -= md_usb_get_epx_packet_count(USB);
		while (RxCount--)
			*pDataBuf++ = *pUSBEPFIFO(EPx);
		md_usb_clear_flag_epx_rx_ready(USB);		//Clear REQPKT
	}
	return(USBH_RESULT_PASS);
#if	HOST_ENUM_MSG
//	printf("Done\r\n");
#endif
}
/****
//!\Brief	USB Endpoint Out
//!\Param	EPx	: Endpoint Index
//!\Param	BCnt	: Byte Count to Write
//!\Param	*pDataBuf	: Data Buffer to Read
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_issue_epx_out(_pUSB_FS_HOST pHost, uint8_t EPx, uint16_t BCnt, uint8_t *pDataBuf)
{
uint16_t	MaxPktSz;
uint16_t	TxCount;

#if	HOST_ENUM_MSG
//	printf("EP%dOut-", EPx);
#endif
	md_usb_set_epx_index(USB, EPx);			//Endpoint x
	MaxPktSz = (md_usb_get_tx_max_packet_size(USB) * 8);
#if	HOST_ENUM_MSG
//	printf("MaxPktSz=%d-", EPxMaxPkt);
#endif
	while (BCnt)
	{
		if (BCnt >= MaxPktSz)
			TxCount = MaxPktSz;
		else
			TxCount = BCnt;
		BCnt -= TxCount;
		while (TxCount--)
			*pUSBEPFIFO(EPx) = *pDataBuf++;
		while(1)
		{
			md_usb_set_epx_index(USB, EPx);			//Endpoint x
			md_usb_enable_epx_tx_ready(USB);		//TXPKTRDY
			while (!(USBIntrEvent & USB_INTR_EPxTX(EPx)));			//Wait for EPx Interrupt
			USBIntrEvent &= ~USB_INTR_EPxTX(EPx);
			if (!md_usb_is_enabled_epx_tx_ready(USB))	//Data Sent
				break;
			if (md_usb_is_active_flag_epx_tx_stalled(USB) || md_usb_is_active_flag_epx_tx_error(USB))	//Error
				return(USBH_RESULT_FAIL);
//			if (md_usb_is_active_txcsrl_nakto())			//Nak Timeout
//				md_usb_clear_txcsrl_nakto();
		}
	}
	return(USBH_RESULT_PASS);
#if	HOST_ENUM_MSG
//	printf("Done\r\n");
#endif
}
/****
//!\Brief	Issue USB Setup
//!\Param	*pRequestData	: USB Request Data
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_issue_setup(_pUSB_FS_HOST pHost, uint8_t *pRequestData)
{
uint8_t	ii;

#if	HOST_ENUM_MSG
//	printf("Setup...");
#endif
	for (ii = 0; ii < sizeof(_USB_SETUP_REQUEST); ii++)
		md_usb_set_ep0_fifo(USB, *pRequestData++);

	md_usb_set_epx_index(USB, USBEP0);		//Endpoint 0
	md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_SETUPPKT_MSK | USB_CSR0L_TXRDY_MSK));	//SETUPPKT+TXPKTRDY
	while (!(USBIntrEvent & USB_INTR_EP0));		//Wait for EP0 Interrupt
	USBIntrEvent &= ~USB_INTR_EP0;
	if (md_usb_is_active_flag_ep0_stalled(USB) || md_usb_is_active_flag_ep0_error(USB))	//Error
	{
		printf("Error\r\n");
		return(USBH_RESULT_FAIL);
	}
#if	HOST_ENUM_MSG
//	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Issue USB Control In
//!\Param	BCnt	: Byte Count to Write
//!\Param	*pDataBuf	: Data Buffer to Read
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_issue_ctrl_in(_pUSB_FS_HOST pHost, uint16_t BCnt, uint8_t *pDataBuf)
{
uint16_t	RxCount;
	
#if	HOST_ENUM_MSG
//	printf("CtrlIn...");
#endif
	while (BCnt)
	{
		while(1)
		{
			md_usb_set_epx_index(USB, USBEP0);		//Endpoint 0
			md_usb_enable_ep0_request_packet(USB);			//REQPKT
			while (!(USBIntrEvent & USB_INTR_EP0));		//Wait for EP0 Interrupt
			USBIntrEvent &= ~USB_INTR_EP0;
			if (md_usb_is_active_flag_host_ep0_rx_ready(USB))	//RxPktRdy
				break;
			if (md_usb_is_active_flag_ep0_stalled(USB) || md_usb_is_active_flag_ep0_error(USB))	//Error
			{
				printf("Error\r\n");
				return(USBH_RESULT_FAIL);
			}
//			if (md_usb_is_active_csr0l_nakto())			//Nak Timeout
//				md_usb_clear_csr0l_nakto();
		}
		RxCount = md_usb_get_ep0_data_count(USB);
		if (RxCount == 0)			//Zero Packet Length
		{
			md_usb_clear_flag_host_ep0_rx_ready(USB);	//Clear RXPKTRDY
			break;
		}
		if ((RxCount < pHost->MaxCtrlPacketSize) || (RxCount > BCnt))
		{
			BCnt = 0;
		}
		else
		{
			if (BCnt != pHost->MaxCtrlPacketSize)
				BCnt -= RxCount;
		}
		while (RxCount--)
			*pDataBuf++ = md_usb_get_ep0_fifo(USB);;
		md_usb_clear_flag_host_ep0_rx_ready(USB);	//Clear RXPKTRDY
	}
#if	HOST_ENUM_MSG
//	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Issue USB Control Out
//!\Param	BCnt	: Byte Count to Write
//!\Param	*pDataBuf	: Data Buffer to Write
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_issue_ctrl_out(_pUSB_FS_HOST pHost, uint16_t BCnt, uint8_t *pDataBuf)
{
uint8_t	len;

#if	HOST_ENUM_MSG
//	printf("CtrlOut...");
#endif
	while (BCnt)
	{
		if (BCnt >= pHost->MaxCtrlPacketSize)
			len = pHost->MaxCtrlPacketSize;
		else
			len = BCnt;
		BCnt -= len;
		while (len--)
			md_usb_set_ep0_fifo(USB, *pDataBuf++);
		while(1)
		{
			md_usb_set_epx_index(USB, USBEP0);		//Endpoint 0
			md_usb_enable_ep0_tx_ready(USB );			//TXPKTRDY
			while (!(USBIntrEvent & USB_INTR_EP0));	//Wait for EP0 Interrupt
			USBIntrEvent &= ~USB_INTR_EP0;
			if (!md_usb_is_enabled_ep0_tx_ready(USB))	//Data Sent
				break;
			if (md_usb_is_active_flag_ep0_stalled(USB) || md_usb_is_active_flag_ep0_error(USB))	//Error
			{
				printf("Error\r\n");
				return(USBH_RESULT_FAIL);
			}
//			if (md_usb_is_active_csr0l_nakto())			//Nak Timeout
//				md_usb_clear_csr0l_nakto();
		}
	}
#if	HOST_ENUM_MSG
//	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Issue USB Status In
//!\Param	None
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_issue_status_in(_pUSB_FS_HOST pHost)
{
#if	HOST_ENUM_MSG
//	printf("StsIn...");
#endif
	md_usb_set_epx_index(USB, USBEP0);		//Endpoint 0
	md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_STATUSPKT_MSK | USB_CSR0L_REQPKT_MSK));	//STATUSPKT+REQPKT
	while (!(USBIntrEvent & USB_INTR_EP0));		//Wait for EP0 Interrupt
	USBIntrEvent &= ~USB_INTR_EP0;
	while (!md_usb_is_active_flag_host_ep0_rx_ready(USB));	//Rx Packet Ready
	md_usb_clear_flag_host_ep0_rx_ready(USB);			//Clear RXPKTRDY	
#if	HOST_ENUM_MSG
//	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Issue USB Status Out
//!\Param	None
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_issue_status_out(_pUSB_FS_HOST pHost)
{
#if	HOST_ENUM_MSG
//	printf("StsOut...");
#endif
	md_usb_set_epx_index(USB, USBEP0);		//Endpoint 0
	md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_STATUSPKT_MSK | USB_CSR0L_TXRDY_MSK));	//STATUSPKT+TXPKTRDY
	while (!(USBIntrEvent & USB_INTR_EP0));		//Wait for EP0 Interrupt
	USBIntrEvent &= ~USB_INTR_EP0;
#if	HOST_ENUM_MSG
//	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	USB Standard Request
//!\		Issue Setup Phase, Data In/Out Phase (if There is Data Phase) and Ststua In/Out
//!\Param	*pSetupReq	: USB Request Data
//!\Param	*pDataBuf	: Data Buffer for Data In/Out Phase
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_issue_request(_pUSB_FS_HOST pHost, _pUSB_SETUP_REQUEST pSetupReq, uint8_t *pDataBuf)
{
#if	HOST_ENUM_MSG
	printf("USB Request : %d\r\n", pSetupReq->bRequest);
#endif
	if (usbh_issue_setup(pHost, (uint8_t *)pSetupReq))
		return(USBH_RESULT_FAIL);
	if (pSetupReq->wLength == 0)		//No Data Phase
	{
		if (usbh_issue_status_in(pHost))
			return(USBH_RESULT_FAIL);
		return(USBH_RESULT_PASS);
	}
	if (pSetupReq->bmRequestType & 0x80)	//Data In Pahse
	{
		if (usbh_issue_ctrl_in(pHost, pSetupReq->wLength, pDataBuf))
			return(USBH_RESULT_FAIL);
		if (usbh_issue_status_out(pHost))
			return(USBH_RESULT_FAIL);
	}
	else					//Data Out Pahse
	{
		if (usbh_issue_ctrl_out(pHost, pSetupReq->wLength, pDataBuf))
			return(USBH_RESULT_FAIL);
		if (usbh_issue_status_in(pHost))
			return(USBH_RESULT_FAIL);
	}
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Get Descriptor on Addressed Phase
//!\Param	DescType	: Descriptor Type
//!\Param	DescIndex	: Descriptor Index
//!\Param	BCnt		: Byte Count to Get
//!\Param	*pDescBuf	: Data Buffer for Descriptor Data In
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_get_descriptor(_pUSB_FS_HOST pHost, uint8_t DescType, uint8_t DescIndex, uint16_t BCnt, uint8_t *pDescBuf)
{
#if	HOST_ENUM_MSG
	printf("GetDescsciptor : %d\r\n",DescType);
#endif

	SetupReq.bmRequestType = (USB_REQ_TYPE_DIR_IN + USB_REQ_TYPE_DEVICE);		//0x80
	SetupReq.bRequest	= USB_REQ_GET_DESCRIPTOR;	//GETDESCRIPTOR
	SetupReq.wValue	= ((DescType << 8) | DescIndex);	//Type/Index
	SetupReq.wIndex	= 0;							//Index
	SetupReq.wLength	= BCnt;					//Length
	
	if (usbh_issue_request(pHost, &SetupReq, pDescBuf))
		return(USBH_RESULT_FAIL);
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Get Interface
//!\Param	Interface	: Interface Number
//!\Param	*pAlternate	: Pointer to alternate value
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_get_interface(_pUSB_FS_HOST pHost, uint8_t Interface, uint8_t *pAlternate)
{
#if	HOST_ENUM_MSG
	printf("GetInterface : %d\r\n",Interface);
#endif

	SetupReq.bmRequestType = (USB_REQ_TYPE_DIR_IN + USB_REQ_TYPE_INTERFACE);	//0x81
	SetupReq.bRequest	= USB_REQ_GET_INTERFACE;	//GETINTERFACE
	SetupReq.wValue	= 0;				//
	SetupReq.wIndex	= Interface;		//Interface
	SetupReq.wLength	= 1;				//Length
	
	if (usbh_issue_request(pHost, &SetupReq, pAlternate))
		return(USBH_RESULT_FAIL);
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Get Configuration
//!\Param	*pConfig	: Pointer to configuration value
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_get_configuration(_pUSB_FS_HOST pHost, uint8_t *pConfig)
{
#if	HOST_ENUM_MSG
	printf("GetConfiguration\r\n");
#endif

	SetupReq.bmRequestType = (USB_REQ_TYPE_DIR_IN + USB_REQ_TYPE_DEVICE);		//0x80
	SetupReq.bRequest	= USB_REQ_GET_CONFIGURATION;	//GETCONFIGURATION
	SetupReq.wValue	= 0;				//
	SetupReq.wIndex	= 0;				//
	SetupReq.wLength	= 1;				//Length
	
	if (usbh_issue_request(pHost, &SetupReq, pConfig))
		return(USBH_RESULT_FAIL);
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Set Address
//!\Param	Addr	: Device Address to Set
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_set_address(_pUSB_FS_HOST pHost, uint8_t Addr)
{
#if	HOST_ENUM_MSG
	printf("SetAddress	: %d\r\n",Addr);
#endif

	SetupReq.bmRequestType = USB_REQ_TYPE_DEVICE;		//0x00
	SetupReq.bRequest	= USB_REQ_SET_ADDRESS;	//SETADDRESS
	SetupReq.wValue	= Addr;				//Address
	SetupReq.wIndex	= 0;					//Index
	SetupReq.wLength	= 0;					//Length
	
	if (usbh_issue_request(pHost, &SetupReq, NULL))
		return(USBH_RESULT_FAIL);
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Set Configuration
//!\Param	Config	: Config Value to Set
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_set_configuration(_pUSB_FS_HOST pHost, uint8_t Config)
{
#if	HOST_ENUM_MSG
	printf("SetConfiguration : %d\r\n",Config);
#endif

	SetupReq.bmRequestType = USB_REQ_TYPE_DEVICE;		//0x00
	SetupReq.bRequest	= USB_REQ_SET_CONFIGURATION;	//SETCONFIGURATION
	SetupReq.wValue	= Config;			//Config Value
	SetupReq.wIndex	= 0;				//Index
	SetupReq.wLength	= 0;				//Length
	
	if (usbh_issue_request(pHost, &SetupReq, NULL))
		return(USBH_RESULT_FAIL);
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Set Alternate Interface
//!\Param	Index		: Interface Index
//!\Param	Alternate	: Alternate Value to Set
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_set_interface(_pUSB_FS_HOST pHost, uint8_t Index, uint8_t Alternate)
{
#if	HOST_ENUM_MSG
	printf("SetInterface : %d/%d\r\n", Index, Alternate);
#endif

	SetupReq.bmRequestType = USB_REQ_TYPE_INTERFACE;		//0x01
	SetupReq.bRequest	= USB_REQ_SET_INTERFACE;	//SETINTERFACE
	SetupReq.wValue	= Alternate;				//Alternate
	SetupReq.wIndex	= Index;					//Index
	SetupReq.wLength	= 0;						//Length
	
	if (usbh_issue_request(pHost, &SetupReq, NULL))
		return(USBH_RESULT_FAIL);
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Clear Feature
//!\Param	Selector	: Feature Selector
//!\Param	Index		: Interface/Endpoint Index
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_clear_feature(_pUSB_FS_HOST pHost, uint8_t Selector, uint8_t Index)		//[00 01 02] 01 XX 00 XX 00 00 00
{
#if	HOST_ENUM_MSG
	printf("ClearFeature : %d/%d\r\n", Selector, Index);
#endif

	if (Selector == USB_FEATURE_REMOTE_WAKE)
		SetupReq.bmRequestType = USB_REQ_TYPE_DEVICE;		//0x00
	if (Selector == USB_FEATURE_EP_HALT)
		SetupReq.bmRequestType = USB_REQ_TYPE_ENDPOINT;	//0x02
	SetupReq.bRequest	= USB_REQ_CLEAR_FEATURE;	//CLEARFEATURE
	SetupReq.wValue	= Selector;				//Selector
	SetupReq.wIndex	= Index;					//Index
	SetupReq.wLength	= 0;						//Length
	
	if (usbh_issue_request(pHost, &SetupReq, NULL))
		return(USBH_RESULT_FAIL);
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Wait for USB Device Plug In
//!\		Issue Reset and Enable Disconnect and Resume Interrupt
//!\Param	None
//!\RetVal	0x0 : Connected, 0x1 : Not Connected
****/
eUSBHResult	usbh_dev_connect(_pUSB_FS_HOST pHost)
{
	if (md_usb_is_active_it_disconnect(USB))		//Device Not Connected
	{
		md_usb_clear_it_disconnect(USB);
		pHost->DevConnected = 0;
		return(USBH_RESULT_FAIL);
	}
	if (pHost->DevConnected == 0)
	{
		if (md_usb_is_active_it_connection(USB))	//Device Connected
		{
			md_usb_clear_it_connection(USB);
			pHost->DevConnected = 1;
		}
		else
			return(USBH_RESULT_FAIL);
	}

	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Issue Reset Signal 20mS to Devie
//!\Param	None
//!\RetVal	eUSBHResult
****/
eUSBHResult	usbh_issue_reset(_pUSB_FS_HOST pHost)
{
	md_tick_waitms(10,2);		//Reset 20mS
	md_usb_enable_reset(USB);
	md_tick_waitms(10,2);		//Reset 20mS
	md_usb_disable_reset(USB);
	md_tick_waitms(10,2);		//Wait 20mS

	md_usb_disable_it_connection(USB);	//Interrupt Enable for Disconnection
	md_usb_enable_it_disconnect(USB);	//Interrupt Enable for Disconnection
	return(USBH_RESULT_PASS);
}
/****
//!\Brief	Dump Descriptor Data
//!\Param	BCnt	: Byte Count to Dump
//!\Param	*pDataBuf	: Data Buffer to Dump
//!\RetVal	None
****/
void	usbh_dump_descriptor(_pUSB_FS_HOST pHost, uint16_t BCnt, uint8_t *pData)
{
uint8_t	Length, DescriptorType;
uint8_t	ii;
	
	while (BCnt)
	{
		Length = *pData++;			//Length
		DescriptorType = *pData++;	//DescriptotType
		printf("Length=0x%02x\r\n",Length);
		printf("Type=0x%02x\r\n",DescriptorType);
		if (Length == 0x0)
			return;
		printf("Data=\r\n");
		for (ii = 0; ii < (Length - 2); ii++)
		{
			printf("0x%02x ",*pData++);
			if ((ii & 0x7) == 0x7)
				printf("\r\n");
		}
		printf("\r\n");
		BCnt -= Length;
	}
}
/****
//!\Brief	Enumeration for USB Device
//!\		Retrive Needed Data for USB Host
//!\Param	*pEnumBuf	: Data Buffer for Enumeration
//!\RetVal	0x00=Pass, 0x1=Fail
****/
eUSBHResult	usbh_enumeration(_pUSB_FS_HOST pHost, uint8_t *pEnumData)
{
_pUSB_DEVICE_DESC			pDeviceDesc;
_pUSB_CONFIGURATION_DESC	pConfigurationDesc;
uint8_t	ii;

	pHost->MaxCtrlPacketSize = 0x40;

	md_usb_set_device_address(USB, DEFAULT_ADDRESS);		//Default Address	
//Addressing Phase
	if (usbh_get_descriptor(pHost, USB_DTYPE_DEVICE, 0, pHost->MaxCtrlPacketSize, pEnumData))
		return(USBH_RESULT_FAIL);
	pDeviceDesc = (_pUSB_DEVICE_DESC)pEnumData;
#if	HOST_ENUM_MSG
	usbh_dump_descriptor(pHost, sizeof(_USB_DEVICE_DESC), pEnumData);
#endif
	if (pDeviceDesc->bLength != sizeof(_USB_DEVICE_DESC))	//bLength Mismatch
	{
		printf("Not Legel Device Descriptor Length\r\n");
		return(USBH_RESULT_FAIL);
	}
	if (pDeviceDesc->bDescriptorType != USB_DTYPE_DEVICE)	//bDescriptorType Mismatch
	{
		printf("Not Legel Device Descriptor Type\r\n");
		return(USBH_RESULT_FAIL);
	}
	pHost->MaxCtrlPacketSize = pDeviceDesc->bMaxPacketSize0;
	if (usbh_set_address(pHost, DEVICE_ADDRESS))
		return(USBH_RESULT_FAIL);
	md_tick_waitms(5,1);			//Wait 5mS
	md_usb_set_device_address(USB, DEVICE_ADDRESS);	//Device Address
//Descriptor Parser
	if (usbh_get_descriptor(pHost, USB_DTYPE_DEVICE, 0, sizeof(_USB_DEVICE_DESC), pEnumData))
		return(USBH_RESULT_FAIL);
#if	HOST_ENUM_MSG
	usbh_dump_descriptor(pHost, sizeof(_USB_DEVICE_DESC), pEnumData);
#endif
	pDeviceDesc=(_pUSB_DEVICE_DESC)pEnumData;
#if	HOST_ENUM_MSG
	printf("VID/PID=0x%04x/0x%04x\r\n", pDeviceDesc->idVendor, pDeviceDesc->idProduct);
#endif
	pEnumData += sizeof(_USB_DEVICE_DESC);
	for (ii = 0; ii < pDeviceDesc->bNumConfigurations; ii++)
	{
		if (usbh_get_descriptor(pHost, USB_DTYPE_CONFIGURATION, ii, sizeof(_USB_CONFIGURATION_DESC), pEnumData))
			return(USBH_RESULT_FAIL);
#if	HOST_ENUM_MSG
		usbh_dump_descriptor(pHost, sizeof(_USB_CONFIGURATION_DESC), pEnumData);
#endif
		pConfigurationDesc = (_pUSB_CONFIGURATION_DESC)pEnumData;
		if (pConfigurationDesc->bLength != sizeof(_USB_CONFIGURATION_DESC))		//bLength Mismatch
		{
			printf("Not Legel Configuration Descriptor Length\r\n");
			return(USBH_RESULT_FAIL);
		}
		if (pConfigurationDesc->bDescriptorType != USB_DTYPE_CONFIGURATION)		//bDescriptorType Mismatch
		{
			printf("Not Legel Configuration Descriptor Type\r\n");
			return(USBH_RESULT_FAIL);
		}
		if (usbh_get_descriptor(pHost, USB_DTYPE_CONFIGURATION, ii, pConfigurationDesc->wTotalLength, pEnumData))
			return(USBH_RESULT_FAIL);
#if	HOST_ENUM_MSG
		usbh_dump_descriptor(pHost, pConfigurationDesc->wTotalLength, pEnumData);
#endif
		pEnumData += pConfigurationDesc->wTotalLength;
	}
	return(USBH_RESULT_PASS);
}

