/**********************************************************************************
 *
 * @file    usb_dev_enum.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_usb.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include <stdio.h>

#pragma pack(1)

#define  DEV_ENUM_MSG  1

static  uint8_t    /*bEPIdle = 0,*/ bEPHalt = 0, bDevRWakeUp = 0;
static  uint8_t    bAlternate = 0;

/****
  * @brief  USB Endpoint Out Available
  * @param  pDev  : Device Instance
  * @param  EPx  : Endpoint Index
  * @retval  Rx Byte Count
****/
uint16_t  usbd_epx_out_available(_pUSB_FS_DEVICE pDev, uint8_t EPx)
{
  md_usb_set_epx_index(USB, EPx);      //Endpoint x
  if (EPx == USBEP0)
    return(md_usb_get_ep0_data_count(USB));
  else
    return(md_usb_get_epx_packet_count(USB));
}
/****
  * @brief  USB Endpoint In
  * @param  pDev  : Device Instance
  * @param  EPx    : Endpoint Index
  * @param  BCnt  : Byte Count to Send
  * @param  *pBuf  : Data Buffer to Write
  * @retval  Idle State
****/
eUSBDState  usbd_handle_epx_in(_pUSB_FS_DEVICE pDev, uint8_t EPx, uint16_t BCnt, uint8_t *pBuf)
{
  md_usb_set_epx_index(USB, EPx);      //Endpoint x
  while (md_usb_is_enabled_epx_tx_ready(USB));  //Wait TxRdy=0
  while (BCnt--)
    *pUSBEPFIFO(EPx) = *pBuf++;
  md_usb_set_epx_index(USB, EPx);      //Endpoint x
  md_usb_enable_epx_tx_ready(USB);    //TxRdy
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Endpoint Out
  * @param  pDev  : Device Instance
  * @param  EPx  : Endpoint Index
  * @param  BCnt  : Byte Count to Receive
  * @param  *pBuf  : Data Buffer to Write
  * @retval  Idle State
****/
eUSBDState  usbd_handle_epx_out(_pUSB_FS_DEVICE pDev, uint8_t EPx, uint16_t BCnt, uint8_t *pBuf)
{
  while (BCnt--)
    *pBuf++ = *pUSBEPFIFO(EPx);
  md_usb_set_epx_index(USB, EPx);      //Endpoint x
  md_usb_clear_flag_epx_rx_ready(USB);    //Clear RxRdy
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Device Control Out
  * @param  pDev  : Device Instance
  * @retval  Control Out State if There are More Packets, Idle State if There is No Packets
****/
eUSBDState  usbd_handle_ctrl_out(_pUSB_FS_DEVICE pDev)
{
uint8_t  Len;
  
  if (pDev->CtrlBurstRemain >= USBD_CTRL_PKTSZ)    //More Packets to Be Transferred
  {
    Len = USBD_CTRL_PKTSZ;
    while (Len--)
      *pDev->pCtrlBurstData++ = USBD->EP0FIFO;
    pDev->CtrlBurstRemain -= USBD_CTRL_PKTSZ;
    md_usb_set_epx_index(USB, USBEP0);    //Endpoint 0
    md_usb_set_csr0l_txcsrl(USB, USB_CSR0L_RXRDYC);  //Clear RxRdy
    while(!(USBIntrEvent & USB_INTR_EP0));
    return(USBD_STATE_CTRLOUT);
  }
  else  //The Last Data Packet or Zero Data Packet
  {
    Len = pDev->CtrlBurstRemain;
    while (Len--)
      *pDev->pCtrlBurstData++ = USBD->EP0FIFO;
    pDev->CtrlBurstRemain = 0;
    md_usb_set_epx_index(USB, USBEP0);    //Endpoint 0
    md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy, DataEnd
    return(USBD_STATE_IDLE);
  }
}
/****
  * @brief  USB Device Control In
  * @param  pDev  : Device Instance
  * @retval  Control In State if There are More Packets, Idle State if There is No Packets
****/
eUSBDState  usbd_handle_ctrl_in(_pUSB_FS_DEVICE pDev)
{
uint8_t  Dlen;
  
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  while (md_usb_is_enabled_ep0_tx_ready(USB));  //Wait TxRdy=0
  if (pDev->CtrlBurstRemain >= USBD_CTRL_PKTSZ)  //More Packets to Be Transferred
  {
    Dlen = USBD_CTRL_PKTSZ;
    while (Dlen--)
      md_usb_set_ep0_fifo(USB, *pDev->pCtrlBurstData++);
    pDev->CtrlBurstRemain -= USBD_CTRL_PKTSZ;
    md_usb_set_epx_index(USB, USBEP0);    //Endpoint 0
    md_usb_enable_ep0_tx_ready(USB);  //TxRdy
    while(!(USBIntrEvent & USB_INTR_EP0));
    return(USBD_STATE_CTRLIN);
  }
  else  //The Last Data Packet or Zero Data Packet
  {
    Dlen = pDev->CtrlBurstRemain;
    while (Dlen--)
      md_usb_set_ep0_fifo(USB, *pDev->pCtrlBurstData++);
    pDev->CtrlBurstRemain = 0;
    md_usb_set_epx_index(USB, USBEP0);    //Endpoint 0
    md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_TXRDY | USB_CSR0L_DATAEND));  //TxRdy & DataEnd
    return(USBD_STATE_IDLE);
  }
}
/****
  * @brief  USB Standard Request - Clear Feature
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_clear_feature(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)    //[00 01 02] 01 XX 00 XX 00 00 00
{
#if  DEV_ENUM_MSG
  printf("ClearFeature\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
  switch (pSetupReq->bmRequestType)
  {
    case USB_FEATURE_TYPE_ENDPOINT:    //Endpoint
      if (pSetupReq->wValue == USB_FEATURE_EP_HALT)
        bEPHalt = 0;
      break;
    case USB_FEATURE_TYPE_DEVICE:    //Device
      if (pSetupReq->wValue == USB_FEATURE_REMOTE_WAKE)
        bDevRWakeUp = 0;
    case USB_FEATURE_TYPE_INTERFACE:  //Interface
    default:
      break;
  }
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Get Configuration
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_get_configuration(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)  //80 08 00 00 00 00 01 00
{
#if  DEV_ENUM_MSG
  printf("GetConfiguration\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);        //Endpoint 0
  md_usb_clear_flag_device_ep0_rx_ready(USB);      //Clear RxRdy
  md_usb_set_ep0_fifo(USB, pDev->ConfigValue);  //Config Value
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_TXRDY | USB_CSR0L_DATAEND));  //TxRdy & DataEnd
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Get Descriptor
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Control In State if There are More Packets, Idle State if There is No Packets
****/
eUSBDState  usbd_request_get_descriptor(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)    //80 06 XX XX 00 00 XX XX
{
eUSBDState  USBState;

#if  DEV_ENUM_MSG
  printf("GetDescriptor 0x%x\r\n",pSetupReq->wValue);
#endif
  if (((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_enum != NULL)
  {
    USBState = (*(((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_enum))(pDev, pSetupReq);
    if (USBState == USBD_STATE_STALL)
    {
      md_usb_set_epx_index(USB, USBEP0);    //Endpoint 0
      md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_STALL));  //Clear RxRdy & Stall
      return(USBD_STATE_IDLE);
    }
  }
  if (pDev->CtrlBurstRemain > pSetupReq->wLength)
    pDev->CtrlBurstRemain = pSetupReq->wLength;
  md_usb_set_epx_index(USB, USBEP0);        //Endpoint 0
  md_usb_clear_flag_device_ep0_rx_ready(USB);      //Clear RxRdy
  USBState = usbd_handle_ctrl_in(pDev);
  return(USBState);
}
/****
  * @brief  USB Standard Request - Get Interface
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_get_interface(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)    //81 0A 00 00 XX 00 01 00
{
#if  DEV_ENUM_MSG
  printf("GetInterface\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_clear_flag_device_ep0_rx_ready(USB);    //Clear RxRdy
  md_usb_set_ep0_fifo(USB, bAlternate);    //Alternayte Interface Number
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_TXRDY | USB_CSR0L_DATAEND));  //TxRdy & DataEnd
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Get Status
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_get_status(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)      //[80 81 82] 00 00 00 XX 00 02 00
{  
#if  DEV_ENUM_MSG
  printf("GetStatus\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_clear_flag_device_ep0_rx_ready(USB);    //Clear RxRdy
  switch (pSetupReq->bmRequestType)
  {
    case USB_STATUS_TYPE_ENDPOINT:    //Endpoint
      if (pSetupReq->wValue == USB_FEATURE_EP_HALT)
        md_usb_set_ep0_fifo(USB, bEPHalt);  //EP Halt
      else
        md_usb_set_ep0_fifo(USB, 0);      //None
      break;
    case USB_STATUS_TYPE_DEVICE:    //Device
      if (pSetupReq->wValue == USB_FEATURE_REMOTE_WAKE)
        md_usb_set_ep0_fifo(USB, bDevRWakeUp);      //Remote Wakeup
      else
        md_usb_set_ep0_fifo(USB, 0);    //None
      break;
    case USB_STATUS_TYPE_INTERFACE:    //Interface
    default:
      md_usb_set_ep0_fifo(USB, 0);      //None
      break;
  }
  md_usb_set_ep0_fifo(USB, 0);  //
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_TXRDY | USB_CSR0L_DATAEND));  //TxRdy & DataEnd
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Set Address
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_set_address(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)    //00 05 XX 00 00 00 00 00
{
#if  DEV_ENUM_MSG
  printf("SetAddress\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
  while (!(USBIntrEvent & USB_INTR_EP0));
  USBIntrEvent &= ~USB_INTR_EP0;
  md_usb_set_device_address(USB, pSetupReq->wValue);  //Device Address  
  if (((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_enum != NULL)
    (*(((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_enum))(pDev, pSetupReq);
#if  DEV_ENUM_MSG
  printf("DevAddr  : %d\r\n",pSetupReq->wValue);
#endif
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Set Configuration
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_set_configuration(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)  //00 09 XX 00 00 00 00 00
{
#if  DEV_ENUM_MSG
  printf("SetConfiguration\r\n");
#endif
  pDev->ConfigValue = (pSetupReq->wValue & 0xff);  //Config Value
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
  if (((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_enum != NULL)
    (*(((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_enum))(pDev, pSetupReq);
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Set Descriptor
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_set_descriptor(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)    //00 07 00 XX 00 00 XX XX
{
#if  DEV_ENUM_MSG
  printf("SetDescriptor\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_STALL));  //Clear RxRdy & Stall
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Set Feature
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_set_feature(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)    //[00 01 02] 03 XX 00 XX 00 00 00
{
#if  DEV_ENUM_MSG
  printf("SetFeature\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
  switch (pSetupReq->bmRequestType)
  {
    case USB_FEATURE_TYPE_ENDPOINT:    //Endpoint
      if (pSetupReq->wValue == USB_FEATURE_EP_HALT)
        bEPHalt = 1;
      break;
    case USB_FEATURE_TYPE_DEVICE:    //Device
      if (pSetupReq->wValue == USB_FEATURE_REMOTE_WAKE)
        bDevRWakeUp = 1;
      break;
    case USB_FEATURE_TYPE_INTERFACE:  //Interface
    default:
      break;
  }
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Set Interface
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_set_interface(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)    //01 0B XX 00 XX 00 00 00
{
#if  DEV_ENUM_MSG
  printf("SetInterface\r\n");
#endif
  bAlternate = (pSetupReq->wValue & 0xff);  //Alternate Value
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
  if (((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_enum != NULL)
    (*(((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_enum))(pDev, pSetupReq);
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Sync Frame
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_sync_frame(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)      //82 0C 00 00 XX 00 02 00
{
#if  DEV_ENUM_MSG
  printf("SyncFrame\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_STALL));  //Clear RxRdy & Stall
  return(USBD_STATE_IDLE);
}
/****
  * @brief  USB Standard Request - Reserved
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  usbd_request_reserved(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
#if  DEV_ENUM_MSG
  printf("Reserved\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
  return(USBD_STATE_IDLE);
}
/****
  * @brief  Functionality Array for USB Standard Request
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
****/
eUSBDState  (*pUSBDStandardRequest[])(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST) =      //Standard Device Request
{
  usbd_request_get_status,
  usbd_request_clear_feature,
  usbd_request_reserved,
  usbd_request_set_feature,
  usbd_request_reserved,
  usbd_request_set_address,
  usbd_request_get_descriptor,
  usbd_request_set_descriptor,
  usbd_request_get_configuration,
  usbd_request_set_configuration,
  usbd_request_get_interface,
  usbd_request_set_interface,
  usbd_request_sync_frame,
};
/****
  * @brief  USB Enumeration State
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @param  USBState  : Current Enumeration State
  * @retval  Updated Enumeration State
****/
eUSBDState  usbd_dev_enum(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq, eUSBDState USBState)
{
uint8_t  ii;

  if (USBIntrEvent == 0)
    return(USBState);
    __set_PRIMASK(1);
  if (USBIntrEvent & USB_INTR_RESET)    //USB Reset
  {
    USBIntrEvent &= ~USB_INTR_RESET;
    __set_PRIMASK(0);
    return(USBD_STATE_RESET);
  }
  if (USBIntrEvent & USB_INTR_RESUME)    //USB Resume
  {
    USBIntrEvent &= ~USB_INTR_RESUME;
    __set_PRIMASK(0);
    return(USBD_STATE_RESUME);
  }
  if (USBIntrEvent & USB_INTR_SUSPEND)  //USB Suspend
  {
    USBIntrEvent &= ~USB_INTR_SUSPEND;
    __set_PRIMASK(0);
    return(USBD_STATE_SUSPEND);
  }
  if (USBIntrEvent & USB_INTR_EP0)    //EP0, Setup or Ctrl In/Out
  {
    USBIntrEvent &= ~USB_INTR_EP0;
    __set_PRIMASK(0);
    md_usb_set_epx_index(USB, USBEP0);    //Endpoint 0
    if (md_usb_is_active_flag_device_ep0_rx_ready(USB))    //Setup or Ctrl Out
    {
      switch (USBState)
      {
        case USBD_STATE_IDLE:
#if  DEV_ENUM_MSG
          printf("Setup\r\n");
#endif
          USBState = USBD_STATE_SETUP;
          for (ii = 0; ii < sizeof(_USB_SETUP_REQUEST); ii++)
          {
            *((uint8_t *)pSetupReq + ii) = md_usb_get_ep0_fifo(USB);
#if  DEV_ENUM_MSG
            printf("0x%02x ",*((uint8_t *)pSetupReq + ii));
#endif
          }
#if  DEV_ENUM_MSG
          printf("\r\n");
#endif
          if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) == USB_REQ_TYPE_STANDARD)  //bmRequestType=Standard
            USBState = (*pUSBDStandardRequest[pSetupReq->bRequest])(pDev, pSetupReq);
          else
            USBState = USBD_STATE_SETUP;
          return(USBState);
        case USBD_STATE_SETUP:
        case USBD_STATE_CTRLOUT:
#if  DEV_ENUM_MSG
//          printf("CtrlOut\r\n");
#endif
          return(USBD_STATE_CTRLOUT);
        default:
          return(USBD_STATE_IDLE);
      }
    }
    else
    {
      switch (USBState)
      {
        case USBD_STATE_CTRLIN:  
#if  DEV_ENUM_MSG
          printf("CtrlIn\r\n");
#endif
          USBState = usbd_handle_ctrl_in(pDev);
          return(USBState);
        default:
          return(USBD_STATE_IDLE);
      }
    }
  }
  if (USBIntrEvent & USB_INTR_EP1TX)  //EP1 TX
  {
#if  DEV_ENUM_MSG
//    printf("EP1TX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP1TX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP1IN);
  }
  if (USBIntrEvent & USB_INTR_EP1RX)  //EP1 RX
  {
#if  DEV_ENUM_MSG
//    printf("EP1RX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP1RX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP1OUT);
  }
  if (USBIntrEvent & USB_INTR_EP2TX)  //EP2 TX
  {
#if  DEV_ENUM_MSG
//    printf("EP2TX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP2TX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP2IN);
  }
  if (USBIntrEvent & USB_INTR_EP2RX)  //EP2 RX
  {
#if  DEV_ENUM_MSG
//    printf("EP2RX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP2RX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP2OUT);
  }
  if (USBIntrEvent & USB_INTR_EP3TX)  //EP3 TX
  {
#if  DEV_ENUM_MSG
//    printf("EP3TX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP3TX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP3IN);
  }
  if (USBIntrEvent & USB_INTR_EP3RX)  //EP3 RX
  {
#if  DEV_ENUM_MSG
//    printf("EP3RX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP3RX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP3OUT);
  }
  if (USBIntrEvent & USB_INTR_EP4TX)  //EP4 TX
  {
#if  DEV_ENUM_MSG
//    printf("EP4TX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP4TX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP4IN);
  }
  if (USBIntrEvent & USB_INTR_EP4RX)  //EP4 RX
  {
#if  DEV_ENUM_MSG
//    printf("EP4RX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP4RX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP4OUT);
  }
  if (USBIntrEvent & USB_INTR_EP5TX)  //EP5 TX
  {
#if  DEV_ENUM_MSG
//    printf("EP5TX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP5TX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP5IN);
  }
  if (USBIntrEvent & USB_INTR_EP5RX)  //EP5 RX
  {
#if  DEV_ENUM_MSG
//    printf("EP5RX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP5RX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP5OUT);
  }
  if (USBIntrEvent & USB_INTR_EP6TX)  //EP6 TX
  {
#if  DEV_ENUM_MSG
//    printf("EP6TX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP6TX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP6IN);
  }
  if (USBIntrEvent & USB_INTR_EP6RX)  //EP6 RX
  {
#if  DEV_ENUM_MSG
//    printf("EP6RX\r\n");
#endif
    USBIntrEvent &= ~USB_INTR_EP6RX;
    __set_PRIMASK(0);
    return(USBD_STATE_EP6OUT);
  }
  __set_PRIMASK(0);
  return(USBD_STATE_IDLE);
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/

