/**********************************************************************************
 *
 * @file    usb_core.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_usb.h"
#include "usb_core.h"
#include <stdio.h>

#define  CORE_MSG  0

void  (*cb_app_usb_handler)(void) = NULL;
static  uint8_t  BackupEPIdx;

volatile  _USB_SETUP_REQUEST  SetupReq;
volatile  uint32_t  USBIntrEvent = 0;

/****
  * @brief  USB IRQ Handler
  * @param  None
  * @retval  None
****/
void USB_IRQHandler(void)
{
  BackupEPIdx = md_usb_get_epx_index(USB);

  USBIntrEvent |= (md_usb_get_ifm(USB) << 16);
  USBIntrEvent |= (md_usb_get_txifm(USB) << 8);
  USBIntrEvent |= md_usb_get_rxifm(USB);
  
  md_usb_set_icr(USB, ((USBIntrEvent >> 16) & 0xff));
  md_usb_set_txicr(USB, ((USBIntrEvent >> 8) & 0xff));
  md_usb_set_rxicr(USB, ((USBIntrEvent >> 0) & 0xff));
  
  if (cb_app_usb_handler != NULL)
    (*cb_app_usb_handler)();
  
  md_usb_set_epx_index(USB, BackupEPIdx);
}
/****
  * @brief  Allows dual mode application to switch between USB device and host modes
  *      and provides a method to force the controller into the desired mode.
  * @param  iUSBMode indicates the mode that the application wishes to operate
  *      in.  Valid values are \b eUSBModeDevice to operate as a USB device and
  *          \b eUSBModeHost to operate as a USB host.
  * @retval  None.
****/
void  usb_core_mode_set(tUSBMode iUSBMode)
{
  //
  // If going to unconfigured mode
  //
  if (iUSBMode == eUSBModeNone)
  {
  }
  //
  // If going to device mode
  //
  if (iUSBMode == eUSBModeDevice)
  {
  }
  //
  // If going to host mode
  //
  if (iUSBMode == eUSBModeHost)
  {
  }
  //
  // If going to otg mode
  //
  if (iUSBMode == eUSBModeOTG)
  {
  }
  //
  // If going to force device mode
  //
  if (iUSBMode == eUSBModeForceDevice)
  {
    md_usb_set_dpdmcon(USB, ((MD_USB_DPDMCON_DPPUD_PULLUP << USB_DPDMCON_DPPUD_POSS) | (MD_USB_DPDMCON_DMPUD_PULLDOWN << USB_DPDMCON_DMPUD_POSS)));
                  //DP = pull up, DM = Pull Down
    md_usb_enable_phy_power(USB);  //USB Phy Enable
    md_usb_set_software_control_device_mode(USB);  //CID=1 (Device)
    md_usb_enable_software_control_cid(USB);  //Enable SW CID
  }
  //
  // If going to force host mode
  //
  if (iUSBMode == eUSBModeForceHost)
  {
    md_usb_set_devcon(USB, 0);
    md_usb_set_dpdmcon(USB, ((MD_USB_DPDMCON_DPPUD_PULLDOWN << USB_DPDMCON_DPPUD_POSS) | (MD_USB_DPDMCON_DMPUD_PULLDOWN << USB_DPDMCON_DMPUD_POSS)));
                  //DP,DM = Pull Down
    md_usb_enable_phy_power(USB);  //USB Phy Enable
    md_usb_set_software_control_host_mode(USB);    //CID=0 (Host)
    md_usb_enable_software_control_cid(USB);  //Enable SW CID

    md_usb_set_devcon(USB, (USB_DEVCON_HOSTREQ | USB_DEVCON_SESSION));
    while (!md_usb_is_active_flag_host_mode(USB));
    md_usb_set_vbus_software_control(USB);          
  }
}
/****
  * @brief  USB device soft connect.
  * @param  None.
  * @retval  None.
****/
void  usb_core_dev_connect()
{
  md_usb_set_dp_pull_up_pull_down(USB, MD_USB_DPDMCON_DPPUD_PULLUP);  //DP=Pull up
  md_usb_enable_phy_power(USB);  //USB Phy Enable
  md_usb_set_software_control_device_mode(USB);  //CID=1 (Device)
  md_usb_enable_software_control_cid(USB);    //Enable SW CID
}
/****
  * @brief  USB device soft disconnect.
  * @param  None.
  * @retval  None.
****/
void  usb_core_dev_disconnect()
{
  md_usb_disable_software_control_cid(USB);  //Disable SW CID
  md_usb_set_dp_pull_up_pull_down(USB, MD_USB_DPDMCON_DPPUD_FLOATING);  //DP=Floating
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/


