/**********************************************************************************
 *
 * @file    usbuac1.h
 * @brief   Head File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */  

#ifndef	_USBUAC1_H
#define	_USBUAC1_H

#pragma pack(1)

//*****************************************************************************
//
// Audio Interface Subclass Codes
//
//*****************************************************************************
#define UAC1_SUBCLASS_UNDEFINED			0x00
#define UAC1_SUBCLASS_AUDIO_CONTROL		0x01
#define UAC1_SUBCLASS_AUDIO_STREAMING	0x02
#define UAC1_SUBCLASS_MIDI_STREAMING	0x03

//*****************************************************************************
//
// Standard Audio terminal types used with audio terminal descriptors.  These
// are defined in the "Universal Serial Bus Device Class Definition for
// Terminal Types" version 1.0 released March 18, 1998.
//
//*****************************************************************************
#define UAC1_TTYPE_UNDEFINED			0x0100
#define UAC1_TTYPE_STREAMING			0x0101
#define UAC1_TTYPE_VENDOR				0x01ff

#define UAC1_TTYPE_OUT_UNDEF			0x0300
#define UAC1_TTYPE_OUT_SPEAKER			0x0301
#define UAC1_TTYPE_OUT_HEADPHONE		0x0302
#define UAC1_TTYPE_OUT_DESK_SPKR		0x0304
#define UAC1_TTYPE_OUT_ROOM_SPKR		0x0305
#define UAC1_TTYPE_OUT_COMM_SPKR		0x0306
#define UAC1_TTYPE_OUT_LFE				0x0307

#define UAC1_TTYPE_EXT_UNDEF			0x0600
#define UAC1_TTYPE_EXT_ANALOG			0x0601
#define UAC1_TTYPE_EXT_DIGITAL			0x0602
#define UAC1_TTYPE_EXT_LINE				0x0603
#define UAC1_TTYPE_EXT_LEGACY			0x0604
#define UAC1_TTYPE_EXT_SPDIF			0x0605
#define UAC1_TTYPE_EXT_1394_DA			0x0606
#define UAC1_TTYPE_EXT_1394_DV			0x0607

//*****************************************************************************
//
// Audio Class-Specific Descriptor Types
// (Table A-4)
//
//*****************************************************************************
#define UAC1_DTYPE_UNDEFINED			0x20
#define UAC1_DTYPE_DEVICE				0x21
#define UAC1_DTYPE_CONFIGURATION		0x22
#define UAC1_DTYPE_STRING				0x23
#define UAC1_DTYPE_INTERFACE			0x24
#define UAC1_DTYPE_ENDPOINT				0x25

//*****************************************************************************
//
// Audio Class-Specific AC Interface Descriptor Subtypes
// (Table A-5)
//
//*****************************************************************************
#define UAC1_AC_DSUBTYPE_UNDEFINED			0x00
#define UAC1_AC_DSUBTYPE_HEADER				0x01
#define UAC1_AC_DSUBTYPE_INPUT_TERMINAL		0x02
#define UAC1_AC_DSUBTYPE_OUTPUT_TERMINAL	0x03
#define UAC1_AC_DSUBTYPE_MIXER_UNIT			0x04
#define UAC1_AC_DSUBTYPE_SELECTOR_UNIT		0x05
#define UAC1_AC_DSUBTYPE_FEATURE_UNIT		0x06
#define UAC1_AC_DSUBTYPE_PROCESSING_UNIT	0x07
#define UAC1_AC_DSUBTYPE_EXTENSION_UNIT		0x08

//*****************************************************************************
//
// Audio Class-Specific AS Interface Descriptor Subtypes
// (Table A-6)
//
//*****************************************************************************
#define UAC1_AS_DSUBTYPE_UNDEFINED			0x00
#define UAC1_AS_DSUBTYPE_GENERAL			0x01
#define UAC1_AS_DSUBTYPE_FORMAT_TYPE		0x02
#define UAC1_AS_DSUBTYPE_FORMAT_SPECIFIC	0x03

//*****************************************************************************
//
// Audio Data Format Type I Codes.
//
//*****************************************************************************
#define UAC1_DFORMAT_UNDEFINED				0x0000
#define UAC1_DFORMAT_PCM					0x0001
#define UAC1_DFORMAT_PCM8					0x0002
#define UAC1_DFORMAT_IEEE_FLOAT				0x0003
#define UAC1_DFORMAT_ALAW					0x0004
#define UAC1_DFORMAT_MULAW					0x0005

//*****************************************************************************
//
// Audio Format Type Codes
//
//*****************************************************************************
#define UAC1_FTYPE_UNDEFINED				0x00
#define UAC1_FTYPE_TYPE_I					0x01
#define UAC1_FTYPE_TYPE_II					0x02
#define UAC1_FTYPE_TYPE_III					0x03

//*****************************************************************************
//
// Audio Class-Specific  Feature Uint controls used with bmaControls values.
//
//*****************************************************************************
#define UAC1_FUCONTROL_MUTE					0x0001	// Mute
#define UAC1_FUCONTROL_VOLUME				0x0002	// Volume
#define UAC1_FUCONTROL_BASS					0x0004	// Bass
#define UAC1_FUCONTROL_MID					0x0008	// Mid
#define UAC1_FUCONTROL_TREBLE				0x0010	// Treble
#define UAC1_FUCONTROL_EQ					0x0020	// Graphic Equalizer
#define UAC1_FUCONTROL_AGC					0x0040	// Automatic Gain
#define UAC1_FUCONTROL_DELAY				0x0080	// Delay
#define UAC1_FUCONTROL_BASS_BOOST			0x0100	// Bass Boost
#define UAC1_FUCONTROL_LOUD					0x0200	// Loudness

//*****************************************************************************
//
// Standard Audio Input terminal types used with audio terminal descriptors.  These
// are defined in the "Universal Serial Bus Device Class Definition for
// Terminal Types" version 1.0 released March 18, 1998.
//
//*****************************************************************************
#define UAC1_ITTYPE_UNDEFINED				0x0200
#define UAC1_ITTYPE_MICROPHONE				0x0201
#define UAC1_ITTYPE_DESKTOP_MICROPHONE		0x0202
#define UAC1_ITTYPE_PERSONAL_MICROPHONE		0x0203
#define UAC1_ITTYPE_OMNI_DIRECTIONAL_MICROPHONE	0x0204
#define UAC1_ITTYPE_MICROPHONE_ARRAY		0x0205
#define UAC1_ITTYPE_PROCESSING_MICROPHONE_ARRAY	0x0206

//*****************************************************************************
//
// Audio Class-Specific Output terminal types.
//
//*****************************************************************************
#define UAC1_OTTYPE_UNDEFINED				0x0300
#define UAC1_OTTYPE_SPEAKER					0x0301
#define UAC1_OTTYPE_HEADPHONES				0x0302
#define UAC1_OTTYPE_HEAD_MOUNTED_DISPLAY_AUDIO	0x0302
#define UAC1_OTTYPE_DESKTOP_SPEAKER			0x0304
#define UAC1_OTTYPE_ROOM_SPEAKER			0x0305
#define UAC1_OTTYPE_COMMUNICATION_SPEAKER	0x0306
#define UAC1_OTTYPE_LOW_FREQUENCY_EFFECTS_SPEAKER	0x0307
// frequencies.

//*****************************************************************************
//
// USB Audio channel configuration bits for wChannelConfig values.
// wChannelConfig: a bit field that indicates which spatial locations are
// present in the cluster. The bit allocations are as follows:
//
//*****************************************************************************
#define UAC1_CHANNEL_L						0x0001	// Left Front (L)
#define UAC1_CHANNEL_R						0x0002	// Right Front (R)
#define UAC1_CHANNEL_C						0x0004	// Center Front (C)
#define UAC1_CHANNEL_LFE					0x0008	// Low Frequency Enhancement (LFE)
#define UAC1_CHANNEL_LS						0x0010	// Left Surround (LS)
#define UAC1_CHANNEL_RS						0x0020	// Right Surround (RS)
#define UAC1_CHANNEL_LC						0x0040	// Left of Center (LC)
#define UAC1_CHANNEL_RC						0x0080	// Right of Center (RC)
#define UAC1_CHANNEL_S						0x0100	// Surround (S)
#define UAC1_CHANNEL_SL						0x0200	// Side Left (SL)
#define UAC1_CHANNEL_SR						0x0400	// Side Right (SR)
#define UAC1_CHANNEL_T						0x0800	// Top (T)

//*****************************************************************************
//
// Endpoint attributes for Audio Class General type.
//
//*****************************************************************************
#define UAC1_EP_ATTR_SAMPLING				0x01	// Sampling Frequency
#define UAC1_EP_ATTR_PITCH					0x02	// Pitch
#define UAC1_EP_ATTR_MAXPACKET				0x80	// MaxPacketsOnly

//*****************************************************************************
//
// Indicates the units used for the wLockDelay field for Audio Class General
// type.
//
//*****************************************************************************
#define UAC1_EP_LOCKDELAY_UNDEF				0x00	// Undefined
#define UAC1_EP_LOCKDELAY_MS				0x01	// Milliseconds
#define UAC1_EP_LOCKDELAY_PCM				0x02	// Decoded PCM samples

//*****************************************************************************
//
// Audio Class-Specific Request Codes
//
//*****************************************************************************
#define UAC1_REQ_SET_CUR					0x01
#define UAC1_REQ_SET_MIN					0x02
#define UAC1_REQ_SET_MAX					0x03
#define UAC1_REQ_SET_RES					0x04
#define UAC1_REQ_SET_MEM					0x05
#define UAC1_REQ_GET_CUR					0x81
#define UAC1_REQ_GET_MIN					0x82
#define UAC1_REQ_GET_RES					0x84
#define UAC1_REQ_GET_MAX					0x83
#define UAC1_REQ_GET_MEM					0x85
#define UAC1_REQ_GET_STAT					0xFF

#define UAC1_CS_CONTROL_MASK				0xFF00
#define UAC1_CS_CHANNEL_MASK				0x00FF

//*****************************************************************************
//
// Endpoint Control Selectors
//
//*****************************************************************************
#define EP_CONTROL_UNDEFINED				0x0000
#define SAMPLING_FREQ_CONTROL				0x0100
#define PITCH_CONTROL						0x0200

//*****************************************************************************
//
// Feature Unit Control Selectors
//
//*****************************************************************************
#define FU_CONTROL_UNDEFINED				0x0000
#define MUTE_CONTROL						0x0100
#define VOLUME_CONTROL						0x0200
#define BASS_CONTROL						0x0300
#define MID_CONTROL							0x0400
#define TREBLE_CONTROL						0x0500
#define EQUALIZER_CONTROL					0x0600
#define AUTOMATIC_GAIN_CONTROL				0x0700
#define DELAY_CONTROL						0x0800
#define BASS_BOOST_CONTROL					0x0900
#define LOUDNESS_CONTROL					0x0A00


//audio endpoint control selector
#define UAC1_CSFREQ							0x01

//*****************************************************************************
//
//! This structure describes the Class-Specific Audio Class Interface Header
//! Descriptor as defined in Universal Serial Bus Device Class Definition
//! for Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AC_HEADER_DESC_
{
	//
	//! The length of this descriptor in bytes.
	//
	uint8_t		bLength;

	//
	//! The type of the descriptor.  For an interface descriptor, this will
	//! be USB_DTYPE_CS_INTERFACE (36).
	//
	uint8_t		bDescriptorType;

	//
	//! This will be USB_DSUBTYPE_HEADER for the header.
	//
	uint8_t		bDescriptorSubtype;

	//
	//! Audio Device Class Specification Release Number in Binary-Coded
	//! Decimal.
	//
	uint16_t bcdADC;

	//
	//! Total number of bytes returned for the class-specific AudioControl
	//! interface descriptor. Includes the combined length of this descriptor
	//! header and all Unit and Terminal descriptors.
	//
	uint16_t	wTotalLength;

	//
	//! The number of AudioStreaming and MIDIStreaming interfaces in the Audio
	//! Interface Collection to which this AudioControl interface belongs.
	//
	uint8_t		bInCollection;

	//
	//! Interface number of the first AudioStreaming or MIDIStreaming interface
	//! in the Collection.
	//
	uint8_t		baInterfaceNr1;

	//
	//! Interface number of the second AudioStreaming or MIDIStreaming interface
	//! in the Collection.
	//
	uint8_t		baInterfaceNr2;
}_UAC1_AC_HEADER_DESC,*_pUAC1_AC_HEADER_DESC;

//*****************************************************************************
//
// These are the possible bits set in the tACInputTerminal.wChannelConfig
// value.
//
//*****************************************************************************
#define USB_AC_CC_LEFT          0x0001
#define USB_AC_CC_RIGHT         0x0002
#define USB_AC_CC_CENTER        0x0004
#define USB_AC_CC_LFE           0x0008
#define USB_AC_CC_LEFTSURROUND  0x0010
#define USB_AC_CC_RIGHTSURROUND 0x0020
#define USB_AC_CC_LEFT_CENTER   0x0040
#define USB_AC_CC_RIGHT_CENTER  0x0080
#define USB_AC_CC_SURROUND      0x0100
#define USB_AC_CC_SIDE_LEFT     0x0200
#define USB_AC_CC_SIDE_RIGHT    0x0400
#define USB_AC_CC_TOP           0x0800

//*****************************************************************************
//
//! This structure describes the Input Terminal Descriptor as defined in
//! Universal Serial Bus Device Class Definition for Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AC_INPUT_TERMINAL_DESC_
{
	//
	//! The length of this descriptor in bytes.
	//
	uint8_t		bLength;

	//
	//! The type of the descriptor.  For an interface descriptor, this will
	//! be USB_DTYPE_CS_INTERFACE (36).
	//
	uint8_t		bDescriptorType;

	//
	//! This will be USB_DSUBTYPE_OUT_TERM for the header.
	//
	uint8_t		bDescriptorSubtype;

	//
	//! Constant uniquely identifying the Terminal within the audio function.
	//! This value is used in all requests to address this Terminal.
	//
	uint8_t		bTerminalID;

	//
	//! Constant characterizing the type of Terminal. See USB Audio Terminal
	//! Types.
	//
	uint16_t	wTerminalType;

	//
	//! Constant, identifying the Input Terminal to which this Output Terminal
	//! is associated.
	//
	uint8_t		bAssocTerminal;

	//
	//! Number of logical output channels in the Terminal's output audio
	//! channel cluster.
	//
	uint8_t		bNrChannels;

	//
	//! Describes the spatial location of the logical channels.
	//
	uint16_t	wChannelConfig;

	//
	//! Index of a string descriptor, describing the name of the first logical
	//! channel.
	//
	uint8_t iChannelNames;

	//
	//! Index of a string descriptor, describing the Output Terminal.
	//
	uint8_t iTerminal;
}_UAC1_AC_INPUT_TERMINAL_DESC, *_pUAC1_AC_INPUT_TERMINAL_DESC;

//*****************************************************************************
//
//! This structure describes the Output Terminal Descriptor as defined in
//! Universal Serial Bus Device Class Definition for Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AC_OUTPUT_TERMINAL_DESC_
{
	//
	//! The length of this descriptor in bytes.
	//
	uint8_t		bLength;

	//
	//! The type of the descriptor.  For an interface descriptor, this will
	//! be USB_DTYPE_CS_INTERFACE (36).
	//
	uint8_t		bDescriptorType;

	//
	//! This will be USB_DSUBTYPE_OUT_TERM for the header.
	//
	uint8_t		bDescriptorSubtype;

	//
	//! Constant uniquely identifying the Terminal within the audio function.
	//! This value is used in all requests to address this Terminal.
	//
	uint8_t		bTerminalID;

	//
	//! Constant characterizing the type of Terminal. See USB Audio Terminal
	//! Types.
	//
	uint16_t	wTerminalType;

	//
	//! Constant, identifying the Input Terminal to which this Output Terminal
	//! is associated.
	//
	uint8_t		bAssocTerminal;

	//
	//! ID of the Unit or Terminal to which this Terminal is connected.
	//
	uint8_t		bSourceID;

	//
	//! Index of a string descriptor, describing the Output Terminal.
	//
	uint8_t iTerminal;
} _UAC1_AC_OUTPUT_TERMINAL_DESC, *_pUAC1_AC_OUTPUT_TERMINAL_DESC;

//*****************************************************************************
//
//! This structure describes the Feature Unit Descriptor as defined in
//! Universal Serial Bus Device Class Definition for Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AC_FEATURE_UNIT_DESC_
{
	//
	//! The length of this descriptor in bytes.
	//
	uint8_t		bLength;

	//
	//! The type of the descriptor.  For an interface descriptor, this will
	//! be USB_DTYPE_CS_INTERFACE (36).
	//
	uint8_t		bDescriptorType;

	//
	//! This will be USB_DSUBTYPE_IN_TERM for the header.
	//
	uint8_t		bDescriptorSubtype;

	//
	//! Constant uniquely identifying the Unit within the audio function. This
	//! value is used in all requests to address this Unit.
	//
	uint8_t		bUnitID;

	//
	//! ID of the Unit or Terminal to which this Feature Unit is connected.
	//
	uint8_t		bSourceID;

	//
	//! ID of the Output Terminal to which this Input Terminal is associated.
	//
	uint8_t		bControlSize;

	//
	//! A bit set to 1 indicates that the mentioned Control is supported for
	//! a given channel.  See the USB_FU_* defines for the bit definitions.
	//! This actually an array of elements of size bControlSize so be
	//! careful when using this value directly.
	//
	uint16_t bmaControl0;

	//
	//! A bit set to 1 indicates that the mentioned Control is supported for
	//! a given channel.  See the USB_FU_* defines for the bit definitions.
	//! This actually an array of elements of size bControlSize so be
	//! careful when using this value directly.
	//
	uint16_t bmaControl1;

	//
	//! A bit set to 1 indicates that the mentioned Control is supported for
	//! a given channel.  See the USB_FU_* defines for the bit definitions.
	//! This actually an array of elements of size bControlSize so be
	//! careful when using this value directly.
	//
	uint16_t bmaControl2;

	//
	//! Index of a string descriptor, describing the Feature Unit.
	//
	uint8_t		iFeature;
} _UAC1_AC_FEATURE_UNIT_DESC, *_pUAC1_AC_FEATURE_UNIT_DESC;

//*****************************************************************************
//
//! This structure describes the Mixer Uint Descriptor as defined in Universal
//! Serial Bus Device Class Definition for Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AC_MIXER_UNIT_DESC_
{
	//
	//! The length of this descriptor in bytes.
	//
	uint8_t		bLength;

	//
	//! The type of the descriptor.  For an interface descriptor, this will
	//! be USB_DTYPE_CS_INTERFACE (36).
	//
	uint8_t		bDescriptorType;
	
	//
	//! This will be USB_AI_MIXER_UNIT for the header.
	//
	uint8_t		bDescriptorSubtype;
	
	//
	//! Constant uniquely identifying the Unit within the audio function. This
	//! value is used in all requests to address this Unit.
	//
	uint8_t		bUnitID;
	
	//
	//! Number of Input Pins of this Unit.
	//
	uint8_t		bNrInPins;
	
	//
	//! ID of the Unit or Terminal to which the first Input Pin of this Mixer
	//! Unit is connected.
	//
	uint8_t		baSourceID;
} _UAC1_AC_MIXER_UNIT_DESC, *_pUAC1_AC_MIXER_UNIT_DESC;

//*****************************************************************************
//
//! This structure describes the Selector Unit Descriptor as defined in Universal
//! Serial Bus Device Class Definition for Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AC_SELECTOR_UNIT_DESC_
{
	//
	//! The length of this descriptor in bytes.
	//
	uint8_t		bLength;
	
	//
	//! The type of the descriptor.  For an interface descriptor, this will
	//! be USB_DTYPE_CS_INTERFACE (36).
	//
	uint8_t		bDescriptorType;
	
	//
	//! This will be USB_AI_MIXER_UNIT for the header.
	//
	uint8_t		bDescriptorSubtype;
	
	//
	//! Constant uniquely identifying the Unit within the audio function. This
	//! value is used in all requests to address this Unit.
	//
	uint8_t		bUnitID;
	
	//
	//! Number of Input Pins of this Unit.
	//
	uint8_t		bNrInPins;
	
	//
	//! ID of the Unit or Terminal to which the first Input Pin of this Mixer
	//! Unit is connected.
	//
	uint8_t		baSourceID;
} _UAC1_AC_SELECTOR_UNIT_DESC, *_pUAC1_AC_SELECTOR_UNIT_DESC;

//*****************************************************************************
//
//! This structure describes the Audio Stream General Descriptor as defined in
//! Universal Serial Bus Device Class Definition for Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AS_GENERAL_DESC_
{
	//
	//! Size of this descriptor in bytes: 7
	//
	uint8_t		bLength;
	
	//
	//! CS_INTERFACE descriptor type.
	//
	uint8_t		bDescriptorType;
	
	//
	//! AS_GENERAL descriptor subtype.
	//
	uint8_t		bDescriptorSubtype;
	
	//
	//! The Terminal ID of the Terminal to which
	//! the endpoint of this interface is
	//! connected.
	//
	uint8_t		bTerminalLink;
	
	//
	//! Delay (d) introduced by the data path
	//! Expressed in number of frames.
	//
	uint8_t		bDelay;
	
	//
	//! The Audio Data Format that has to be
	//! used to communicate with this interface.
	//
	uint16_t	wFormatTag;
}_UAC1_AS_GENERAL_DESC, *_pUAC1_AS_GENERAL_DESC;

//*****************************************************************************
//
//! This structure describes the Type I Audio format descriptors defined in
//! USB Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AS_FORMAT_DESC_
{
  //
  //! The length of this descriptor in bytes.
  //
  uint8_t		bLength;

  //
  //! The type of the descriptor.  For an interface descriptor, this will
  //! be USB_DTYPE_CS_INTERFACE (36).
  //
  uint8_t		bDescriptorType;

  //
  //! This will be USB_AS_FORMAT_TYPE.
  //
  uint8_t		bDescriptorSubtype;

  //
  //! This will be USB_AS_FORMAT_TYPE_I.
  //
  uint8_t		bFormatType;

  //
  //! Number of channels on this streaming interface.
  //
  uint8_t		bNrChannels;

  //
  //! Number of bytes per audio sub-frame or channel.
  //
  uint8_t		bSubFrameSize;

  //
  //! Number of bits per sample.
  //
  uint8_t		bBitResolution;

  //
  //! Number of sample rates that are supported.
  //
  uint8_t		bSamFreqType;

	//
  //! The sample rate of first supported, Byte 0.
  //
	uint8_t		tSamFreq10;

	//
  //! The sample rate of first supported, Byte 1.
  //
	uint8_t		tSamFreq11;

	//
  //! The sample rate of first supported, Byte 2.
  //
	uint8_t		tSamFreq12;
	
	//
  //! The sample rate of second supported, Byte 0.
  //
	uint8_t		tSamFreq20;

	//
  //! The sample rate of second supported, Byte 1.
  //	
	uint8_t		tSamFreq21;

	//
  //! The sample rate of second supported, Byte 2.
  //
	uint8_t		tSamFreq22;
}_UAC1_AS_FORMAT_DESC, *_pUAC1_AS_FORMAT_DESC;

//*****************************************************************************
//
//! This structure describes the Standard Audio Class Endpoint Descriptor 
//! as defined in Universal Serial Bus Device Class Definition
//! for Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AS_DATA_ENDPOINT_DESC_
{
	//
	//! Size of this descriptor, in bytes: 9
	//
	uint8_t		bLength;
	
	//
	//! ENDPOINT descriptor type.
	//
	uint8_t		bDescriptorType;
	
	//
	//! The address of the endpoint on the USB
	//! device described by this descriptor. The
	//! address is encoded as follows:
	//! D7: Direction.
	//! 0 = OUT endpoint
	//! 1 = IN endpoint
	//! D6..4: Reserved, reset to zero
	//! D3..0: The endpoint number,
	//! determined by the designer.
	//
	uint8_t		bEndpointAddress;
	
	//
	//! D3..2: Synchronization type
	//! 01 = Asynchronous
	//! 10 = Adaptive
	//! 11 = Synchronous
	//! D1..0: Transfer type
	//! 01 = Isochronous
	//! All other bits are reserved.
	//
	uint8_t		bmAttribute;
	
	//
	//! Maximum packet size this endpoint is
	//! capable of sending or receiving when this
	//! configuration is selected.
	//! This is determined by the audio
	//! bandwidth constraints of the endpoint.
	//
	uint16_t	wMaxPacketSize;
	
	//
	//! Interval for polling endpoint for data
	//! transfers expressed in milliseconds.
	//! Must be set to 1.
	//
	uint8_t		bInterval;
	
	//
	//! Reset to 0.
	//
	uint8_t		bRefresh;
	
	//
	//! The address of the endpoint used to
	//! communicate synchronization information
	//! if required by this endpoint. Reset to zero
	//! if no synchronization pipe is used.
	//
	uint8_t		bSynchAddress;
}_UAC1_AS_DATA_ENDPOINT_DESC, *_pUAC1_AS_DATA_ENDPOINT_DESC;

//*****************************************************************************
//
//! This structure describes the Class-Specific Audio Class Endpoint Descriptor 
//! Descriptor as defined in Universal Serial Bus Device Class Definition
//! for Audio Devices Release 1.0.
//
//*****************************************************************************
typedef	struct	_UAC1_AS_CLASS_ENDPOINT_DESC_
{
	//
	//! Size of this descriptor, in bytes: 7
	//
	uint8_t		bLength;
	
	//
	//! CS_ENDPOINT descriptor type.
	//
	uint8_t		bDescriptorType;
	
	//
	//! EP_GENERAL descriptor subtype.
	//
	uint8_t		bDescriptorSubtype;
	
	//
	//! bit in the range D6..0 set to 1 indicates
	//! that the mentioned Control is supported
	//! by this endpoint.
	//! D0: Sampling Frequency
	//! D1: Pitch
	//! D6..2: Reserved
	//! Bit D7 indicates a requirement for wMaxPacketSize packets.
	//! D7: MaxPacketsOnly
	//
	uint8_t		bmAttributes;

	//
	//! Indicates the units used for the wLockDelay field:
	//! 0: Undefined
	//! 1: Milliseconds
	//! 2: Decoded PCM samples
	//! 3..255: Reserved
	//
	uint8_t		bLockDelayUnits;

	//
	//! Indicates the time it takes this endpoint to reliably lock
	//! its internal clock recovery circuitry.
	//! Units used depend on the value of the bLockDelayUnits field.
	//
	uint16_t	wLockDelay;
}_UAC1_AS_CLASS_ENDPOINT_DESC, *_pUAC1_AS_CLASS_ENDPOINT_DESC;

#endif	//_USBUAC1_H

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/

