/**********************************************************************************
 *
 * @file    usbmsc.h
 * @brief   Head File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/**
  **************************************************************************************
  * @file    usbmsc.h
  * @brief   Generic types and defines use by the mass storage class.
  * @data    12/21/2018
  * @author  Vega AE Team
  * @note
  *
  * Copyright (C) 2018 Vega Integration Co., Ltd. ALL rights reserved.
  *
  **************************************************************************************
  */

#ifndef  _USBMSC_H
#define  _USBMSC_H

#pragma pack(1)

//*****************************************************************************
//
// The subclass for the mass storage class.
//
//*****************************************************************************
#define  MSC_SUBCLASS_RESERVED    0x00  //Reserved
#define  MSC_SUBCLASS_RBC      0x01  //Reduced Block Cpmmands
#define  MSC_SUBCLASS_ATAPI      0x02  //ATAPI
#define  MSC_SUBCLASS_QIC157      0x03  //QIC-157
#define  MSC_SUBCLASS_UFI      0x04  //UFI
#define  MSC_SUBCLASS_SFF8070i    0x05  //SFF 8070-i
#define  MSC_SUBCLASS_SCSI      0x06  //SCSI

//*****************************************************************************
//
// The request for the maximum number of logical units on a mass storage device.
//
//*****************************************************************************
#define  MSC_REQ_GET_MAX_LUN      0xFE
//*****************************************************************************
//
// The request for the bulk only reset of a mass storage device.
//
//*****************************************************************************
#define MSC_REQ_STORAGE_RESET    0xFF


//Define MSC Constant
#define  CBW_SIGNATURE        0x43425355
#define  CSW_SIGNATURE        0x53425355

#define  CSW_STATUS_SUCCESS      0x00    
#define  CSW_STATUS_FAILED      0x01    
#define  CSW_STATUS_ERROR      0x02    

//*****************************************************************************
//
//  Define SCSI Command
//
//*****************************************************************************
#define  SCSI_FORMAT_UNIT      0x04  //Format Unformatted Media
#define  SCSI_INQUIRY          0x12  //Get Device Information
#define  SCSI_START_STOP        0x1B  //Request Device to Load or Unload Media
#define  SCSI_MODE_SELECT_6      0x15  //
#define  SCSI_MODE_SELECT_10      0x55  //
#define  SCSI_MODE_SENSE_6      0x1A  //
#define  SCSI_MODE_SENSE_10      0x5A  //
#define  SCSI_READ_10          0x28  //
#define  SCSI_READ_12          0xA8  //
#define  SCSI_READ_CAPACITY      0x25  //
#define  SCSI_READ_FORMAT_CAPACITIES  0x23  //
#define  SCSI_REQUEST_SENSE      0x03  //
#define  SCSI_REZERO_UNIT      0x01  //
#define  SCSI_SEEK_10          0x2B  //
#define  SCSI_SEND_DIAGNOSTIC    0x1D  //
#define  SCSI_TEST_UNIT_READY    0x00  //
#define  SCSI_VERIFY            0x2F  //
#define  SCSI_WRITE_10          0x2A  //
#define  SCSI_WRITE_12          0xAA  //
#define  SCSI_WRITE_VERIFY      0x2E  //

//*****************************************************************************
//
// SCSI Command return codes.
//
//*****************************************************************************
#define SCSI_CMD_STATUS_PASS    0x00
#define SCSI_CMD_STATUS_FAIL    0x01

//*****************************************************************************
//
// Offset 0 in the Request Sense response.
//
//*****************************************************************************
#define SCSI_RS_VALID        0x80  // Response is valid.
#define SCSI_RS_CUR_ERRORS      0x70  // Current errors returned.
#define SCSI_RS_DEFER_ERRORS    0x71  // Deferred errors returned.

//*****************************************************************************
//
// Offset 2 in the Request Sense response.
//
//*****************************************************************************
#define SCSI_RS_KEY_M             0x0f  // Sense Key.
#define SCSI_RS_KEY_NO_SENSE      0x00  // No Sense Data.
#define SCSI_RS_KEY_RECOVRD_ERR   0x01  // Recovered Error.
#define SCSI_RS_KEY_NOT_READY     0x02  // Not Ready.
#define SCSI_RS_KEY_MEDIUM_ERR    0x03  // Error in the media.
#define SCSI_RS_KEY_HW_ERR        0x04  // Hardware Error, non recoverable.
#define SCSI_RS_KEY_ILGL_RQST     0x05  // Illegal request.
#define SCSI_RS_KEY_UNIT_ATTN    0x06  // Unit changed or reset.
#define SCSI_RS_KEY_DATA_PROT     0x07  // Write Protect error.
#define SCSI_RS_KEY_BLANK_CHK     0x08  // Write once error, block not clear.
#define SCSI_RS_KEY_ABORT         0x0b  // Last command was aborted.
#define SCSI_RS_ILI               0x20  // Incorrect length indicator.
#define SCSI_RS_EOM               0x40  // End of medium condition.
#define SCSI_RS_FILEMARK      0x80  // Command has read a filemark/setmark.
#define SCSI_RS_MED_NOT_PRSNT    0x003a  // Medium not present.
#define SCSI_RS_MED_NOTRDY2RDY    0x0028  // Not ready to ready transition.
#define SCSI_RS_PV_INVALID      0x0226  // Parameter Value Invalid.

//*****************************************************************************
//
// Additional information for SCSI_RS_KEY_NOT_READY
//
//*****************************************************************************
#define SCSI_RS_KEY_NOTPRSNT    0x3A  // Media Not Present.

//*****************************************************************************
//
// Prevent/Allow Medium Removal Definitions.
//
//*****************************************************************************

//*****************************************************************************
//
// Offset 4 in the Prevent/Allow Medium Removal command.
//
//*****************************************************************************
#define SCSI_PE_MEDRMV_M      0x03
#define SCSI_PE_MEDRMV_ALLOW    0x00
#define SCSI_PE_MEDRMV_PREVENT    0x01

//MSC CBW Structure
typedef  struct  _MSC_COMMAND_BLOCK_
{
  //
  // Signature that helps identify this data packet as a CBW.  The signature
  // field shall contain the value 0x43425355 (little endian), indicating a
  // CBW.
  //
  uint32_t  dSignature;

  //
  // The Command Block Tag sent by the host controller.  The device shall
  // echo the contents of this field back to the host in the dCSWTag field
  // of the associated CSW.  The dCSWTag positively associates a CSW with the
  // corresponding CBW.
  //
  uint32_t  dTag;

  //
  // The number of bytes of data that the host expects to transfer on the
  // Bulk-In or Bulk-Out endpoint (as indicated by the Direction bit) during
  // the execution of this command.  If this field is zero, the device and
  // the host will not transfer data between the CBW and the associated CSW,
  // and the device will ignore the value of the Direction bit in
  // bmCBWFlags.
  //
  uint32_t  sDataTransferLength;

  //
  // The device will ignore these bits if the dCBWDataTransferLength value
  // is set to 0.
  //
  // The bits of this field are defined as follows:
  // Bit 7 Direction
  //   0 = Data-Out from host to the device,
  //   1 = Data-In from the device to the host.
  // Bit 6 Obsolete - The host shall set this bit to zero.
  // Bits 5..0 Reserved - the host shall set these bits to zero.
  //
  uint8_t    bmFlags;

  //
  // The device Logical Unit Number (LUN) to which the command block is being
  // sent.  For devices that support multiple LUNs, the host shall place into
  // this field the LUN to which this command block is addressed.  Otherwise,
  // the host shall set this field to zero.
  //
  uint8_t    bLUN;

  //
  // The valid length of the CBWCB in bytes.  This defines the valid length
  // of the command block.  The only legal values are 1 through 16.  All
  // other values are reserved.
  //
  uint8_t    bCBLength;

  //
  // This array holds the command block to be executed by the device.  The
  // MSC device will interpret the first bCBWCBLength bytes in this field as
  // a command block as defined by the command set identified by
  // bInterfaceSubClass.  If the command set supported by the device uses
  // command blocks of fewer than 16 bytes in length, the significant bytes
  // shall be transferred first, beginning with the byte at offset 15.  The
  // device will ignore the content of the CBWCB field past the byte at
  // offset (15 + bCBWCBLength - 1).
  //
  uint8_t    CB[16];
} _MSC_COMMAND_BLOCK, *_pMSC_COMMAND_BLOCK;

//MSC CSW Structure
typedef  struct  _MSC_COMMAND_STATUS_
{
  //
  // Signature that identifies this data packet as a CSW.  The signature
  // field must contain the value 53425355h (little endian) to indicate CSW.
  //
  uint32_t  dSignature;

  //
  // The device will set this field to the value received in the dCBWTag of
  // the associated CBW.
  //
  uint32_t  dTag;

  //
  // For OUT transactions the device will fill the dCSWDataResidue field with
  // the difference between the amount of data expected as stated in the
  // dCBWDataTransferLength, and the actual amount of data processed by the
  // device.  For IN transactions the device will fill the dCSWDataResidue
  // field with the difference between the amount of data expected as stated
  // in the dCBWDataTransferLength and the actual amount of relevant data
  // sent by the device.  The dCSWDataResidue will not exceed the value sent
  // in the dCBWDataTransferLength.
  //
  uint32_t  sDataResidue;

  //
  // The bCSWStatus field indicates the success or failure of the command.
  // The device shall set this byte to zero if the command completed
  // successfully.  A non-zero value shall indicate a failure during command
  // execution.
  //
  uint8_t    bStatus;
}_MSC_COMMAND_STATUS, *_pMSC_COMMAND_STATUS;

//SCSI Command Block
typedef  struct  _SCSI_COMMAND_BLOCK_
{
  uint8_t    OPCode;
  uint8_t    LUN;
  uint8_t    LBA3;
  uint8_t    LBA2;
  uint8_t    LBA1;
  uint8_t    LBA0;
  uint8_t    Length3;
  uint8_t    Length2;
  uint8_t    Length1;
  uint8_t    Length0;
  uint8_t    Res0[2];
}_SCSI_COMMAND_BLOCK, *_pSCSI_COMMAND_BLOCK;

//*****************************************************************************
//
// SCSI Test Unit Ready definitions.
//
//*****************************************************************************
typedef  struct  _SCSI_TEST_UNIT_READY_COMMAND_
{
  uint8_t    OPCode;
  uint8_t    LUN;
  uint8_t    Res0[10];
}_SCSI_TEST_UNIT_READY_COMMAND, *_pSCSI_TEST_UNIT_READY_COMMAND;

//*****************************************************************************
//
// SCSI Inquiry command definitions.
//
//*****************************************************************************
typedef  struct  _SCSI_INQUIRY_COMMAND_
{
  uint8_t    OPCode;
  uint8_t    LUN_EVPD;
  uint8_t    PageCode;
  uint8_t    Res0;
  uint8_t    AllocationLength;
  uint8_t    Res1[7];
}_SCSI_INQUIRY_COMMAND, *_pSCSI_INQUIRY_COMMAND;

//SCSI Inquiry Data
typedef  struct  _SCSI_INQUIRY_DATA_
{
  uint8_t    PeripheralDeviceType;
  uint8_t    RMB;
  uint8_t    ISO_ECMA_ANSI;
  uint8_t    ResponseDataFormat;
  uint8_t    AdditionalLength;
  uint8_t    Res0[3];
  uint8_t    VendorInfo[8];
  uint8_t    ProductInfo[16];
  uint8_t    ProductRevision[4];
}_SCSI_INQUIRY_DATA, *_pSCSI_INQUIRY_DATA;

//SCSI Read Format Capacity Command
typedef  struct  _SCSI_READ_FORMAT_CAPACITIES_COMMAND_
{
  uint8_t    OPCode;
  uint8_t    LUN;
  uint8_t    Res0[5];
  uint8_t    AllocationLength1;
  uint8_t    AllocationLength0;
  uint8_t    Res1[3];
}_SCSI_READ_FORMAT_CAPACITIES_COMMAND, *_pSCSI_READ_FORMAT_CAPACITIES_COMMAND;

//SCSI Capacity List Header
typedef  struct  _SCSI_CAPACITY_LIST_HEADER_
{
  uint8_t    Res0[3];
  uint8_t    CapacityListLength;
}_SCSI_CAPACITY_LIST_HEADER, *_pSCSI_CAPACITY_LIST_HEADER;

//SCSI Capacity List Descriptor
typedef  struct  _SCSI_CAPACITY_DESCRIPTOR_
{
  uint8_t    NumberofBlocks3;
  uint8_t    NumberofBlocks2;
  uint8_t    NumberofBlocks1;
  uint8_t    NumberofBlocks0;
  uint8_t    DescriptorCode;
  uint8_t    BlockLength2;
  uint8_t    BlockLength1;
  uint8_t    BlockLength0;
}_SCSI_CAPACITY_DESCRIPTOR, *_pSCSI_CAPACITY_DESCRIPTOR;

//SCSI Capacity List Data
typedef  struct  _SCSI_CAPACITY_LIST_DATA_
{
  _SCSI_CAPACITY_LIST_HEADER  CapacityListHeader;
  _SCSI_CAPACITY_DESCRIPTOR  CapacityDescriptor;
}_SCSI_CAPACITY_LIST_DATA, *_pSCSI_CAPACITY_LIST_DATA;

//*****************************************************************************
//
// SCSI Read Capacity definitions.
//
//*****************************************************************************
typedef  struct  _SCSI_READ_CAPACITY_COMMAND_
{
  uint8_t    OPCode;
  uint8_t    LUN_RelAdr;
  uint8_t    LBA3;
  uint8_t    LBA2;
  uint8_t    LBA1;
  uint8_t    LBA0;
  uint8_t    Res0[2];
  uint8_t    PMI;
  uint8_t    Res1[3];
}_SCSI_READ_CAPACITY_COMMAND, *_pSCSI_READ_CAPACITY_COMMAND;

//*****************************************************************************
//
// Size of the SCSI Read Capacity response data.
//
//*****************************************************************************
typedef  struct  _SCSI_READ_CAPACITY_DATA_
{
  uint8_t    LastLBA3;
  uint8_t    LastLBA2;
  uint8_t    LastLBA1;
  uint8_t    LastLBA0;
  uint8_t    BlockLength3;
  uint8_t    BlockLength2;
  uint8_t    BlockLength1;
  uint8_t    BlockLength0;
}_SCSI_READ_CAPACITY_DATA, *_pSCSI_READ_CAPACITY_DATA;

//SCSI Mode Select 6 Command
typedef  struct  _SCSI_MODE_SELECT_6_COMMAND_
{
  uint8_t    OPCode;
  uint8_t    LUN_PF_SP;
  uint8_t    Res0;
  uint8_t    Res1;
  uint8_t    ParameterLictLength;
  uint8_t    Res2;
}_SCSI_MODE_SELECT_6_COMMAND, *_pSCSI_MODE_SELECT_6_COMMAND;

//*****************************************************************************
//
// SCSI Mode Sense definitions, these are passed in via the ui32Flags parameter
// of the SCSIModeSense() function call.
//
//*****************************************************************************
typedef  struct  _SCSI_MODE_SENSE_6_COMMAND_
{
  uint8_t    OPCode;
  uint8_t    LUN_DBD;
  uint8_t    PC_PageCode;
  uint8_t    Res0;
  uint8_t    AllocationLength;
  uint8_t    Res1;
}_SCSI_MODE_SENSE_6_COMMAND, *_pSCSI_MODE_SENSE_6_COMMAND;

//SCSI Mode Sense 6 Header
typedef  struct  _SCSI_MODE_SENSE_6_HEADER_
{
  uint8_t    ModeDataLength;
  uint8_t    MediumType;
  uint8_t    DeviceSpecificParameter;
  uint8_t    BlockDescriptorLength;
}_SCSI_MODE_SENSE_6_HEADER, *_pSCSI_MODE_SENSE_6_HEADER;

//SCSI Mode Sense 6 Data
typedef  struct  _SCSI_MODE_SENSE_6_DATA_
{
  uint8_t    ModeDataLength;
  uint8_t    MediumType;
  uint8_t    DeviceSpecificParameter;
  uint8_t    BlockDescriptorLength;
}_SCSI_MODE_SENSE_6_DATA, *_pSCSI_MODE_SENSE_6_DATA;

//SCSI Read 10 Command
typedef  struct  _SCSI_READ_10_COMMAND_
{
  uint8_t    OPCode;
  uint8_t    LUN_RelAdr;
  uint8_t    LBA3;
  uint8_t    LBA2;
  uint8_t    LBA1;
  uint8_t    LBA0;
  uint8_t    Res0;
  uint8_t    TransferLength1;
  uint8_t    TransferLength0;
  uint8_t    Res1[3];
}_SCSI_READ_10_COMMAND, *_pSCSI_READ_10_COMMAND;

//SCSI Write 10 Command
typedef  struct  _SCSI_WRITE_10_COMMAND_
{
  uint8_t    OPCode;
  uint8_t    LUN_RelAdr;
  uint8_t    LBA3;
  uint8_t    LBA2;
  uint8_t    LBA1;
  uint8_t    LBA0;
  uint8_t    Res0;
  uint8_t    TransferLength1;
  uint8_t    TransferLength0;
  uint8_t    Res1[3];
}_SCSI_WRITE_10_COMMAND, *_pSCSI_WRITE_10_COMMAND;

//SCSI Request Sense Command
typedef  struct  _SCSI_REQUEST_SENSE_COMMAND_
{
  uint8_t    OPCode;
  uint8_t    LUN;
  uint8_t    Res0[2];
  uint8_t    AllocationLength;
  uint8_t    Res1[7];
}_SCSI_REQUEST_SENSE_COMMAND, *_pSCSI_REQUEST_SENSE_COMMAND;

//SCSI Request Sense Data
typedef  struct  _SCSI_REQUEST_SENSE_DATA_
{
  uint8_t    ResponseCode;
  uint8_t    Obsolete;
  uint8_t    SenseKey;
  uint8_t    Information[4];
  uint8_t    AdditionalSenseLength;
  uint8_t    CommandSpecificInfo[4];
  uint8_t    AdditionalSenseCode;
  uint8_t    AdditionalSenseCodeQualifier;
  uint8_t    FieldReplaceableUnitCode;
  uint8_t    SenseKeySpecific[3];
}_SCSI_REQUEST_SENSE_DATA, *_pSCSI_REQUEST_SENSE_DATA;

#endif  //_USBMSC_H

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/

