/**********************************************************************************
 *
 * @file    usb_host_msc.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_usb.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_host_enum.h"
#include "usbmsc.h"
#include "usb_host_msc.h"
#include <stdio.h>

#pragma pack(1)

#define	HOST_MSC_MSG	1


extern	uint8_t		CBWData[];
extern	uint8_t		CSWData[];
extern	uint8_t		MSCData[];

static	uint32_t	MSCTag = ESMSCTAG;

static	_pMSC_COMMAND_BLOCK		pMSC_CBW;
static	_pMSC_COMMAND_STATUS	pMSC_CSW;

/****
	* @brief	Get MAX LUN
	* @param	pHost		: Host Instance
	* @param	pMaxLUN	: Pointer to Max LUN
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_get_max_lun(_pUSB_FS_HOST pHost, uint8_t *pMaxLUN)			//A1 FE 00 00 XX 00 01 00
{
#if	USBMSG
	printf("GetMaxLUN\r\n");
#endif

	SetupReq.bmRequestType = 0xa1;		//Class Specified
	SetupReq.bRequest = MSC_REQ_GET_MAX_LUN;	//GETMAXLUN
	SetupReq.wValue	= 0;				//wValue
	SetupReq.wIndex	= 0;				//wIndex
	SetupReq.wLength = 1;				//wLength
	
	if (usbh_issue_request(pHost, &SetupReq, pMaxLUN))
		return(USBH_RESULT_FAIL);
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC CBW Out
	* @param	pHost		: Host Instance
	* @param	pCBWData	: Pointer to Command Block Structure
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_cbw_out(_pUSB_FS_HOST pHost, _pMSC_COMMAND_BLOCK pCBWData)
{
#if	HOST_MSC_MSG
	printf("CBW Out-");
#endif
	pCBWData->dSignature = CBW_SIGNATURE;	//Signature
	pCBWData->dTag		= MSCTag++;		//Tag
	pMSC_CBW->bmFlags	= 0x80;
	pMSC_CBW->bLUN		= 0;
	if (usbh_issue_epx_out(pHost, MSC_BULK_OUT_EP, sizeof(_MSC_COMMAND_BLOCK), (uint8_t *)pCBWData))
	{
#if	HOST_MSC_MSG
		printf("Error-\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
#if	HOST_MSC_MSG
//	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC CSW In
	* @param	pHost		: Host Instance
	* @param	pCSWData	: Pointer to Command Block Status
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_csw_in(_pUSB_FS_HOST pHost, _pMSC_COMMAND_STATUS pCSWData)
{
#if	HOST_MSC_MSG
	printf("CSW In-");
#endif
	if (usbh_issue_epx_in(pHost, MSC_BULK_IN_EP, sizeof(_MSC_COMMAND_STATUS), (uint8_t *)pCSWData))
	{
#if	HOST_MSC_MSG
		printf("Error\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
	if (pCSWData->dSignature != CSW_SIGNATURE)
	{
#if	HOST_MSC_MSG
		printf("Error\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
#if	HOST_MSC_MSG
//	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC Data In
	* @param	pHost		: Host Instance
	* @param	BCnt	: Byte Count for In Transaction
	* @param	pBuf	: Pointer to uint8_t Data Buffer
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_data_in(_pUSB_FS_HOST pHost, uint16_t BCnt, uint8_t *pBuf)
{
#if	HOST_MSC_MSG
	printf("Data In-");
#endif
	if (usbh_issue_epx_in(pHost, MSC_BULK_IN_EP, BCnt, pBuf))
	{
		md_usb_set_epx_index(USB, MSC_BULK_IN_EP);			//Endpoint 1
		if (md_usb_is_active_flag_epx_rx_stalled(USB))		//STALL
		{
#if	HOST_MSC_MSG
			printf("Stalled-");
#endif
			usbh_clear_feature(pHost, USB_FEATURE_EP_HALT, (MSC_BULK_IN_EP | 0x80));		//clearerr EPHalt for In Endpoint
			md_usb_clear_flag_epx_rx_stalled(USB);
			return(USBH_RESULT_PASS);
		}
		else
		{
#if	HOST_MSC_MSG
			printf("Data Error\r\n");
#endif
			return(USBH_RESULT_FAIL);
		}
	}
#if	HOST_MSC_MSG
//	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC RequestSense
	* @param	pHost		: Host Instance
	* @param	*pCBWData	: CBW Command Data
	* @param	*pCSWData	: CSW Status Data
	* @param	*pMSCData	: RequestSense Data In Buffer
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_request_sense(_pUSB_FS_HOST pHost, uint8_t *pCBWData, uint8_t *pCSWData, uint8_t *pMSCData)
{
_pSCSI_REQUEST_SENSE_COMMAND	pSCSIRequestSenseCommand;
_pSCSI_REQUEST_SENSE_DATA		pSCSIRequestSenseData;
uint8_t	ii;
	
#if	HOST_MSC_MSG
	printf("RequestSense...");
#endif
	pMSC_CBW = (_pMSC_COMMAND_BLOCK)pCBWData;
	pMSC_CBW->sDataTransferLength = sizeof(_SCSI_REQUEST_SENSE_DATA);
	pMSC_CBW->bCBLength=sizeof(_SCSI_REQUEST_SENSE_COMMAND);
	for (ii = 0; ii < sizeof(pMSC_CBW->CB); ii++)
		pMSC_CBW->CB[ii] = 0;
	pSCSIRequestSenseCommand = (_pSCSI_REQUEST_SENSE_COMMAND)pMSC_CBW->CB;
	pSCSIRequestSenseCommand->OPCode = SCSI_REQUEST_SENSE;
	pSCSIRequestSenseCommand->LUN = 0;
	pSCSIRequestSenseCommand->AllocationLength=sizeof(_SCSI_REQUEST_SENSE_DATA);
	if (msc_issue_cbw_out(pHost, pMSC_CBW))
		return(USBH_RESULT_FAIL);
	if (msc_issue_data_in(pHost, sizeof(_SCSI_REQUEST_SENSE_DATA), (uint8_t *)pMSCData))
	{
#if	HOST_MSC_MSG
			printf("Data Error\r\n");
#endif
			return(USBH_RESULT_FAIL);
	}
	pMSC_CSW = (_pMSC_COMMAND_STATUS)pCSWData;
	if (msc_issue_csw_in(pHost, pMSC_CSW))
		return(USBH_RESULT_FAIL);
#if	HOST_MSC_MSG
	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC TestUnitReady
	* @param	pHost		: Host Instance
	* @param	*pCBWData	: CBW Command Data
	* @param	*pCSWData	: CSW Status Data
	* @param	*pMSCData	: Inquiry Data In Buffer
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_test_unit_ready(_pUSB_FS_HOST pHost, uint8_t *pCBWData, uint8_t *pCSWData, uint8_t *pMSCData)
{
_pSCSI_TEST_UNIT_READY_COMMAND	pSCSITestUnitReadyCommand;
uint8_t	ii;
	
#if	HOST_MSC_MSG
	printf("Inquiry...");
#endif
	pMSC_CBW = (_pMSC_COMMAND_BLOCK)pCBWData;
	pMSC_CBW->sDataTransferLength = 0;
	pMSC_CBW->bCBLength = sizeof(_SCSI_TEST_UNIT_READY_COMMAND);
	for (ii = 0; ii < sizeof(pMSC_CBW->CB); ii++)
		pMSC_CBW->CB[ii] = 0;
	pSCSITestUnitReadyCommand = (_pSCSI_TEST_UNIT_READY_COMMAND)pMSC_CBW->CB;
	pSCSITestUnitReadyCommand->OPCode = SCSI_TEST_UNIT_READY;
	pSCSITestUnitReadyCommand->LUN = 0;
	if (msc_issue_cbw_out(pHost, pMSC_CBW))
	{
#if	HOST_MSC_MSG
		printf("CBW Error\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
	pMSC_CSW = (_pMSC_COMMAND_STATUS)pCSWData;
	if (msc_issue_csw_in(pHost, pMSC_CSW))
	{
#if	HOST_MSC_MSG
		printf("CSW Error\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
	if (pMSC_CSW->bStatus != CSW_STATUS_SUCCESS)
	{
		if (msc_issue_request_sense(pHost, pCBWData, pCSWData, pMSCData))
			return(USBH_RESULT_FAIL);
	}
#if	HOST_MSC_MSG
	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC Inquiry
	* @param	pHost		: Host Instance
	* @param	*pCBWData	: CBW Command Data
	* @param	*pCSWData	: CSW Status Data
	* @param	*pMSCData	: Inquiry Data In Buffer
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_inquiry(_pUSB_FS_HOST pHost, uint8_t *pCBWData, uint8_t *pCSWData, uint8_t *pMSCData)
{
_pSCSI_INQUIRY_COMMAND	pSCSIInquiryCommand;
_pSCSI_INQUIRY_DATA		pSCSIInquiryData;
uint8_t	ii;
	
#if	HOST_MSC_MSG
	printf("Inquiry...");
#endif
	pMSC_CBW = (_pMSC_COMMAND_BLOCK)pCBWData;
	pMSC_CBW->sDataTransferLength = sizeof(_SCSI_INQUIRY_DATA);
	pMSC_CBW->bCBLength = sizeof(_SCSI_INQUIRY_COMMAND);
	for (ii = 0; ii < sizeof(pMSC_CBW->CB); ii++)
		pMSC_CBW->CB[ii] = 0;
	pSCSIInquiryCommand = (_pSCSI_INQUIRY_COMMAND)pMSC_CBW->CB;
	pSCSIInquiryCommand->OPCode = SCSI_INQUIRY;
	pSCSIInquiryCommand->LUN_EVPD = 0;
	pSCSIInquiryCommand->Res0 = 0;
	pSCSIInquiryCommand->AllocationLength=sizeof(_SCSI_INQUIRY_DATA);
	if (msc_issue_cbw_out(pHost, pMSC_CBW))
	{
#if	HOST_MSC_MSG
		printf("CBW Error\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
	if (msc_issue_data_in(pHost, sizeof(_SCSI_INQUIRY_DATA), (uint8_t *)pMSCData))
	{
#if	HOST_MSC_MSG
		printf("Data Error\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
	pSCSIInquiryData = (_pSCSI_INQUIRY_DATA)pMSCData;
	pMSC_CSW = (_pMSC_COMMAND_STATUS)pCSWData;
	if (msc_issue_csw_in(pHost, pMSC_CSW))
	{
#if	HOST_MSC_MSG
		printf("CSW Error\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
	if (pMSC_CSW->bStatus != CSW_STATUS_SUCCESS)
	{
		if (msc_issue_request_sense(pHost, pCBWData, pCSWData, pMSCData))
			return(USBH_RESULT_FAIL);
	}
#if	HOST_MSC_MSG
	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC ReadFormatCapacities
	* @param	pHost		: Host Instance
	* @param	*pCBWData	: CBW Command Data
	* @param	*pCSWData	: Status Data
	* @param	*pMSCData	: ReadFormatCapacities Data In Buffer
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_read_format_capacities(_pUSB_FS_HOST pHost, uint8_t *pCBWData, uint8_t *pCSWData, uint8_t *pMSCData)
{
_pSCSI_READ_FORMAT_CAPACITIES_COMMAND	pSCSIReadFormatCapacitiesCommand;
//_pSCSIREADFORMATCAPACITIESDATA		pSCSIReadFormatCapacitiesData;
uint8_t	ii;
	
#if	HOST_MSC_MSG
	printf("ReadFormatCapacities...");
#endif
	pMSC_CBW = (_pMSC_COMMAND_BLOCK)pCBWData;
	pMSC_CBW->sDataTransferLength = (sizeof(_SCSI_CAPACITY_LIST_HEADER) + sizeof(_SCSI_CAPACITY_DESCRIPTOR));
	pMSC_CBW->bCBLength = sizeof(_SCSI_READ_FORMAT_CAPACITIES_COMMAND);
	for (ii = 0; ii < sizeof(pMSC_CBW->CB); ii++)
		pMSC_CBW->CB[ii] = 0;
	pSCSIReadFormatCapacitiesCommand = (_pSCSI_READ_FORMAT_CAPACITIES_COMMAND)pMSC_CBW->CB;
	pSCSIReadFormatCapacitiesCommand->OPCode = SCSI_READ_FORMAT_CAPACITIES;
	pSCSIReadFormatCapacitiesCommand->LUN = 0;
	pSCSIReadFormatCapacitiesCommand->AllocationLength1 = ((pMSC_CBW->sDataTransferLength >> 8) & 0xff);
	pSCSIReadFormatCapacitiesCommand->AllocationLength0 = ((pMSC_CBW->sDataTransferLength >> 0) & 0xff);
	if (msc_issue_cbw_out(pHost, pMSC_CBW))
		return(USBH_RESULT_FAIL);
	if (msc_issue_data_in(pHost, sizeof(_SCSI_CAPACITY_LIST_HEADER)+sizeof(_SCSI_CAPACITY_DESCRIPTOR), (uint8_t *)pMSCData))
	{
#if	HOST_MSC_MSG
			printf("Data Error\r\n");
#endif
			return(USBH_RESULT_FAIL);
	}
	pMSC_CSW=(_pMSC_COMMAND_STATUS)pCSWData;
	if (msc_issue_csw_in(pHost, pMSC_CSW))
		return(USBH_RESULT_FAIL);
	if (pMSC_CSW->bStatus!=CSW_STATUS_SUCCESS)
	{
		if (msc_issue_request_sense(pHost, pCBWData, pCSWData, pMSCData))
			return(USBH_RESULT_FAIL);
	}
#if	HOST_MSC_MSG
	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC ReadCapacity
	* @param	pHost		: Host Instance
	* @param	*pCBWData	: CBW Command Data
	* @param	*pCSWData	: CSW Status Data
	* @param	*pMSCData	: ReadCapacity Data In Buffer
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_read_capacity(_pUSB_FS_HOST pHost, uint8_t *pCBWData, uint8_t *pCSWData, uint8_t *pMSCData)
{
_pSCSI_READ_CAPACITY_COMMAND	pSCSIReadCapacityCommand;
_pSCSI_READ_CAPACITY_DATA		pSCSIReadCapacityData;
uint8_t	ii;
	
#if	HOST_MSC_MSG
	printf("ReadCapacity...");
#endif
	pMSC_CBW = (_pMSC_COMMAND_BLOCK)pCBWData;
	pMSC_CBW->sDataTransferLength = sizeof(_SCSI_READ_CAPACITY_DATA);
	pMSC_CBW->bCBLength = sizeof(_SCSI_READ_CAPACITY_COMMAND);
	for (ii = 0; ii < sizeof(pMSC_CBW->CB); ii++)
		pMSC_CBW->CB[ii] = 0;
	pSCSIReadCapacityCommand = (_pSCSI_READ_CAPACITY_COMMAND)pMSC_CBW->CB;
	pSCSIReadCapacityCommand->OPCode = SCSI_READ_CAPACITY;
	pSCSIReadCapacityCommand->LUN_RelAdr = 0;
	pSCSIReadCapacityCommand->LBA3 = 0;
	pSCSIReadCapacityCommand->LBA2 = 0;
	pSCSIReadCapacityCommand->LBA1 = 0;
	pSCSIReadCapacityCommand->LBA0 = 0;
	pSCSIReadCapacityCommand->PMI = 0;
	if (msc_issue_cbw_out(pHost, pMSC_CBW))
		return(USBH_RESULT_FAIL);
	if (msc_issue_data_in(pHost, sizeof(_SCSI_READ_CAPACITY_DATA), (uint8_t *)pMSCData))
	{
#if	HOST_MSC_MSG
			printf("Data Error\r\n");
#endif
			return(USBH_RESULT_FAIL);
	}
	pMSC_CSW=(_pMSC_COMMAND_STATUS)pCSWData;
	if (msc_issue_csw_in(pHost, pMSC_CSW))
		return(USBH_RESULT_FAIL);
	if (pMSC_CSW->bStatus != CSW_STATUS_SUCCESS)
	{
		if (msc_issue_request_sense(pHost, pCBWData, pCSWData, pMSCData))
			return(USBH_RESULT_FAIL);
	}
#if	HOST_MSC_MSG
	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC ModeSense6
	* @param	pHost		: Host Instance
	* @param	*pCBWData	: CBW Command Data
	* @param	*pCSWData	: CSW Status Data
	* @param	*pMSCData	: ModeSense6 Data In Buffer
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_mode_sense_6(_pUSB_FS_HOST pHost, uint8_t *pCBWData, uint8_t *pCSWData, uint8_t *pMSCData)
{
_pSCSI_MODE_SENSE_6_COMMAND		pSCSIModeSense6Command;
_pSCSI_MODE_SENSE_6_DATA		pSCSIModeSense6Data;
uint8_t	ii;
	
#if	HOST_MSC_MSG
	printf("ModeSense6...");
#endif
	pMSC_CBW = (_pMSC_COMMAND_BLOCK)pCBWData;
	pMSC_CBW->sDataTransferLength = 0xc0;
	pMSC_CBW->bCBLength = sizeof(_SCSI_MODE_SENSE_6_COMMAND);
	for (ii = 0 ;ii < sizeof(pMSC_CBW->CB); ii++)
		pMSC_CBW->CB[ii] = 0;
	pSCSIModeSense6Command = (_pSCSI_MODE_SENSE_6_COMMAND)pMSC_CBW->CB;
	pSCSIModeSense6Command->OPCode = SCSI_MODE_SENSE_6;
	pSCSIModeSense6Command->LUN_DBD = 0;
	pSCSIModeSense6Command->PC_PageCode = 0x1c;
	pSCSIModeSense6Command->AllocationLength = 0x1c;
	if (msc_issue_cbw_out(pHost, pMSC_CBW))
		return(USBH_RESULT_FAIL);
	if (msc_issue_data_in(pHost, sizeof(_SCSI_MODE_SENSE_6_DATA), (uint8_t *)pMSCData))
	{
#if	HOST_MSC_MSG
			printf("Data Error\r\n");
#endif
			return(USBH_RESULT_FAIL);
	}
	pMSC_CSW=(_pMSC_COMMAND_STATUS)pCSWData;
	if (msc_issue_csw_in(pHost, pMSC_CSW))
		return(USBH_RESULT_FAIL);
	if (pMSC_CSW->bStatus != CSW_STATUS_SUCCESS)
	{
		if (msc_issue_request_sense(pHost, pCBWData, pCSWData, pMSCData))
			return(USBH_RESULT_FAIL);
	}
#if	HOST_MSC_MSG
	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	Issue MSC Read10
	* @param	pHost		: Host Instance
	* @param	*pCBWData	: CBW Command Data
	* @param	*pCSWData	: CSW Status Data
	* @param	*pMSCData	: Read10 Data In Buffer
	* @retval	eUSBHResult
****/
eUSBHResult	msc_issue_read_10(_pUSB_FS_HOST pHost, uint8_t *pCBWData, uint8_t *pCSWData, uint8_t *pMSCData)
{
_pSCSI_READ_10_COMMAND	pSCSIRead10Command;
uint8_t	ii;
	
#if	HOST_MSC_MSG
	printf("Read10...");
#endif
	pMSC_CBW = (_pMSC_COMMAND_BLOCK)pCBWData;
	pMSC_CBW->sDataTransferLength = 0x200;
	pMSC_CBW->bCBLength = sizeof(_SCSI_READ_10_COMMAND);
	for (ii = 0; ii < sizeof(pMSC_CBW->CB); ii++)
		pMSC_CBW->CB[ii] = 0;
	pSCSIRead10Command = (_pSCSI_READ_10_COMMAND)pMSC_CBW->CB;
	pSCSIRead10Command->OPCode = SCSI_READ_10;
	if (msc_issue_cbw_out(pHost, pMSC_CBW))
		return(USBH_RESULT_FAIL);
	if (msc_issue_data_in(pHost, MSC_BLOCK_SZ, pMSCData))
	{
#if	HOST_MSC_MSG
		printf("Data Error\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
	pMSC_CSW=(_pMSC_COMMAND_STATUS)pCSWData;
	if (msc_issue_csw_in(pHost, pMSC_CSW))
		return(USBH_RESULT_FAIL);
	if (pMSC_CSW->bStatus != CSW_STATUS_SUCCESS)
	{
		if (msc_issue_request_sense(pHost, pCBWData, pCSWData, pMSCData))
			return(USBH_RESULT_FAIL);
	}
#if	HOST_MSC_MSG
	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/****
	* @brief	MSC Read LBA
	* @param	pHost		: Host Instance
	* @param	LBAIndex	: LBA Index
	* @param	LBACnt	: Count of LBA
	* @param	*pLBABuf	: Data Buffer to Read
	* @retVal	eUSBHResult
****/
eUSBHResult	msc_read_lba(_pUSB_FS_HOST pHost, uint32_t LBAIndex, uint8_t LBACnt, uint8_t *pLBABuf)
{
_pSCSI_READ_10_COMMAND	pSCSIRead10Command;
uint8_t	ii;
	
#if	HOST_MSC_MSG
	printf("LBA=0x%x, Count=1\r\n", LBAIndex);
#endif
	pMSC_CBW = (_pMSC_COMMAND_BLOCK)CBWData;
	pMSC_CBW->sDataTransferLength = MSC_BLOCK_SZ;
	pMSC_CBW->bCBLength = sizeof(_SCSI_READ_10_COMMAND);
	for (ii = 0; ii < sizeof(pMSC_CBW->CB); ii++)
		pMSC_CBW->CB[ii] = 0;
	pSCSIRead10Command = (_pSCSI_READ_10_COMMAND)pMSC_CBW->CB;
	pSCSIRead10Command->OPCode = SCSI_READ_10;
	pSCSIRead10Command->LUN_RelAdr = 0;
	pSCSIRead10Command->LBA3 = ((LBAIndex >> 24) & 0xff);
	pSCSIRead10Command->LBA2 = ((LBAIndex >> 16) & 0xff);
	pSCSIRead10Command->LBA1 = ((LBAIndex >> 8) & 0xff);
	pSCSIRead10Command->LBA0 = ((LBAIndex >> 0) & 0xff);
	pSCSIRead10Command->TransferLength0 = LBACnt;
	if (msc_issue_cbw_out(pHost, pMSC_CBW))
		return(USBH_RESULT_FAIL);
	if (msc_issue_data_in(pHost, MSC_BLOCK_SZ, pLBABuf))
	{
#if	HOST_MSC_MSG
		printf("Data Error\r\n");
#endif
		return(USBH_RESULT_FAIL);
	}
	pMSC_CSW = (_pMSC_COMMAND_STATUS)CSWData;
	if (msc_issue_csw_in(pHost, pMSC_CSW))
		return(USBH_RESULT_FAIL);
	if (pMSC_CSW->bStatus != CSW_STATUS_SUCCESS)
	{
		if (msc_issue_request_sense(pHost, CBWData, CSWData, MSCData))
			return(USBH_RESULT_FAIL);
	}
#if	HOST_MSC_MSG
	printf("Done\r\n");
#endif
	return(USBH_RESULT_PASS);
}
/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/
