/**********************************************************************************
 *
 * @file    usb_dev_hid.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_usb.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include "usbhid.h"
#include "usb_dev_hid.h"
#include <stdio.h>

#pragma pack(1)
#define  DEV_HID_MSG    1

#define  HID_IAP_START  0x45533332
#define  HID_IAP_FEATURE  0xA0
#define  IAP_BLDR_BASE  (0xD000)

static  uint8_t    bEPIdle = 0/*, bProtocol = 0*/;

/****
  * @brief  HID Class Request - Get Report
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  hid_request_get_report(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
  eUSBDState  USBState;
#if  DEV_HID_MSG
  printf("HIDGetReport\r\n");
#endif
   md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_clear_flag_device_ep0_rx_ready(USB);    //Clear RxRdy
  if (((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_class != NULL)
    USBState = (*(((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_class))(pDev, pSetupReq);
//  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_DATAEND | USB_CSR0L_TXRDY));  //TxRdy & DataEnd
  return(USBState);
}
/****
  * @brief  HID Class Request - Get Idle
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  hid_request_get_idle(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
#if  DEV_HID_MSG
  printf("HIDGetIdle\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_clear_flag_device_ep0_rx_ready(USB);    //Clear RxRdy
  md_usb_set_ep0_fifo(USB, bEPIdle);    //Idle Rate
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_DATAEND | USB_CSR0L_TXRDY));  //TxRdy & DataEnd
  return(USBD_STATE_IDLE);
}
/****
  * @brief  HID Class Request - Get Protocol
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  hid_request_get_protocol(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
#if  DEV_HID_MSG
  printf("HIDGetProtocol\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
  return(USBD_STATE_IDLE);
}
/****
  * @brief  HID Class Request - Set Report
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  hid_request_set_report(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
eUSBDState  USBState;

#if  DEV_HID_MSG
  printf("HIDSetReport\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_clear_flag_device_ep0_rx_ready(USB);    //Clear RxRdy
  USBState = USBD_STATE_CTRLOUT;
  if (((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_class != NULL)
    USBState = (*(((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_class))(pDev, pSetupReq);
  return(USBState);
}
/****
  * @brief  HID Class Request - Set Idle
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  hid_request_set_idle(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
eUSBDState  USBState;

#if  DEV_HID_MSG
  printf("HIDSetIdle\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
  bEPIdle = (pSetupReq->wValue >> 8);
  USBState = USBD_STATE_IDLE;
  if (((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_class != NULL)
    USBState = (*(((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_class))(pDev, pSetupReq);
  return(USBState);
}
/****
  * @brief  HID Class Request - Set Protocol
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  hid_request_set_protocol(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
eUSBDState  USBState;

#if  DEV_HID_MSG
  printf("HIDSetProtocol\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
//  bProtocol = (pSetupReq->wValue >> 8);
  USBState = USBD_STATE_IDLE;
  if (((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_class != NULL)
    USBState = (*(((_pCB_APP_DEV_REQUEST)pDev->cb_app_dev_request)->cb_app_dev_class))(pDev, pSetupReq);
  return(USBState);
}
/****
  * @brief  HID Class Request - Reserved
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
  * @retval  Idle State
****/
eUSBDState  hid_request_reserved(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
#if  DEV_HID_MSG
  printf("Reserved\r\n");
#endif
  md_usb_set_epx_index(USB, USBEP0);      //Endpoint 0
  md_usb_set_csr0l_txcsrl(USB, (USB_CSR0L_RXRDYC | USB_CSR0L_DATAEND));  //Clear RxRdy & Dataend
  return(USBD_STATE_IDLE);
}
/****
  * @brief  Functionality Array for HID Class Request
  * @param  pDev    : Device Instance
  * @param  pSetupReq  : Pointer to Standard Request Data
****/
eUSBDState  (*pHIDClassRequest[])(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST) =  //HID Class Request
{
  hid_request_reserved,
  hid_request_get_report,
  hid_request_get_idle,
  hid_request_get_protocol,
  hid_request_reserved,
  hid_request_reserved,
  hid_request_reserved,
  hid_request_reserved,
  hid_request_reserved,
  hid_request_set_report,
  hid_request_set_idle,
  hid_request_set_protocol,
};

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/

