/**
  *********************************************************************************
  *
  * @file    usb_utils.h
  * @brief   Header file for the USB UTILS.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USB_UTILS_H__
#define __USB_UTILS_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "uart_stdio.h"
#include "usb_lowlayer_api.h"

/** @addtogroup USB_LIBRARY
  * @{
  */

/** @addtogroup UTILS
  * @{
  */

/**
  * @defgroup UTILS_Private_Macros Utils Private Macros
  * @{
  */

#ifndef USB_MAX_DEVICES
#define USB_MAX_DEVICES	5
#endif
#ifndef USBLIB_NUM_EP
#define USBLIB_NUM_EP	8
#endif

#define USB_LANG_CHINESE_PRC	0x0804U
#define USB_LANG_CHINESE_TAIWAN	0x0404U
#define USB_LANG_EN_US		0x0409U
#define USB_LANG_EN_UK		0x0809U
#define USB_LANG_EN_AUS		0x0C09U
#define USB_LANG_EN_CA		0x1009U
#define USB_LANG_EN_NZ		0x1409U
#define USB_LANG_FRENCH		0x040CU
#define USB_LANG_GERMAN		0x0407U
#define USB_LANG_HINDI		0x0439U
#define USB_LANG_ITALIAN	0x0410U
#define USB_LANG_JAPANESE	0x0411U
#define USB_LANG_KOREAN		0x0412U
#define USB_LANG_ES_TRAD	0x040AU
#define USB_LANG_ES_MODERN	0x0C0AU
#define USB_LANG_SWAHILI	0x0441U
#define USB_LANG_URDU_IN	0x0820U
#define USB_LANG_URDU_PK	0x0420U

#define USB_RTYPE_DIR_IN	0x80U
#define USB_RTYPE_DIR_OUT	0x00U
#define USB_RTYPE_TYPE_M	0x60U
#define USB_RTYPE_VENDOR	0x40U
#define USB_RTYPE_CLASS		0x20U
#define USB_RTYPE_STANDARD	0x00U
#define USB_RTYPE_RECIPIENT_M	0x1fU
#define USB_RTYPE_OTHER		0x03U
#define USB_RTYPE_ENDPOINT	0x02U
#define USB_RTYPE_INTERFACE	0x01U
#define USB_RTYPE_DEVICE	0x00U
#define USBREQ_GET_STATUS	0x00U
#define USBREQ_CLEAR_FEATURE	0x01U
#define USBREQ_SET_FEATURE	0x03U
#define USBREQ_SET_ADDRESS	0x05U
#define USBREQ_GET_DESCRIPTOR	0x06U
#define USBREQ_SET_DESCRIPTOR	0x07U
#define USBREQ_GET_CONFIG	0x08U
#define USBREQ_SET_CONFIG	0x09U
#define USBREQ_GET_INTERFACE	0x0aU
#define USBREQ_SET_INTERFACE	0x0bU
#define USBREQ_SYNC_FRAME	0x0cU

#define USB_STATUS_SELF_PWR	0x0001U
#define USB_STATUS_BUS_PWR	0x0000U
#define USB_STATUS_PWR_M	0x0001U
#define USB_STATUS_REMOTE_WAKE	0x0002U

#define USB_FEATURE_EP_HALT	0x0000U
#define USB_FEATURE_REMOTE_WAKE	0x0001U
#define USB_FEATURE_TEST_MODE	0x0002U

#define USB_REQ_EP_NUM_M	0x007FU
#define USB_REQ_EP_DIR_M	0x0080U
#define USB_REQ_EP_DIR_IN	0x0080U
#define USB_REQ_EP_DIR_OUT	0x0000U

#define USB_DTYPE_DEVICE	1
#define USB_DTYPE_CONFIGURATION	2
#define USB_DTYPE_STRING	3
#define USB_DTYPE_INTERFACE	4
#define USB_DTYPE_ENDPOINT	5
#define USB_DTYPE_DEVICE_QUAL	6
#define USB_DTYPE_OSPEED_CONF	7
#define USB_DTYPE_INTERFACE_PWR	8
#define USB_DTYPE_OTG		9
#define USB_DTYPE_INTERFACE_ASC	11
#define USB_DTYPE_CS_INTERFACE	36
#define USB_DTYPE_HUB		41

#define USB_CLASS_DEVICE	0x00U
#define USB_CLASS_AUDIO		0x01U
#define USB_CLASS_CDC		0x02U
#define USB_CLASS_HID		0x03U
#define USB_CLASS_PHYSICAL	0x05U
#define USB_CLASS_IMAGE		0x06U
#define USB_CLASS_PRINTER	0x07U
#define USB_CLASS_MASS_STORAGE	0x08U
#define USB_CLASS_HUB		0x09U
#define USB_CLASS_CDC_DATA	0x0aU
#define USB_CLASS_SMART_CARD	0x0bU
#define USB_CLASS_SECURITY	0x0dU
#define USB_CLASS_VIDEO		0x0eU
#define USB_CLASS_HEALTHCARE	0x0fU
#define USB_CLASS_DIAG_DEVICE	0xdcU
#define USB_CLASS_WIRELESS	0xe0U
#define USB_CLASS_MISC		0xefU
#define USB_CLASS_APP_SPECIFIC	0xfeU
#define USB_CLASS_VEND_SPECIFIC	0xffU
#define USB_CLASS_EVENTS	0xffffffffU

#define USB_SUBCLASS_UNDEFINED		0x00U
#define USB_PROTOCOL_UNDEFINED		0x00U
#define USB_MISC_SUBCLASS_SYNC		0x01U
#define USB_MISC_SUBCLASS_COMMON	0x02U
#define USB_MISC_PROTOCOL_IAD		0x01U
#define USB_HUB_PROTOCOL_FS		0x00U
#define USB_HUB_PROTOCOL_SINGLE		0x01U
#define USB_HUB_PROTOCOL_MULTI		0x02U

#define USB_CONF_ATTR_PWR_M	0xC0U
#define USB_CONF_ATTR_SELF_PWR	0xC0U
#define USB_CONF_ATTR_BUS_PWR	0x80U
#define USB_CONF_ATTR_RWAKE	0xA0U

#define USB_EP_DESC_OUT			0x00
#define USB_EP_DESC_IN			0x80
#define USB_EP_DESC_NUM_M		0x0F
#define USB_EP_MAX_PACKET_COUNT_M	0x07FF
#define USB_EP_ATTR_CONTROL		0x00
#define USB_EP_ATTR_ISOC		0x01
#define USB_EP_ATTR_BULK		0x02
#define USB_EP_ATTR_INT			0x03
#define USB_EP_ATTR_TYPE_M		0x03
#define USB_EP_ATTR_ISOC_M		0x0C
#define USB_EP_ATTR_ISOC_NOSYNC		0x00
#define USB_EP_ATTR_ISOC_ASYNC		0x04
#define USB_EP_ATTR_ISOC_ADAPT		0x08
#define USB_EP_ATTR_ISOC_SYNC		0x0C
#define USB_EP_ATTR_USAGE_M		0x30
#define USB_EP_ATTR_USAGE_DATA		0x00
#define USB_EP_ATTR_USAGE_FEEDBACK	0x10
#define USB_EP_ATTR_USAGE_IMPFEEDBACK	0x20

#define USBShort(x)	(x & 0xff), (x >> 8)
#define USB3Byte(x)	(x & 0xff), ((x >> 8) & 0xff), ((x >> 16) & 0xff)
#define USBLong(x)	(x & 0xff), ((x >> 8) & 0xff), ((x >> 16) & 0xff), ((x >> 24) & 0xff)

#define NEXT_USB_DESCRIPTOR(x)	(desc_head_t *)(((uint8_t *)(x)) + *((uint8_t *)(x)))
#define USB_DESC_ANY 0xFFFFFFFFU

#define USBERR_HOST_IN_PID_ERROR	0x01000000U
#define USBERR_HOST_IN_NOT_COMP		0x00100000U
#define USBERR_HOST_IN_STALL		0x00400000U
#define USBERR_HOST_IN_DATA_ERROR	0x00080000U
#define USBERR_HOST_IN_NAK_TO		0x00080000U
#define USBERR_HOST_IN_ERROR		0x00040000U
#define USBERR_HOST_IN_FIFO_FULL	0x00020000U
#define USBERR_HOST_OUT_NAK_TO		0x00000080U
#define USBERR_HOST_OUT_NOT_COMP	0x00000080U
#define USBERR_HOST_OUT_STALL		0x00000020U
#define USBERR_HOST_OUT_ERROR		0x00000004U
#define USBERR_HOST_EP0_NAK_TO		0x00000080U
#define USBERR_HOST_EP0_ERROR		0x00000010U
#define USBERR_DEV_RX_DATA_ERROR	0x00080000U
#define USBERR_DEV_RX_OVERRUN		0x00040000U
#define USBERR_DEV_RX_FIFO_FULL		0x00020000U

#define USB_EVENT_BASE			0x0000U
#define USB_CLASS_EVENT_BASE		0x8000U
#define USBD_CDC_EVENT_BASE		(USB_CLASS_EVENT_BASE  + 0)
#define USBD_HID_EVENT_BASE		(USB_CLASS_EVENT_BASE  + 0x1000)
#define USBD_HID_KEYB_EVENT_BASE	(USBD_HID_EVENT_BASE   + 0x100)
#define USBD_BULK_EVENT_BASE		(USB_CLASS_EVENT_BASE  + 0x2000)
#define USBD_MSC_EVENT_BASE		(USB_CLASS_EVENT_BASE  + 0x3000)
#define USBD_AUDIO_EVENT_BASE		(USB_CLASS_EVENT_BASE  + 0x4000)
#define USBD_DFU_EVENT_BASE		(USB_CLASS_EVENT_BASE  + 0x5000)
#define USBD_PRINTER_EVENT_BASE		(USB_CLASS_EVENT_BASE  + 0x6000)
#define USBH_CDC_EVENT_BASE		(USBD_CDC_EVENT_BASE   + 0x800)
#define USBH_HID_EVENT_BASE		(USBD_HID_EVENT_BASE   + 0x800)
#define USBH_BULK_EVENT_BASE		(USBD_BULK_EVENT_BASE  + 0x800)
#define USBH_MSC_EVENT_BASE		(USBD_MSC_EVENT_BASE   + 0x800)
#define USBH_AUDIO_EVENT_BASE		(USBD_AUDIO_EVENT_BASE + 0x800)

#define USB_EVENT_CONNECTED		(USB_EVENT_BASE + 0)
#define USB_EVENT_DISCONNECTED		(USB_EVENT_BASE + 1)
#define USB_EVENT_RX_AVAILABLE		(USB_EVENT_BASE + 2)
#define USB_EVENT_DATA_REMAINING	(USB_EVENT_BASE + 3)
#define USB_EVENT_REQUEST_BUFFER	(USB_EVENT_BASE + 4)
#define USB_EVENT_TX_COMPLETE		(USB_EVENT_BASE + 5)
#define USB_EVENT_ERROR			(USB_EVENT_BASE + 6)
#define USB_EVENT_SUSPEND		(USB_EVENT_BASE + 7)
#define USB_EVENT_RESUME		(USB_EVENT_BASE + 8)
#define USB_EVENT_SCHEDULER		(USB_EVENT_BASE + 9)
#define USB_EVENT_STALL			(USB_EVENT_BASE + 10)
#define USB_EVENT_POWER_FAULT		(USB_EVENT_BASE + 11)
#define USB_EVENT_POWER_ENABLE		(USB_EVENT_BASE + 12)
#define USB_EVENT_POWER_DISABLE		(USB_EVENT_BASE + 13)
#define USB_EVENT_COMP_IFACE_CHANGE	(USB_EVENT_BASE + 14)
#define USB_EVENT_COMP_EP_CHANGE	(USB_EVENT_BASE + 15)
#define USB_EVENT_COMP_STR_CHANGE	(USB_EVENT_BASE + 16)
#define USB_EVENT_COMP_CONFIG		(USB_EVENT_BASE + 17)
#define USB_EVENT_UNKNOWN_CONNECTED	(USB_EVENT_BASE + 18)
#define USB_EVENT_SOF			(USB_EVENT_BASE + 19)
#define USB_EVENT_LPM_SLEEP		(USB_EVENT_BASE + 20)
#define USB_EVENT_LPM_RESUME		(USB_EVENT_BASE + 21)
#define USB_EVENT_LPM_ERROR		(USB_EVENT_BASE + 22)
#define USB_EVENT_CONFIG_CHANGE		(USB_EVENT_BASE + 23)

#define USB_FEATURE_LPM			0x1U
#define USB_FEATURE_LPM_RMT_WAKE	0x2U
#define USB_FEATURE_LPM_EN		0x1U
#define USB_FEATURE_LPM_DIS		0x0U
#define USB_FEATURE_CPUCLK		0x2U
#define USB_FEATURE_USBPLL		0x3U
#define USB_FEATURE_USBULPI		0x4U
#define USB_FEATURE_POWER		0x5U
#define USB_FEATURE_POWER_SELF		0x1U
#define USB_FEATURE_POWER_BUS		0x0U
#define USB_FEATURE_REMOTE_WAKEUP	0x2U

#define USB_PID_MOUSE			0x0000U
#define USB_PID_KEYBOARD		0x0001U
#define USB_PID_SERIAL			0x0002U
#define USB_PID_BULK			0x0003U
#define USB_PID_SCOPE			0x0004U
#define USB_PID_MSC			0x0005U
#define USB_PID_AUDIO			0x0006U
#define USB_PID_COMP_SERIAL		0x0007U
#define USB_PID_COMP_AUDIO_HID		0x0008U
#define USB_PID_COMP_HID_SER		0x0009U
#define USB_PID_COMP_HID_DFU		0x000AU
#define USB_PID_DATA_LOGGER		0x000BU
#define USB_PID_COMP_HID_HID		0x000DU
#define USB_PID_GAMEPAD			0x000FU
#define USB_PID_LP_CGAMEPAD		0x0010U
#define USB_PID_PRINTER			0x0011U
#define USB_PID_DFU			0x00FFU
#define USB_VID_EASTSOFT_30CC		0x30CCU

#define USB_SOF_TICK_DIVIDE		0x5U
#define USB_MAX_TICK_HANDLER		0x6U

#define USB_DMA_CH_MAX			0x8U
#define USB_DMA_EP_TX_TYPE		0x0U
#define USB_DMA_EP_RX_TYPE		0x1U
/**
  * @}
  */

/**
  * @defgroup UTILS_Public_Types Utils Public Types
  * @{
  */
#if defined(ccs)             || \
    defined(codered)         || \
    defined(gcc)             || \
    defined(rvmdk)           || \
    defined(__ARMCC_VERSION) || \
    defined(sourcerygxx)
#define PACKED __attribute__ ((packed))
#elif defined(__ICCARM__)
#define PACKED
#else
#error Unrecognized COMPILER!
#endif

#ifdef __ICCARM__
#pragma pack(1)
#endif

/**
  * @brief The standard USB request
  */
typedef struct {
	uint8_t bmRequestType;	/**< Type of the request */
	uint8_t bRequest;	/**< Mode of the request */
	uint16_t wValue;	/**< Value field */
	uint16_t wIndex;	/**< Index field */
	uint16_t wLength;	/**< Length field */
} PACKED usb_request_t;

/**
  * @brief Header of the standard USB descriptor
  */
typedef struct {
	uint8_t bLength;		/**< Length of this descriptor */
	uint8_t bDescriptorType;	/**< Type of this descriptor */
} PACKED desc_head_t;

/**
  * @brief Device descriptor
  */
typedef struct {
	uint8_t bLength;		/**< Length field */
	uint8_t bDescriptorType;	/**< Type field */
	uint16_t bcdUSB;		/**< Specification Release Number in BCD format */
	uint8_t bDeviceClass;		/**< Class code */
	uint8_t bDeviceSubClass;	/**< Sub class code */
	uint8_t bDeviceProtocol;	/**< Protocol code */
	uint8_t bMaxPacketSize0;	/**< The maximum packet size for endpoint zero */
	uint16_t idVendor;		/**< VID */
	uint16_t idProduct;		/**< PID */
	uint16_t bcdDevice;		/**< Device release number in BCD format */
	uint8_t iManufacturer;		/**< Index of a string descriptor */
	uint8_t iProduct;		/**< Index of a string descriptor */
	uint8_t iSerialNumber;		/**< Index of a string descriptor */
	uint8_t bNumConfigurations;	/**< Number of configuration descriptors */
} PACKED device_desc_t;

/**
  * @brief Configuration descriptor
  */
typedef struct {
	uint8_t bLength;		/**< Length field */
	uint8_t bDescriptorType;	/**< Type field */
	uint16_t wTotalLength;		/**< Total length of this configuration */
	uint8_t bNumInterfaces;		/**< Number of interfaces */
	uint8_t bConfigurationValue;	/**< Index for setting */
	uint8_t iConfiguration;		/**< Index of a string descriptor */
	uint8_t bmAttributes;		/**< Attributes */
	uint8_t bMaxPower;		/**< Maximum power, This is expressed in units of 2mA */
} PACKED config_desc_t;

/**
  * @brief Interface descriptor
  */
typedef struct {
	uint8_t bLength;		/**< Length field */
	uint8_t bDescriptorType;	/**< Type field */
	uint8_t bInterfaceNumber;	/**< Number of this interface */
	uint8_t bAlternateSetting;	/**< Alternate */
	uint8_t bNumEndpoints;		/**< Number of endpoints */
	uint8_t bInterfaceClass;	/**< Class code */
	uint8_t bInterfaceSubClass;	/**< Sub class code */
	uint8_t bInterfaceProtocol;	/**< Protocol code */
	uint8_t iInterface;		/**< Index of a string descriptor */
} PACKED interface_desc_t;

/**
  * @brief Endpoint descriptor
  */
typedef struct {
	uint8_t bLength;		/**< Length field */
	uint8_t bDescriptorType;	/**< Type field */
	uint8_t bEndpointAddress;	/**< Address of the endpoint */
	uint8_t bmAttributes;		/**< bmAttributes */
	uint16_t wMaxPacketSize;	/**< Maximum packet size */
	uint8_t bInterval;		/**< Polling interval */
} PACKED endpoint_desc_t;

/**
  * @brief String descriptor for index 0
  */
typedef struct {
	uint8_t bLength;		/**< Length field */
	uint8_t bDescriptorType;	/**< Type field */
	uint16_t wLANGID[1];		/**< Language code */
} PACKED string0_desc_t;

/**
  * @brief String descriptor
  */
typedef struct {
	uint8_t bLength;		/**< Length field */
	uint8_t bDescriptorType;	/**< Type field */
	uint8_t bString;		/**< First byte of the UNICODE string */
} PACKED string_desc_t;

#ifdef __ICCARM__
#pragma pack()
#endif

/**
  * @brief USB mode
  */
typedef enum {
	USB_LIB_MODE_DEVICE = 0x0,	/**< Device */
	USB_LIB_MODE_HOST   = 0x1,	/**< Host */
	USB_LIB_MODE_OTG    = 0x2,	/**< OTG */
	USB_LIB_MODE_NONE   = 0x3,	/**< None */
} usb_mode_t;

typedef void (* std_request)(void *inst, usb_request_t *req);
typedef void (* info_cbk)(void *inst, uint32_t info);
typedef void (* interface_cbk)(void *inst, uint8_t nr, uint8_t setting);
typedef void (* usb_int_handler)(void *inst);
typedef void (* usb_ep_int_handler)(void *inst, uint32_t status);
typedef void (* usb_device_handler)(void *inst, uint32_t req, void *data);
typedef void (*usb_mode_cbk)(uint32_t index, usb_mode_t mode);
typedef uint32_t (* usb_cbk)(void *arg, uint32_t event, uint32_t param, void *data);
typedef uint32_t (* usb_packet_send)(void *handle, uint8_t *data, uint32_t len, uint8_t last);
typedef uint32_t (* usb_packet_avail)(void *handle);
typedef void (* usb_tick_handler)(void *inst, uint32_t tick);

/**
  * @brief OTG status
  */
typedef enum {
	USB_OTG_MODE_IDLE      = 0,     /**< Idle */
	USB_OTG_MODE_WAIT      = 1,	/**< Waiting */
	USB_OTG_MODE_WAIT_ID   = 2,	/**< Waiting for ID */
	USB_OTG_MODE_WAIT_CONN = 3,	/**< Waiting for connectting */
	USB_OTG_MODE_DEVICE    = 4,	/**< Device */
	USB_OTG_MODE_HOST      = 5,	/**< Host */
	USB_OTG_MODE_ERR       = 6,	/**< VBUS error */
} usb_otg_state_t;

/**
  * @brief Basic handler functions[Device]
  */
typedef struct {
	std_request get_desc;			/**< Host requests a non-standard descriptor */
	std_request request_handler;		/**< Host makes a non-standard request */
	interface_cbk interface_change;		/**< Response to a SetInterface request from the host */
	info_cbk config_change;			/**< Response to a SetConfiguration request from the host */
	info_cbk data_recv;			/**< Received data from endpoint 0 */
	info_cbk data_send;			/**< Send data to endpoint 0 */
	usb_int_handler reset_handler;		/**< USB reset */
	usb_int_handler suspend_handler;	/**< USB suspend */
	usb_int_handler resume_handler;		/**< USB resume */
	usb_int_handler disconnect_handler;	/**< USB disconnected */
	usb_ep_int_handler ep_handler;		/**< Handle endpoint event expect endpoint 0 */
	usb_device_handler device_handler;	/**< Handle device event */
} base_handler_t;

/**
  * @brief Config section
  */
typedef struct {
	uint16_t size;		/**< Size of this section */
	const uint8_t *data;	/**< Data of this section */
} config_section_t;

/**
  * @brief Config header
  */
typedef struct {
	uint8_t nr_section;				/**< Number of sections */
	const config_section_t * const *section;	/**< Section pointer */
} config_head_t;

/**
  * @brief Event infonation
  */
typedef struct {
	uint32_t event;		/**< Event */
	uint32_t instance;	/**< Instance value */
} event_info_t;

/**
  * @brief Low power manage feature
  */
typedef struct {
	uint32_t hird;		/**< HIRD */
	uint32_t feature;	/**< Feature */
} lpm_feature_t;

/**
  * @brief Callback for internal DMA
  */
typedef struct {
#ifdef USB_DMA_N_SUPPORT
	void (*cplt_cbk)(void *arg);	/**< DMA transfers complete callback */
	void (*err_cbk)(void* arg);	/**< DMA occurs error callback */
#else
	void (*cplt_cbk)(uint8_t ch, void *arg);	/**< DMA transfers complete callback */
	void (*err_cbk)(uint8_t ch, void* arg);		/**< DMA occurs error callback */
#endif
	void *cplt_arg;					/**< The parameter of cplt_cbk() */
	void *err_arg;					/**< The parameter of err_cbk() */
} usb_dma_callback_t;

/**
  * @brief DMA environmental variable
  */
typedef struct usb_dma_env_s {
#ifdef USB_DMA_N_SUPPORT
#define USB_E_DMA_TYPE_MSC_TX	0
#define USB_E_DMA_TYPE_MSC_RX	1
#define USB_E_DMA_TYPE_AUDIO_RX	2
#define USB_E_DMA_TYPE_BULK_TX	3
#define USB_E_DMA_TYPE_BULK_RX	4
	uint8_t usbd_msc_tx_ch;			/**< External DMA channel for MSC TX */
	uint8_t usbd_msc_rx_ch;			/**< External DMA channel for MSC RX */
	uint8_t usbd_audio_rx_ch;		/**< External DMA channel for AUDIO RX */
	uint8_t usbd_bulk_tx_ch;		/**< External DMA channel for BULK TX */
	uint8_t usbd_bulk_rx_ch;		/**< External DMA channel for BULK RX */
	map_dma_handle_t msc_hdma_tx;		/**< Handle for MSC TX */
	map_dma_handle_t msc_hdma_rx;		/**< Handle for MSC RX */
	map_dma_handle_t audio_hdma_rx;		/**< Handle for AUDIO RX */
	map_dma_handle_t bulk_hdma_tx;		/**< Handle for BULK TX */
	map_dma_handle_t bulk_hdma_rx;		/**< Handle for BULK RX */
#else
	uint32_t usb_idx;			/**< Index of USB */
	uint32_t alloc;				/**< Bit mapping */
	uint8_t ep2ch[2][7];			/**< Get channel from endpoint */
	uint8_t ch2ep[USB_DMA_CH_MAX];		/**< Get endpoint from channel */
	uint8_t ep_type[USB_DMA_CH_MAX];	/**< Type of endpoint */
	usb_dma_callback_t cbk[USB_DMA_CH_MAX];	/**< Callback functions */
#endif
} usb_dma_env_t;
/**
  * @}
  */

/** @addtogroup UTILS_Public_Variables
  * @{
  */
extern volatile usb_mode_t __usb_mode;
extern uint32_t __usb_current_tick;
extern uint32_t __usb_sof_count;
/**
  * @}
  */
/** @addtogroup UTILS_Public_Functions
  * @{
  */
/** @addtogroup UTILS_Public_Functions_Group1
  * @{
  */
/* Descriptor parsing functions */
extern uint32_t usb_desc_get_nr(desc_head_t *desc, uint32_t size, uint32_t type);
extern desc_head_t *usb_desc_get(desc_head_t *desc, uint32_t size, uint32_t type, uint32_t idx);
extern uint32_t usb_desc_get_nr_alt_interface(config_desc_t *config, uint8_t nr_interface);
extern interface_desc_t *usb_desc_get_interface(config_desc_t *config, uint32_t idx, uint32_t alt);
extern endpoint_desc_t *usb_desc_get_ep(interface_desc_t *interface, uint32_t idx, uint32_t size);
/**
  * @}
  */
/** @addtogroup UTILS_Public_Functions_Group2
  * @{
  */
/* Tick timer functions */
extern void usb_tick_init(void);
extern void usb_tick_reset(void);
extern uint32_t usb_tick_get(void);
extern int32_t usb_tick_handler_register(usb_tick_handler handler, void *inst);
extern void usb_tick_inc(uint32_t tick);
/**
  * @}
  */
/** @addtogroup UTILS_Public_Functions_Group3
  * @{
  */
/* Internal DMA functions */
extern void usb_dma_init(uint32_t usb_idx);
extern void usb_dma_channel_free(uint8_t ep, uint32_t ep_type);
#ifdef USB_DMA_N_SUPPORT
extern map_dma_handle_t *usb_dma_channel_alloc(uint8_t ep, uint32_t type, usb_dma_callback_t *cbk);
extern int usb_dma_config(uint32_t type, uint32_t addr, uint32_t count);
#else
extern uint8_t usb_dma_channel_alloc(uint8_t ep, uint32_t ep_type, usb_dma_callback_t *cbk);
extern int usb_dma_config(uint8_t ch, uint32_t addr, uint32_t count);
#endif
extern int8_t usb_dma_ep_get(uint8_t ch);
extern void usb0_dma_int_handler(void);

/**
  * @}
  */
/** @addtogroup UTILS_Public_Functions_Group4
  * @{
  */
/* OTG mode functions */
extern void usb_mode_set(uint32_t idx, usb_mode_t mode, usb_mode_cbk cbk);
extern void usb_mode_dual_init(uint32_t idx);
extern void usb_mode_dual_term(uint32_t idx);
extern void usb_mode_dual_int_handler(void);
extern void usb_mode_otg_init(uint32_t idx, uint32_t rate, void *pool, uint32_t size);
extern void usb_mode_otg_term(uint32_t idx);
extern void usb_mode_otg_poll_rate_set(uint32_t idx, uint32_t rate);
extern void usb_mode_otg_int_handler(void);
extern void usb_mode_otg_device_disconnect(uint32_t idx);
extern void usb_mode_otg_main(void);
extern uint8_t usb_mode_otg_feature_set(uint32_t idx, uint32_t feature, void *arg);
/**
  * @}
  */
/**
  * @}
  */
/** @defgroup UTILS_Extern_Functions Extern Functions
  * @brief Extern functions
  * @{
  */
extern uint8_t usb_hcd_feature_set(uint32_t idx, uint32_t feature, void *arg);
extern uint8_t usb_dcd_feature_set(uint32_t idx, uint32_t feature, void *arg);
extern uint8_t usb_dcd_feature_get(uint32_t idx, uint32_t feature, void *arg);
extern void _usb_device_int_handler(uint32_t idx, uint32_t status);
extern void _usb_host_int_handler(uint32_t idx, uint32_t status);
extern void _usb_hcd_event_send(uint32_t idx, event_info_t *event, uint32_t flag);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
