/**
  *********************************************************************************
  *
  * @file    usb_utils.c
  * @brief   Functions related to basic operation.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include <string.h>
#include "usb_utils.h"
#include "usbh_core.h"

/** @defgroup USB_LIBRARY Eastsoft USB Stack
  * @brief Shanghai Eastsoft Microelectronics Cortex-M Chip USB Stack
  * @{
  */

/** @defgroup UTILS Utils
  * @brief Utils module driver
  * @{
  */
/** @defgroup UTILS_Private_Variables Private Variables
  * @{
  */
volatile uint32_t __poll_rate;
volatile uint32_t __wait_tick = 0;
volatile usb_otg_state_t __otg_mode_state;
static volatile usb_mode_t __dual_mode = USB_LIB_MODE_NONE;
static usb_mode_cbk __usb_mode_cbk;
static void *__pool;
static uint32_t __pool_size;
usb_tick_handler __tick_handler[USB_MAX_TICK_HANDLER];
void *__tick_instance[USB_MAX_TICK_HANDLER];
uint8_t __dma_init  = 0;
uint8_t __tick_init = 0;
usb_dma_env_t usb_dma_env[1];
/**
  * @}
  */

/** @defgroup UTILS_Public_Variables Public Variables
  * @{
  */
volatile usb_mode_t __usb_mode = USB_LIB_MODE_NONE;
uint32_t __usb_current_tick = 0;
uint32_t __usb_sof_count = 0;
/**
  * @}
  */

/** @defgroup UTILS_Private_Functions Private Functions
  * @{
  */
/**
  * @brief  Switching between host, device and unconfigured mode.
  * @param  mode: Mode.
  * @retval None
  */
static void usb_otg_mode_set(usb_mode_t mode)
{
	if ((mode != __dual_mode) || (__dual_mode == USB_LIB_MODE_NONE)) {
		if (mode == USB_LIB_MODE_NONE)
			__wait_tick = __poll_rate;
		if ((__usb_mode_cbk) && (__dual_mode != mode))
			__usb_mode_cbk(0, mode);

		__dual_mode = mode;
	}
}

/**
  * @brief  Get alternate interface
  * @param  config: Configuration descriptor.
  * @param  nr: Number of the descriptor.
  * @param  idx: Index of the descriptor.
  * @retval Pointer to the interface descriptor.
  */
static interface_desc_t *usb_desc_get_alt_interface(config_desc_t *config, uint8_t nr, uint32_t idx)
{
	desc_head_t *__desc;
	uint32_t len, cnt;

	__desc = (desc_head_t *)config;
	len    = 0;
	cnt    = 0;

	while (len < config->wTotalLength) {
		if ((__desc->bDescriptorType == USB_DTYPE_INTERFACE) && (((interface_desc_t *)__desc)->bInterfaceNumber == nr)) {
			if (cnt++ == idx)
				return (interface_desc_t *)__desc;
		}

		len   += (uint32_t)__desc->bLength;
		__desc = NEXT_USB_DESCRIPTOR(__desc);
	}

	return (interface_desc_t *)0;
}
/**
  * @}
  */
  
/** @defgroup UTILS_Public_Functions Public Functions
  * @{
  */
/** @defgroup UTILS_Public_Functions_Group1 descriptor parsing functions
  * @brief Descriptor parsing functions
  * @{
  */
/**
  * @brief  Get the number of individual descriptors.
  * @param  desc: The descriptor.
  * @param  size: Number of bytes of descriptor.
  * @param  type: Type of descriptor that is to be counted.
  * @retval Number of descriptors.
  */
uint32_t usb_desc_get_nr(desc_head_t *desc, uint32_t size, uint32_t type)
{
	desc_head_t *__desc;
	uint32_t len, cnt;

	__desc = desc;
	len    = 0;
	cnt    = 0;

	while (len < size) {
		if ((type == USB_DESC_ANY) || __desc->bDescriptorType == (type & 0xFF))
			cnt++;

		len   += (uint32_t)__desc->bLength;
		__desc = NEXT_USB_DESCRIPTOR(__desc);
	}

	return cnt;
}

/**
  * @brief  Get the descriptor.
  * @param  desc: The descriptor.
  * @param  size: Number of bytes of descriptor.
  * @param  type: Type of descriptor that is to be found.
  * @param  idx: Index of the descriptor.
  * @retval The descriptor.
  */
desc_head_t *usb_desc_get(desc_head_t *desc, uint32_t size, uint32_t type, uint32_t idx)
{
	desc_head_t *__desc;
	uint32_t len, cnt;

	__desc = desc;
	len    = 0;
	cnt    = 0;

	while (len < size) {
		if ((type == USB_DESC_ANY) || __desc->bDescriptorType == (type & 0xFF)) {
			if (cnt++ == idx)
				return __desc;
		}

		len   += (uint32_t)__desc->bLength;
		__desc = NEXT_USB_DESCRIPTOR(__desc);
	}

	return (desc_head_t *)0;
}

/**
  * @brief  Get the number of different alternate configurations.
  * @param  config: The configuration descriptor.
  * @param  nr: Number of alternate configurations is to be counted.
  * @retval Number of alternate.
  */
uint32_t usb_desc_get_nr_alt_interface(config_desc_t *config, uint8_t nr)
{
	desc_head_t *__desc;
	uint32_t len, cnt;

	__desc = (desc_head_t *)config;
	len    = 0;
	cnt    = 0;

	while (len < config->wTotalLength) {
		if ((__desc->bDescriptorType == USB_DTYPE_INTERFACE) && (((interface_desc_t *)__desc)->bInterfaceNumber == nr))
			++cnt;

		len   += (uint32_t)__desc->bLength;
		__desc = NEXT_USB_DESCRIPTOR(__desc);
	}

	return cnt;
}

/**
  * @brief  Get the interface descriptor from configuration descriptor.
  * @param  config: The configuration descriptor.
  * @param  idx: Index of the interface.
  * @param  alt: Alternate setting number.
  * @retval Interface descriptor.
  */
interface_desc_t *usb_desc_get_interface(config_desc_t *config, uint32_t idx, uint32_t alt)
{
	if (alt == USB_DESC_ANY)
		return (interface_desc_t *)usb_desc_get((desc_head_t *)config , config->wTotalLength, USB_DTYPE_INTERFACE, idx);
	else
		return usb_desc_get_alt_interface(config, idx, alt);
}

/**
  * @brief  Get the endpoint descriptor from interface descriptor.
  * @param  interface: The interface descriptor.
  * @param  idx: Index of the endpoint.
  * @param  size: Maximum number of bytes that the function may search beyond.
  * @retval Endpoint descriptor.
  */
endpoint_desc_t *usb_desc_get_ep(interface_desc_t *interface, uint32_t idx, uint32_t size)
{
	if (idx >= interface->bNumEndpoints)
		return (endpoint_desc_t *)0;
	else
		return (endpoint_desc_t *)usb_desc_get((desc_head_t *)interface, size, USB_DTYPE_ENDPOINT, idx);
}
/**
  * @}
  */

/** @defgroup UTILS_Public_Functions_Group2 tick timer functions
  * @brief Tick timer functions
  * @{
  */
/**
  * @brief  Initialize the variables.
  * @retval None.
  */
void usb_tick_init(void)
{
	int32_t i;

	if (__tick_init)
		return;

	for (i = 0; i < USB_MAX_TICK_HANDLER; ++i) {
		__tick_handler[i]  = (usb_tick_handler)0;
		__tick_instance[i] = 0;
	}

	__tick_init = 1;
	return;
}

/**
  * @brief  Resets the tick variables.
  * @retval None.
  */
void usb_tick_reset(void)
{
	__tick_init = 0;
}

/**
  * @brief  Gets the USB tick.
  * @retval Tick.
  */
uint32_t usb_tick_get(void)
{
	return __usb_current_tick;
}

/**
  * @brief  Registering timer handler functions.
  * @param  handler: The callback function.
  * @param  inst: Instance.
  * @retval 0-success, other value indicates failed.
  */
int32_t usb_tick_handler_register(usb_tick_handler handler, void *inst)
{
	int32_t i;

	for (i = 0; i < USB_MAX_TICK_HANDLER; ++i) {
		if (__tick_handler[i] == 0) {
			__tick_handler[i]  = handler;
			__tick_instance[i] = inst;

			break;
		}
	}

	if (i == USB_MAX_TICK_HANDLER)
		return -1;

	return 0;
}

/**
  * @brief  Increase tick.
  * @param  tick: How many milliseconds have passed.
  * @retval None
  */
void usb_tick_inc(uint32_t tick)
{
	int32_t i;

	__usb_current_tick += tick;

	for (i = 0; i < USB_MAX_TICK_HANDLER; ++i) {
		if (__tick_handler[i])
			__tick_handler[i](__tick_instance[i], tick);
	}
}
/**
  * @}
  */

/** @defgroup UTILS_Public_Functions_Group3 Internal DMA functions
  * @brief Internal DMA functions
  * @{
  */
/**
  * @brief  Initialize internal DMA.
  * @param  usb_idx: Index of USB.
  * @retval None
  */
void usb_dma_init(uint32_t usb_idx)
{
#ifndef USB_DMA_N_SUPPORT
	int i;
#endif

	if (__dma_init)
		return;

	memset(&usb_dma_env, 0x0, sizeof(usb_dma_env_t));

#ifdef USB_DMA_N_SUPPORT
	usb_dma_env[0].usbd_msc_tx_ch   = USBD_E_DMA_MSC_TX_CH;
	usb_dma_env[0].usbd_msc_rx_ch   = USBD_E_DMA_MSC_RX_CH;
	usb_dma_env[0].usbd_audio_rx_ch = USBD_E_DMA_AUDIO_RX_CH;
	usb_dma_env[0].usbd_bulk_tx_ch  = USBD_E_DMA_BULK_TX_CH;
	usb_dma_env[0].usbd_bulk_rx_ch  = USBD_E_DMA_BULK_RX_CH;
#else
	for (i = 0; i < 7; ++i) {
		usb_dma_env[0].ep2ch[USB_DMA_EP_TX_TYPE][i] = 0xFF;
		usb_dma_env[0].ep2ch[USB_DMA_EP_RX_TYPE][i] = 0xFF;
	}
#endif
	__dma_init = 1;
	return;
}


#ifdef USB_DMA_N_SUPPORT
/**
  * @brief  Alloc a external DMA channel.
  * @param  ep: Endpoint.
  * @param  type: Type of transfer.
  * @param  cbk: Callback function.
  * @retval Pointer to the handle.
  */
map_dma_handle_t *usb_dma_channel_alloc(uint8_t ep, uint32_t type, usb_dma_callback_t *cbk)
{
	map_dma_handle_t *hdl;

	switch (type) {
	case USB_E_DMA_TYPE_MSC_TX:
		hdl = &usb_dma_env[0].msc_hdma_tx;
		break;

	case USB_E_DMA_TYPE_MSC_RX:
		hdl = &usb_dma_env[0].msc_hdma_rx;
		break;

	case USB_E_DMA_TYPE_AUDIO_RX:
		hdl = &usb_dma_env[0].audio_hdma_rx;
		break;

	case USB_E_DMA_TYPE_BULK_TX:
		hdl = &usb_dma_env[0].bulk_hdma_tx;
		break;

	case USB_E_DMA_TYPE_BULK_RX:
		hdl = &usb_dma_env[0].bulk_hdma_rx;
		break;

	default:
		return NULL;
	}


	hdl->perh = EXT_DMA;
	hdl->cplt_cbk = cbk->cplt_cbk;
	hdl->cplt_arg = cbk->cplt_arg;
	hdl->err_cbk  = cbk->err_cbk;
	hdl->err_arg  = cbk->err_arg;
	map_dma_config_struct(&hdl->config);

	switch (type) {
	case USB_E_DMA_TYPE_MSC_TX:
		hdl->config.dst     = (void *)map_usb_fifo_addr_get(ep);
		hdl->config.dst_inc = DMA_DATA_DST_INC_NONE;
		hdl->config.src_inc = DMA_DATA_SRC_INC_BYTE;
		hdl->config.channel = usb_dma_env[0].usbd_msc_tx_ch;
		break;

	case USB_E_DMA_TYPE_MSC_RX:
		hdl->config.src     = (void *)map_usb_fifo_addr_get(ep);
		hdl->config.src_inc = DMA_DATA_SRC_INC_NONE;
		hdl->config.dst_inc = DMA_DATA_DST_INC_BYTE;
		hdl->config.channel = usb_dma_env[0].usbd_msc_rx_ch;
		break;

	case USB_E_DMA_TYPE_AUDIO_RX:
		hdl->config.src     = (void *)map_usb_fifo_addr_get(ep);
		hdl->config.src_inc = DMA_DATA_SRC_INC_NONE;
		hdl->config.dst_inc = DMA_DATA_DST_INC_BYTE;
		hdl->config.channel = usb_dma_env[0].usbd_audio_rx_ch;
		break;

	case USB_E_DMA_TYPE_BULK_TX:
		hdl->config.dst     = (void *)map_usb_fifo_addr_get(ep);
		hdl->config.dst_inc = DMA_DATA_DST_INC_NONE;
		hdl->config.src_inc = DMA_DATA_SRC_INC_BYTE;
		hdl->config.channel = usb_dma_env[0].usbd_bulk_tx_ch;
		break;

	case USB_E_DMA_TYPE_BULK_RX:
		hdl->config.src     = (void *)map_usb_fifo_addr_get(ep);
		hdl->config.src_inc = DMA_DATA_SRC_INC_NONE;
		hdl->config.dst_inc = DMA_DATA_DST_INC_BYTE;
		hdl->config.channel = usb_dma_env[0].usbd_bulk_rx_ch;
		break;

	default:
		break;
	}

	return hdl;
}
#else

/**
  * @brief  Alloc a DMA channel.
  * @param  ep: Endpoint.
  * @param  ep_type: Type of endpoint.
  * @param  cbk: Callback function.
  * @retval Channel.
  */
uint8_t usb_dma_channel_alloc(uint8_t ep, uint32_t ep_type, usb_dma_callback_t *cbk)
{
	int i, idx;

	idx = (ep_type & 0x01) ? 0 : 1;

	if (usb_dma_env[0].ep2ch[idx][ep] != 0xFF)
		return usb_dma_env[0].ep2ch[idx][ep];

	for (i = 0; i < USB_DMA_CH_MAX; ++i) {
		if ((usb_dma_env[0].alloc & (1 << i)) == 0)
			break;
	}

	if (i == USB_DMA_CH_MAX)
		return 0xFF;

	usb_dma_env[0].alloc         |= (1 << i);
	usb_dma_env[0].ep_type[i]     = ep_type;
	usb_dma_env[0].ep2ch[idx][ep] = i;
	usb_dma_env[0].ch2ep[i]       = ep;

	usb_dma_env[0].cbk[i].cplt_cbk = cbk->cplt_cbk;
	usb_dma_env[0].cbk[i].err_cbk  = cbk->err_cbk;
	usb_dma_env[0].cbk[i].cplt_arg = cbk->cplt_arg;
	usb_dma_env[0].cbk[i].err_arg  = cbk->err_arg;

	return i;
}
#endif

/**
  * @brief  Free the DMA channel.
  * @param  ep: Endpoint.
  * @param  ep_type: Type of endpoint.
  * @retval None
  */
void usb_dma_channel_free(uint8_t ep, uint32_t ep_type)
{
#ifndef USB_DMA_N_SUPPORT
	int idx;

	idx = (ep_type & 0x01) ? 0 : 1;
	usb_dma_env[0].alloc &= ~(1 << usb_dma_env[0].ep2ch[idx][ep]);
	usb_dma_env[0].ep2ch[idx][ep] = 0xFF;
#endif
	return;
}

#ifdef USB_DMA_N_SUPPORT
/**
  * @brief  Configure the external DMA channel.
  * @param  type: Type of transfer
  * @param  addr: Address of the buffer.
  * @param  count: Count for carrying.
  * @retval 0-success, other value indicates failed.
  */
int usb_dma_config(uint32_t type, uint32_t addr, uint32_t count)
{
	map_dma_handle_t *hdl;

	switch (type) {
	case USB_E_DMA_TYPE_MSC_TX:
		hdl = &usb_dma_env[0].msc_hdma_tx;
		hdl->config.src  = (void *)addr;
		hdl->config.size = count;
		break;

	case USB_E_DMA_TYPE_MSC_RX:
		hdl = &usb_dma_env[0].msc_hdma_rx;
		hdl->config.dst  = (void *)addr;
		hdl->config.size = count;
		break;

	case USB_E_DMA_TYPE_AUDIO_RX:
		hdl = &usb_dma_env[0].audio_hdma_rx;
		hdl->config.dst  = (void *)addr;
		hdl->config.size = count;
		break;

	case USB_E_DMA_TYPE_BULK_TX:
		hdl = &usb_dma_env[0].bulk_hdma_tx;
		hdl->config.src  = (void *)addr;
		hdl->config.size = count;
		break;

	case USB_E_DMA_TYPE_BULK_RX:
		hdl = &usb_dma_env[0].bulk_hdma_rx;
		hdl->config.dst  = (void *)addr;
		hdl->config.size = count;
		break;

	default:
		return -1;
	}

#ifdef USB_DRIVER_ALD
	map_dma_config_auto(hdl);
#endif
#ifdef USB_DRIVER_MD
	hdl->config.mode = ALD_DMA_MODE_MEM_TO_MEM;
	map_dma_config_basic(hdl);
#endif
	return 0;
}
#else

/**
  * @brief  Configure the DMA channel.
  * @param  ch: DMA channel.
  * @param  addr: Address of the buffer.
  * @param  count: Count for carrying.
  * @retval 0-success, other value indicates failed.
  */
int usb_dma_config(uint8_t ch, uint32_t addr, uint32_t count)
{
	uint32_t max, flag;
	uint32_t ctrl = 0;

	if ((addr & 0x3) || (count == 0))
		return -1;

	ctrl |= USB_DMA_CFG_START;
	ctrl |= USB_DMA_CFG_IE_EN;
	ctrl |= (usb_dma_env[0].ch2ep[ch] << USB_DMA_CFG_EP_POS);
	ctrl |= USB_DMA_CFG_BURST_NONE;

	if (usb_dma_env[0].ep_type[ch] == USB_DMA_EP_CFG_TX)
		ctrl |= USB_DMA_CFG_DIR_RD;

	map_usb_dev_ep_get_config(usb_dma_env[0].ch2ep[ch], &max, &flag);
	
	if (count <= max) {	/* Single Packet --> MODE0*/
		map_usb_dma_channel_config(ch, addr, count, ctrl);
	}
	else {	/* Multiple Packet --> MODE1 */
		ctrl |= USB_DMA_CFG_MODE_1;
		map_usb_ep_dma_config(usb_dma_env[0].ch2ep[ch], usb_dma_env[0].ep_type[ch], ENABLE);
		map_usb_dma_channel_config(ch, addr, count, ctrl);
	}

	return 0;
}
#endif


/**
  * @brief  Get endpoint from channel.
  * @param  ch: DMA channel.
  * @retval Endpoint.
  */
int8_t usb_dma_ep_get(uint8_t ch)
{
#ifdef USB_DMA_N_SUPPORT
	return 0;
#else
	if ((usb_dma_env[0].alloc & (1 << ch)) == 0)
		return -1;
	
	return usb_dma_env[0].ch2ep[ch];
#endif
}

/**
  * @brief  Internal DMA interrupt handler.
  * @retval None
  */
void usb0_dma_int_handler(void)
{
#ifndef USB_DMA_N_SUPPORT
	uint32_t i, err;
	uint32_t flag = map_usb_dma_get_interrupt_flag();

	for (i = 0; i < USB_DMA_CH_MAX; ++i) {
		err = map_usb_dma_get_channel_error(i);
		
		if (err && usb_dma_env[0].cbk[i].err_cbk) {
			usb_dma_env[0].cbk[i].err_cbk(i, usb_dma_env[0].cbk[i].err_arg);
			map_usb_dma_clear_channel_error(i);
			continue;
		}

		if ((flag & (1 << i)) && (usb_dma_env[0].cbk[i].cplt_cbk))
			usb_dma_env[0].cbk[i].cplt_cbk(i, usb_dma_env[0].cbk[i].cplt_arg);
	}
#endif
	return;
}

/**
  * @}
  */

/** @defgroup UTILS_Public_Functions_Group4 OTG mode functions
  * @brief OTG mode functions
  * @{
  */
/**
  * @brief  Set USB mode.
  * @param  idx: Index of USB.
  * @param  mode: USB mode:DEVICE/HOST/OTG.
  * @param  cbk: callback function.
  * @retval None
  */
void usb_mode_set(uint32_t idx, usb_mode_t mode, usb_mode_cbk cbk)
{
	__usb_mode = mode;
	__usb_mode_cbk = cbk;

	if ((mode == USB_LIB_MODE_DEVICE) || (mode == USB_LIB_MODE_HOST)) {
		if(__usb_mode_cbk)
			__usb_mode_cbk(0, mode);
	}
}

/**
  * @brief  Initialize dual mode.
  * @param  idx: Index of USB.
  * @retval None
  */
void usb_mode_dual_init(uint32_t idx)
{
	map_usb_host_ep_config(USB_EP_0, 64, 0, 0,
			(USB_EP_MODE_CTRL  |
			 USB_EP_SPEED_FULL |
			 USB_EP_HOST_OUT));

	map_usb_int_enable(USB_INTCTRL_RESET | USB_INTCTRL_DISCONNECT |
			USB_INTCTRL_SESSION | USB_INTCTRL_BABBLE |
			USB_INTCTRL_CONNECT | USB_INTCTRL_RESUME |
			USB_INTCTRL_SUSPEND | USB_INTCTRL_VBUS_ERR);

	map_usb_int_enable_ep(USB_INTEP_ALL);
	usb_tick_init();
	map_usb_int_register();
	map_usb_otg_session_request(true);

	return;
}

/**
  * @brief  terminal dual mode.
  * @param  idx: Index of USB.
  * @retval None
  */
void usb_mode_dual_term(uint32_t idx)
{
	map_usb_int_unregister();
	map_usb_int_disable(USB_INTCTRL_ALL);
	map_usb_int_disable_ep(USB_INTEP_ALL);
}

/**
  * @brief  USB interrupt handler.
  * @retval None
  */
void usb_mode_dual_int_handler(void)
{
	uint32_t state;

	state = map_usb_int_status_get();

	switch (__usb_mode) {
	case USB_LIB_MODE_HOST:
		_usb_host_int_handler(0, state);
		break;

	case USB_LIB_MODE_DEVICE:
		_usb_device_int_handler(0, state);
		break;

	default:
		break;
	}
}

/**
  * @brief  Initialize OTG mode.
  * @param  idx: Index of USB.
  * @param  rate: Rate in milliseconds to poll the controller for changes mode.
  * @param  pool: Data to use as a memory pool for host mode.
  * @param  size: Size of data.
  * @retval None
  */
void usb_mode_otg_init(uint32_t idx, uint32_t rate, void *pool, uint32_t size)
{
	__usb_mode  = USB_LIB_MODE_OTG;
	__dual_mode = USB_LIB_MODE_NONE;
	__poll_rate = rate;
	__pool      = pool;
	__pool_size = size;

#ifdef USB_DRIVER_MD
	map_usb_controller_enable();
	map_usb_clk_phy_enable();
	map_usb_mode_force_otg();
#endif	
	usb_hcd_init(idx, pool, size);
	map_usb_host_ep_config(USB_EP_0, 64, 0, 0, (USB_EP_MODE_CTRL |
				USB_EP_SPEED_FULL | USB_EP_HOST_OUT));
	map_usb_int_enable(USB_INTCTRL_RESET | USB_INTCTRL_DISCONNECT |
		       USB_INTCTRL_SESSION | USB_INTCTRL_BABBLE |
		       USB_INTCTRL_CONNECT | USB_INTCTRL_RESUME |
		       USB_INTCTRL_SUSPEND | USB_INTCTRL_VBUS_ERR |
		       USB_INTCTRL_MODE_DETECT | USB_INTCTRL_SOF);

	map_usb_int_enable_ep(USB_INTEP_ALL);
	map_usb_int_register();
	usb_hcd_power_config_set(idx, usb_hcd_power_config_get(idx));
	map_usb_host_pwr_enable();
	map_usb_otg_session_request(true);
	__otg_mode_state = USB_OTG_MODE_IDLE;
}

/**
  * @brief  terminal OTG mode.
  * @param  idx: Index of USB.
  * @retval None
  */
void usb_mode_otg_term(uint32_t idx)
{
	map_usb_int_unregister();
	map_usb_int_disable(USB_INTCTRL_ALL);
	map_usb_int_disable_ep(USB_INTEP_ALL);
	usb_otg_mode_set(USB_LIB_MODE_NONE);
}

/**
  * @brief  Set poll rate.
  * @param  idx: Index of USB.
  * @param  rate: Rate in milliseconds to poll the controller for changes mode.
  * @retval None
  */
void usb_mode_otg_poll_rate_set(uint32_t idx, uint32_t rate)
{
	__poll_rate = rate;
}

/**
  * @brief  USB interrupt handler.
  * @retval None
  */
void usb_mode_otg_int_handler(void)
{
	uint32_t state;

	state = map_usb_int_status_get();

	if (state & USB_INTCTRL_VBUS_ERR) {
		usb_otg_mode_set(USB_LIB_MODE_NONE);
		__otg_mode_state = USB_OTG_MODE_ERR;
	}

	if ((state & USB_INTCTRL_SUSPEND) && (__otg_mode_state == USB_OTG_MODE_DEVICE)) {
		usb_otg_mode_set(USB_LIB_MODE_NONE);
		__otg_mode_state = USB_OTG_MODE_WAIT;

		return;
	}
	
	if ((state & USB_INTCTRL_RESET) && (__otg_mode_state != USB_OTG_MODE_DEVICE)) {
		__otg_mode_state = USB_OTG_MODE_DEVICE;
		usb_otg_mode_set(USB_LIB_MODE_DEVICE);
	}

	if (state & USB_INTCTRL_CONNECT) {
		__otg_mode_state = USB_OTG_MODE_HOST;
		usb_otg_mode_set(USB_LIB_MODE_HOST);
	}

	switch(__otg_mode_state) {
	case USB_OTG_MODE_HOST:
		_usb_host_int_handler(0, state);
		break;

	case USB_OTG_MODE_DEVICE:
		_usb_device_int_handler(0, state);
		break;

	default:
		break;
	}
}

/**
  * @brief  Disconnect from USB bus.
  * @param  idx: Index of USB.
  * @retval None
  */
void usb_mode_otg_device_disconnect(uint32_t idx)
{
	if (__otg_mode_state == USB_OTG_MODE_HOST) {
		usb_otg_mode_set(USB_LIB_MODE_NONE);
		__otg_mode_state = USB_OTG_MODE_WAIT;
	}
}

/**
  * @brief  Main routine for the OTG mode.
  * @retval None
  */
void usb_mode_otg_main(void)
{
	switch (__otg_mode_state) {
	case USB_OTG_MODE_IDLE:
		__otg_mode_state = USB_OTG_MODE_WAIT_ID;
		map_usb_otg_session_request(true);
		break;

	case USB_OTG_MODE_HOST:
		usb_hcd_main();
		break;
	
	case USB_OTG_MODE_ERR:
		map_usb_control_reset();
		map_delay_ms(100);
		usb_mode_otg_init(0, 100, __pool, __pool_size);
		break;

	default:
		break;
	}
}

/**
  * @brief  Set feature for OTG mode
  * @param  idx: Index of USB.
  * @param  feature: Feature.
  * @param  arg: Paramter.
  * @retval Status.
  */
uint8_t usb_mode_otg_feature_set(uint32_t idx, uint32_t feature, void *arg)
{
	uint32_t ret = 0;

	usb_dcd_feature_set(idx, feature, arg);
	ret = usb_hcd_feature_set(idx, feature, arg);

	return ret;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
