/**
  *********************************************************************************
  *
  * @file    usb_hid.h
  * @brief   Header file for the USB HID.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USB_HID_H__
#define __USB_HID_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usb_utils.h"


/** @addtogroup USB_LIBRARY
  * @{
  */

/** @addtogroup UTILS
  * @{
  */

/**
  * @defgroup HID_Private_Macros HID Private Macros
  * @{
  */
#define USB_HID_SCLASS_NONE		0x00
#define USB_HID_SCLASS_BOOT		0x01
#define USB_HID_PROTOCOL_NONE		0
#define USB_HID_PROTOCOL_KEYB		1
#define USB_HID_PROTOCOL_MOUSE		2
#define USB_HID_DTYPE_HID		0x21
#define USB_HID_DTYPE_REPORT		0x22
#define USB_HID_DTYPE_PHYSICAL		0x23

#define USBREQ_GET_REPORT		0x01
#define USBREQ_GET_IDLE			0x02
#define USBREQ_GET_PROTOCOL		0x03
#define USBREQ_SET_REPORT		0x09
#define USBREQ_SET_IDLE			0x0a
#define USBREQ_SET_PROTOCOL		0x0b

#define USB_HID_REPORT_IN		0x01
#define USB_HID_REPORT_OUTPUT		0x02
#define USB_HID_REPORT_FEATURE		0x03
#define USB_HID_PROTOCOL_BOOT		0
#define USB_HID_PROTOCOL_REPORT		1
#define USB_HID_GENERIC_DESKTOP		0x01
#define USB_HID_BUTTONS			0x09
#define USB_HID_X			0x30
#define USB_HID_Y			0x31
#define USB_HID_Z			0x32
#define USB_HID_RX			0x33
#define USB_HID_RY			0x34
#define USB_HID_RZ			0x35
#define USB_HID_POINTER			0x01
#define USB_HID_MOUSE			0x02
#define USB_HID_JOYSTICK		0x04
#define USB_HID_GAME_PAD		0x05
#define USB_HID_KEYBOARD		0x06
#define USB_HID_PHYSICAL		0x00
#define USB_HID_APPLICATION		0x01
#define USB_HID_LOGICAL			0x02
#define USB_HID_USAGE_POINTER		0x0109
#define USB_HID_USAGE_BUTTONS		0x0509
#define USB_HID_USAGE_LEDS		0x0508
#define USB_HID_USAGE_KEYCODES		0x0507

#define HID_MOUSE_BUTTON_1		0x01
#define HID_MOUSE_BUTTON_2		0x02
#define HID_MOUSE_BUTTON_3		0x04

#define HID_KEYB_NUM_LOCK		0x01
#define HID_KEYB_CAPS_LOCK		0x02
#define HID_KEYB_SCROLL_LOCK		0x04
#define HID_KEYB_COMPOSE		0x08
#define HID_KEYB_KANA			0x10
#define HID_KEYB_LEFT_CTRL		0x01
#define HID_KEYB_LEFT_SHIFT		0x02
#define HID_KEYB_LEFT_ALT		0x04
#define HID_KEYB_LEFT_GUI		0x08
#define HID_KEYB_RIGHT_CTRL		0x10
#define HID_KEYB_RIGHT_SHIFT		0x20
#define HID_KEYB_RIGHT_ALT		0x40
#define HID_KEYB_RIGHT_GUI		0x80
#define HID_KEYB_USAGE_RESERVED		0x00
#define HID_KEYB_USAGE_ROLLOVER		0x01
#define HID_KEYB_USAGE_A		0x04
#define HID_KEYB_USAGE_B		0x05
#define HID_KEYB_USAGE_C		0x06
#define HID_KEYB_USAGE_D		0x07
#define HID_KEYB_USAGE_E		0x08
#define HID_KEYB_USAGE_F		0x09
#define HID_KEYB_USAGE_G		0x0A
#define HID_KEYB_USAGE_H		0x0B
#define HID_KEYB_USAGE_I		0x0C
#define HID_KEYB_USAGE_J		0x0D
#define HID_KEYB_USAGE_K		0x0E
#define HID_KEYB_USAGE_L		0x0F
#define HID_KEYB_USAGE_M		0x10
#define HID_KEYB_USAGE_N		0x11
#define HID_KEYB_USAGE_O		0x12
#define HID_KEYB_USAGE_P		0x13
#define HID_KEYB_USAGE_Q		0x14
#define HID_KEYB_USAGE_R		0x15
#define HID_KEYB_USAGE_S		0x16
#define HID_KEYB_USAGE_T		0x17
#define HID_KEYB_USAGE_U		0x18
#define HID_KEYB_USAGE_V		0x19
#define HID_KEYB_USAGE_W		0x1A
#define HID_KEYB_USAGE_X		0x1B
#define HID_KEYB_USAGE_Y		0x1C
#define HID_KEYB_USAGE_Z		0x1D
#define HID_KEYB_USAGE_1		0x1E
#define HID_KEYB_USAGE_2		0x1F
#define HID_KEYB_USAGE_3		0x20
#define HID_KEYB_USAGE_4		0x21
#define HID_KEYB_USAGE_5		0x22
#define HID_KEYB_USAGE_6		0x23
#define HID_KEYB_USAGE_7		0x24
#define HID_KEYB_USAGE_8		0x25
#define HID_KEYB_USAGE_9		0x26
#define HID_KEYB_USAGE_0		0x27
#define HID_KEYB_USAGE_ENTER		0x28
#define HID_KEYB_USAGE_ESCAPE		0x29
#define HID_KEYB_USAGE_BACKSPACE	0x2A
#define HID_KEYB_USAGE_TAB		0x2B
#define HID_KEYB_USAGE_SPACE		0x2C
#define HID_KEYB_USAGE_MINUS		0x2D
#define HID_KEYB_USAGE_EQUAL		0x2E
#define HID_KEYB_USAGE_LBRACKET		0x2F
#define HID_KEYB_USAGE_RBRACKET		0x30
#define HID_KEYB_USAGE_BSLASH		0x31
#define HID_KEYB_USAGE_SEMICOLON	0x33
#define HID_KEYB_USAGE_FQUOTE		0x34
#define HID_KEYB_USAGE_BQUOTE		0x35
#define HID_KEYB_USAGE_COMMA		0x36
#define HID_KEYB_USAGE_PERIOD		0x37
#define HID_KEYB_USAGE_FSLASH		0x38
#define HID_KEYB_USAGE_CAPSLOCK		0x39
#define HID_KEYB_USAGE_F1		0x3A
#define HID_KEYB_USAGE_F2		0x3B
#define HID_KEYB_USAGE_F3		0x3C
#define HID_KEYB_USAGE_F4		0x3D
#define HID_KEYB_USAGE_F5		0x3E
#define HID_KEYB_USAGE_F6		0x3F
#define HID_KEYB_USAGE_F7		0x40
#define HID_KEYB_USAGE_F8		0x41
#define HID_KEYB_USAGE_F9		0x42
#define HID_KEYB_USAGE_F10		0x43
#define HID_KEYB_USAGE_F11		0x44
#define HID_KEYB_USAGE_F12		0x45
#define HID_KEYB_USAGE_SCROLLOCK	0x47
#define HID_KEYB_USAGE_PAGE_UP		0x4B
#define HID_KEYB_USAGE_PAGE_DOWN	0x4E
#define HID_KEYB_USAGE_RIGHT_ARROW	0x4F
#define HID_KEYB_USAGE_LEFT_ARROW	0x50
#define HID_KEYB_USAGE_DOWN_ARROW	0x51
#define HID_KEYB_USAGE_UP_ARROW		0x52
#define HID_KEYB_USAGE_NUMLOCK		0x53
#define HID_KEYB_USAGE_KEYPAD_SLASH	0x54
#define HID_KEYB_USAGE_KEYPAD_STAR	0x55
#define HID_KEYB_USAGE_KEYPAD_MINUS	0x56
#define HID_KEYB_USAGE_KEYPAD_PLUS	0x57
#define HID_KEYB_USAGE_KEPAD_ENTER	0x58
#define HID_KEYB_USAGE_KEYPAD_1		0x59
#define HID_KEYB_USAGE_KEYPAD_2		0x5A
#define HID_KEYB_USAGE_KEYPAD_3		0x5B
#define HID_KEYB_USAGE_KEYPAD_4		0x5C
#define HID_KEYB_USAGE_KEYPAD_5		0x5D
#define HID_KEYB_USAGE_KEYPAD_6		0x5E
#define HID_KEYB_USAGE_KEYPAD_7		0x5F
#define HID_KEYB_USAGE_KEYPAD_8		0x60
#define HID_KEYB_USAGE_KEYPAD_9		0x61
#define HID_KEYB_USAGE_KEYPAD_0		0x62
#define HID_KEYB_USAGE_KEPAD_PERIOD	0x63

#define USB_HID_COUNTRY_NONE		0x00
#define USB_HID_COUNTRY_ARABIC		0x01
#define USB_HID_COUNTRY_BELGIAN		0x02
#define USB_HID_COUNTRY_CANADA_BI	0x03
#define USB_HID_COUNTRY_CANADA_FR	0x04
#define USB_HID_COUNTRY_CZECH_REPUBLIC	0x05
#define USB_HID_COUNTRY_DANISH		0x06
#define USB_HID_COUNTRY_FINNISH		0x07
#define USB_HID_COUNTRY_FRENCH		0x08
#define USB_HID_COUNTRY_GERMAN		0x09
#define USB_HID_COUNTRY_GREEK		0x0A
#define USB_HID_COUNTRY_HEBREW		0x0B
#define USB_HID_COUNTRY_HUNGARY		0x0C
#define USB_HID_COUNTRY_INTERNATIONAL_ISO \
                                        0x0D
#define USB_HID_COUNTRY_ITALIAN		0x0E
#define USB_HID_COUNTRY_JAPAN_KATAKANA	0x0F
#define USB_HID_COUNTRY_KOREAN		0x10
#define USB_HID_COUNTRY_LATIN_AMERICAN	0x11
#define USB_HID_COUNTRY_NETHERLANDS	0x12
#define USB_HID_COUNTRY_NORWEGIAN	0x13
#define USB_HID_COUNTRY_PERSIAN		0x14
#define USB_HID_COUNTRY_POLAND		0x15
#define USB_HID_COUNTRY_PORTUGUESE	0x16
#define USB_HID_COUNTRY_RUSSIA		0x17
#define USB_HID_COUNTRY_SLOVAKIA	0x18
#define USB_HID_COUNTRY_SPANISH		0x19
#define USB_HID_COUNTRY_SWEDISH		0x1A
#define USB_HID_COUNTRY_SWISS_FRENCH	0x1B
#define USB_HID_COUNTRY_SWISS_GERMAN	0x1C
#define USB_HID_COUNTRY_SWITZERLAND	0x1D
#define USB_HID_COUNTRY_TAIWAN		0x1E
#define USB_HID_COUNTRY_TURKISH_Q	0x1F
#define USB_HID_COUNTRY_UK		0x20
#define USB_HID_COUNTRY_US		0x21
#define USB_HID_COUNTRY_YUGOSLAVIA	0x22
#define USB_HID_COUNTRY_TURKISH_F	0x23

#define USB_HID_INPUT_DATA		0x0000
#define USB_HID_INPUT_CONSTANT		0x0001
#define USB_HID_INPUT_ARRAY		0x0000
#define USB_HID_INPUT_VARIABLE		0x0002
#define USB_HID_INPUT_ABS		0x0000
#define USB_HID_INPUT_RELATIVE		0x0004
#define USB_HID_INPUT_NOWRAP		0x0000
#define USB_HID_INPUT_WRAP		0x0008
#define USB_HID_INPUT_LINEAR		0x0000
#define USB_HID_INPUT_NONLINEAR		0x0010
#define USB_HID_INPUT_PREFER		0x0000
#define USB_HID_INPUT_NONPREFER		0x0020
#define USB_HID_INPUT_NONULL		0x0000
#define USB_HID_INPUT_NULL		0x0040
#define USB_HID_INPUT_BITF		0x0100
#define USB_HID_INPUT_BYTES		0x0000

#define USB_HID_FEATURE_DATA		0x0000
#define USB_HID_FEATURE_CONSTANT	0x0001
#define USB_HID_FEATURE_ARRAY		0x0000
#define USB_HID_FEATURE_VARIABLE	0x0002
#define USB_HID_FEATURE_ABS		0x0000
#define USB_HID_FEATURE_RELATIVE	0x0004
#define USB_HID_FEATURE_NOWRAP		0x0000
#define USB_HID_FEATURE_WRAP		0x0008
#define USB_HID_FEATURE_LINEAR		0x0000
#define USB_HID_FEATURE_NONLINEAR	0x0010
#define USB_HID_FEATURE_PREFER		0x0000
#define USB_HID_FEATURE_NONPREFER	0x0020
#define USB_HID_FEATURE_NONULL		0x0000
#define USB_HID_FEATURE_NULL		0x0040
#define USB_HID_FEATURE_BITF		0x0100
#define USB_HID_FEATURE_BYTES		0x0000

#define USB_HID_OUTPUT_DATA		0x0000
#define USB_HID_OUTPUT_CONSTANT		0x0001
#define USB_HID_OUTPUT_ARRAY		0x0000
#define USB_HID_OUTPUT_VARIABLE		0x0002
#define USB_HID_OUTPUT_ABS		0x0000
#define USB_HID_OUTPUT_RELATIVE		0x0004
#define USB_HID_OUTPUT_NOWRAP		0x0000
#define USB_HID_OUTPUT_WRAP		0x0008
#define USB_HID_OUTPUT_LINEAR		0x0000
#define USB_HID_OUTPUT_NONLINEAR	0x0010
#define USB_HID_OUTPUT_PREFER		0x0000
#define USB_HID_OUTPUT_NONPREFER	0x0020
#define USB_HID_OUTPUT_NONULL		0x0000
#define USB_HID_OUTPUT_NULL		0x0040
#define USB_HID_OUTPUT_BITF		0x0100
#define USB_HID_OUTPUT_BYTES		0x0000

#define USB_HID_BIAS_NOT_APPLICABLE	0x00
#define USB_HID_BIAS_RIGHT_HAND		0x01
#define USB_HID_BIAS_LEFT_HAND		0x02
#define USB_HID_BIAS_BOTH_HANDS		0x03
#define USB_HID_BIAS_EITHER_HAND	0x04

#define USB_HID_DESIGNATOR_NONE		0x00
#define USB_HID_DESIGNATOR_HAND		0x01
#define USB_HID_DESIGNATOR_EYEBALL	0x02
#define USB_HID_DESIGNATOR_EYEBROW	0x03
#define USB_HID_DESIGNATOR_EYELID	0x04
#define USB_HID_DESIGNATOR_EAR		0x05
#define USB_HID_DESIGNATOR_NOSE		0x06
#define USB_HID_DESIGNATOR_MOUTH	0x07
#define USB_HID_DESIGNATOR_UPPER_LIP	0x08
#define USB_HID_DESIGNATOR_LOWER_LIP	0x09
#define USB_HID_DESIGNATOR_JAW		0x0A
#define USB_HID_DESIGNATOR_NECK		0x0B
#define USB_HID_DESIGNATOR_UPPER_ARM	0x0C
#define USB_HID_DESIGNATOR_ELBOW	0x0D
#define USB_HID_DESIGNATOR_FOREARM	0x0E
#define USB_HID_DESIGNATOR_WRIST	0x0F
#define USB_HID_DESIGNATOR_PALM		0x10
#define USB_HID_DESIGNATOR_THUMB	0x11
#define USB_HID_DESIGNATOR_INDEX_FINGER	0x12
#define USB_HID_DESIGNATOR_MIDDLE_FINGER \
                                        0x13
#define USB_HID_DESIGNATOR_RING_FINGER	0x14
#define USB_HID_DESIGNATOR_LITTLE_FINGER \
                                        0x15
#define USB_HID_DESIGNATOR_HEAD		0x16
#define USB_HID_DESIGNATOR_SHOULDER	0x17
#define USB_HID_DESIGNATOR_HIP		0x18
#define USB_HID_DESIGNATOR_WAIST	0x19
#define USB_HID_DESIGNATOR_THIGH	0x1A
#define USB_HID_DESIGNATOR_KNEE		0x1B
#define USB_HID_DESIGNATOR_CALF		0x1C
#define USB_HID_DESIGNATOR_ANKLE	0x1D
#define USB_HID_DESIGNATOR_FOOT		0x1E
#define USB_HID_DESIGNATOR_HEEL		0x1F
#define USB_HID_DESIGNATOR_BALL_OF_FOOT	0x20
#define USB_HID_DESIGNATOR_BIG_TOE	0x21
#define USB_HID_DESIGNATOR_SECOND_TOE	0x22
#define USB_HID_DESIGNATOR_THIRD_TOE	0x23
#define USB_HID_DESIGNATOR_FOURTH_TOE	0x24
#define USB_HID_DESIGNATOR_LITTLE_TOE	0x25
#define USB_HID_DESIGNATOR_BROW		0x26
#define USB_HID_DESIGNATOR_CHEEK	0x27

#define USB_HID_QUALIFIER_NOT_APPLICABLE \
                                        (0x00 << 5)
#define USB_HID_QUALIFIER_RIGHT		(0x01 << 5)
#define USB_HID_QUALIFIER_LEFT		(0x02 << 5)
#define USB_HID_QUALIFIER_BOTH		(0x03 << 5)
#define USB_HID_QUALIFIER_EITHER	(0x04 << 5)
#define USB_HID_QUALIFIER_CENTER	(0x05 << 5)

#define USBH_HID_MAX_USAGE		256
#define USBH_HID_CAPS_ARRAY_SZ		(USBH_HID_MAX_USAGE / sizeof(uint32_t))
/**
  * @}
  */

  /**
  * @defgroup HID_Public_Types HID Public Types
  * @{
  */
#ifdef __ICCARM__
#pragma pack(1)
#endif

/**
  * @brief Descriptor information
  */
typedef struct {
	uint8_t type;	/**< Type of HID class descriptor */
	uint16_t len;	/**< Length of the HID class descriptor */
} PACKED hid_class_desc_info_t;

/**
  * @brief HID descriptor
  */
typedef struct {
	uint8_t len;			/**< Length */
	uint8_t type;			/**< Type */
	uint16_t bcd;			/**< HID BCD code */
	uint8_t country;		/**< Country */
	uint8_t nr;			/**< Number of class-specific descriptors */
	hid_class_desc_info_t desc[1];	/**< Table of descriptors */
} PACKED hid_desc_t;

#ifdef __ICCARM__
#pragma pack()
#endif

/**
  * @brief Defines the mapping of USB usage identifiers
  */
typedef struct {
	uint8_t byte_of_char;				/**< Number of bytes per character */
	uint32_t caps_lock[USBH_HID_CAPS_ARRAY_SZ];	/**< Bit mapping */
	void *mapping;					/**< Table of Usage ID to character value */
} hid_keyb_usage_table_t;

extern const hid_keyb_usage_table_t __keyb_map;
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
