/**
  *********************************************************************************
  *
  * @file    usb_cdc.h
  * @brief   Header file for the USB CDC.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USB_CDC_H__
#define __USB_CDC_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usb_utils.h"


/** @addtogroup USB_LIBRARY
  * @{
  */

/** @addtogroup UTILS
  * @{
  */

/**
  * @defgroup CDC_Private_Macros CDC Private Macros
  * @{
  */
#define BYTE(x)		(*(uint8_t *)(x))
#define SHORT(x)	(*(uint16_t *)(x))
#define LONG(x)		(*(uint32_t *)(x))

#define USB_CDC_SUBCLASS_DIRECT_LINE_MODEL		0x01U
#define USB_CDC_SUBCLASS_ABSTRACT_MODEL			0x02U
#define USB_CDC_SUBCLASS_TELEPHONE_MODEL		0x03U
#define USB_CDC_SUBCLASS_MULTI_CHANNEL_MODEL		0x04U
#define USB_CDC_SUBCLASS_CAPI_MODEL			0x05U
#define USB_CDC_SUBCLASS_ETHERNET_MODEL			0x06U
#define USB_CDC_SUBCLASS_ATM_MODEL			0x07U
#define USB_CDC_PROTOCOL_NONE				0x00U
#define USB_CDC_PROTOCOL_V25TER				0x01U
#define USB_CDC_PROTOCOL_VENDOR				0xFFU
#define USB_CDC_PROTOCOL_NONE				0x00U
#define USB_CDC_PROTOCOL_I420				0x30U
#define USB_CDC_PROTOCOL_TRANSPARENT			0x32U
#define USB_CDC_PROTOCOL_Q921M				0x50U
#define USB_CDC_PROTOCOL_Q921				0x51U
#define USB_CDC_PROTOCOL_Q921TM				0x52U
#define USB_CDC_PROTOCOL_V42BIS				0x90U
#define USB_CDC_PROTOCOL_Q921EURO			0x91U
#define USB_CDC_PROTOCOL_V120				0x92U
#define USB_CDC_PROTOCOL_CAPI20				0x93U
#define USB_CDC_PROTOCOL_HOST_DRIVER			0xFDU
#define USB_CDC_PROTOCOL_CDC_SPEC			0xFEU
#define USB_CDC_PROTOCOL_VENDOR				0xFFU
#define USB_CDC_CS_INTERFACE				0x24U
#define USB_CDC_CS_ENDPOINT				0x25U
#define USB_CDC_FD_SUBTYPE_HEADER			0x00U
#define USB_CDC_FD_SUBTYPE_CALL_MGMT			0x01U
#define USB_CDC_FD_SUBTYPE_ABSTRACT_CTL_MGMT		0x02U
#define USB_CDC_FD_SUBTYPE_DIRECT_LINE_MGMT		0x03U
#define USB_CDC_FD_SUBTYPE_TELEPHONE_RINGER		0x04U
#define USB_CDC_FD_SUBTYPE_LINE_STATE_CAPS		0x05U
#define USB_CDC_FD_SUBTYPE_UNION			0x06U
#define USB_CDC_FD_SUBTYPE_COUNTRY			0x07U
#define USB_CDC_FD_SUBTYPE_TELEPHONE_MODES		0x08U
#define USB_CDC_FD_SUBTYPE_USB_TERMINAL			0x09U
#define USB_CDC_FD_SUBTYPE_NETWORK_TERMINAL		0x0AU
#define USB_CDC_FD_SUBTYPE_PROTOCOL_UNIT		0x0BU
#define USB_CDC_FD_SUBTYPE_EXTENSION_UNIT		0x0CU
#define USB_CDC_FD_SUBTYPE_MULTI_CHANNEL_MGMT		0x0DU
#define USB_CDC_FD_SUBTYPE_CAPI_MGMT			0x0EU
#define USB_CDC_FD_SUBTYPE_ETHERNET			0x0FU
#define USB_CDC_FD_SUBTYPE_ATM				0x10U
#define USB_CDC_CALL_MGMT_VIA_DATA			0x02U
#define USB_CDC_CALL_MGMT_HANDLED			0x01U
#define USB_CDC_ACM_SUPPORTS_NETWORK_CONNECTION		0x08U
#define USB_CDC_ACM_SUPPORTS_SEND_BREAK			0x04U
#define USB_CDC_ACM_SUPPORTS_LINE_PARAMS		0x02U
#define USB_CDC_ACM_SUPPORTS_COMM_FEATURE		0x01U
#define USB_CDC_DLM_NEEDS_EXTRA_PULSE_SETUP		0x04U
#define USB_CDC_DLM_SUPPORTS_AUX			0x02U
#define USB_CDC_DLM_SUPPORTS_PULSE			0x01U
#define USB_CDC_TELEPHONE_SUPPORTS_COMPUTER		0x04U
#define USB_CDC_TELEPHONE_SUPPORTS_STANDALONE		0x02U
#define USB_CDC_TELEPHONE_SUPPORTS_SIMPLE		0x01U
#define USB_CDC_LINE_STATE_CHANGES_NOTIFIED		0x20U
#define USB_CDC_LINE_STATE_REPORTS_DTMF			0x10U
#define USB_CDC_LINE_STATE_REPORTS_DIST_RING		0x08U
#define USB_CDC_LINE_STATE_REPORTS_CALLERID		0x04U
#define USB_CDC_LINE_STATE_REPORTS_BUSY			0x02U
#define USB_CDC_LINE_STATE_REPORTS_INT_DIALTONE		0x01U
#define USB_CDC_TERMINAL_NO_WRAPPER_USED		0x00U
#define USB_CDC_TERMINAL_WRAPPER_USED			0x01U
#define USB_CDC_MCM_SUPPORTS_SET_UNIT_PARAM		0x04U
#define USB_CDC_MCM_SUPPORTS_CLEAR_UNIT_PARAM		0x02U
#define USB_CDC_MCM_UNIT_PARAMS_NON_VOLATILE		0x01U
#define USB_CDC_CAPI_INTELLIGENT			0x01U
#define USB_CDC_CAPI_SIMPLE				0x00U
#define USB_CDC_ETHERNET_XMIT_OK			0x01000000U
#define USB_CDC_ETHERNET_RCV_OK				0x02000000U
#define USB_CDC_ETHERNET_XMIT_ERROR			0x04000000U
#define USB_CDC_ETHERNET_RCV_ERROR			0x08000000U
#define USB_CDC_ETHERNET_RCV_NO_BUFFER			0x10000000U
#define USB_CDC_ETHERNET_DIRECTED_BYTES_XMIT		0x20000000U
#define USB_CDC_ETHERNET_DIRECTED_FRAMES_XMIT		0x40000000U
#define USB_CDC_ETHERNET_MULTICAST_BYTES_XMIT		0x80000000U
#define USB_CDC_ETHERNET_MULTICAST_FRAMES_XMIT		0x00010000U
#define USB_CDC_ETHERNET_BROADCAST_BYTES_XMIT		0x00020000U
#define USB_CDC_ETHERNET_BROADCAST_FRAMES_XMIT		0x00040000U
#define USB_CDC_ETHERNET_DIRECTED_BYTES_RCV		0x00080000U
#define USB_CDC_ETHERNET_DIRECTED_FRAMES_RCV		0x00100000U
#define USB_CDC_ETHERNET_MULTICAST_BYTES_RCV		0x00200000U
#define USB_CDC_ETHERNET_MULTICAST_FRAMES_RCV		0x00400000U
#define USB_CDC_ETHERNET_BROADCAST_BYTES_RCV		0x00800000U
#define USB_CDC_ETHERNET_BROADCAST_FRAMES_RCV		0x00000100U
#define USB_CDC_ETHERNET_RCV_CRC_ERROR			0x00000200U
#define USB_CDC_ETHERNET_TRANSMIT_QUEUE_LENGTH		0x00000400U
#define USB_CDC_ETHERNET_RCV_ERROR_ALIGNMENT		0x00000800U
#define USB_CDC_ETHERNET_XMIT_ONE_COLLISION		0x00001000U
#define USB_CDC_ETHERNET_XMIT_MORE_COLLISIONS		0x00002000U
#define USB_CDC_ETHERNET_XMIT_DEFERRED			0x00004000U
#define USB_CDC_ETHERNET_XMIT_MAX_COLLISIONS		0x00008000U
#define USB_CDC_ETHERNET_RCV_OVERRUN			0x00000001U
#define USB_CDC_ETHERNET_XMIT_UNDERRUN			0x00000002U
#define USB_CDC_ETHERNET_XMIT_HEARTBEAT_FAILURE		0x00000004U
#define USB_CDC_ETHERNET_XMIT_TIMES_CRS_LOST		0x00000010U
#define USB_CDC_ATM_TYPE_3				0x08U
#define USB_CDC_ATM_TYPE_2				0x04U
#define USB_CDC_ATM_TYPE_1				0x02U
#define USB_CDC_ATM_VC_US_CELLS_SENT			0x10U
#define USB_CDC_ATM_VC_US_CELLS_RECEIVED		0x08U
#define USB_CDC_ATM_DS_CELLS_HEC_ERR_CORRECTED		0x04U
#define USB_CDC_ATM_US_CELLS_SENT			0x02U
#define USB_CDC_ATM_US_CELLS_RECEIVED			0x01U
#define USB_CDC_SEND_ENCAPSULATED_COMMAND		0x00U
#define USB_CDC_GET_ENCAPSULATED_RESPONSE		0x01U
#define USB_CDC_SET_COMM_FEATURE			0x02U
#define USB_CDC_GET_COMM_FEATURE			0x03U
#define USB_CDC_CLEAR_COMM_FEATURE			0x04U
#define USB_CDC_SET_AUX_LINE_STATE			0x10U
#define USB_CDC_SET_HOOK_STATE				0x11U
#define USB_CDC_PULSE_SETUP				0x12U
#define USB_CDC_SEND_PULSE				0x13U
#define USB_CDC_SET_PULSE_TIME				0x14U
#define USB_CDC_RING_AUX_JACK				0x15U
#define USB_CDC_SET_LINE_CODING				0x20U
#define USB_CDC_GET_LINE_CODING				0x21U
#define USB_CDC_SET_CONTROL_LINE_STATE			0x22U
#define USB_CDC_SEND_BREAK				0x23U
#define USB_CDC_SET_RINGER_PARMS			0x30U
#define USB_CDC_GET_RINGER_PARMS			0x31U
#define USB_CDC_SET_OPERATION_PARMS			0x32U
#define USB_CDC_GET_OPERATION_PARMS			0x33U
#define USB_CDC_SET_LINE_PARMS				0x34U
#define USB_CDC_GET_LINE_PARMS				0x35U
#define USB_CDC_DIAL_DIGITS				0x36U
#define USB_CDC_SET_UNIT_PARAMETER			0x37U
#define USB_CDC_GET_UNIT_PARAMETER			0x38U
#define USB_CDC_CLEAR_UNIT_PARAMETER			0x39U
#define USB_CDC_GET_PROFILE				0x3AU
#define USB_CDC_SET_ETHERNET_MULTICAST_FILTERS		0x40U
#define USB_CDC_SET_ETHERNET_POWER_MANAGEMENT_PATTERN_FILTER	0x41U
#define USB_CDC_GET_ETHERNET_POWER_MANAGEMENT_PATTERN_FILTER	0x42U
#define USB_CDC_SET_ETHERNET_PACKET_FILTER		0x43U
#define USB_CDC_GET_ETHERNET_STATISTIC			0x44U
#define USB_CDC_SET_ATM_DATA_FORMAT			0x50U
#define USB_CDC_GET_ATM_DEVICE_STATISTICS		0x51U
#define USB_CDC_SET_ATM_DEFAULT_VC			0x52U
#define USB_CDC_GET_ATM_VC_STATISTICS			0x53U
#define USB_CDC_SIZE_COMM_FEATURE			2U
#define USB_CDC_SIZE_LINE_CODING			7U
#define USB_CDC_SIZE_RINGER_PARMS			4U
#define USB_CDC_SIZE_OPERATION_PARMS			2U
#define USB_CDC_SIZE_UNIT_PARAMETER			2U
#define USB_CDC_SIZE_PROFILE				64U
#define USB_CDC_SIZE_ETHERNET_POWER_MANAGEMENT_PATTERN_FILTER	2U
#define USB_CDC_SIZE_ETHERNET_STATISTIC			4U
#define USB_CDC_SIZE_ATM_DEVICE_STATISTICS		4U
#define USB_CDC_SIZE_ATM_VC_STATISTICS			4U
#define USB_CDC_SIZE_LINE_PARMS				10U
#define USB_CDC_ABSTRACT_STATE				0x0001U
#define USB_CDC_COUNTRY_SETTING				0x0002U
#define USB_CDC_ABSTRACT_CALL_DATA_MULTIPLEXED		0x0002U
#define USB_CDC_ABSTRACT_ENDPOINTS_IDLE			0x0001U

#define SetResponseCommFeature(buf, data)		\
do {							\
	(*(uint16_t *)(buf)) = data;			\
} while(0)

#define USB_CDC_AUX_DISCONNECT				0x0000U
#define USB_CDC_AUX_CONNECT				0x0001U
#define USB_CDC_ON_HOOK					0x0000U
#define USB_CDC_OFF_HOOK				0x0001U
#define USB_CDC_SNOOPING				0x0002U
#define USB_CDC_STOP_BITS_1				0x00U
#define USB_CDC_STOP_BITS_1_5				0x01U
#define USB_CDC_STOP_BITS_2				0x02U
#define USB_CDC_PARITY_NONE				0x00U
#define USB_CDC_PARITY_ODD				0x01U
#define USB_CDC_PARITY_EVEN				0x02U
#define USB_CDC_PARITY_MARK				0x03U
#define USB_CDC_PARITY_SPACE				0x04U

#define SetResponseLineCoding(buf, rate, stop, parity, data_bits)	\
do {									\
	(*(uint32_t *)(buf)) = rate;					\
	(*((uint8_t *)(buf) + 4)) = stop;				\
	(*((uint8_t *)(buf) + 5)) = parity;				\
	(*((uint8_t *)(buf) + 6)) = data_bits;				\
} while(0)

#define USB_CDC_DEACTIVATE_CARRIER			0x00U
#define USB_CDC_ACTIVATE_CARRIER			0x02U
#define USB_CDC_DTE_NOT_PRESENT				0x00U
#define USB_CDC_DTE_PRESENT				0x01U
#define USB_CDC_RINGER_EXISTS				0x80000000U
#define USB_CDC_RINGER_DOES_NOT_EXIST			0x00000000U

#define SetResponseRingerParms(buf, pattern, volume, exist)		\
do {									\
	*(uint32_t *)(buf) = ((pattern) + ((volume & 0xFF) << 8)	\
			+ (exist & USB_CDC_RINGER_EXISTS));		\
} while(0)

#define GetRingerVolume(x)		(BYTE((x) + 1))
#define GetRingerPattern(x)		(BYTE(x))
#define GetRingerExists(x)		((LONG(x)) & USB_CDC_RINGER_EXISTS)
#define USB_CDC_SIMPLE_MODE		0x0000U
#define USB_CDC_STANDALONE_MODE		0x0001U
#define USB_CDC_HOST_CENTRIC_MODE	0x0002U

#define SetResponseOperationParms(buf, data)	\
do {						\
	WORD(buf) = data;			\
} while(0)

#define USB_CDC_DROP_ACTIVE_CALL		0x0000U
#define USB_CDC_START_NEW_CALL			0x0001U
#define USB_CDC_APPLY_RINGING			0x0002U
#define USB_CDC_REMOVE_RINGING			0x0003U
#define USB_CDC_SWITCH_CALL			0x0004U
#define USB_CDC_LINE_IS_ACTIVE			0x80000000U
#define USB_CDC_LINE_IS_IDLE			0x00000000U
#define USB_CDC_LINE_NO_ACTIVE_CALL		0x000000FFU
#define USB_CDC_CALL_ACTIVE			0x80000000U
#define USB_CDC_CALL_IDLE			0x00000000U
#define USB_CDC_CALL_TYPICAL_DIALTONE		0x00000001U
#define USB_CDC_CALL_INTERRUPTED_DIALTONE	0x00000002U
#define USB_CDC_CALL_DIALING			0x00000003U
#define USB_CDC_CALL_RINGBACK			0x00000004U
#define USB_CDC_CALL_CONNECTED			0x00000005U
#define USB_CDC_CALL_INCOMING			0x00000006U
#define USB_CDC_CALL_STATE_IDLE			0x01U
#define USB_CDC_CALL_STATE_DIALING		0x02U
#define USB_CDC_CALL_STATE_RINGBACK		0x03U
#define USB_CDC_CALL_STATE_CONNECTED		0x04U
#define USB_CDC_CALL_STATE_INCOMING		0x05U
#define USB_CDC_VOICE				0x00U
#define USB_CDC_ANSWERING_MACHINE		0x01U
#define USB_CDC_FAX				0x02U
#define USB_CDC_MODEM				0x03U
#define USB_CDC_UNKNOWN				0xFFU
#define GetCallIndex(x)   			(BYTE(x))

#define SetResponseCallState(buf, idx, active, change, state)		\
do {									\
	(LONG((uint8_t *)(buf) + (10 + (4 * (idx))))) =			\
			(((active) & USB_CDC_CALL_IS_ACTIVE) +		\
			(((change) & 0xFF) << 8) +			\
			((state) & 0xFF));				\
} while(0)

#define SetResponseLineParms(buf, len, pattern, volume, exist,		\
				line_active, line_idx, call_active,	\
				change, state)				\
do {									\
	(WORD(buf)) = len;						\
	SetResponseRingerParams(((uint8_t *)(buf) + 2),			\
					pattern, volume, exist);	\
	(LONG((uint8_t *)(buf) + 6)) =					\
		(((line_active) & USB_CDC_LINE_IS_ACTIVE) +		\
		((line_idx) & 0xFF));					\
	SetResponseCallState(buf, 0, call_active, change, state);	\
} while(0)

#define USB_CDC_NOTIFY_NETWORK_CONNECTION	0x00U
#define USB_CDC_NOTIFY_RESPONSE_AVAILABLE	0x01U
#define USB_CDC_NOTIFY_AUX_JACK_HOOK_STATE	0x08U
#define USB_CDC_NOTIFY_RING_DETECT		0x09U
#define USB_CDC_NOTIFY_SERIAL_STATE		0x20U
#define USB_CDC_NOTIFY_CALL_STATE_CHANGE	0x28U
#define USB_CDC_NOTIFY_LINE_STATE_CHANGE	0x29U
#define USB_CDC_NOTIFY_CONNECTION_SPEED_CHANGE	0x2AU
#define USB_CDC_NETWORK_DISCONNECTED		0x0000U
#define USB_CDC_NETWORK_CONNECTED		0x0001U
#define USB_CDC_AUX_JACK_ON_HOOK		0x0000U
#define USB_CDC_AUX_JACK_OFF_HOOK		0x0001U
#define USB_CDC_NOTIFY_SERIAL_STATE_SIZE	2U
#define USB_CDC_SERIAL_STATE_OVERRUN		0x0040U
#define USB_CDC_SERIAL_STATE_PARITY		0x0020U
#define USB_CDC_SERIAL_STATE_FRAMING		0x0010U
#define USB_CDC_SERIAL_STATE_RING_SIGNAL	0x0008U
#define USB_CDC_SERIAL_STATE_BREAK		0x0004U
#define USB_CDC_SERIAL_STATE_TXCARRIER		0x0002U
#define USB_CDC_SERIAL_STATE_RXCARRIER		0x0001U

#define SetNotifyCallStatewValue(result, state, idx)			\
do {									\
	(WORD(result)) = (((state) & 0xFF) + (((idx) & 0xFF) << 8));	\
} while(0)

#define USB_CDC_LINE_STATE_IDLE			0x0000U
#define USB_CDC_LINE_STATE_HOLD			0x0001U
#define USB_CDC_LINE_STATE_OFF_HOOK		0x0002U
#define USB_CDC_LINE_STATE_ON_HOOK		0x0003U

#define SetNotifyConnectionSpeedChange(buf, us_rate, ds_rate)		\
do {									\
	LONG(buf) = us_rate;						\
	LONG((uint8_t *)(buf) + 4) = ds_rate;				\
} while(0)
/**
  * @}
  */
/**
  * @defgroup CDC_Public_Types CDC Public Types
  * @{
  */
#ifdef __ICCARM__
#pragma pack(1)
#endif

/**
  * @brief Request specific data
  */
typedef struct {
	uint32_t rate;		/**< Buad rate */
	uint8_t stop;		/**< Number of stop bits */
	uint8_t parity;		/**< Parity */
	uint8_t data_bit;	/**< Number of data bits per character */
} PACKED line_coding_t;

#ifdef __ICCARM__
#pragma pack()
#endif
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
