/**
  *********************************************************************************
  *
  * @file    uart_stdio.c
  * @brief   stdio file for usb stack
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "uart_stdio.h"
#ifdef USB_DRIVER_ALD
#if ((defined USB_DEBUG) && (defined(ES32F36xx) || defined(ES32F336x) || defined(ES32F326x)))
	#include "ald_cmu.h"
#elif ((defined USB_DEBUG) && (defined(ES32F0271) || defined(ES32F027x)))
	#include "ald_rcu.h"
#else
#endif
#include "ald_gpio.h"
#include "ald_uart.h"
#endif

/** @addtogroup USB_LIBRARY
  * @{
  */

/** @addtogroup UTILS
  * @{
  */
/** @addtogroup UTILS_Private_Variables
  * @{
  */
#ifdef USB_DRIVER_ALD
#if ((defined USB_DEBUG) && (defined(ES32F36xx) || defined(ES32F336x) || defined(ES32F326x)))
uart_handle_t h_uart;
#elif ((defined USB_DEBUG) && (defined(ES32F0271) || defined(ES32F027x)))
ald_uart_handle_t h_uart;
#else
#endif
#ifdef USB_DEBUG
char uart_buf[64];
#endif
#endif
/**
  * @}
  */

/** @addtogroup UTILS_Public_Functions
  * @{
  */
/** @defgroup UTILS_Public_Functions_Group5 Debug functions
  * @brief Debug functions
  * @{
  */
#ifdef USB_DRIVER_ALD
/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
#if ((defined USB_DEBUG) && (defined(ES32F36xx) || defined(ES32F336x) || defined(ES32F326x)))
	va_list args;

	va_start(args, fmt);
	vsnprintf(uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
	
#elif ((defined USB_DEBUG) && (defined(ES32F0271) || defined(ES32F027x)))
	va_list args;

	va_start(args, fmt);
	vsnprintf(uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);	
#else
	
#endif
	return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
#if ((defined USB_DEBUG) && (defined(ES32F36xx) || defined(ES32F336x) || defined(ES32F326x)))
	gpio_init_t x;

	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_UART1, ENABLE);
	
	/* Initialize tx pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_0, &x);

	/* Initialize rx pin */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_1, &x);

	/* Initialize uart */
	h_uart.perh             = UART1;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);
  
#elif ((defined USB_DEBUG) && (defined(ES32F0271) || defined(ES32F027x)))
	ald_gpio_handle_t gpiox = {0};
	
	/* Initialize mcu */
	/* config gpiob pin6 (Tx) */
	gpiox.perh  = GPIOB;
	gpiox.pin   = ALD_GPIO_PIN_6;
	gpiox.mod   = ALD_GPIO_MOD_FUNCTION;
	gpiox.pud   = ALD_GPIO_PUD_FLOATING;
	gpiox.ot    = ALD_GPIO_OT_PUSHPULL;
	gpiox.af    = ALD_GPIO_AF_AF2;
	ald_gpio_init(&gpiox);
	ald_gpio_pin_config(&gpiox);

	/* config gpiob pin7 (Rx) */
	gpiox.perh  = GPIOB;
	gpiox.pin   = ALD_GPIO_PIN_7;
	gpiox.mod   = ALD_GPIO_MOD_FUNCTION;
	gpiox.pud   = ALD_GPIO_PUD_FLOATING;
	gpiox.ot    = ALD_GPIO_OT_PUSHPULL;
	gpiox.af    = ALD_GPIO_AF_AF2;
	ald_gpio_pin_config(&gpiox);

	/* Initialize uart */
	h_uart.perh             = UART1;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = ALD_UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = ALD_UART_STOP_BITS_1;
	h_uart.init.parity      = ALD_UART_PARITY_NONE;
	h_uart.init.mode        = ALD_UART_MODE_UART;
	h_uart.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);
#else
#endif

	return;
}

#endif

#ifdef USB_DRIVER_MD
/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
#ifdef USB_DEBUG

#endif
	return;
}

/**
  * @brief  Initialize the UART1
  * @retval None
  */
void uart_stdio_init(void)
{
	return;
}
#endif
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
