/**
  *********************************************************************************
  *
  * @file    usbh_scsi.c
  * @brief   SCSI layer used by the USB host MSC driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usb_msc.h"
#include "usbh_scsi.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @defgroup Host_SCSI SCSI
  * @brief Host SCSI driver
  * @{
  */
/** @defgroup Host_SCSI_Private_Functions Private Functions
  * @{
  */
/**
  * @brief  Send SCSI commands.
  * @param  in_pipe: BULK IN pipe.
  * @param  out_pipe: BULK OUT pipe.
  * @param  cmd: SCSI command to send.
  * @param  data: SCSI command data.
  * @param  size: Size of the data.
  * @retval Status.
  */
static uint32_t usbh_scsi_send_cmd(uint32_t in_pipe, uint32_t out_pipe, msc_cbw_t *cmd, uint8_t *data, uint32_t *size)
{
	uint32_t len;
	msc_csw_t cmd_status;

	cmd_status.dCSWSignature = 0;
	cmd_status.dCSWTag       = 0;
	cmd_status.bCSWStatus    = SCSI_CMD_STATUS_FAIL;

	cmd->dCBWSignature = CBW_SIGNATURE;
	cmd->dCBWTag       = CBW_TAG_VALUE;
	cmd->dCBWDataTransferLength = *size;
	len = usb_hcd_pipe_write(out_pipe, (uint8_t*)cmd, sizeof(msc_cbw_t));

	if (len == 0)
		return SCSI_CMD_STATUS_FAIL;

	if (cmd->dCBWDataTransferLength != 0) {
		if (cmd->bmCBWFlags & CBWFLAGS_DIR_IN)
			*size = usb_hcd_pipe_read(in_pipe, data, *size);
		else
			*size = usb_hcd_pipe_write(out_pipe, data, *size);
	}

	len = usb_hcd_pipe_read(in_pipe, (uint8_t *)&cmd_status, sizeof(msc_csw_t));


	if ((len == 0) || (cmd_status.dCSWSignature != CSW_SIGNATURE) || (cmd_status.dCSWTag != CBW_TAG_VALUE))
		return SCSI_CMD_STATUS_FAIL;

	return (uint32_t)cmd_status.bCSWStatus;
}
/**
  * @}
  */

/** @defgroup Host_SCSI_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Send SCSI inquiry command.
  * @param  in_pipe: BULK IN pipe.
  * @param  out_pipe: BULK OUT pipe.
  * @param  data: SCSI command data.
  * @param  size: Size of the data.
  * @retval Status.
  */
uint32_t usbh_scsi_inquiry(uint32_t in_pipe, uint32_t out_pipe, uint8_t *data, uint32_t *size)
{
	msc_cbw_t cmd;
	uint32_t *buf;

	buf   = (uint32_t *)cmd.CBWCB;
	*size = SCSI_INQUIRY_DATA_SZ;

	cmd.bmCBWFlags   = CBWFLAGS_DIR_IN;
	cmd.bCBWLUN      = SLAVE_LUN;
	cmd.bCBWCBLength = 6;

	buf[0] = SCSI_INQUIRY_CMD;
	buf[1] = SCSI_INQUIRY_DATA_SZ;
	buf[2] = 0;
	buf[3] = 0;

	return usbh_scsi_send_cmd(in_pipe, out_pipe, &cmd, data, size);
}

/**
  * @brief  Send SCSI read capacity command.
  * @param  in_pipe: BULK IN pipe.
  * @param  out_pipe: BULK OUT pipe.
  * @param  data: SCSI command data.
  * @param  size: Size of the data.
  * @retval Status.
  */
uint32_t usbh_scsi_read_capacity(uint32_t in_pipe, uint32_t out_pipe, uint8_t *data, uint32_t *size)
{
	msc_cbw_t cmd;
	uint32_t *buf;

	buf   = (uint32_t *)cmd.CBWCB;
	*size = SCSI_READ_CAPACITY_SZ;

	cmd.bmCBWFlags   = CBWFLAGS_DIR_IN;
	cmd.bCBWLUN      = SLAVE_LUN;
	cmd.bCBWCBLength = 12;

	buf[0] = SCSI_READ_CAPACITY;
	buf[1] = 0;
	buf[2] = 0;
	buf[3] = 0;

	return usbh_scsi_send_cmd(in_pipe, out_pipe, &cmd, data, size);
}

/**
  * @brief  Send SCSI read capacities command.
  * @param  in_pipe: BULK IN pipe.
  * @param  out_pipe: BULK OUT pipe.
  * @param  data: SCSI command data.
  * @param  size: Size of the data.
  * @retval Status.
  */
uint32_t usbh_scsi_read_capacities(uint32_t in_pipe, uint32_t out_pipe, uint8_t *data, uint32_t *size)
{
	msc_cbw_t cmd;
	uint32_t *buf;

	buf = (uint32_t *)cmd.CBWCB;
	cmd.bmCBWFlags   = CBWFLAGS_DIR_IN;
	cmd.bCBWLUN      = SLAVE_LUN;
	cmd.bCBWCBLength = 12;

	buf[0] = SCSI_READ_CAPACITIES;
	buf[1] = 0;
	buf[2] = 0;
	buf[3] = 0;

	return usbh_scsi_send_cmd(in_pipe, out_pipe, &cmd, data, size);
}

/**
  * @brief  Send SCSI Mode Sense(6) command.
  * @param  in_pipe: BULK IN pipe.
  * @param  out_pipe: BULK OUT pipe.
  * @param  flag: Flags.
  * @param  data: SCSI command data.
  * @param  size: Size of the data.
  * @retval Status.
  */
uint32_t usbh_scsi_mode_sense6(uint32_t in_pipe, uint32_t out_pipe, uint32_t flag, uint8_t *data, uint32_t *size)
{
	msc_cbw_t cmd;
	uint32_t *buf;

	buf = (uint32_t *)cmd.CBWCB;

	cmd.bmCBWFlags   = CBWFLAGS_DIR_IN;
	cmd.bCBWLUN      = SLAVE_LUN;
	cmd.bCBWCBLength = 6;

	buf[0] = (SCSI_MODE_SENSE_6 | flag);
	buf[1] = (uint8_t)*size;
	buf[2] = 0;
	buf[3] = 0;

	return usbh_scsi_send_cmd(in_pipe, out_pipe, &cmd, data, size);
}

/**
  * @brief  Send SCSI Test Unit Ready command.
  * @param  in_pipe: BULK IN pipe.
  * @param  out_pipe: BULK OUT pipe.
  * @retval Status.
  */
uint32_t usbh_scsi_test_unit_ready(uint32_t in_pipe, uint32_t out_pipe)
{
	msc_cbw_t cmd;
	uint32_t size = 0;
	uint32_t *buf;

	buf = (uint32_t *)cmd.CBWCB;

	cmd.bmCBWFlags   = CBWFLAGS_DIR_IN;
	cmd.bCBWLUN      = SLAVE_LUN;
	cmd.bCBWCBLength = 6;

	buf[0] = SCSI_TEST_UNIT_READY;
	buf[1] = 0;
	buf[2] = 0;
	buf[3] = 0;

	return usbh_scsi_send_cmd(in_pipe, out_pipe, &cmd, 0, &size);
}

/**
  * @brief  Send SCSI Request Sense command.
  * @param  in_pipe: BULK IN pipe.
  * @param  out_pipe: BULK OUT pipe.
  * @param  data: SCSI command data.
  * @param  size: Size of the data.
  * @retval Status.
  */
uint32_t usbh_scsi_request_sense(uint32_t in_pipe, uint32_t out_pipe, uint8_t *data, uint32_t *size)
{
	msc_cbw_t cmd;
	uint32_t *buf;

	buf = (uint32_t *)cmd.CBWCB;

	cmd.bmCBWFlags   = CBWFLAGS_DIR_IN;
	cmd.bCBWLUN      = SLAVE_LUN;
	cmd.bCBWCBLength = 12;

	buf[0] = SCSI_REQUEST_SENSE;
	buf[1] = 18;
	buf[2] = 0;
	buf[3] = 0;

	return usbh_scsi_send_cmd(in_pipe, out_pipe, &cmd, data, size);
}

/**
  * @brief  Send SCSI Read(10) command.
  * @param  in_pipe: BULK IN pipe.
  * @param  out_pipe: BULK OUT pipe.
  * @param  lba: Logical block address.
  * @param  data: Buffer to return the data.
  * @param  size: Size of the buffer.
  * @param  nr: Number of contiguous blocks.
  * @retval Status.
  */
uint32_t usbh_scsi_read10(uint32_t in_pipe, uint32_t out_pipe, uint32_t lba, uint8_t *data, uint32_t *size, uint32_t nr)
{
	msc_cbw_t cmd;
	int32_t i;

	for (i = 0; i < sizeof(cmd.CBWCB); ++i)
		cmd.CBWCB[i] = 0;

	cmd.bmCBWFlags   = CBWFLAGS_DIR_IN;
	cmd.bCBWLUN      = SLAVE_LUN;
	cmd.bCBWCBLength = 10;

	cmd.CBWCB[0] = SCSI_READ_10;
	cmd.CBWCB[2] = (uint8_t)(lba >> 24);
	cmd.CBWCB[3] = (uint8_t)(lba >> 16);
	cmd.CBWCB[4] = (uint8_t)(lba >> 8);
	cmd.CBWCB[5] = (uint8_t)lba;
	cmd.CBWCB[7] = (uint8_t)(nr >> 8);
	cmd.CBWCB[8] = (uint8_t)nr;

	return usbh_scsi_send_cmd(in_pipe, out_pipe, &cmd, data, size);
}

/**
  * @brief  Send SCSI Write(10) command.
  * @param  in_pipe: BULK IN pipe.
  * @param  out_pipe: BULK OUT pipe.
  * @param  lba: Logical block address.
  * @param  data: Buffer to write out.
  * @param  size: Size of the buffer.
  * @param  nr: Number of contiguous blocks.
  * @retval Status.
  */
uint32_t usbh_scsi_write10(uint32_t in_pipe, uint32_t out_pipe, uint32_t lba, uint8_t *data, uint32_t *size, uint32_t nr)
{
	msc_cbw_t cmd;
	uint32_t *buf;

	buf = (uint32_t *)cmd.CBWCB;

	cmd.bmCBWFlags   = CBWFLAGS_DIR_OUT;
	cmd.bCBWLUN      = SLAVE_LUN;
	cmd.bCBWCBLength = 10;

	cmd.CBWCB[0] = SCSI_WRITE_10;
	cmd.CBWCB[1] = 0;
	cmd.CBWCB[2] = (uint8_t)(lba >> 24);
	cmd.CBWCB[3] = (uint8_t)(lba >> 16);
	cmd.CBWCB[4] = (uint8_t)(lba >> 8);
	cmd.CBWCB[5] = (uint8_t)lba;
	cmd.CBWCB[6] = 0;
	cmd.CBWCB[7] = (nr & 0xFF00) >> 8;

	buf[2] = (nr & 0xFF);
	buf[3] = 0;

	return usbh_scsi_send_cmd(in_pipe, out_pipe, &cmd, data, size);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
