/**
  *********************************************************************************
  *
  * @file    usbh_msc.c
  * @brief   USB MSC host driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usb_msc.h"
#include "usbh_msc.h"
#include "usbh_scsi.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @defgroup Host_MSC MSC
  * @brief MSC host driver
  * @{
  */
/** @defgroup Host_MSC_Private_Functions Private Functions
  * @{
  */
static void *usbh_msc_open(usbh_device_t *dev);
static void usbh_msc_close(void *inst);
/**
  * @}
  */
/** @defgroup Host_MSC_Private_Variables Private Variables
  * @{
  */
static usbh_msc_inst_t __msc_dev = {
	0
};

const usbh_class_driver_t __usbh_msc_driver = {
	USB_CLASS_MASS_STORAGE,
	usbh_msc_open,
	usbh_msc_close,
	0
};
/**
  * @}
  */

/** @addtogroup Host_MSC_Private_Functions
  * @{
  */
/**
  * @brief  Open an instance of the MSC driver.
  * @param  dev: Device information structure.
  * @retval None
  */
static void *usbh_msc_open(usbh_device_t *dev)
{
	int32_t i;
	endpoint_desc_t *ep;
	interface_desc_t *iface;

	if (__msc_dev.dev)
		return 0;

	__msc_dev.dev = dev;
	iface = usb_desc_get_interface(dev->desc_config, 0, 0);

	for (i = 0; i < 3; ++i) {
		ep = usb_desc_get_ep(iface, i, dev->size_config);

		if (ep == NULL)
			break;

		if ((ep->bmAttributes & USB_EP_ATTR_TYPE_M) != USB_EP_ATTR_BULK)
			continue;

		if (ep->bEndpointAddress & USB_EP_DESC_IN) {
			__msc_dev.in_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_BULK_IN, dev, ep->wMaxPacketSize, 0);
			usb_hcd_pipe_config(__msc_dev.in_pipe, ep->wMaxPacketSize, 0, (ep->bEndpointAddress & USB_EP_DESC_NUM_M));
		}
		else {
			__msc_dev.out_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_BULK_OUT, dev, ep->wMaxPacketSize, 0);
			usb_hcd_pipe_config(__msc_dev.out_pipe, ep->wMaxPacketSize, 0, (ep->bEndpointAddress & USB_EP_DESC_NUM_M));
		}
	}

	if (__msc_dev.cbk)
		__msc_dev.cbk(&__msc_dev, MSC_EVENT_OPEN, 0);

	__msc_dev.max_lun = 0xffffffff;
	return &__msc_dev;
}

/**
  * @brief  Release an instance of the MSC driver.
  * @param  inst: Device that needs to be released.
  * @retval None
  */
static void usbh_msc_close(void *inst)
{
	if (__msc_dev.dev == 0)
		return;

	__msc_dev.dev = 0;

	if (__msc_dev.in_pipe != 0)
		usb_hcd_pipe_free(__msc_dev.in_pipe);
	if (__msc_dev.out_pipe != 0)
		usb_hcd_pipe_free(__msc_dev.out_pipe);
	if (__msc_dev.cbk)
		__msc_dev.cbk(&__msc_dev, MSC_EVENT_CLOSE, 0);

	return;
}

/**
  * @brief  Get the maximum number of the logical units.
  * @param  dev: Device information structure.
  * @param  interface: Interface number.
  * @param  max_lun: Device's maximum logical unit.
  * @retval None
  */
static void usbh_msc_get_max_lun(usbh_device_t *dev, uint32_t interface, uint8_t *max_lun)
{
	usb_request_t req;

	req.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_CLASS | USB_RTYPE_INTERFACE;
	req.bRequest      = USBREQ_GET_MAX_LUN;
	req.wValue        = 0;
	req.wIndex        = (uint16_t)interface;
	req.wLength       = 1;

	if ((usb_hcd_ctrl_transfer(0, &req, dev, max_lun, 1, MAX_PACKET_SIZE_EP0)) != 1)
		*max_lun = 0;

	return;
}
/**
  * @}
  */

/** @defgroup Host_MSC_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Checks if a drive is ready to be accessed.
  * @param  inst: Device instance to use for this read.
  * @retval Status.
  */
int32_t usbh_msc_driver_ready(usbh_msc_inst_t *inst)
{
	uint8_t max_lun = 0, buf[SCSI_INQUIRY_DATA_SZ];
	uint32_t size;

	if (inst->dev == 0)
		return -1;

	if (__msc_dev.max_lun == 0xffffffff) {
		usbh_msc_get_max_lun(__msc_dev.dev, __msc_dev.dev->interface, &max_lun);
		__msc_dev.max_lun = max_lun;
	}
	printf_e("Max LUN: %d\n\r", max_lun);

	size = SCSI_REQUEST_SENSE_SZ;
	if ((usbh_scsi_request_sense(inst->in_pipe, inst->out_pipe, buf, &size)) != SCSI_CMD_STATUS_PASS)
		return -2;
	if ((buf[SCSI_RS_SKEY] == SCSI_RS_KEY_UNIT_ATTN) && (buf[SCSI_RS_SKEY_AD_SKEY] == SCSI_RS_KEY_NOTPRSNT))
		return -3;

	size = SCSI_INQUIRY_DATA_SZ;
	if ((usbh_scsi_inquiry(inst->in_pipe, inst->out_pipe, buf, &size) != SCSI_CMD_STATUS_PASS))
		return -4;

	size = SCSI_INQUIRY_DATA_SZ;
	if (usbh_scsi_read_capacity(inst->in_pipe, inst->out_pipe, buf, &size) != SCSI_CMD_STATUS_PASS) {
		size = SCSI_REQUEST_SENSE_SZ;
		usbh_scsi_request_sense(inst->in_pipe, inst->out_pipe, buf, &size);

		if (usbh_scsi_test_unit_ready(inst->in_pipe, inst->out_pipe) != SCSI_CMD_STATUS_PASS) {
			size = SCSI_REQUEST_SENSE_SZ;
			usbh_scsi_request_sense(inst->in_pipe, inst->out_pipe, buf, &size);
		}

		return -5;
	}
	else {
		inst->nr_block   = (buf[3] | (buf[2] << 8) | buf[1] << 16 | (buf[0] << 24));
		inst->size_block = (buf[7] | (buf[6] << 8) | buf[5] << 16 | (buf[4] << 24));
	}

	if (usbh_scsi_test_unit_ready(inst->in_pipe, inst->out_pipe) != SCSI_CMD_STATUS_PASS) {
		size = SCSI_REQUEST_SENSE_SZ;
		usbh_scsi_request_sense(inst->in_pipe, inst->out_pipe, buf, &size);

		return -6;
	}

	return 0;
}

/**
  * @brief  Enable the mass storage device class driver.
  * @param  drv: Index of driver.
  * @param  cbk: Callback for the mass storage events.
  * @retval The driver instance.
  */
usbh_msc_inst_t *usbh_msc_driver_open(uint32_t drv, usbh_msc_cbk cbk)
{
	if ((drv != 0) || (__msc_dev.cbk))
		return 0;

	__msc_dev.cbk = cbk;
	return &__msc_dev;
}

/**
  * @brief  Disable the mass storage device class driver.
  * @param  inst: Device that is to be released.
  * @retval None
  */
void usbh_msc_driver_close(usbh_msc_inst_t *inst)
{
	usbh_msc_close((void *)inst);
	inst->cbk = 0;

	return;
}

/**
  * @brief  Read block from an MSC device.
  * @param  inst: Device to use for this read.
  * @param  lba: Logical block address.
  * @param  data: The buffer.
  * @param  nr_block: Number of blocks to read from the device.
  * @retval Status
  */
int32_t usbh_msc_block_read(usbh_msc_inst_t *inst, uint32_t lba, uint8_t *data, uint32_t nr_block)
{
	uint32_t size;

	if (inst->dev == 0)
		return -1;

	size = inst->size_block * nr_block;

	if (usbh_scsi_read10(inst->in_pipe, inst->out_pipe, lba, data, &size, nr_block) != SCSI_CMD_STATUS_PASS)
		return -2;

	return 0;
}

/**
  * @brief  Write block to an MSC device.
  * @param  inst: Device to use for this write.
  * @param  lba: Logical block address.
  * @param  data: The buffer.
  * @param  nr_block: Number of blocks to write to the device.
  * @retval Status
  */
int32_t usbh_msc_block_write(usbh_msc_inst_t *inst, uint32_t lba, uint8_t *data, uint32_t nr_block)
{
	uint32_t size;

	if (inst->dev == 0)
		return -1;

	size = inst->size_block * nr_block;

	if (usbh_scsi_write10(inst->in_pipe, inst->out_pipe, lba, data, &size, nr_block) != SCSI_CMD_STATUS_PASS)
		return -2;

	return 0;
}

/**
  * @brief  Generates an LPM request enter sleep.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_msc_lpm_sleep(usbh_msc_inst_t *inst)
{
	return usb_hcd_lpm_sleep(inst->dev);
}

/**
  * @brief  Get the status of the LPM.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_msc_lpm_status(usbh_msc_inst_t *inst)
{
	return usb_hcd_lpm_status(inst->dev);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
