/**
  *********************************************************************************
  *
  * @file    usbh_hub.h
  * @brief   Header file for the HUB[HOST].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBH_HUB_H__
#define __USBH_HUB_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbh_core.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @addtogroup Host_HUB
  * @{
  */
/**
  * @defgroup Host_Hub_Public_Macros Public Macros
  * @{
  */
#define ROOT_HUB_MAX_PORTS		7U
#define HUB_FEATURE_C_HUB_LOCAL_POWER	0U
#define HUB_FEATURE_C_HUB_OVER_CURRENT	1U
#define HUB_FEATURE_PORT_CONNECTION	0U
#define HUB_FEATURE_PORT_ENABLE		1U
#define HUB_FEATURE_PORT_SUSPEND	2U
#define HUB_FEATURE_PORT_OVER_CURRENT	3U
#define HUB_FEATURE_PORT_RESET		4U
#define HUB_FEATURE_PORT_POWER		8U
#define HUB_FEATURE_PORT_LOW_SPEED	9U
#define HUB_FEATURE_C_PORT_CONNECTION	16U
#define HUB_FEATURE_C_PORT_ENABLE	17U
#define HUB_FEATURE_C_PORT_SUSPEND	18U
#define HUB_FEATURE_C_PORT_OVER_CURRENT	19U
#define HUB_FEATURE_C_PORT_RESET	20U
#define HUB_FEATURE_PORT_TEST		21U
#define HUB_FEATURE_PORT_INDICATOR	22U
#define HUB_STATUS_PWR_LOST		1U
#define HUB_STATUS_OVER_CURRENT		2U
#define HUB_PORT_STATUS_DEVICE_PRESENT	0x0001U
#define HUB_PORT_STATUS_ENABLED		0x0002U
#define HUB_PORT_STATUS_SUSPENDED	0x0004U
#define HUB_PORT_STATUS_OVER_CURRENT	0x0008U
#define HUB_PORT_STATUS_RESET		0x0010U
#define HUB_PORT_STATUS_POWERED		0x0100U
#define HUB_PORT_STATUS_LOW_SPEED	0x0200U
#define HUB_PORT_STATUS_HIGH_SPEED	0x0400U
#define HUB_PORT_STATUS_TEST_MODE	0x0800U
#define HUB_PORT_STATUS_INDICATOR_CTRL	0x1000U
#define HUB_PORT_CHANGE_DEVICE_PRESENT	0x0001U
#define HUB_PORT_CHANGE_ENABLED		0x0002U
#define HUB_PORT_CHANGE_SUSPENDED	0x0004U
#define HUB_PORT_CHANGE_OVER_CURRENT	0x0008U
#define HUB_PORT_CHANGE_RESET		0x0010U
#define USB_LIB_HUB_ACTIVE		0x1U
#define USB_LIB_HUB_HS			0x2U
#define USB_LIB_HUB_MULTI_TT		0x4U
#define HUB_POLLING_INTERVAL		0x1000U
/**
  * @}
  */

/**
  * @defgroup Host_Hub_Public_Types Public Types
  * @{
  */

#ifdef __ICCARM__
#pragma pack(1)
#endif

/**
  * @brief Standard hub descriptor
  */
typedef struct {
	uint8_t bLength;		/**< Length */
	uint8_t bDescType;		/**< Type */
	uint8_t bNbrPorts;		/**< Number of downstream-facing ports */
	uint16_t wHubCharacteristics;	/**< Characteristics */
	uint8_t bPwrOn2PwrGood;		/**< Time between the start and becoming stable */
	uint8_t bHubContrCurrent;	/**< Maximum circuitry in mA */
	uint8_t PortInfo[((ROOT_HUB_MAX_PORTS + 7) / 8) * 2];	/**< Port information */
} PACKED hub_desc_t;

#ifdef __ICCARM__
#pragma pack()
#endif

/**
  * @brief Port information
  */
typedef struct {
	uint32_t handle;			/**< Handle used by the HCD layer */
	volatile hub_port_state_t state;	/**< Status */
	volatile uint32_t count;		/**< counter */
	uint8_t low_peed;			/**< Low speed device */
	uint32_t speed;				/**< Speed of the device on this port */
	volatile uint8_t change;		/**< Change is pending on this port */
} hub_port_t;

/**
  * @brief Instance of a Hub device
  */
typedef struct {
	usbh_device_t *dev;			/**< Device instance */
	usb_cbk cbk;				/**< Callback function */
	void *arg;				/**< Parameter */
	uint32_t in_pipe;			/**< IN pipe */
	uint16_t hub_char;			/**< characteristics */
	uint8_t nr_port;			/**< Number of downstream-facing ports */
	uint8_t nr_port_use;			/**< Number of ports that we can use */
	uint8_t size;				/**< Size of a status change packet */
	uint32_t flag;				/**< Flags */
	volatile uint8_t busy;			/**< In process of being enumerated */
	uint8_t idx_enum;			/**< Index of device */
	hub_port_t port[USB_MAX_DEVICES];	/**< Ports */
	uint32_t int_num;			/**< Interrupt number for this instance */
} hub_inst_t;
/**
  * @}
  */
/** @addtogroup Host_Hub_Public_Variables
  * @{
  */
extern const usbh_class_driver_t __hub_class_driver;
/**
  * @}
  */
/** @addtogroup Host_Hub_Public_Functions
  * @{
  */
extern hub_inst_t * usbh_hub_open(usb_cbk cbk);
extern void usbh_hub_close(hub_inst_t *inst);
extern uint32_t usbh_hub_lpm_sleep(hub_inst_t *inst);
extern uint32_t usbh_hub_lpm_status(hub_inst_t *inst);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
