/**
  *********************************************************************************
  *
  * @file    usbh_hub.c
  * @brief   Functions related to hub.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbh_hub.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @defgroup Host_HUB Hub
  * @brief Host hub driver
  * @{
  */
/** @addtogroup Host_Hub_Private_Functions
  * @{
  */
static void *hub_driver_open(usbh_device_t *dev);
static void hub_driver_close(void *device);
/**
  * @}
  */
/** @defgroup Host_Hub_Public_Variables Public Variables
  * @{
  */
const usbh_class_driver_t __hub_class_driver = {
	USB_CLASS_HUB,
	hub_driver_open,
	hub_driver_close,
	0
};
/**
  * @}
  */
/** @defgroup Host_Hub_Private_Variables Private Variables
  * @{
  */
static hub_inst_t __root_hub;
static volatile uint32_t __c_flag;
static uint32_t __c_hub;
/**
  * @}
  */

/** @defgroup Host_Hub_Private_Functions Private Functions
  * @{
  */
/**
  * @brief  Set the operating speed of a given port.
  * @param  port: The port.
  * @param  speed: The speed.
  * @retval None
  */
static void hub_set_port_speed(uint8_t port, uint32_t speed)
{
	__root_hub.port[port].speed = speed;
	return;
}

/**
  * @brief  Set the feature on a given port.
  * @param  inst: The hub device instance.
  * @param  port: Port.
  * @param  feature: Feature.
  * @retval None
  */
static void hub_set_port_feature(hub_inst_t *inst, uint8_t port, uint16_t feature)
{
	usb_request_t req;
	usbh_device_t *dev = inst->dev;

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS | USB_RTYPE_OTHER;
	req.bRequest      = USBREQ_SET_FEATURE;
	req.wValue        = feature;
	req.wIndex        = port;
	req.wLength       = 0;

	usb_hcd_ctrl_transfer(0, &req, dev, 0, 0,dev->desc_device.bMaxPacketSize0);
	return;
}

/**
  * @brief  Clear the feature on a given port.
  * @param  inst: The hub device instance.
  * @param  port: Port.
  * @param  feature: Feature.
  * @retval None
  */
static void hub_clear_port_feature(hub_inst_t *inst, uint8_t port, uint16_t feature)
{
	usb_request_t req;
	usbh_device_t *dev = inst->dev;

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS | USB_RTYPE_OTHER;
	req.bRequest      = USBREQ_CLEAR_FEATURE;
	req.wValue        = feature;
	req.wIndex        = port;
	req.wLength       = 0;

	usb_hcd_ctrl_transfer(0, &req, dev, 0, 0, dev->desc_device.bMaxPacketSize0);
	return;
}

/**
  * @brief  Get the current status of a port.
  * @param  inst: The hub device instance.
  * @param  port: Port.
  * @param  status: Store the current status.
  * @param  change: Store the current change status.
  * @retval Status.
  */
static uint8_t hub_get_port_status(hub_inst_t *inst, uint8_t port, uint16_t *status, uint16_t *change)
{
	uint32_t data, read;
	usb_request_t req;
	usbh_device_t *dev = inst->dev;

	req.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_CLASS | USB_RTYPE_OTHER;
	req.bRequest      = USBREQ_GET_STATUS;
	req.wValue        = 0;
	req.wIndex        = (uint16_t)port;
	req.wLength       = 4;

	read = usb_hcd_ctrl_transfer(0, &req, dev, (uint8_t *)&data, 4, dev->desc_device.bMaxPacketSize0);

	if (read != 4) {
		return 0;
	}
	else {
		*status = (uint16_t)(data & 0xFFFF);
		*change = (uint16_t)(data >> 16);
	}

	return 1;
}

/**
  * @brief  Callback for the interrupt IN endpoint.
  * @param  pipe: Pipe.
  * @param  event: Event.
  * @retval None
  */
static void hub_int_in_cbk(uint32_t pipe, uint32_t event)
{
	switch (event) {
	case USB_EVENT_SCHEDULER:
		usb_hcd_pipe_schedule(pipe, (uint8_t *)&__c_hub, (uint32_t)__root_hub.size);
		break;

	case USB_EVENT_RX_AVAILABLE:
		usb_hcd_pipe_data_ack(pipe);
		__c_flag |= __c_hub;

		if (__root_hub.cbk)
			__root_hub.cbk(__root_hub.arg, USB_EVENT_RX_AVAILABLE, pipe, &__c_hub);

		break;
	default:
		break;
	}

	return;
}

/**
  * @brief  Get the class-specific hub descriptor.
  * @param  desc: Store the hub descriptor.
  * @retval Status
  */
static uint8_t hub_get_desc(hub_desc_t *desc)
{
	uint32_t ret;
	usb_request_t req;
	usbh_device_t *dev = __root_hub.dev;

	req.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_CLASS |USB_RTYPE_DEVICE;
	req.bRequest      = USBREQ_GET_DESCRIPTOR;
	req.wValue        = (USB_DTYPE_HUB << 8);
	req.wIndex        = 0;
	req.wLength       = sizeof(hub_desc_t);

	ret = usb_hcd_ctrl_transfer(0, &req, dev, (void *)desc, sizeof(hub_desc_t), dev->desc_device.bMaxPacketSize0);

	if (ret == 0)
		return 0;

	return 1;
}

/**
  * @brief  Open an instance of the hub driver.
  * @param  dev: Device.
  * @retval None
  */
static void *hub_driver_open(usbh_device_t *dev)
{
	uint8_t ret;
	uint32_t i;
	endpoint_desc_t *ep;
	interface_desc_t *iface;
	hub_desc_t hub;

	if (__root_hub.flag & USB_LIB_HUB_ACTIVE)
		return 0;

	iface = usb_desc_get_interface(dev->desc_config, 0, 0);
	ep = usb_desc_get_ep(iface, 0, dev->size_config);

	if (ep == 0)
		return 0;

	if ((iface->bInterfaceClass != USB_CLASS_HUB) || (iface->bInterfaceSubClass != 0))
		return 0;

	if (iface->bInterfaceProtocol == USB_HUB_PROTOCOL_SINGLE)
		__root_hub.flag |= USB_LIB_HUB_HS;
	else if (iface->bInterfaceProtocol == USB_HUB_PROTOCOL_MULTI)
		__root_hub.flag |= USB_LIB_HUB_HS | USB_LIB_HUB_MULTI_TT;

	__root_hub.dev = dev;

	if ((ep->bmAttributes & USB_EP_ATTR_TYPE_M) == USB_EP_ATTR_INT) {
		if (ep->bEndpointAddress & USB_EP_DESC_IN) {
			__root_hub.in_pipe = usb_hcd_pipe_alloc(0, USBHCD_PIPE_INTR_IN, dev, hub_int_in_cbk);
			usb_hcd_pipe_config(__root_hub.in_pipe, ep->wMaxPacketSize, ep->bInterval, ep->bEndpointAddress & USB_EP_DESC_NUM_M);
		}
	}

	if(!__root_hub.in_pipe)
		return 0;

	if (__root_hub.cbk != 0)
		__root_hub.cbk(__root_hub.arg, USB_EVENT_CONNECTED, (uint32_t)&__root_hub, 0);

	ret = hub_get_desc(&hub);
	if (ret) {
		__root_hub.nr_port     = hub.bNbrPorts;
		__root_hub.hub_char    = hub.wHubCharacteristics;
		__root_hub.nr_port_use = (hub.bNbrPorts > USB_MAX_DEVICES) ? USB_MAX_DEVICES : hub.bNbrPorts;
		__root_hub.size        = ((hub.bNbrPorts + 1) + 7) / 8;

		for (i = 1; i <= hub.bNbrPorts; ++i)
			hub_set_port_feature(&__root_hub, i, HUB_FEATURE_PORT_POWER);

		for (i = 0; i < USB_MAX_DEVICES; ++i) {
			__root_hub.port[i].change = 0;
			__root_hub.port[i].state  = HUB_PORT_IDLE;
		}
	}
	else {
		usb_hcd_pipe_free(__root_hub.in_pipe);
		__root_hub.cbk   = NULL;
		__root_hub.flag &= ~USB_LIB_HUB_ACTIVE;
		return 0;
	}

	__root_hub.flag |= USB_LIB_HUB_ACTIVE;
	return (void *)&__root_hub;
}

/**
  * @brief  Close an instance of the hub driver.
  * @param  device: Device.
  * @retval None
  */
static void hub_driver_close(void *device)
{
	uint32_t i;

	if (__root_hub.dev == NULL)
		return;

	for (i = 0; i < USB_MAX_DEVICES; ++i) {
		if ((__root_hub.port[i].state == HUB_PORT_ACTIVE) ||
				(__root_hub.port[i].state == HUB_PORT_RESET_WAIT) ||
				(__root_hub.port[i].state == HUB_PORT_ENUMERATED) ||
				(__root_hub.port[i].state == HUB_PORT_ERROR)) {
			usb_hcd_hub_device_disconnect(0, __root_hub.port[i].handle);

		}

		__root_hub.port[i].state = HUB_PORT_IDLE;
	}

	__root_hub.dev         = NULL;
	__root_hub.flag       &= ~USB_LIB_HUB_ACTIVE;
	__root_hub.busy        = 0;
	__root_hub.nr_port     = 0;
	__root_hub.hub_char    = 0;
	__root_hub.nr_port_use = 0;
	__root_hub.size        = 0;

	if (__root_hub.in_pipe != 0)
		usb_hcd_pipe_free(__root_hub.in_pipe);

	if (__root_hub.cbk != 0)
		__root_hub.cbk(__root_hub.arg, USB_EVENT_DISCONNECTED, (uint32_t)&__root_hub, 0);

	return;
}

/**
  * @brief  Reset an hub driver.
  * @param  port: Port.
  * @param  active: Status.
  * @retval None
  */
static void hub_driver_reset(uint8_t port, uint8_t active)
{
	if(!active) {
		__root_hub.port[port].state = HUB_PORT_RESET_WAIT;
		__root_hub.port[port].count = 10;
	}
	else {
		if (__root_hub.port[port].state == HUB_PORT_ACTIVE)
			usb_hcd_hub_device_disconnect(0, __root_hub.port[port].handle);

		__root_hub.port[port].state = HUB_PORT_RESET_ACTIVE;
	}

	return;
}

/**
  * @brief  Start the process of enumerating a new device by issuing a reset.
  * @param  port: Port.
  * @retval None
  */
static void hub_driver_dev_reset(uint8_t port)
{
	printf_e("Start enumeration for port %d\n\r", port);

	__root_hub.busy = 1;
	__root_hub.idx_enum = port;
	__root_hub.port[port].state = HUB_PORT_RESET_ACTIVE;
	hub_set_port_feature(&__root_hub, port, HUB_FEATURE_PORT_RESET);

	return;
}

/**
  * @brief  New device has been connected to the hub.
  * @param  port: Port.
  * @retval None
  */
static void hub_driver_dev_connect(uint8_t port)
{
	printf_e("Device connect at port %d \n\r", port);

	__root_hub.port[port].change = 0;
	__root_hub.port[port].state  = HUB_PORT_CONNECTED;
	__root_hub.port[port].count  = 100;

	return;
}

/**
  * @brief  An existing device has been removed from the hub.
  * @param  port: Port.
  * @retval None
  */
static void hub_driver_dev_disconnect(uint8_t port)
{
	if ((__root_hub.port[port].state == HUB_PORT_ACTIVE) ||
			(__root_hub.port[port].state == HUB_PORT_RESET_WAIT) ||
			(__root_hub.port[port].state == HUB_PORT_ENUMERATED) ||
			(__root_hub.port[port].state == HUB_PORT_ERROR)) {
		usb_hcd_hub_device_disconnect(0, __root_hub.port[port].handle);
	}

	if ((__root_hub.port[port].state == HUB_PORT_RESET_ACTIVE) ||
			(__root_hub.port[port].state == HUB_PORT_RESET_WAIT) ||
			(__root_hub.port[port].state == HUB_PORT_ACTIVE)) {
		__root_hub.busy = 0;
	}

	__root_hub.port[port].state = HUB_PORT_IDLE;
	return;
}
/**
  * @}
  */

/** @defgroup Host_Hub_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Main routine for Hub.  This function is called periodically by usb_hcd_main()
  * @retval None
  */
void usbh_hub_main(void)
{
	uint16_t status = 0, change = 0;
	uint8_t k, ret;

	if ((__root_hub.flag & USB_LIB_HUB_ACTIVE) == 0)
		return;

	for (k = 0; k <= __root_hub.nr_port_use; ++k) {
		if (__root_hub.port[k].count != 0)
			__root_hub.port[k].count--;

		if ((__root_hub.port[k].state == HUB_PORT_CONNECTED) && (!__root_hub.busy) && (__root_hub.port[k].count == 0))
			hub_driver_dev_reset(k);

		if ((__root_hub.port[k].state == HUB_PORT_RESET_WAIT) && (__root_hub.port[k].count == 0)) {
			__root_hub.port[k].state  = HUB_PORT_ACTIVE;
			__root_hub.port[k].handle = usb_hcd_hub_device_connect(0, 1, k, __root_hub.port[k].speed);
		}

		if (__root_hub.busy && (__root_hub.idx_enum != k))
			continue;

		if ((__root_hub.flag == USB_LIB_HUB_ACTIVE) && (__root_hub.port[k].count == 0)) {
			__root_hub.port[k].count = HUB_POLLING_INTERVAL;
			__c_flag |= (1 << k);
		}
		
		if (__c_flag & (1 << k)) {
			ret = hub_get_port_status(&__root_hub, k, &status, &change);

			map_usb_int_unregister();
			__c_flag &= ~(1 << k);
			map_usb_int_register();

			if (!ret)
				continue;

			if (change & HUB_PORT_CHANGE_DEVICE_PRESENT) {
				printf_e("Connect change on port %d\n\r", k);
				hub_clear_port_feature(&__root_hub, k, HUB_FEATURE_C_PORT_CONNECTION);

				if (status & HUB_PORT_STATUS_DEVICE_PRESENT) {
					printf_e("Connected\n\r");
					hub_driver_dev_connect(k);
				}
				else {
					printf_e("Disconnected\n\r");
					hub_driver_dev_disconnect(k);
				}
			}

			if (change & HUB_PORT_CHANGE_RESET) {
				hub_clear_port_feature(&__root_hub, k, HUB_FEATURE_C_PORT_RESET);
				hub_get_port_status(&__root_hub, k, &status, &change);
				printf_e("Reset %s for port %d\n\r", ((status & HUB_PORT_STATUS_RESET) ? "assert" : "deassert"), k);
				hub_driver_reset(k, (status & HUB_PORT_STATUS_RESET) ? 1 : 0);

				if (status & HUB_PORT_STATUS_LOW_SPEED)
					hub_set_port_speed(k, USB_EP_SPEED_LOW);
				else if (status & HUB_PORT_STATUS_HIGH_SPEED)
					hub_set_port_speed(k, USB_EP_SPEED_HIGH);
				else
					hub_set_port_speed(k, USB_EP_SPEED_FULL);
			}

			if (change & HUB_PORT_CHANGE_OVER_CURRENT) {
				printf_e("Port %d over current.\n\r", k);
				hub_clear_port_feature(&__root_hub, k, HUB_FEATURE_C_PORT_OVER_CURRENT);
			}

			if (change & HUB_PORT_CHANGE_ENABLED) {
				printf_e("Enable change for port %d.\n\r", k);
				hub_clear_port_feature(&__root_hub, k, HUB_FEATURE_C_PORT_ENABLE);
			}

			if (change & HUB_PORT_CHANGE_SUSPENDED) {
				printf_e("Suspend change for port %d.\n\r", k);
				hub_clear_port_feature(&__root_hub, k, HUB_FEATURE_C_PORT_SUSPEND);
			}
		}
	}

	return;
}

/**
  * @brief  Downstream device has been enumerated.
  * @param  hub: Address of the hub.
  * @param  port: Port.
  * @retval None
  */
void usbh_hub_enum_cplt(uint8_t hub, uint8_t port)
{
	printf_e("Enumeration complete for hub %d, port %d\n\r", hub, port);

	__root_hub.port[port].state = HUB_PORT_ENUMERATED;
	__root_hub.busy = 0;
	return;
}

/**
  * @brief  Downstream device failed to enumerate.
  * @param  hub: Address of the hub.
  * @param  port: Port.
  * @retval None
  */
void usbh_hub_enum_error(uint8_t hub, uint8_t port)
{
	printf_e("\rEnumeration error for hub %d, port %d\n\r", hub, port);

	__root_hub.port[port].state = HUB_PORT_ERROR;
	__root_hub.busy = 0;

	return;
}

/**
  * @brief  Enable the host hub class driver.
  * @param  cbk: Callback for the hub events.
  * @retval Driver instance
  */
hub_inst_t *usbh_hub_open(usb_cbk cbk)
{
	if (__root_hub.cbk) {
		printf_e("HUB open failed - already connected.\n\r");
		return 0;
	}

	__root_hub.cbk = cbk;
	printf_e("HUB open completed.\n\r");

	return &__root_hub;
}

/**
  * @brief  Release a hub device instance.
  * @param  inst: Device instance.
  * @retval None
  */
void usbh_hub_close(hub_inst_t *inst)
{
	inst->dev = NULL;
	inst->cbk = NULL;

	printf_e("HUB close completed.\n\r");
	return;
}

/**
  * @brief  Initialize the Hub driver.
  * @retval None
  */
void usbh_hub_init(void)
{
	__c_flag = 0;
	__c_hub  = 0;

	if (__root_hub.dev != 0)
		__root_hub.int_num = map_usb_int_num_get();

	return;
}

/**
  * @brief  Request for a device to enter sleep.
  * @param  inst: Device instance.
  * @retval Status
  */
uint32_t usbh_hub_lpm_sleep(hub_inst_t *inst)
{
	return usb_hcd_lpm_sleep(inst->dev);
}

/**
  * @brief  Get current status of an LPM request.
  * @param  inst: Device instance.
  * @retval Status
  */
uint32_t usbh_hub_lpm_status(hub_inst_t *inst)
{
	return usb_hcd_lpm_status(inst->dev);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
