/**
  *********************************************************************************
  *
  * @file    usbh_hid_mouse.c
  * @brief   Functions related to host mouse driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbh_hid_mouse.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @defgroup Host_Mouse Mouse
  * @brief Host Mouse driver
  * @{
  */
/** @defgroup Host_Mouse_Private_Variables Private Variables
  * @{
  */
static usbh_mouse_t __usbh_mouse = {
	0
};
/**
  * @}
  */

/** @defgroup Host_Mouse_Private_Functions Private Functions
  * @{
  */
/**
  * @brief  Update the state of the mouse buttons and axis.
  * @param  inst: Instance value.
  * @retval None
  */
static void usbh_mouse_state_update(usbh_mouse_t *inst)
{
	uint32_t k;

	if (inst->buf[0] != inst->button) {
		for (k = 1; k <= 0x4; k <<= 1) {
			if (((inst->buf[0] & k) != 0) && ((inst->button & k) == 0))
				inst->cbk(0, USBH_EVENT_HID_MS_PRESS, k, 0);
			if (((inst->buf[0] & k) == 0) && ((inst->button & k) != 0))
				inst->cbk(0, USBH_EVENT_HID_MS_REL, k, 0);
		}

		inst->button = inst->buf[0];
	}

	if (inst->buf[1] != 0)
		inst->cbk(0, USBH_EVENT_HID_MS_X, (uint32_t)inst->buf[1], 0);
	if (inst->buf[2] != 0)
		inst->cbk(0, USBH_EVENT_HID_MS_Y, (uint32_t)inst->buf[2], 0);

	return;
}

/**
  * @brief  Handles event callbacks.
  * @param  mouse: Instance value.
  * @param  event: Event that has been passed.
  * @param  param: Parameter.
  * @param  data: Message data.
  * @retval Always 0.
  */
static uint32_t usbh_mouse_handle(void *mouse, uint32_t event, uint32_t param, void *data)
{
	usbh_mouse_t *inst = (usbh_mouse_t *)mouse;

	switch(event) {
	case USB_EVENT_CONNECTED:
		inst->flag |= USBHMS_DEVICE_PRESENT;
		inst->cbk(0, event, param, data);
		break;

	case USB_EVENT_DISCONNECTED:
		inst->flag &= ~USBHMS_DEVICE_PRESENT;
		inst->cbk(0, event, param, data);
		break;

	case USB_EVENT_RX_AVAILABLE:
		usbh_hid_report_get(inst->hid_inst, 0, inst->buf, USBHMS_REPORT_SIZE);
		usbh_mouse_state_update(inst);
		break;
	}

	return 0;
}
/**
  * @}
  */

/** @defgroup Host_Mouse_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Open an instance of a mouse.
  * @param  cbk: Callback function.
  * @param  buf: Buffer.
  * @param  size: Size of the buffer.
  * @retval Mouse instance.
  */
usbh_mouse_t *usbh_mouse_open(usbh_mouse_cbk cbk, uint8_t *buf, uint32_t size)
{
	usbh_hid_inst_t *inst;

	__usbh_mouse.cbk = cbk;
	inst = usbh_hid_open(USBH_HID_CLASS_MOUSE, usbh_mouse_handle, (void *)&__usbh_mouse);
	__usbh_mouse.hid_inst = inst;
	__usbh_mouse.heap     = buf;
	__usbh_mouse.size     = size;

	return &__usbh_mouse;
}

/**
  * @brief  Close an instance of a mouse.
  * @param  inst: Instance value.
  * @retval Always 0.
  */
uint32_t usbh_mouse_close(usbh_mouse_t *inst)
{
	inst->cbk = NULL;
	usbh_hid_close(inst->hid_inst);

	return 0;
}

/**
  * @brief  Initialize a mouse interface.
  * @param  inst: Instance value.
  * @retval Always 0.
  */
uint32_t usbh_mouse_init(usbh_mouse_t *inst)
{
	usbh_hid_idle_set(inst->hid_inst, 0, 0);
	usbh_hid_report_desc_get(inst->hid_inst, inst->heap, inst->size);
	usbh_hid_protocol_set(inst->hid_inst, 1);

	return 0;
}

/**
  * @brief  Generates an LPM request enter sleep.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_mouse_lpm_sleep(usbh_mouse_t *inst)
{
	return usbh_hid_lpm_sleep(inst->hid_inst);
}

/**
  * @brief  Get the status of the LPM.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_mouse_lpm_status(usbh_mouse_t *inst)
{
	return usbh_hid_lpm_status(inst->hid_inst);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
