/**
  *********************************************************************************
  *
  * @file    usbh_hid_keyb.c
  * @brief   Functions related to host keyboard driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbh_hid_keyb.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @defgroup Host_Keyb Keyboard
  * @brief Host Keyboard driver
  * @{
  */
/** @defgroup Host_Keyb_Private_Variables Private Variables
  * @{
  */
static usbh_keyb_t __usbh_keyb = {
	0
};
/**
  * @}
  */

/** @defgroup Host_Keyb_Private_Functions Private Functions
  * @{
  */
/**
  * @brief  Modify the current keyboard state.
  * @param  inst: Instance value.
  * @retval None
  */
static void usbh_keyb_state_update(usbh_keyb_t *inst)
{
	int32_t i, k;

	if (inst->buf[2] == 0x01)
		return;

	if (inst->mod_state != inst->buf[0]) {
		inst->cbk(0, USBH_EVENT_HID_KB_MOD, inst->buf[0], 0);
		inst->mod_state = inst->buf[0];
	}

	for (k = 0; k < 6; k++) {
		if (inst->key_state[k] == 0)
			continue;

		for (i = 2; i < 8; i++) {
			if (inst->buf[i] == inst->key_state[k])
				break;
		}

		if (i == 8) {
			inst->cbk(0, USBH_EVENT_HID_KB_REL, inst->key_state[k], 0);
			inst->key_state[k] = 0;
		}
	}

	for (i = 2; i < 8; i++) {
		if (inst->buf[i] == 0)
			break;

		for (k = 0; k < 6; k++) {
			if (inst->buf[i] == inst->key_state[k])
				break;
		}
		if (k == 6) {
			for (k = 0; k < 6; k++) {
				if (inst->key_state[k] == 0) {
					inst->key_state[k] = inst->buf[i];
					inst->cbk(0, USBH_EVENT_HID_KB_PRESS, inst->buf[i], 0);

					break;
				}
			}
		}
	}

	return;
}

/**
  * @brief  Handles event callbacks.
  * @param  keyboard: Instance value.
  * @param  event: Event that has been passed.
  * @param  param: Parameter.
  * @param  data: Message data.
  * @retval Always 0.
  */
static uint32_t usbh_keyb_handle(void *keyboard, uint32_t event, uint32_t param, void *data)
{
	usbh_keyb_t *inst = (usbh_keyb_t *)keyboard;

	switch (event) {
	case USB_EVENT_CONNECTED:
		inst->flag |= USBHKEYB_DEVICE_PRESENT;
		inst->cbk(0, event, param, data);
		break;

	case USB_EVENT_DISCONNECTED:
		inst->flag &= ~USBHKEYB_DEVICE_PRESENT;
		inst->cbk(0, event, param, data);
		break;

	case USB_EVENT_RX_AVAILABLE:
		usbh_hid_report_get(inst->hid_inst, 0, inst->buf, USBHKEYB_REPORT_SIZE);
		usbh_keyb_state_update(inst);
		break;

	}

	return 0;
}
/**
  * @}
  */

/** @defgroup Host_Keyb_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Open an instance of a keyboard.
  * @param  cbk: Callback function.
  * @param  buf: Buffer.
  * @param  size: Size of the buffer.
  * @retval Keyboard instance.
  */
usbh_keyb_t *usbh_keyb_open(usbh_keyb_cbk cbk, uint8_t *buf, uint32_t size)
{
	usbh_hid_inst_t *inst;

	__usbh_keyb.cbk = cbk;
	inst = usbh_hid_open(USBH_HID_CLASS_KEYB, usbh_keyb_handle, (void *)&__usbh_keyb);
	__usbh_keyb.hid_inst = inst;

	return &__usbh_keyb;
}

/**
  * @brief  Close an instance of a keyboard.
  * @param  inst: Instance value.
  * @retval Always 0.
  */
uint32_t usbh_keyb_close(usbh_keyb_t *inst)
{
	inst->cbk = NULL;
	usbh_hid_close(inst->hid_inst);

	return 0;
}

/**
  * @brief  Initialize a keyboard interface.
  * @param  inst: Instance value.
  * @retval Always 0.
  */
uint32_t usbh_keyb_init(usbh_keyb_t *inst)
{
	int32_t i;
	uint8_t data = 0;

	usbh_hid_idle_set(inst->hid_inst, 0, 0);
	usbh_hid_report_desc_get(inst->hid_inst, inst->buf, USBHKEYB_REPORT_SIZE);
	usbh_hid_protocol_set(inst->hid_inst, 1);
	usbh_hid_report_set(inst->hid_inst, 0, &data, 1);

	for(i = 0; i < 6; ++i)
		inst->key_state[i] = 0;

	return 0;
}

/**
  * @brief  Set one of the fixed modifier keys.
  * @param  inst: Instance value.
  * @param  modifier: Bit mask of the modifiers.
  * @retval Always 0.
  */
uint32_t usbh_keyb_modifier_set(usbh_keyb_t *inst, uint32_t modifier)
{
	inst->mod_sticky = (uint8_t)modifier;
	usbh_hid_report_set(inst->hid_inst, 0, (uint8_t *)&modifier, 1);

	return 0;
}

/**
  * @brief  Set the automatic poll rate.
  * @param  inst: Instance value.
  * @param  rate: The poll rate.
  * @retval Always 0.
  */
uint32_t usbh_keyb_poll_rate_set(usbh_keyb_t *inst, uint32_t rate)
{
	usbh_hid_idle_set(inst->hid_inst, rate, 0);
	return 0;
}

/**
  * @brief  Map a USB usage ID to a printable character.
  * @param  inst: Instance value.
  * @param  table: Table to use to map the usage ID to characters.
  * @param  usage: USB usage ID to map to a character.
  * @retval Printable character.
  */
uint32_t usbh_keyb_usage_to_char(usbh_keyb_t *inst, const hid_keyb_usage_table_t *table, uint8_t usage)
{
	uint32_t value, offset, shift = 0;
	const uint8_t *map8;
	const uint16_t *map16;

	offset = (usage * table->byte_of_char * 2);

	if (inst->mod_sticky &= HID_KEYB_CAPS_LOCK) {
		if ((table->caps_lock[usage >> 5]) >> (usage & 0x1f) & 1)
			shift = table->byte_of_char;
	}

	if ((inst->mod_state & 0x22) != 0) {
		if (shift == 0)
			shift = table->byte_of_char;
		else
			shift = 0;
	}

	if (table->byte_of_char == 1) {
		map8 = table->mapping;
		value = map8[offset + shift];
	}
	else if (table->byte_of_char == 2) {
		map16 = (uint16_t *)table->mapping;
		value = map16[offset + shift];
	}
	else {
		value = 0;
	}

	return value;
}

/**
  * @brief  Generates an LPM request enter sleep.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_keyb_lpm_sleep(usbh_keyb_t *inst)
{
	return usbh_hid_lpm_sleep(inst->hid_inst);
}

/**
  * @brief  Get the status of the LPM.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_keyb_lpm_status(usbh_keyb_t *inst)
{
	return usbh_hid_lpm_status(inst->hid_inst);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
