/**
  *********************************************************************************
  *
  * @file    usbh_hid.c
  * @brief   Functions related to host HID driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbh_hid.h"
#include "usb_hid.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @defgroup Host_HID HID
  * @brief Host HID driver
  * @{
  */
/** @defgroup Host_HID_Private_Functions Private Functions
  * @{
  */
static void *hid_driver_open(usbh_device_t *dev);
static void hid_driver_close(void *instance);
/**
  * @}
  */
/** @defgroup Host_HID_Private_Variables Private Variables
  * @{
  */
static usbh_hid_inst_t __hid_dev[HID_MAX_DEVICES];
const usbh_class_driver_t __usbh_hid_driver = {
	USB_CLASS_HID,
	hid_driver_open,
	hid_driver_close,
	0
};
/**
  * @}
  */

/** @addtogroup Host_HID_Private_Functions
  * @{
  */
/**
  * @brief  Callbacks for the interrupt IN endpoint.
  * @param  pipe: Interrupt IN endpoint.
  * @param  event: The event.
  * @retval None
  */
static void hid_in_cbk(uint32_t pipe, uint32_t event)
{
	int32_t i;

	switch (event) {
	case USB_EVENT_SCHEDULER:
		usb_hcd_pipe_schedule(pipe, 0, 1);
		break;

	case USB_EVENT_RX_AVAILABLE:
		for (i = 0; i < HID_MAX_DEVICES; ++i) {
			if (__hid_dev[i].in_pipe != pipe)
				continue;

			__hid_dev[i].cbk(__hid_dev[i].arg, USB_EVENT_RX_AVAILABLE, pipe, 0);
		}

		break;
	}

	return;
}

/**
  * @brief  Open an instance of a HID device.
  * @param  dev: Device information structure.
  * @retval None
  */
static void *hid_driver_open(usbh_device_t *dev)
{
	int32_t i, k;
	endpoint_desc_t *ep;
	interface_desc_t *iface;

	iface = usb_desc_get_interface(dev->desc_config, 0, 0);

	for (k = 0; k < HID_MAX_DEVICES; ++k) {
		if (__hid_dev[k].type == iface->bInterfaceProtocol) {
			__hid_dev[k].dev = dev;
	
			for (i = 0; i < 3; ++i) {
				ep = usb_desc_get_ep(iface, i, 256);
	
				if (ep == NULL)
					break;
	
				if ((ep->bmAttributes & USB_EP_ATTR_TYPE_M) != USB_EP_ATTR_INT)
					continue;
	
				if (ep->bEndpointAddress & USB_EP_DESC_IN) {
					__hid_dev[k].in_pipe = usb_hcd_pipe_alloc(0, USBHCD_PIPE_INTR_IN, dev, hid_in_cbk);
					usb_hcd_pipe_config(__hid_dev[k].in_pipe, ep->wMaxPacketSize, ep->bInterval,
									(ep->bEndpointAddress & USB_EP_DESC_NUM_M));
				}
			}
	
			if (__hid_dev[k].cbk != 0)
				__hid_dev[k].cbk(__hid_dev[k].arg, USB_EVENT_CONNECTED, (uint32_t)&__hid_dev[k], 0);
	
			__hid_dev[k].dev = dev;
			return &__hid_dev[k];
		}
	}

	return NULL;
}

/**
  * @brief  Close an instance of a HID device.
  * @param  instance: Device information structure.
  * @retval None
  */
static void hid_driver_close(void *instance)
{
	usbh_hid_inst_t *inst;

	inst = (usbh_hid_inst_t *)instance;
	inst->dev = 0;

	if (inst->in_pipe != 0)
		usb_hcd_pipe_free(inst->in_pipe);

	if (inst->cbk)
		inst->cbk(inst->arg, USB_EVENT_DISCONNECTED, (uint32_t)instance, 0);

	return;
}
/**
  * @}
  */

/** @defgroup Host_HID_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Open an instance of a HID device.
  * @param  type: Type of HID device.
  * @param  cbk: Callback that will be called when any event occurs.
  * @param  arg: Parameter of the callback.
  * @retval Instance value
  */
usbh_hid_inst_t * usbh_hid_open(hid_sub_class_t type, usb_cbk cbk, void *arg)
{
	uint32_t i;

	for(i = 0; i < HID_MAX_DEVICES; ++i) {
		if(__hid_dev[i].type == USBH_HID_CLASS_NONE) {
			__hid_dev[i].cbk  = cbk;
			__hid_dev[i].type = type;
			__hid_dev[i].arg  = arg;

			return &__hid_dev[i];
		}
	}

	return NULL;
}

/**
  * @brief  Release an instance of a HID device.
  * @param  inst: Instance value.
  * @retval None
  */
void usbh_hid_close(usbh_hid_inst_t *inst)
{
	inst->cbk = NULL;
	inst->type = USBH_HID_CLASS_NONE;

	return;
}

/**
  * @brief  Retrieve the report descriptor.
  * @param  inst: Instance value.
  * @param  buf: Buffer to store the report descriptor.
  * @param  size: Size of buffer.
  * @retval Length of the report descriptor.
  */
uint32_t usbh_hid_report_desc_get(usbh_hid_inst_t *inst, uint8_t *buf, uint32_t size)
{
	uint32_t len;
	usb_request_t req;

	req.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_STANDARD | USB_RTYPE_INTERFACE;
	req.bRequest      = USBREQ_GET_DESCRIPTOR;
	req.wValue        = USB_HID_DTYPE_REPORT << 8;
	req.wIndex        = 0;
	req.wLength       = size;

	len = usb_hcd_ctrl_transfer(0, &req, inst->dev, buf, size, inst->dev->desc_device.bMaxPacketSize0);
	return len;
}

/**
  * @brief  Sets the idle timeout.
  * @param  inst: Instance value.
  * @param  dur: Duration of the timeout.
  * @param  report_id: Report identifier.
  * @retval Always 0.
  */
uint32_t usbh_hid_idle_set(usbh_hid_inst_t *inst, uint8_t dur, uint8_t report_id)
{
	usb_request_t req;

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS | USB_RTYPE_INTERFACE;
	req.bRequest      = USBREQ_SET_IDLE;
	req.wValue        = (dur << 8) | report_id;
	req.wIndex        = 0;
	req.wLength       = 0;

	return usb_hcd_ctrl_transfer(0, &req, inst->dev, 0, 0, MAX_PACKET_SIZE_EP0);
}

/**
  * @brief  Set/Clear the boot protocol state.
  * @param  inst: Instance value.
  * @param  protocol: The protocol.
  * @retval Always 0.
  */
uint32_t usbh_hid_protocol_set(usbh_hid_inst_t *inst, uint32_t protocol)
{
	usb_request_t req;

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS | USB_RTYPE_INTERFACE;
	req.bRequest      = USBREQ_SET_PROTOCOL;
	req.wValue        = protocol ? 0 : 1;
	req.wIndex        = 0;
	req.wLength       = 0;

	usb_hcd_ctrl_transfer(0, &req, inst->dev, 0, 0, inst->dev->desc_device.bMaxPacketSize0);
	return 0;
}

/**
  * @brief  Send a report to a HID device.
  * @param  inst: Instance value.
  * @param  interface: The interface.
  * @param  data: Buffer to use to store the report.
  * @param  size: Size of the buffer.
  * @retval Length.
  */
uint32_t usbh_hid_report_set(usbh_hid_inst_t *inst, uint32_t interface, uint8_t *data, uint32_t size)
{
	usb_request_t req;

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS | USB_RTYPE_INTERFACE;
	req.bRequest      = USBREQ_SET_REPORT;
	req.wValue        = USB_HID_REPORT_OUTPUT << 8;
	req.wIndex        = (uint16_t)interface;
	req.wLength       = size;

	usb_hcd_ctrl_transfer(0, &req, inst->dev, data, size, inst->dev->desc_device.bMaxPacketSize0);
	return size;
}

/**
  * @brief  Retrieve a report to a HID device.
  * @param  inst: Instance value.
  * @param  interface: The interface.
  * @param  data: Buffer to use to store the report.
  * @param  size: Size of the buffer.
  * @retval Length.
  */
uint32_t usbh_hid_report_get(usbh_hid_inst_t *inst, uint32_t interface, uint8_t *data, uint32_t size)
{
	return usb_hcd_pipe_read_n_blocking(inst->in_pipe, data, size);
}

/**
  * @brief  Generates an LPM request enter sleep.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_hid_lpm_sleep(usbh_hid_inst_t *inst)
{
	return usb_hcd_lpm_sleep(inst->dev);
}

/**
  * @brief  Get the status of the LPM.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_hid_lpm_status(usbh_hid_inst_t *inst)
{
	return usb_hcd_lpm_status(inst->dev);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
