/**
  *********************************************************************************
  *
  * @file    usbh_core.h
  * @brief   Header file for the CORE[HOST].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBH_CORE_H__
#define __USBH_CORE_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usb_utils.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @addtogroup Host_Core
  * @{
  */
/**
  * @defgroup Host_Core_Public_Macros Public Macros
  * @{
  */
#define USBHCD_PIPE_UNUSED		0x00100000U
#define USBHCD_PIPE_CONTROL		0x00130000U
#define USBHCD_PIPE_BULK_OUT		0x00210000U
#define USBHCD_PIPE_BULK_IN		0x00220000U
#define USBHCD_PIPE_INTR_OUT		0x00410000U
#define USBHCD_PIPE_INTR_IN		0x00420000U
#define USBHCD_PIPE_ISOC_OUT		0x00810000U
#define USBHCD_PIPE_ISOC_IN		0x00820000U
#define USBHCD_PIPE_ISOC_OUT_DMA	0x01810000U
#define USBHCD_PIPE_ISOC_IN_DMA		0x01820000U
#define USBHCD_PIPE_BULK_OUT_DMA	0x01210000U
#define USBHCD_PIPE_BULK_IN_DMA		0x01220000U
#define USBHCD_FAULT_LOW		0x10U
#define USBHCD_FAULT_HIGH		0x30U
#define USBHCD_FAULT_VBUS_NONE		0x0U
#define USBHCD_FAULT_VBUS_TRI		0x140U
#define USBHCD_FAULT_VBUS_DIS		0x400U
#define USBHCD_VBUS_MANUAL		0x4U
#define USBHCD_VBUS_AUTO_LOW		0x2U
#define USBHCD_VBUS_AUTO_HIGH		0x3U
#define USBHCD_VBUS_FILTER		0x10000U
#define USBHCD_LPM_AVAIL		0x0U
#define USBHCD_LPM_ERROR		0x1U
#define USBHCD_LPM_PENDING		0x2U
#define USBHCD_EVFLAG_SOF		0x1U
#define USBHCD_EVFLAG_CONNECT		0x2U
#define USBHCD_EVFLAG_UNKCNCT		0x4U
#define USBHCD_EVFLAG_DISCNCT		0x8U
#define USBHCD_EVFLAG_PWRFAULT		0x10U
#define USBHCD_EVFLAG_PWRDIS		0x20U
#define USBHCD_EVFLAG_PWREN		0x40U
#define USBHCD_DMA_UNUSED		0xffU
#define INT_EVENT_VBUS_ERR		0x01U
#define INT_EVENT_CONNECT		0x02U
#define INT_EVENT_DISCONNECT		0x04U
#define INT_EVENT_POWER_FAULT		0x08U
#define INT_EVENT_SOF			0x10U
#define INT_EVENT_ENUM			0x20U
#define INT_EVENT_LPM_PEND		0x40U
#define INT_EVENT_LPM			0x80U
#define USBHDEV_FLAG_NOTIFYINT		0x1U
#define USBHDEV_FLAG_LPMPEND		0x2U
#define USBHDEV_FLAG_LPMERROR		0x4U
#define USBHDEV_FLAG_ALLOCATED		0x80000000U
#define EP_PIPE_TYPE_LOW_SPEED		0x02000000U
#define EP_PIPE_USE_DMA			0x01000000U
#define EP_PIPE_TYPE_ISOC		0x00800000U
#define EP_PIPE_TYPE_INTR		0x00400000U
#define EP_PIPE_TYPE_BULK		0x00200000U
#define EP_PIPE_TYPE_CONTROL		0x00100000U
#define EP_PIPE_TYPE_IN			0x00020000U
#define EP_PIPE_TYPE_OUT		0x00010000U
#define EP_PIPE_IDX_M			0x0000ffffU
#define MAX_NUM_PIPES			15U
#define MAX_NUM_DMA_CHANNELS		6U
#define USB_HOST_PWREN_AUTO		0x2U

#define OUT_PIPE_HANDLE(idx)		(__hcd.out_pipe[idx].type | (idx))
#define IN_PIPE_HANDLE(idx)		(__hcd.in_pipe[idx].type | (idx))

#define DECLARE_EVENT_DRIVER(name, open, close, event)	\
const usbh_class_driver_t name =			\
{							\
	USB_CLASS_EVENTS,				\
	0,						\
	0,						\
	event						\
}
/**
  * @}
  */

/**
  * @defgroup Host_Core_Public_Types Public Types
  * @{
  */
/**
  * @brief Endpoint 0 status
  */
typedef enum {
	EP0_STATE_IDLE      = 0x0,	/**< Idle */
	EP0_STATE_SETUP_IN  = 0x1,	/**< Setup in stage */
	EP0_STATE_SETUP_OUT = 0x2,	/**< Setup out stage */
	EP0_STATE_RX        = 0x3,	/**< Receive */
	EP0_STATE_STATUS    = 0x4,	/**< Status stage */
	EP0_STATE_STATUS_IN = 0x5,	/**< Status in stage */
	EP0_STATE_STALL     = 0x6,	/**< Stall */
	EP0_STATE_ERROR     = 0x7,	/**< Error */
} ep0_status_t;

/**
  * @brief Pipe status
  */
typedef enum {
	PIPE_STATE_READING        = 0x0,	/**< Reading */
	PIPE_STATE_READ_DMA       = 0x1,	/**< Read via DMA */
	PIPE_STATE_READ_DMA_WAIT  = 0x2,	/**< Read waitting for DMA */
	PIPE_STATE_DATA_READY     = 0x3,	/**< Data ready */
	PIPE_STATE_DATA_SENT      = 0x4,	/**< Data sent */
	PIPE_STATE_WRITING        = 0x5,	/**< Writing */
	PIPE_STATE_WRITE_DMA      = 0x6,	/**< Writing via DMA */
	PIPE_STATE_WRITE_DMA_WAIT = 0x7,	/**< Writing waitting for DMA */
	PIPE_STATE_STALL          = 0x8,	/**< Stall */
	PIPE_STATE_ERROR          = 0x9,	/**< Error */
	PIPE_STATE_IDLE           = 0xa,	/**< Idle */
} pipe_state_t;

/**
  * @brief DEVICE status
  */
typedef enum {
	HCD_DEV_DISCONNECT  = 0x0,	/**< Disconnect */
	HCD_DEV_CONNECT     = 0x1,	/**< Connect */
	HCD_DEV_CONNECT_HUB = 0x2,	/**< Connect via hub */
	HCD_DEV_REQUEST     = 0x3,	/**< Request */
	HCD_DEV_RESET       = 0x4,	/**< Reset */
	HCD_DEV_ADDRESS     = 0x5,	/**< Address */
	HCD_DEV_CONFIGURE   = 0x6,	/**< Configure */
	HCD_DEV_GET_STRING  = 0x7,	/**< Get string */
	HCD_DEV_ERROR       = 0x8,	/**< Error */
	HCD_VBUS_ERROR      = 0x9,	/**< VBUS error */
	HCD_POWER_FAULT     = 0xa,	/**< Power Fault */
	HCD_IDLE            = 0xb,	/**< Idle */
} usbh_device_state_t;

/**
  * @brief Hub port status
  */
typedef enum {
	HUB_PORT_IDLE         = 0x0,	/**< Idle */
	HUB_PORT_CONNECTED    = 0x1,	/**< Disconnect */
	HUB_PORT_RESET_ACTIVE = 0x2,	/**< Reset Active */
	HUB_PORT_RESET_WAIT   = 0x3,	/**< Reset Wait */
	HUB_PORT_ACTIVE       = 0x4,	/**< Active */
	HUB_PORT_ENUMERATED   = 0x5,	/**< Enumerated */
	HUB_PORT_ERROR        = 0x6,	/**< Error */
} hub_port_state_t;

typedef void (* hcd_pipe_cbk)(uint32_t pipe, uint32_t event);

/**
  * @brief Device information
  */
typedef struct {
	uint32_t addr;			/**< Address */
	uint32_t interface;		/**< Interface */
	uint8_t low_speed;		/**< Low speed */
	uint32_t speed;			/**< High/Full/Low speed */
	uint8_t config_read;		/**< Configure descriptor */
	uint8_t hub;			/**< HUB */
	uint8_t hub_port;		/**< HUB port */
	device_desc_t desc_device;	/**< Device descriptor */
	config_desc_t *desc_config;	/**< Configure descriptor */
	uint32_t size_config;		/**< Size of Configure descriptor */
	uint32_t flag;			/**< Bit mapping */
} usbh_device_t;

/**
  * @brief State for the device enumeration
  */
typedef struct {
	uint8_t *data;			/**< Current data being sent out or received */
	volatile uint32_t remaining;	/**< Number of bytes that remain to be sent or received */
	uint32_t size;			/**< Amount of data */
	uint32_t addr;			/**< Current device address */
	uint32_t p_max;			/**< Maximum packet size */
	ep0_status_t state;		/**< Status */
} host_state_t;

/**
  * @brief Pipe infomation
  */
typedef struct {
	usbh_device_t *dev;	/**< Current address for this pipe */
	uint8_t ep;		/**< Current address for this pipe */
	uint8_t dma_ch;		/**< DMA channel assigned to this pipe */
	uint32_t type;		/**< Type */
	uint32_t interval;	/**< Millisecond interval for this pipe */
	uint32_t next_tick;	/**< Next tick value to trigger */
	hcd_pipe_cbk cbk;	/**< Current call back for this pipe */
	uint8_t *p_read;	/**< Pointer to IN data */
	uint32_t size;		/**< Amount of the buffer */
	uint32_t count;		/**< Number of bytes read */
	pipe_state_t state;	/**< Status */
	uint16_t fifo_addr;	/**< Actual FIFO offset */
	uint8_t fifo_size;	/**< Size of the FIFO */
	uint8_t fifo_offset;	/**< Bit offset */
} usb_hcd_pipe_t;

/**
  * @brief Host class driver interface
  */
typedef struct {
	uint32_t interface_class;		/**< Interface class */
	void *(*open)(usbh_device_t *device);	/**< When this class of device has been detected */
	void (*close)(void *inst);		/**< When this class of device has been disconnected */
	void (*int_handler)(void *inst);	/**< interrupt callback function */
} usbh_class_driver_t;

/**
  * @brief Host controller infomation
  */
typedef struct {
	uint32_t usb_idx;						/**< Index of USB controller */
	usb_hcd_pipe_t ctrl_pipe;					/**< EP0 pipe */
	usb_hcd_pipe_t out_pipe[MAX_NUM_PIPES];				/**< OUT pipe */
	usb_hcd_pipe_t in_pipe[MAX_NUM_PIPES];				/**< IN pipe */
	usbh_device_t dev[USB_MAX_DEVICES + 1];				/**< Device array */
	volatile usbh_device_state_t dev_state[USB_MAX_DEVICES + 1];	/**< Device status */
	void *pool;							/**< Memory pool */
	uint32_t size;							/**< Size of pool */
	uint32_t nr_ep;							/**< Number of endpoint */
	const usbh_class_driver_t * const *driver;			/**< Class drivers */
	uint32_t nr_driver;						/**< Number of drivers */
	int32_t event_driver;						/**< Index in the driver list */
	uint32_t event_en;						/**< Enable event */
	uint32_t _class;						/**< Class */
	uint32_t int_num;						/**< Index of interrupt */
	uint32_t int_event;						/**< Interrupt event */
	uint32_t feature;						/**< Feature */
	uint32_t lpm;							/**< Low power management */
} usb_hcd_t;
/**
  * @}
  */
/** @defgroup Host_Core_Public_Variables Public Variables
  * @{
  */
extern const usbh_class_driver_t __usbh_msc_driver;
extern const usbh_class_driver_t __usbh_hid_driver;
extern const usbh_class_driver_t __usbh_audio_driver;
extern const usbh_class_driver_t __usbh_bulk_driver;
/**
  * @}
  */

/** @addtogroup Host_Core_Public_Functions
  * @{
  */
extern void usb_hcd_main(void);
extern int32_t usb_hcd_event_enable(uint32_t idx, void *driver, uint32_t event);
extern int32_t usb_hcd_event_disable(uint32_t idx, void *driver, uint32_t event);
extern void usb_hcd_init(uint32_t idx, void *data, uint32_t size);
extern void usb_hcd_power_config_init(uint32_t idx, uint32_t flag);
extern uint32_t usb_hcd_power_config_get(uint32_t idx);
extern uint32_t usb_hcd_power_config_set(uint32_t idx, uint32_t config);
extern uint32_t usb_hcd_power_auto(uint32_t idx);
extern void usb_hcd_register_driver(uint32_t idx, const usbh_class_driver_t * const *drv, uint32_t nr);
extern void usb_hcd_term(uint32_t idx);
extern void usb_hcd_set_config(uint32_t idx, usbh_device_t *dev, uint32_t config);
extern void usb_hcd_set_interface(uint32_t idx, usbh_device_t *dev, uint32_t interface, uint32_t alt);
extern void usb_hcd_suspend(uint32_t idx);
extern void usb_hcd_resume(uint32_t idx);
extern void usb_hcd_reset(uint32_t idx);
extern void usb_hcd_pipe_free(uint32_t pipe);
extern uint32_t usb_hcd_pipe_alloc(uint32_t idx, uint32_t ep_type, usbh_device_t *dev, hcd_pipe_cbk cbk);
extern uint32_t usb_hcd_pipe_alloc_size(uint32_t idx, uint32_t ep_type, usbh_device_t *dev, uint32_t size, hcd_pipe_cbk cbk);
extern uint32_t usb_hcd_pipe_config(uint32_t pipe, uint32_t p_max, uint32_t interval, uint32_t t_ep);
extern uint32_t usb_hcd_pipe_status(uint32_t pipe);
extern uint32_t usb_hcd_pipe_write(uint32_t pipe, uint8_t *data, uint32_t size);
extern uint32_t usb_hcd_pipe_read(uint32_t pipe, uint8_t *data, uint32_t size);
extern uint32_t usb_hcd_pipe_schedule(uint32_t pipe, uint8_t *data, uint32_t size);
extern uint32_t usb_hcd_pipe_transfer_size_get(uint32_t pipe);
extern void usb_hcd_pipe_data_ack(uint32_t pipe);
extern uint32_t usb_hcd_pipe_read_n_blocking(uint32_t pipe, uint8_t *data, uint32_t size);

extern uint32_t usb_hcd_get_string_desc(usbh_device_t *dev, uint8_t *buf, uint32_t size, uint32_t lang, uint32_t idx);
extern uint32_t usb_hcd_ctrl_transfer(uint32_t idx, usb_request_t *packet, usbh_device_t *dev,
                                                uint8_t *data, uint32_t size, uint32_t p_max);
extern void _usb_host_int_handler(uint32_t idx, uint32_t status);
extern void usb0_host_int_handler(void);
extern uint8_t usb_hcd_dev_hub_port(uint32_t inst);
extern uint8_t usb_hcd_dev_addr(uint32_t inst);
extern uint8_t usb_hcd_dev_class(uint32_t inst, uint32_t interface);
extern uint8_t usb_hcd_dev_sub_class(uint32_t inst, uint32_t interface);
extern uint8_t usb_hcd_dev_protocol(uint32_t inst, uint32_t interface);
extern uint32_t usb_hcd_lpm_sleep(usbh_device_t *dev);
extern void usb_hcd_lpm_resume(uint32_t idx);
extern uint32_t usb_hcd_lpm_status(usbh_device_t *dev);
extern void usbh_hub_main(void);
extern void usbh_hub_init(void);
extern void usbh_hub_enum_cplt(uint8_t hub, uint8_t port);
extern void usbh_hub_enum_error(uint8_t hub, uint8_t port);
extern uint32_t usb_hcd_hub_device_connect(uint32_t idx, uint8_t hub, uint8_t port, uint32_t speed);
extern void usb_hcd_hub_device_disconnect(uint32_t idx, uint32_t num);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
