/**
  *********************************************************************************
  *
  * @file    usbh_core.c
  * @brief   Functions related to host mode.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbh_core.h"
#include "usbh_hub.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @defgroup HOST Host
  * @brief Host mode driver
  * @{
  */
/** @defgroup Host_Core Core
  * @brief Host core driver
  * @{
  */

/** @defgroup Host_Core_Private_Variables Private Variables
  * @{
  */
static uint32_t __alloc[2];
static usb_hcd_t __hcd;
static uint32_t __power_config = USBHCD_VBUS_AUTO_HIGH;
static volatile uint32_t __c_tick = 0;
static int32_t __usbh_active[USB_MAX_DEVICES + 1];
static void *__driver_inst[USB_MAX_DEVICES + 1];
#ifndef USB_DMA_N_SUPPORT
static usb_dma_callback_t dma_tx_cbk;
static usb_dma_callback_t dma_rx_cbk;
#endif
static volatile host_state_t __usbh_ep0 = {
	0,
	0,
	0,
	0,
	0,
	EP0_STATE_IDLE
};
/**
  * @}
  */

/** @defgroup Host_Core_Private_Functions Private Functions
  * @{
  */
static void usb_hcd_ep0_state_tx(void);
static void usb_hcd_enum_handler(void);
static void usb_hcd_clear_feature(uint32_t addr, uint32_t ep, uint32_t feature);
static void usbh_device_machine(usbh_device_state_t old_state, uint32_t idx);

#ifndef USB_DMA_N_SUPPORT
/**
  * @brief  Callback function when DMA send complete.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbh_pipe_dma_tx_cplt(uint8_t ch, void *arg)
{
	int8_t ep = usb_dma_ep_get(ch);
	__hcd.out_pipe[ep - 1].state = PIPE_STATE_WRITE_DMA_WAIT;
	map_usb_ep_data_send(ep, USB_TRANS_OUT);

	return;
}

/**
  * @brief  Callback function when DMA send error.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbh_pipe_dma_tx_err(uint8_t ch, void *arg)
{
	int8_t ep = usb_dma_ep_get(ch);
	__hcd.out_pipe[ep - 1].state = PIPE_STATE_ERROR;

	return;
}

/**
  * @brief  Callback function when DMA receive complete.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbh_pipe_dma_rx_cplt(uint8_t ch, void *arg)
{
	int8_t ep = usb_dma_ep_get(ch);
	__hcd.in_pipe[ep - 1].state = PIPE_STATE_DATA_READY;
	
	if (__hcd.in_pipe[ep - 1].cbk)
		__hcd.in_pipe[ep - 1].cbk(IN_PIPE_HANDLE(ep - 1), USB_EVENT_RX_AVAILABLE);

	return;
}

/**
  * @brief  Callback function when DMA receive error.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbh_pipe_dma_rx_err(uint8_t ch, void *arg)
{
	int8_t ep = usb_dma_ep_get(ch);
	__hcd.in_pipe[ep - 1].state = PIPE_STATE_ERROR;

	return;
}
#endif
/**
  * @brief  Allocates the memory to support configuration descriptors.
  * @param  dev: Device.
  * @param  size: Size of memory.
  * @retval Size
  */
static uint32_t alloc_config_desc(usbh_device_t *dev, uint32_t size)
{
	uint32_t i, block_len, pool_size;
	uint8_t *pool;

	if (__hcd.dev[0].desc_config != 0)
		return size;

	size = (size + 3) & ~3;

	if (__hcd.size < size)
		return 0;

	__hcd.dev[0].desc_config = __hcd.pool;
	__hcd.dev[0].size_config = size;

	pool = (uint8_t *)__hcd.pool + size;
	pool_size = __hcd.size - size;
	block_len = (pool_size / USB_MAX_DEVICES) & ~3;

	for (i = 1; i < USB_MAX_DEVICES; ++i) {
		__hcd.dev[i].desc_config = (config_desc_t *)(pool + (i * block_len));
		__hcd.dev[i].size_config = block_len;
	}

	return size;
}

/**
  * @brief  Free the memory to support configuration descriptors.
  * @param  dev: Device.
  * @retval None
  */
static void free_config_desc(usbh_device_t *dev)
{
	uint32_t i;

	if (&__hcd.dev[0] != dev)
		return;

	for (i = 0; i < USB_MAX_DEVICES; ++i) {
		__hcd.dev[i].flag &= ~USBHDEV_FLAG_ALLOCATED;
		__hcd.dev[i].desc_config = NULL;
		__hcd.dev[i].size_config = 0;
	}

	return;
}

/**
  * @brief  Get Index of the device from instance.
  * @param  inst: Instance.
  * @retval Index of the device
  */
static uint8_t hcd_inst_to_dev_idx(uint32_t inst)
{
	uint32_t idx = (inst & 0xff);

	return idx > USB_MAX_DEVICES ? 0xff : idx;
}

/**
  * @brief  Get flags from event.
  * @param  event: Event.
  * @retval Flags
  */
static uint32_t get_event_flag(uint32_t event)
{
	uint32_t flag = 0;

	switch (event) {
	case USB_EVENT_SOF:
		flag |= USBHCD_EVFLAG_SOF;
		break;
	case USB_EVENT_CONNECTED:
		flag |= USBHCD_EVFLAG_CONNECT;
		break;
	case USB_EVENT_DISCONNECTED:
		flag |= USBHCD_EVFLAG_DISCNCT;
		break;
	case USB_EVENT_UNKNOWN_CONNECTED:
		flag |= USBHCD_EVFLAG_UNKCNCT;
		break;
	case USB_EVENT_POWER_FAULT:
		flag |= USBHCD_EVFLAG_PWRFAULT;
		break;
	case USB_EVENT_POWER_DISABLE:
		flag |= USBHCD_EVFLAG_PWRDIS;
		break;
	case USB_EVENT_POWER_ENABLE:
		flag |= USBHCD_EVFLAG_PWREN;
		break;
	default:
		break;
	}

	return flag;
}

/**
  * @brief  Send event to the upper layer.
  * @param  idx: Index.
  * @param  info: Information.
  * @param  flag: Flags.
  * @retval None
  */
static void usbh_send_event(uint32_t idx, event_info_t *info, uint32_t flag)
{
	if ((__hcd.event_driver != -1) && (__hcd.driver[__hcd.event_driver]->int_handler) && (__hcd.event_en & flag))
		__hcd.driver[__hcd.event_driver]->int_handler(info);

	return;
}

/**
  * @brief  Send unknown connect to the upper layer.
  * @param  idx: Index.
  * @param  class: Class.
  * @retval None
  */
static void send_unknown_connect(uint32_t idx, uint32_t class)
{
	event_info_t info;

	info.event    = USB_EVENT_UNKNOWN_CONNECTED;
	info.instance = class;
	usbh_send_event(0, &info, USBHCD_EVFLAG_UNKCNCT);

	return;
}

/**
  * @brief  Release FIFO resources.
  * @param  pipe: The pipe.
  * @retval None
  */
static void free_fifo(usb_hcd_pipe_t *pipe)
{
	uint32_t mask;

	mask = (1U << (1U << (pipe->fifo_size - 3))) - 1;
	mask = mask << pipe->fifo_offset;

	if (pipe->fifo_size > USB_FIFO_SZ_64)
		__alloc[1] &= ~mask;
	else
		__alloc[0] &= ~mask;

	return;
}

/**
  * @brief  Allocate FIFO resources.
  * @param  pipe: The pipe.
  * @param  size: Size of memory.
  * @retval Size of allocated memory.
  */
static uint32_t alloc_fifo(usb_hcd_pipe_t *pipe, uint32_t size)
{
	uint32_t i, mask, __size, tmp, idx;

	idx = size > 64 ? 1 : 0;
	mask = 1;
	__size = 64;

	pipe->fifo_offset = 0;
	pipe->fifo_size   = 3;

	for (i = 0; i < 32; ++i) {
		if ((__alloc[idx] & mask) == 0) {
			if (__size >= size) {
				__alloc[idx] |= mask;
				break;
			}

			pipe->fifo_size++;
			__size <<= 1;
			mask = mask | (mask << (1U << (pipe->fifo_size - 4)));

		}
		else {
			pipe->fifo_offset = i;
			pipe->fifo_size = 3;
			__size = 64;
			mask = 1 << i;
		}
	}

	if (i == 32) {
		__size = 0;
		pipe->fifo_addr   = 0;
		pipe->fifo_offset = 0;
		pipe->fifo_size   = 0;
	}
	else {
		tmp = pipe->fifo_offset << 6;
		tmp = idx == 0 ? tmp : tmp + 2048;
		pipe->fifo_addr = tmp;
	}

	return __size;
}

/**
  * @brief  Initialize the host controller.
  * @param  idx: Index of USB controller.
  * @param  pool: The pool.
  * @param  size: Size of pool.
  * @retval None
  */
static void _usb_hcd_init(uint32_t idx, void *pool, uint32_t size)
{
	int32_t i;

	assert_param(idx == 0);

	__alloc[0]    = 1;
	__alloc[1]    = 0;
	__hcd.nr_ep   = map_usb_num_ep_get();
	__hcd.usb_idx = 0;
	__hcd.int_num = map_usb_int_num_get();

	for (i = 0; i < MAX_NUM_PIPES; ++i) {
		__hcd.in_pipe[i].dev     = NULL;
		__hcd.in_pipe[i].type    = USBHCD_PIPE_UNUSED;
		__hcd.in_pipe[i].dma_ch  = USBHCD_DMA_UNUSED;
		__hcd.out_pipe[i].dev    = NULL;
		__hcd.out_pipe[i].type   = USBHCD_PIPE_UNUSED;
		__hcd.out_pipe[i].dma_ch = USBHCD_DMA_UNUSED;
	}

	usbh_hub_init();
#ifndef USB_DMA_N_SUPPORT
	usb_dma_init(idx);
	dma_tx_cbk.cplt_cbk = usbh_pipe_dma_tx_cplt;
	dma_tx_cbk.err_cbk  = usbh_pipe_dma_tx_err;
	dma_tx_cbk.cplt_arg = NULL;
	dma_tx_cbk.err_arg  = NULL;
	dma_rx_cbk.cplt_cbk = usbh_pipe_dma_rx_cplt;
	dma_rx_cbk.err_cbk  = usbh_pipe_dma_rx_err;
	dma_rx_cbk.cplt_arg = NULL;
	dma_rx_cbk.err_arg  = NULL;
#endif
	for (i = 0; i <= USB_MAX_DEVICES; ++i) {
		__hcd.dev_state[i] = HCD_IDLE;
		__usbh_active[i]   = -1;
		__hcd.dev[i].desc_config = NULL;
		__hcd.dev[i].config_read = 0;
		__hcd.dev[i].desc_device.bLength = 0;
		__hcd.dev[i].desc_device.bMaxPacketSize0 = 0;
		__hcd.dev[i].addr = 0;
		__hcd.dev[i].interface = 0;
		__hcd.dev[i].flag = 0;
	}

	__hcd.pool   = pool;
	__hcd.size   = size;
	__hcd._class = USB_CLASS_EVENTS;

	__hcd.event_en = USBHCD_EVFLAG_CONNECT | USBHCD_EVFLAG_UNKCNCT | USBHCD_EVFLAG_DISCNCT |
					USBHCD_EVFLAG_PWRFAULT | USBHCD_EVFLAG_PWREN | USBHCD_EVFLAG_PWRDIS;
	usb_tick_init();

	if (__usb_mode == USB_LIB_MODE_HOST) {
		usb_hcd_power_config_set(idx, (__power_config & ~USB_HOST_PWREN_AUTO));
		map_usb_host_components_init();
	}

	if (__hcd.feature & USB_FEATURE_LPM_EN) {
		if (__hcd.feature & USB_FEATURE_LPM_RMT_WAKE)
			map_usb_host_lpm_config(__hcd.lpm, USB_DEV_LPM_LS_L1 | USB_DEV_LPM_LS_RMTWAKE);
		else
			map_usb_host_lpm_config(__hcd.lpm, USB_DEV_LPM_LS_L1);

		map_usb_lpm_int_enable(USB_INTLPM_ERROR | USB_INTLPM_RESUME | USB_INTLPM_INCOMPLETE |
						USB_INTLPM_ACK | USB_INTLPM_NYET | USB_INTLPM_STALL);
	}

	return;
}

/**
  * @brief  Get configuration descriptor.
  * @param  idx: Index of USB controller.
  * @param  dev: Current device.
  * @retval Size of descriptor.
  */
static uint32_t usbh_get_config_desc(uint32_t idx, usbh_device_t *dev)
{
	uint32_t size = 0;
	usb_request_t req;
	config_desc_t desc;

	assert_param(idx == 0);

	req.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_STANDARD | USB_RTYPE_DEVICE;
	req.bRequest      = USBREQ_GET_DESCRIPTOR;
	req.wValue        = USB_DTYPE_CONFIGURATION << 8;
	req.wIndex        = 0;

	if (!dev->config_read) {
		req.wLength = sizeof(config_desc_t);
		size = usb_hcd_ctrl_transfer(0, &req, dev, (uint8_t *)&desc, sizeof(config_desc_t), dev->desc_device.bMaxPacketSize0);
	}

	if (size == sizeof(config_desc_t)) {
		req.wLength = desc.wTotalLength;

		if (alloc_config_desc(dev, desc.wTotalLength) == 0)
			return 0;
		if (req.wLength > dev->size_config)
			return 0;

		size = usb_hcd_ctrl_transfer(0, &req, dev, (uint8_t *)dev->desc_config, req.wLength, dev->desc_device.bMaxPacketSize0);

		if (size)
			dev->config_read = 1;
	}

	return size;
}

/**
  * @brief  Get device descriptor.
  * @param  idx: Index of USB controller.
  * @param  dev: Current device.
  * @retval Size of descriptor.
  */
static uint32_t usbh_get_device_desc(uint32_t idx, usbh_device_t *dev)
{
	uint32_t size = 0;
	usb_request_t req;

	assert_param(idx == 0);

	req.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_STANDARD | USB_RTYPE_DEVICE;
	req.bRequest      = USBREQ_GET_DESCRIPTOR;
	req.wValue        = USB_DTYPE_DEVICE << 8;
	req.wIndex        = 0;
	req.wLength       = sizeof(device_desc_t);

	if (dev->desc_device.bMaxPacketSize0 == 0)
		size = usb_hcd_ctrl_transfer(0, &req, dev, (uint8_t *)&(dev->desc_device), sizeof(device_desc_t), 8);

	if (size < sizeof(device_desc_t)) {
		req.wLength = (uint16_t)sizeof(device_desc_t);
		size = usb_hcd_ctrl_transfer(0, &req, dev, (uint8_t *)&(dev->desc_device), sizeof(device_desc_t), dev->desc_device.bMaxPacketSize0);
	}

	return size;
}

/**
  * @brief  Set device's address.
  * @param  idx: Index of USB controller.
  * @param  addr: Address.
  * @retval None
  */
static void usbh_set_addr(uint32_t idx, uint32_t addr)
{
	usb_request_t req;

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_STANDARD | USB_RTYPE_DEVICE;
	req.bRequest      = USBREQ_SET_ADDRESS;
	req.wValue        = addr;
	req.wIndex        = 0;
	req.wLength       = 0;

	usb_hcd_ctrl_transfer(0, &req, &__hcd.dev[idx], 0, 0, MAX_PACKET_SIZE_EP0);
	map_delay_ms(2);

	return;
}

/**
  * @brief  Clear feature.
  * @param  addr: Address.
  * @param  pipe: The pipe.
  * @param  feature: Feature.
  * @retval None
  */
static void usb_hcd_clear_feature(uint32_t addr, uint32_t pipe, uint32_t feature)
{
	usb_request_t req;
	uint32_t idx;

	idx = (pipe & EP_PIPE_IDX_M);

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_STANDARD | USB_RTYPE_ENDPOINT;
	req.bRequest      = USBREQ_CLEAR_FEATURE;
	req.wValue        = feature;
	req.wLength       = 0;

	if (pipe & EP_PIPE_TYPE_IN)
		req.wIndex = __hcd.in_pipe[idx].ep | 0x80;
	else
		req.wIndex = __hcd.out_pipe[idx].ep;

	usb_hcd_ctrl_transfer(0, &req, &__hcd.dev[addr - 1], 0, 0, MAX_PACKET_SIZE_EP0);

	if (pipe & EP_PIPE_TYPE_IN)
		map_usb_ep_data_toggle_clear(idx + 1, USB_EP_HOST_IN);
	else
		map_usb_ep_data_toggle_clear(idx + 1, USB_EP_HOST_OUT);

	map_delay_ms(2);
	return;
}

/**
  * @brief  Check pipe.
  * @retval None
  */
static void usbh_pipe_check(void)
{
	int32_t i;

	__c_tick++;

	for (i = 0; i < __hcd.nr_ep; ++i) {
		if (__hcd.in_pipe[i].type == USBHCD_PIPE_UNUSED)
			continue;

		if ((__hcd.in_pipe[i].interval != 0) && (__hcd.in_pipe[i].next_tick == __c_tick)) {
			__hcd.in_pipe[i].next_tick += __hcd.in_pipe[i].interval;

			if ((__hcd.in_pipe[i].state == PIPE_STATE_IDLE) && (__hcd.in_pipe[i].cbk))
				__hcd.in_pipe[i].cbk(IN_PIPE_HANDLE(i), USB_EVENT_SCHEDULER);
		}
	}
}

/**
  * @brief  Open driver.
  * @param  idx: Index of USB controller.
  * @param  num: Index in the device list.
  * @retval Index in the driver list.
  */
static int32_t usbh_open_driver(uint32_t idx, uint32_t num)
{
	int32_t i;
	uint32_t class;
	interface_desc_t *interface;
	event_info_t info;

	assert_param(idx == 0);

	interface = usb_desc_get_interface(__hcd.dev[num].desc_config, __hcd.dev[num].interface, USB_DESC_ANY);
	class = interface->bInterfaceClass;

	for (i = 0; i < __hcd.nr_driver; ++i) {
		if (__hcd.driver[i]->interface_class == class) {
			__driver_inst[num] = __hcd.driver[i]->open(&__hcd.dev[num]);

			if (__driver_inst[num] != 0)
				break;
		}
	}

	if (i == __hcd.nr_driver) {
		send_unknown_connect(idx, (idx << 16) | num);
		i = -1;
	}
	else {
		info.event = USB_EVENT_CONNECTED;
		info.instance = (idx << 16) | num;
		usbh_send_event(0, &info, USBHCD_EVFLAG_CONNECT);
	}

	return i;
}

/**
  * @brief  Disconnected device.
  * @param  idx: Index of USB controller.
  * @param  num: Index in the device list.
  * @retval None
  */
static void usbh_device_disconnect(uint32_t idx, uint32_t num)
{
	event_info_t info;

	assert_param(idx == 0);
	assert_param(num <= USB_MAX_DEVICES);

	info.event    = USB_EVENT_DISCONNECTED;
	info.instance = (idx << 16) | num;
	usbh_send_event(0, &info, USBHCD_EVFLAG_DISCNCT);

	__hcd._class = USB_CLASS_EVENTS;
	free_config_desc(&__hcd.dev[num]);

	__hcd.dev[num].flag = 0;
	__hcd.dev[num].config_read = false;
	__hcd.dev[num].desc_device.bMaxPacketSize0 = 0;
	__hcd.dev[num].desc_device.bLength = 0;
	__hcd.dev[num].addr = 0;

	if (__usbh_active[num] >= 0) {
		__hcd.driver[__usbh_active[num]]->close(__driver_inst[num]);

		__usbh_active[num] = -1;
		__driver_inst[num] = 0;
	}

	if ((num == 0) && (__usb_mode == USB_LIB_MODE_OTG))
		usb_mode_otg_device_disconnect(0);
}

/**
  * @brief  Handle event.
  * @param  state: Device status.
  * @param  idx: Index in the device list.
  * @retval None
  */
static void usbh_device_machine(usbh_device_state_t state, uint32_t idx)
{
	switch(__hcd.dev_state[idx]) {
	case HCD_POWER_FAULT:
		break;

	case HCD_VBUS_ERROR:
		map_usb_int_unregister();

		if ((state != HCD_IDLE) && (state != HCD_POWER_FAULT))
		{
			usbh_device_disconnect(0, idx);
		}

		map_usb_control_reset();
		map_delay_ms(100);
		_usb_hcd_init(0, __hcd.pool, __hcd.size);
		break;

	case HCD_DEV_RESET:
		if (!idx) {
			printf_e("\rGenerate bus reset.\r\n");
			usb_hcd_reset(0);
		}

		__hcd.dev_state[0] = HCD_DEV_CONNECT;
		__hcd.dev[0].config_read = 0;
		break;

	case HCD_DEV_CONNECT:
	case HCD_DEV_CONNECT_HUB:
		if (__hcd.dev[idx].desc_device.bLength == 0) {
			printf_e("\rDevice#%d. Get device descriptor\r\n", idx);

			if (__hcd.dev_state[idx] == HCD_DEV_CONNECT) {
				switch (map_usb_host_speed_get()) {
				case USB_HIGH_SPEED:
					__hcd.dev[idx].speed = USB_EP_SPEED_HIGH;
					break;
				case USB_FULL_SPEED:
					__hcd.dev[idx].speed = USB_EP_SPEED_FULL;
					break;
				default:
					__hcd.dev[idx].speed = USB_EP_SPEED_LOW;
					break;
				}
			}

			if (usbh_get_device_desc(0, &__hcd.dev[idx]) == 0) {
				__hcd.dev_state[idx] = HCD_DEV_ERROR;
				printf_e("\rDevice#%d. Failed to get descriptor\r\n", idx);
				send_unknown_connect(0, 0);

				if (__hcd.dev[idx].hub)
					usbh_hub_enum_error(__hcd.dev[idx].hub, __hcd.dev[idx].hub_port);
			}
		}
		else if (__hcd.dev[idx].addr == 0) {
			printf_e("\rDevice#%d. Set address#%d\r\n", idx, idx + 1);
			usbh_set_addr(idx, (idx + 1));
			__hcd.dev[idx].addr = (idx + 1);
			__hcd.dev_state[idx] = HCD_DEV_ADDRESS;
		}
		break;

	case HCD_DEV_ADDRESS:
		if (!__hcd.dev[idx].config_read) {
			printf_e("\rDevice#%d. Get config descriptor\r\n", idx);

			if (usbh_get_config_desc(0, &__hcd.dev[idx]) == 0) {
				__hcd.dev_state[idx] = HCD_DEV_ERROR;

				printf_e("\rDevice#%d. Failed to get descriptor\r\n", idx);
				send_unknown_connect(0, 0);

				if (__hcd.dev[idx].hub)
					usbh_hub_enum_error(__hcd.dev[idx].hub, __hcd.dev[idx].hub_port);
			}
		}
		else {
			printf_e("\rDevice#%d. Set config\r\n", idx);
			usb_hcd_set_config(0, &__hcd.dev[idx], 1);
			__hcd.dev_state[idx] = HCD_DEV_CONFIGURE;
			__usbh_active[idx] = usbh_open_driver(0, idx);

			if (__hcd.dev[idx].hub) {
				usbh_hub_enum_cplt(__hcd.dev[idx].hub, __hcd.dev[idx].hub_port);
			}
		}
		break;

	case HCD_DEV_REQUEST:
		__hcd.dev_state[idx] = HCD_DEV_CONNECT;
		break;

	case HCD_DEV_GET_STRING:
		break;

	case HCD_DEV_DISCONNECT:
		printf_e("\rDevice#%d. disconnected\r\n", idx);
		usbh_device_disconnect(0, idx);
		__hcd.dev_state[idx] = HCD_IDLE;
		break;

	case HCD_DEV_CONFIGURE:
		break;

	case HCD_DEV_ERROR:
		printf_e("Connection %d - Error!\r\n", idx);

		if (idx == 0) {
			__hcd.int_event |= INT_EVENT_DISCONNECT;
			__hcd.dev_state[idx] = HCD_IDLE;
		}
		break;

	default:
		break;
	}
}

/**
  * @brief  Handle enumeration.
  * @retval None
  */
static void usb_hcd_enum_handler(void)
{
	uint32_t status, size;

	status = map_usb_ep_status(USB_EP_0);

	if (status == USB_HOST_EP0_ERROR) {
		map_usb_host_ep_status_clear(USB_EP_0, USB_HOST_EP0_ERROR);
		map_usb_fifo_flush(USB_EP_0, 0);
		__usbh_ep0.state = EP0_STATE_ERROR;

		return;
	}

	switch(__usbh_ep0.state) {
	case EP0_STATE_STATUS:
		if (status & (USB_HOST_EP0_RXPKTRDY | USB_HOST_EP0_STATUS))
			map_usb_host_ep_status_clear(USB_EP_0, (USB_HOST_EP0_RXPKTRDY | USB_HOST_EP0_STATUS));

		__usbh_ep0.state = EP0_STATE_IDLE;
		break;

	case EP0_STATE_STATUS_IN:
		map_usb_host_request_status();
		__usbh_ep0.state =  EP0_STATE_STATUS;
		break;

	case EP0_STATE_IDLE:
		break;

	case EP0_STATE_SETUP_OUT:
		usb_hcd_ep0_state_tx();
		break;

	case EP0_STATE_SETUP_IN:
		map_usb_host_request_in(USB_EP_0);
		__usbh_ep0.state = EP0_STATE_RX;

		break;

	case EP0_STATE_RX:
		if (status & USB_HOST_EP0_RX_STALL) {
			__usbh_ep0.state = EP0_STATE_IDLE;
			map_usb_host_ep_status_clear(USB_EP_0, status & USB_HOST_IN_STATUS);
			break;
		}

		size = __usbh_ep0.remaining;

		if (size > __usbh_ep0.p_max)
			size = MAX_PACKET_SIZE_EP0;

		if (size != 0)
			map_usb_ep_data_get(USB_EP_0, __usbh_ep0.data, &size);

		__usbh_ep0.data += size;
		__usbh_ep0.remaining -= size;
		map_usb_host_ep_data_ack(USB_EP_0);

		if ((size < __usbh_ep0.p_max) || (__usbh_ep0.remaining == 0)) {
			__usbh_ep0.state = EP0_STATE_STATUS;
			__usbh_ep0.data  = NULL;
			map_usb_ep_data_send(USB_EP_0, USB_TRANS_STATUS);
		}
		else {
			map_usb_host_request_in(USB_EP_0);
		}
		break;

	case EP0_STATE_STALL:
		__usbh_ep0.state = EP0_STATE_IDLE;
		break;

	default:
		break;
	}
}

/**
  * @brief  Send data via EP0
  * @retval None
  */
static void usb_hcd_ep0_state_tx(void)
{
	uint32_t size;
	uint8_t *data;

	__usbh_ep0.state = EP0_STATE_SETUP_OUT;
	size = __usbh_ep0.remaining;

	if (size > 64)
		size = 64;

	data = (uint8_t *)__usbh_ep0.data;
	__usbh_ep0.remaining -= size;
	__usbh_ep0.data      += size;
	map_usb_ep_data_put(USB_EP_0, data, size);

	if (size == 64) {
		map_usb_ep_data_send(USB_EP_0, USB_TRANS_OUT);
	}
	else {
		map_usb_ep_data_send(USB_EP_0, USB_TRANS_OUT);
		__usbh_ep0.state = EP0_STATE_STATUS_IN;
	}
}
/**
  * @}
  */

/** @defgroup Host_Core_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Enable some events.
  * @param  idx: Index of the USB controller.
  * @param  driver: Device.
  * @param  event: Event which will be enabled.
  * @retval Status.
  */
int32_t usb_hcd_event_enable(uint32_t idx, void *driver, uint32_t event)
{
	int32_t ret = 0;
	uint32_t flag;

	assert_param(idx == 0);

	flag = get_event_flag(event);

	if (flag) {
		__hcd.event_en |= flag;
		ret = 1;
	}

	return ret;
}

/**
  * @brief  Disable some events.
  * @param  idx: Index of the USB controller.
  * @param  driver: Device.
  * @param  event: Event which will be disenabled.
  * @retval Status.
  */
int32_t usb_hcd_event_disable(uint32_t idx, void *driver, uint32_t event)
{
	int32_t ret = 0;
	uint32_t flag;

	assert_param(idx == 0);

	flag = get_event_flag(event);

	if (flag) {
		__hcd.event_en &= ~flag;
		ret = 1;
	}

	return ret;
}

/**
  * @brief  Get transfer size.
  * @param  pipe: The pipe.
  * @retval Size.
  */
uint32_t usb_hcd_pipe_transfer_size_get(uint32_t pipe)
{
	return __hcd.in_pipe[pipe & EP_PIPE_IDX_M].count;
}

/**
  * @brief  Allocate pipe.
  * @param  idx: Index of the USB controller.
  * @param  ep_type: Type of endpoint
  * @param  dev: Current device.
  * @param  size: Size of the FIFO.
  * @param  cbk: Callback function.
  * @retval The pipe.
  */
uint32_t usb_hcd_pipe_alloc_size(uint32_t idx, uint32_t ep_type, usbh_device_t *dev, uint32_t size, hcd_pipe_cbk cbk)
{
	int32_t i;
	uint32_t hub_addr;

	assert_param(idx == 0);

	for (i = 0; i < MAX_NUM_PIPES; ++i) {
		if (ep_type & EP_PIPE_TYPE_OUT) {
			if (__hcd.out_pipe[i].dev != 0)
				continue;

			if (ep_type & EP_PIPE_USE_DMA) {
#ifndef USB_DMA_N_SUPPORT
				__hcd.out_pipe[i].dma_ch = usb_dma_channel_alloc(i + 1, USB_DMA_EP_CFG_TX, &dma_tx_cbk);
#endif
			}

			__hcd.out_pipe[i].type = ep_type;
			__hcd.out_pipe[i].dev  = dev;
			__hcd.out_pipe[i].cbk  = cbk;

			map_usb_host_ep_status_clear(i + 1, USB_HOST_OUT_STATUS);
			map_usb_ep_data_toggle_clear(i + 1, USB_EP_HOST_OUT);
			__hcd.out_pipe[i].state = PIPE_STATE_IDLE;

			if (alloc_fifo(&__hcd.out_pipe[i], size) != 0)
				map_usb_fifo_config_set(i + 1, __hcd.out_pipe[i].fifo_addr, __hcd.out_pipe[i].fifo_size, USB_EP_HOST_OUT);

			map_usb_host_addr_set(i + 1, dev->addr, USB_EP_HOST_OUT);
			hub_addr = dev->hub | (dev->hub_port << 8);
			map_usb_host_hub_addr_set(i + 1, hub_addr, (USB_EP_HOST_OUT | dev->speed));
			break;
		}
		else if (ep_type & EP_PIPE_TYPE_IN) {
			if (__hcd.in_pipe[i].dev != 0)
				continue;

			if (ep_type & EP_PIPE_USE_DMA) {
#ifndef USB_DMA_N_SUPPORT
				__hcd.in_pipe[i].dma_ch = usb_dma_channel_alloc(i + 1, USB_DMA_EP_CFG_RX_HOST, &dma_rx_cbk);
#endif
			}

			__hcd.in_pipe[i].type = ep_type;
			__hcd.in_pipe[i].dev  = dev;
			__hcd.in_pipe[i].cbk  = cbk;

			map_usb_host_ep_status_clear(i + 1, USB_HOST_IN_STATUS);
			map_usb_ep_data_toggle_clear(i + 1, USB_EP_HOST_IN);

			if (alloc_fifo(&__hcd.in_pipe[i], size) != 0)
				map_usb_fifo_config_set(i + 1, __hcd.in_pipe[i].fifo_addr, __hcd.in_pipe[i].fifo_size, USB_EP_HOST_IN);

			map_usb_host_addr_set(i + 1, dev->addr, USB_EP_HOST_IN);
			hub_addr = dev->hub | (dev->hub_port << 8);
			map_usb_host_hub_addr_set(i + 1, hub_addr, (USB_EP_HOST_IN | dev->speed));
			__hcd.in_pipe[i].state = PIPE_STATE_IDLE;

			break;
		}
	}

	if (i == MAX_NUM_PIPES)
		return 0;

	return (ep_type | i);
}

/**
  * @brief  Allocate pipe.
  * @param  idx: Index of the USB controller.
  * @param  ep_type: Type of endpoint
  * @param  dev: Current device.
  * @param  cbk: Callback function.
  * @retval The pipe.
  */
uint32_t usb_hcd_pipe_alloc(uint32_t idx, uint32_t ep_type, usbh_device_t *dev, hcd_pipe_cbk cbk)
{
	return usb_hcd_pipe_alloc_size(idx, ep_type, dev, 64, cbk);
}

/**
  * @brief  Configure the pipe.
  * @param  pipe: Allocated endpoint to modify.
  * @param  p_max: Aaximum payload.
  * @param  interval: Polling interval.
  * @param  t_ep: Target endpoint.
  * @retval Status.
  */
uint32_t usb_hcd_pipe_config(uint32_t pipe, uint32_t p_max, uint32_t interval, uint32_t t_ep)
{
	uint32_t flag, idx = pipe & EP_PIPE_IDX_M;

	if (pipe & EP_PIPE_TYPE_OUT) {
		if (__hcd.out_pipe[idx].type & EP_PIPE_TYPE_BULK)
			flag = USB_EP_MODE_BULK;
		else if (__hcd.out_pipe[idx].type & EP_PIPE_TYPE_INTR)
			flag = USB_EP_MODE_INT;
		else if (__hcd.out_pipe[idx].type & EP_PIPE_TYPE_ISOC)
			flag = USB_EP_MODE_ISOC;
		else
			flag = USB_EP_MODE_CTRL;

		flag |= USB_EP_HOST_OUT;

		__hcd.out_pipe[idx].ep        = (uint8_t)t_ep;
		__hcd.out_pipe[idx].interval  = interval;
		__hcd.out_pipe[idx].next_tick = interval + __c_tick;

		flag |= (__hcd.out_pipe[idx].dev->speed);
	}
	else {
		if (__hcd.in_pipe[idx].type & EP_PIPE_TYPE_BULK)
			flag = USB_EP_MODE_BULK;
		else if (__hcd.in_pipe[idx].type & EP_PIPE_TYPE_INTR)
			flag = USB_EP_MODE_INT;
		else if (__hcd.in_pipe[idx].type & EP_PIPE_TYPE_ISOC)
			flag = USB_EP_MODE_ISOC;
		else
			flag = USB_EP_MODE_CTRL;

		flag |= USB_EP_HOST_IN;

		__hcd.in_pipe[idx].ep        = (uint8_t)t_ep;
		__hcd.in_pipe[idx].interval  = interval;
		__hcd.in_pipe[idx].next_tick = interval + __c_tick;

		flag |= __hcd.in_pipe[idx].dev->speed;
	}

	map_usb_host_ep_config(((pipe & EP_PIPE_IDX_M) + 1), p_max, interval, t_ep, flag);

	return 0;
}

/**
  * @brief  Get current status of the pipe.
  * @param  pipe: The pipe
  * @retval Status.
  */
uint32_t usb_hcd_pipe_status(uint32_t pipe)
{
	return 0;
}

/**
  * @brief  Write data to the pipe.
  * @param  pipe: The pipe
  * @param  data: Data to send.
  * @param  size: Amount of data.
  * @retval Size.
  */
uint32_t usb_hcd_pipe_write(uint32_t pipe, uint8_t *data, uint32_t size)
{
	uint32_t ep, __len, len, idx, max;

	idx   = pipe & EP_PIPE_IDX_M;
	ep    = idx + 1;
	__len = size;
	max   = __hcd.out_pipe[idx].fifo_size;
	max   = 1u << (max + 3);

	while (__len) {
		len = __len >= max ? max : __len;
		
		
		if (pipe & EP_PIPE_USE_DMA) {
			__hcd.out_pipe[idx].state = PIPE_STATE_WRITE_DMA;
			usb_dma_config(__hcd.out_pipe[idx].dma_ch, (uint32_t)data, len);
		}
		else {
			__hcd.out_pipe[idx].state = PIPE_STATE_WRITING;
			map_usb_ep_data_put(ep, data, len);
			map_usb_ep_data_send(ep, USB_TRANS_OUT);
		}

		while (1) {
			if (__hcd.int_event & (INT_EVENT_DISCONNECT | INT_EVENT_VBUS_ERR | INT_EVENT_POWER_FAULT)) {
				printf_e("#error1!\n\r");
				__hcd.out_pipe[idx].state = PIPE_STATE_ERROR;
				__len = 0;
				break;
			}
			else if (__hcd.out_pipe[idx].state == PIPE_STATE_DATA_SENT) {
				__len -= len;
				data  += len;
				break;
			}
			else if (__hcd.out_pipe[idx].state == PIPE_STATE_STALL) {
				printf_e("PIPE_STATE_STALL!\n\r");
				size  = 0;
				__len = 0;

				usb_hcd_clear_feature(__hcd.out_pipe[idx].dev->addr, pipe, USB_FEATURE_EP_HALT);
				break;
			}
			else if (__hcd.out_pipe[idx].state == PIPE_STATE_ERROR) {
				size  = 0;
				__len = 0;
				break;
			}
		}
	}

	__hcd.out_pipe[idx].state = PIPE_STATE_IDLE;
	return size;
}

/**
  * @brief  Schedule IN transaction.
  * @param  pipe: The pipe
  * @param  data: Data that is received.
  * @param  size: Amount of data.
  * @retval Size.
  */
uint32_t usb_hcd_pipe_schedule(uint32_t pipe, uint8_t *data, uint32_t size)
{
	uint32_t ep, idx;

	idx = pipe & EP_PIPE_IDX_M;
	ep  = idx + 1;

	if (pipe & EP_PIPE_TYPE_OUT) {
		if (pipe & EP_PIPE_USE_DMA) {
			__hcd.out_pipe[idx].state = PIPE_STATE_WRITE_DMA;
			usb_dma_config(__hcd.out_pipe[idx].dma_ch, (uint32_t)data, size);
		}
		else {
			__hcd.out_pipe[idx].state = PIPE_STATE_WRITING;
			map_usb_ep_data_put(ep, data, size);
			map_usb_ep_data_send(ep, USB_TRANS_OUT);
		}
	}
	else {
		if ((pipe & EP_PIPE_USE_DMA) == 0) {
			__hcd.in_pipe[idx].state = PIPE_STATE_READING;
		}
		else {
			__hcd.in_pipe[idx].state = PIPE_STATE_READ_DMA;
		}

		__hcd.in_pipe[idx].p_read = data;
		__hcd.in_pipe[idx].size   = size;
		map_usb_host_request_in(ep);
		size = 0;
	}

	return size;
}

/**
  * @brief  Read data from the pipe.
  * @param  pipe: The pipe
  * @param  data: Data that is received.
  * @param  size: Amount of data.
  * @retval Size.
  */
uint32_t usb_hcd_pipe_read_n_blocking(uint32_t pipe, uint8_t *data, uint32_t size)
{
	uint32_t ep;

	ep = (EP_PIPE_IDX_M & pipe) + 1;
	map_usb_ep_data_get(ep, data, &size);
	map_usb_host_ep_data_ack(ep);
	__hcd.in_pipe[EP_PIPE_IDX_M & pipe].state = PIPE_STATE_IDLE;

	return size;
}

/**
  * @brief  Acknowledges data.
  * @param  pipe: The pipe
  * @retval None
  */
void usb_hcd_pipe_data_ack(uint32_t pipe)
{
	uint32_t ep;

	ep = (EP_PIPE_IDX_M & pipe) + 1;
	map_usb_host_ep_data_ack(ep);
	__hcd.in_pipe[EP_PIPE_IDX_M & pipe].state = PIPE_STATE_IDLE;
}

/**
  * @brief  Read data from the pipe.
  * @param  pipe: The pipe
  * @param  data: Data that is received.
  * @param  size: Amount of data.
  * @retval Size.
  */
uint32_t usb_hcd_pipe_read(uint32_t pipe, uint8_t *data, uint32_t size)
{
	uint32_t ep, __len, len, idx, max;

	len   = 0;
	idx   = pipe & EP_PIPE_IDX_M;
	ep    = idx + 1;
	__len = size;
	max   = __hcd.in_pipe[idx].fifo_size;
	max   = 1u << (max + 3);

	while (__len) {
		__hcd.in_pipe[idx].state = PIPE_STATE_READING;

		if (pipe & EP_PIPE_USE_DMA)
			__hcd.in_pipe[idx].state = PIPE_STATE_READ_DMA;

		__hcd.in_pipe[idx].p_read = data;
		__hcd.in_pipe[idx].size   = (__len < max) ? __len : max;
		map_usb_host_request_in(ep);

		while (1) {
			if (__hcd.in_pipe[idx].state == PIPE_STATE_STALL) {
				size  = 0;
				__len = 0;

				usb_hcd_clear_feature(__hcd.in_pipe[idx].dev->addr, pipe, USB_FEATURE_EP_HALT);
				break;
			}

			if (__hcd.int_event & (INT_EVENT_DISCONNECT | INT_EVENT_VBUS_ERR | INT_EVENT_POWER_FAULT)) {
				__hcd.in_pipe[idx].state = PIPE_STATE_ERROR;
				__len = 0;
				break;
			}

			if (__hcd.in_pipe[idx].state == PIPE_STATE_DATA_READY) {
				len = __len > max ? max : __len;
				map_usb_host_ep_data_ack(ep);
				__len -= len;

				if (len < max)
					size -= __len;
				else
					data += max;

				break;
			}
			else if (__hcd.in_pipe[idx].state == PIPE_STATE_ERROR) {
				size  = 0;
				__len = 0;
				break;
			}
		}
	}

	__hcd.in_pipe[idx].state = PIPE_STATE_IDLE;
	return size;
}

/**
  * @brief  Free the pipe.
  * @param  pipe: Allocated pipe to release.
  * @retval None
  */
void usb_hcd_pipe_free(uint32_t pipe)
{
	uint32_t idx = pipe & EP_PIPE_IDX_M;

	if (pipe & EP_PIPE_TYPE_OUT) {
		__hcd.out_pipe[idx].dev  = NULL;
		__hcd.out_pipe[idx].type = 0;
		__hcd.out_pipe[idx].cbk  = NULL;

		if (__hcd.out_pipe[idx].dma_ch != USBHCD_DMA_UNUSED) {
			usb_dma_channel_free((uint8_t)(idx + 1), USB_DMA_EP_CFG_TX);
			__hcd.out_pipe[idx].dma_ch = USBHCD_DMA_UNUSED;
		}

		if (__hcd.out_pipe[idx].fifo_size)
			free_fifo(&__hcd.out_pipe[idx]);

		map_usb_host_addr_set(idx + 1, 0, USB_EP_HOST_OUT);
		map_usb_host_hub_addr_set(idx + 1, 0, (USB_EP_HOST_OUT | USB_EP_SPEED_LOW));
	}
	else if (pipe & EP_PIPE_TYPE_IN) {
		__hcd.in_pipe[idx].dev  = NULL;
		__hcd.in_pipe[idx].type = 0;
		__hcd.in_pipe[idx].cbk  = NULL;

		if (__hcd.in_pipe[idx].dma_ch != USBHCD_DMA_UNUSED) {
			usb_dma_channel_free((uint8_t)(idx + 1), USB_DMA_EP_CFG_RX_HOST);
			__hcd.in_pipe[idx].dma_ch = USBHCD_DMA_UNUSED;
		}

		if (__hcd.in_pipe[pipe & EP_PIPE_IDX_M].fifo_size)
			free_fifo(&__hcd.in_pipe[pipe & EP_PIPE_IDX_M]);

		map_usb_host_addr_set(idx + 1, 0, USB_EP_HOST_IN);
		map_usb_host_hub_addr_set(idx + 1, 0, (USB_EP_HOST_IN | USB_EP_SPEED_LOW));
		map_usb_host_request_in_clear(idx + 1);
	}

	return;
}

/**
  * @brief  Configure dufalut power mode.
  * @param  idx: Index of the USB controller.
  * @param  flag: Flags.
  * @retval None
  */
void usb_hcd_power_config_init(uint32_t idx, uint32_t flag)
{
	assert_param(idx == 0);

	__power_config = flag;
	return;
}

/**
  * @brief  Get power mode.
  * @param  idx: Index of the USB controller.
  * @retval Status
  */
uint32_t usb_hcd_power_config_get(uint32_t idx)
{
	assert_param(idx == 0);

	return __power_config;
}

/**
  * @brief  Set power mode.
  * @param  idx: Index of the USB controller.
  * @param  config: The power mode.
  * @retval Status
  */
uint32_t usb_hcd_power_config_set(uint32_t idx, uint32_t config)
{
	assert_param(idx == 0);

	__power_config = config;
	config = __power_config & ~(USBHCD_VBUS_MANUAL | USBHCD_FAULT_VBUS_DIS);

	if (__power_config & USBHCD_FAULT_VBUS_DIS) {
		assert_param((USBHCD_VBUS_AUTO_HIGH & 1) == 1);
		assert_param((USBHCD_VBUS_AUTO_LOW & 1) == 0);

		if(__power_config & 1) {
			__power_config |= USB_HOST_PWRFLT_EP_LOW;
			config |= USB_HOST_PWRFLT_EP_LOW;
		}
		else {
			__power_config |= USB_HOST_PWRFLT_EP_HIGH;
			config |= USB_HOST_PWRFLT_EP_HIGH;
		}
	}

	map_usb_host_pwr_config(config);

	if ((__power_config & USBHCD_VBUS_MANUAL) == 0)
		map_usb_host_pwr_enable();

	return 0;
}

/**
  * @brief  Set power automatic.
  * @param  idx: Index of the USB controller.
  * @retval Status
  */
uint32_t usb_hcd_power_auto(uint32_t idx)
{
	if (__power_config & USBHCD_VBUS_MANUAL)
		return 0;

	return 1;
}

/**
  * @brief  Initialize the USB controller.
  * @param  idx: Index of the USB controller.
  * @param  data: Memory pool.
  * @param  size: Size of the pool.
  * @retval None
  */
void usb_hcd_init(uint32_t idx, void *data, uint32_t size)
{
	int32_t i;

	assert_param(idx == 0);
	assert_param(size >= sizeof(config_desc_t));
	assert_param(__usb_mode != USB_LIB_MODE_DEVICE);

	if (__usb_mode == USB_LIB_MODE_NONE)
		__usb_mode = USB_LIB_MODE_HOST;

#ifdef USB_DRIVER_MD
	map_usb_controller_reset();
	map_usb_controller_enable();
	map_usb_clk_phy_enable();
	map_usb_re_config(false);

	if (__usb_mode != USB_LIB_MODE_OTG)
		map_usb_mode_force_host();
#endif
	
	_usb_hcd_init(idx, data, size);
	__hcd.event_driver = -1;

	for (i = 0; i < __hcd.nr_driver; ++i) {
		if (__hcd.driver[i]->interface_class == USB_CLASS_EVENTS)
			__hcd.event_driver = i;
	}

	return;
}

/**
  * @brief  Initialize the driver list.
  * @param  idx: Index of the USB controller.
  * @param  drv: Array ofclass drivers.
  * @param  nr: Number of entries in the array.
  * @retval None
  */
void usb_hcd_register_driver(uint32_t idx, const usbh_class_driver_t * const *drv, uint32_t nr)
{
	assert_param(idx == 0);

	__hcd.driver    = drv;
	__hcd.nr_driver = nr;

	return;
}

/**
  * @brief  Terminate the USB controller.
  * @param  idx: Index of the USB controller.
  * @retval None
  */
void usb_hcd_term(uint32_t idx)
{
	int32_t i;

	assert_param(idx == 0);

	map_usb_otg_session_request(false);
	map_usb_host_pwr_disable();
	map_usb_int_unregister();
	map_usb_int_disable(USB_INTCTRL_ALL);
	map_usb_int_disable_ep(USB_INTEP_ALL);

	for (i = 0; i < MAX_NUM_PIPES; ++i) {
		__hcd.in_pipe[i].type  = USBHCD_PIPE_UNUSED;
		__hcd.out_pipe[i].type = USBHCD_PIPE_UNUSED;
	}

	free_config_desc(&__hcd.dev[0]);

	__hcd.dev_state[0] = HCD_IDLE;
	__hcd.dev[0].desc_config = NULL;
	__hcd.dev[0].config_read = 0;
	__hcd.dev[0].desc_device.bLength = 0;
	__hcd.dev[0].desc_device.bMaxPacketSize0 = 0;
	__hcd.dev[0].addr = 0;
	__hcd.dev[0].interface = 0;
	__hcd.pool = NULL;
	__hcd.size = 0;

	return;
}

/**
  * @brief  Reset the USB controller.
  * @param  idx: Index of the USB controller.
  * @retval None
  */
void usb_hcd_reset(uint32_t idx)
{
	assert_param(idx == 0);

	map_usb_host_reset(1);
	map_delay_ms(20);
	map_usb_host_reset(0);
	map_delay_ms(20);

	return;
}

/**
  * @brief  Suspend the USB controller.
  * @param  idx: Index of the USB controller.
  * @retval None
  */
void usb_hcd_suspend(uint32_t idx)
{
	assert_param(idx == 0);

	map_usb_host_suspend();
	return;
}

/**
  * @brief  Resume the USB controller.
  * @param  idx: Index of the USB controller.
  * @retval None
  */
void usb_hcd_resume(uint32_t idx)
{
	assert_param(idx == 0);

	map_usb_host_resume(1);
	map_delay_ms(100);
	map_usb_host_resume(0);

	return;
}

/**
  * @brief  Request for a string descriptor.
  * @param  dev: Device for this request.
  * @param  buf: Buffer to store the requested string descriptor.
  * @param  size: Size of the buffer.
  * @param  lang: ID of the language.
  * @param  idx: Index for the request.
  * @retval Size.
  */
uint32_t usb_hcd_get_string_desc(usbh_device_t *dev, uint8_t *buf, uint32_t size, uint32_t lang, uint32_t idx)
{
	usb_request_t req;

	req.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_STANDARD | USB_RTYPE_DEVICE;
	req.bRequest      = USBREQ_GET_DESCRIPTOR;
	req.wValue        = (USB_DTYPE_STRING << 8) | (uint16_t)idx;
	req.wIndex        = lang;
	req.wLength       = (uint16_t)size;

	return usb_hcd_ctrl_transfer(0, &req, dev, buf, size, dev->desc_device.bMaxPacketSize0);
}

/**
  * @brief  Set the current configuration.
  * @param  idx: Index of the USB controller.
  * @param  dev: Device.
  * @param  config: Devices valid configurations.
  * @retval None
  */
void usb_hcd_set_config(uint32_t idx, usbh_device_t *dev, uint32_t config)
{
	usb_request_t req;

	assert_param(idx == 0);

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_STANDARD | USB_RTYPE_DEVICE;
	req.bRequest      = USBREQ_SET_CONFIG;
	req.wValue        = config;
	req.wIndex        = 0;
	req.wLength       = 0;

	usb_hcd_ctrl_transfer(0, &req, dev, 0, 0, MAX_PACKET_SIZE_EP0);
	return;
}

/**
  * @brief  Set the current interface.
  * @param  idx: Index of the USB controller.
  * @param  dev: Device.
  * @param  interface: Devices interface configurations.
  * @param  alt: Valid alternate interfaces.
  * @retval None
  */
void usb_hcd_set_interface(uint32_t idx, usbh_device_t *dev, uint32_t interface, uint32_t alt)
{
	usb_request_t req;

	assert_param(idx == 0);

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_STANDARD | USB_RTYPE_INTERFACE;
	req.bRequest      = USBREQ_SET_INTERFACE;
	req.wIndex        = interface;
	req.wValue        = alt;
	req.wLength       = 0;

	usb_hcd_ctrl_transfer(0, &req, dev, 0, 0, MAX_PACKET_SIZE_EP0);
	return;
}

/**
  * @brief  Internal USB interrup handeler.
  * @param  idx: Index of the USB controller.
  * @param  status: Current interrupt status.
  * @retval None
  */
void _usb_host_int_handler(uint32_t idx, uint32_t status)
{
	int32_t drv;
	uint32_t i, k, ep_status;
	static uint32_t _sof_cnt = 0;

	__hcd.dev[0].flag |= USBHDEV_FLAG_NOTIFYINT;

	if (status & USB_INTCTRL_SOF)
		__hcd.int_event |= INT_EVENT_SOF;

	if (status & USB_INTCTRL_POWER_FAULT) {
		__hcd.int_event |= INT_EVENT_POWER_FAULT;
		map_usb_host_pwr_disable();
		map_usb_int_unregister();

		return;
	}

	if (status & USB_INTCTRL_VBUS_ERR){
		__hcd.int_event = INT_EVENT_VBUS_ERR;
		return;
	}

	if (status & USB_INTCTRL_BABBLE) {
	}
	if (status & USB_INTCTRL_SUSPEND) {
	}
	if (status & USB_INTCTRL_RESUME) {
	}

	if (status & USB_INTCTRL_SESSION) {
		map_usb_host_pwr_enable();
		ald_usb_otg_session_request(true);
		printf_e("irq: session\r\n");
	}

	if (status & USB_INTCTRL_CONNECT) {
		__hcd.int_event |= INT_EVENT_CONNECT;
		__hcd.int_event &= ~INT_EVENT_DISCONNECT;
		map_usb_host_pwr_enable();
		printf_e("irq: connect\r\n");
	}

	if (status & USB_INTCTRL_MODE_DETECT) {
		if(__usb_mode == USB_LIB_MODE_HOST) {
			printf_e("map_usb_mode_force_host()");
		}
	}

	if (status & USB_INTCTRL_DISCONNECT) {
		__hcd.int_event |= INT_EVENT_DISCONNECT;
		__hcd.int_event &= ~INT_EVENT_CONNECT;
		printf_e("irq: disconnected\r\n");
	}

	if (status & USB_INTCTRL_SOF) {
		++__usb_sof_count;

		if (++_sof_cnt == USB_SOF_TICK_DIVIDE) {
			_sof_cnt = 0;
			usb_tick_inc(USB_SOF_TICK_DIVIDE);
		}
	}

	status = map_usb_lpm_int_status_get();

	if (status) {
		__hcd.int_event |= INT_EVENT_LPM;
		__hcd.int_event &= ~INT_EVENT_LPM_PEND;

		for (i = 0; i < (USB_MAX_DEVICES + 1); ++i) {
			if ((status != USB_INTLPM_ACK) && (__hcd.dev[i].flag & USBHDEV_FLAG_LPMPEND))
				__hcd.dev[i].flag |= USBHDEV_FLAG_LPMERROR;

			__hcd.dev[i].flag &= ~USBHDEV_FLAG_LPMPEND;
		}
	}

	/* Handle Endpoint */
	status = map_usb_int_status_ep_get();

	if (status & USB_INTEP_0)
		__hcd.int_event |= INT_EVENT_ENUM;

	for (i = 0; i < MAX_NUM_PIPES; ++i) {
		status >>= 1;

		if (status == 0)
			break;

		if (status & 0x10000U) {
			status   &= ~0x10000U;
			ep_status = map_usb_ep_status(i + 1);

			if (ep_status & USB_HOST_IN_STALL) {
				map_usb_host_ep_status_clear(i + 1, USB_HOST_IN_STALL);
				__hcd.in_pipe[i].state = PIPE_STATE_STALL;

				if (__hcd.in_pipe[i].cbk)
					__hcd.in_pipe[i].cbk(IN_PIPE_HANDLE(i), USB_EVENT_STALL);
			}
			else if (ep_status & USB_HOST_IN_ERROR) {
				map_usb_host_ep_status_clear(i + 1, USB_HOST_IN_ERROR);
				__hcd.in_pipe[i].state = PIPE_STATE_ERROR;

				if (__hcd.in_pipe[i].cbk)
					__hcd.in_pipe[i].cbk(IN_PIPE_HANDLE(i), USB_EVENT_ERROR);
			}
			else if (__hcd.in_pipe[i].state == PIPE_STATE_READ_DMA) {  
				__hcd.in_pipe[i].state = PIPE_STATE_READ_DMA_WAIT;
				__hcd.in_pipe[i].count = map_usb_ep_data_avail(i + 1);
				usb_dma_config(__hcd.in_pipe[i].dma_ch, (uint32_t)__hcd.in_pipe[i].p_read, __hcd.in_pipe[i].count);
			}
			else if (__hcd.in_pipe[i].state == PIPE_STATE_READING) {
				__hcd.in_pipe[i].state = PIPE_STATE_DATA_READY;

				if (__hcd.in_pipe[i].p_read) {
					__hcd.in_pipe[i].count = __hcd.in_pipe[i].size;
					map_usb_ep_data_get(i + 1, __hcd.in_pipe[i].p_read, &__hcd.in_pipe[i].count);
				}

				if (__hcd.in_pipe[i].cbk)
					__hcd.in_pipe[i].cbk(IN_PIPE_HANDLE(i), USB_EVENT_RX_AVAILABLE);
			}

			if (__hcd.in_pipe[i].dev)
				__hcd.in_pipe[i].dev->flag |= USBHDEV_FLAG_NOTIFYINT;
		}

		if (status & 1) {
			ep_status = map_usb_ep_status(i + 1);
			if (ep_status & USB_HOST_OUT_STALL) {
				printf_e("out_pipe stall!\n\r");
				map_usb_host_ep_status_clear(i + 1, USB_HOST_OUT_STALL);
				__hcd.out_pipe[i].state = PIPE_STATE_STALL;

				if (__hcd.out_pipe[i].cbk)
					__hcd.out_pipe[i].cbk(OUT_PIPE_HANDLE(i), USB_EVENT_STALL);
			}
			else if (ep_status & USB_HOST_OUT_ERROR) {
				printf_e("out_pipe error!\n\r");
				map_usb_host_ep_status_clear(i + 1, USB_HOST_OUT_ERROR);
				__hcd.out_pipe[i].state = PIPE_STATE_ERROR;

				if (__hcd.out_pipe[i].cbk)
					__hcd.out_pipe[i].cbk(OUT_PIPE_HANDLE(i), USB_EVENT_ERROR);
			}
			else if ((__hcd.out_pipe[i].state == PIPE_STATE_WRITING) || (__hcd.out_pipe[i].state == PIPE_STATE_WRITE_DMA_WAIT)) {
				__hcd.out_pipe[i].state = PIPE_STATE_DATA_SENT;

				if (__hcd.out_pipe[i].cbk)
					__hcd.out_pipe[i].cbk(OUT_PIPE_HANDLE(i), USB_EVENT_TX_COMPLETE);
			}

			map_usb_host_ep_status_clear(i + 1, ep_status & USB_HOST_OUT_STATUS);

			if (__hcd.out_pipe[i].dev)
				__hcd.out_pipe[i].dev->flag |= USBHDEV_FLAG_NOTIFYINT;
		}
	}

	for (k = 0; k <= USB_MAX_DEVICES; ++k) {
		drv = __usbh_active[k];

		if ((drv >= 0) && (__hcd.dev[k].flag & USBHDEV_FLAG_NOTIFYINT) && (__hcd.driver[drv]->int_handler))
			__hcd.driver[drv]->int_handler(__driver_inst[k]);
	}
}

/**
  * @brief  External USB interrup handeler.
  * @retval None
  */
void usb0_host_int_handler(void)
{
	uint32_t status;

	status = map_usb_int_status_get();
	_usb_host_int_handler(0, status);
}

/**
  * @brief  Main routine for USB controller.
  * @retval None
  */
void usb_hcd_main(void)
{
	int32_t i;
	usbh_device_state_t _state;
	event_info_t event;

	_state = __hcd.dev_state[0];

	if (__hcd.int_event) {
		map_usb_int_unregister();

		if (__hcd.int_event & INT_EVENT_POWER_FAULT) {
			event.event    = USB_EVENT_POWER_FAULT;
			event.instance = 0;
			usbh_send_event(0, &event, USBHCD_EVFLAG_PWRFAULT);
			__hcd.dev_state[0] = HCD_POWER_FAULT;
		}
		else if (__hcd.int_event & INT_EVENT_VBUS_ERR) {
			__hcd.dev_state[0] = HCD_VBUS_ERROR;
		}
		else {
			if (__hcd.int_event & INT_EVENT_CONNECT) {
				__hcd.dev_state[0]    = HCD_DEV_RESET;
				__hcd.dev[0].hub      = 0;
				__hcd.dev[0].hub_port = 0;
			}
			else {
				if (__hcd.int_event & INT_EVENT_DISCONNECT)
					__hcd.dev_state[0] = HCD_DEV_DISCONNECT;
			}

			if (__hcd.int_event & INT_EVENT_SOF) {
				event.event    = USB_EVENT_SOF;
				event.instance = 0;
				usbh_send_event(0, &event, USBHCD_EVFLAG_SOF);

				usbh_pipe_check();
				usbh_hub_main();
			}

			if (__hcd.int_event & INT_EVENT_LPM) {
				assert_param((__hcd.int_event & INT_EVENT_LPM_PEND) != 0);

				for (i = 0; i < USB_MAX_DEVICES + 1; ++i) {
					if (__hcd.dev[i].flag & USBHDEV_FLAG_LPMPEND) {
						__hcd.dev[i].flag &= ~USBHDEV_FLAG_LPMPEND;
						__hcd.int_event   &= ~(INT_EVENT_LPM_PEND | INT_EVENT_LPM);
					}
				}
			}
		}

		__hcd.int_event = 0;
		map_usb_int_register();
	}

	for (i = 0; i <= USB_MAX_DEVICES; ++i) {
		if (i != 0)
			_state = __hcd.dev_state[i];

		usbh_device_machine(_state, i);
	}
}

/**
  * @brief  Control transaction to a device.
  * @param  idx: Index of the USB controller.
  * @param  packet: Setup request to be sent.
  * @param  dev: Device.
  * @param  data: Buffer to send for OUT requests or the receive buffer for IN requests.
  * @param  size: Size of the buffer.
  * @param  p_max: Maximum packet size.
  * @retval Size
  */
uint32_t usb_hcd_ctrl_transfer(uint32_t idx, usb_request_t *packet, usbh_device_t *dev,
                                                uint8_t *data, uint32_t size, uint32_t p_max)
{
        uint32_t tmp1,tmp2;

	assert_param(__usbh_ep0.state == EP0_STATE_IDLE);
	assert_param(idx == 0);

	__usbh_ep0.data      = data;
	__usbh_ep0.remaining = size;
	__usbh_ep0.size      = size;
	__usbh_ep0.p_max     = p_max;
	__usbh_ep0.addr      = dev->addr;

	map_usb_host_addr_set(USB_EP_0, __usbh_ep0.addr, USB_EP_HOST_OUT);
	map_usb_host_ep_config(USB_EP_0, 64, 0, 0, (USB_EP_MODE_CTRL | dev->speed | USB_EP_HOST_OUT));
	map_usb_ep_data_put(USB_EP_0, (uint8_t *)packet, sizeof(usb_request_t));

	if (packet->bmRequestType & USB_RTYPE_DIR_IN) {
		__usbh_ep0.state = EP0_STATE_SETUP_IN;
	}
	else {
		if (size != 0)
			__usbh_ep0.state = EP0_STATE_SETUP_OUT;
		else
			__usbh_ep0.state = EP0_STATE_STATUS_IN;
	}

	if (dev->hub == 0)
		map_usb_host_hub_addr_set(USB_EP_0, 0, USB_EP_HOST_OUT | dev->speed);
	else
		map_usb_host_hub_addr_set(USB_EP_0, (dev->hub | (dev->hub_port << 8)), USB_EP_HOST_OUT | dev->speed);

	map_usb_ep_data_send(USB_EP_0, USB_TRANS_SETUP);

	while (__usbh_ep0.state != EP0_STATE_IDLE) {
		map_usb_int_unregister();

		if ((__hcd.int_event & (INT_EVENT_ENUM | INT_EVENT_SOF)) == (INT_EVENT_ENUM | INT_EVENT_SOF)) {
			__hcd.int_event &= ~(INT_EVENT_ENUM | INT_EVENT_SOF);
			usb_hcd_enum_handler();
		}

		map_usb_int_register();

		if (__usbh_ep0.state == EP0_STATE_ERROR)
			return 0xffffffff;

		if (__hcd.int_event & (INT_EVENT_VBUS_ERR | INT_EVENT_DISCONNECT))
			return 0xffffffff;
	}
        
        tmp1 = __usbh_ep0.size;
        tmp2 = tmp1 - __usbh_ep0.remaining;

	return (tmp2);
}

/**
  * @brief  Device connected via the hub.
  * @param  idx: Index of the USB controller.
  * @param  hub: Address of hub.
  * @param  port: Index of hub port.
  * @param  speed: High/Full/Low speed.
  * @retval Index in device list.
  */
uint32_t usb_hcd_hub_device_connect(uint32_t idx, uint8_t hub, uint8_t port, uint32_t speed)
{
	uint32_t i;

	assert_param(idx == 0);
	assert_param(port);

	printf_e("Connection from hub %d, port %d.\n\r", hub, port);

	for (i = 1; i <= USB_MAX_DEVICES; ++i) {
		if ((__hcd.dev[i].flag & USBHDEV_FLAG_ALLOCATED) == 0) {
			__hcd.dev[i].flag = USBHDEV_FLAG_ALLOCATED;
			__hcd.dev[i].desc_config->bLength = 0;
			__hcd.dev[i].hub      = hub;
			__hcd.dev[i].hub_port = port;
			__hcd.dev[i].speed    = speed;
			__hcd.dev[i].desc_device.bLength = 0;
			__hcd.dev_state[i] = HCD_DEV_CONNECT_HUB;

			printf_e("Allocating device %d\n\r", i);
			return i;
		}
	}

	return 0;
}

/**
  * @brief  Device disconnected from the hub.
  * @param  idx: Index of the USB controller.
  * @param  num: Index in device list.
  * @retval None
  */
void usb_hcd_hub_device_disconnect(uint32_t idx, uint32_t num)
{
	assert_param(idx == 0);
	assert_param(num && (num <= USB_MAX_DEVICES));

	printf_e("Disconnection from hub %d, port %d, device %d\n\r",
					__hcd.dev[num].hub,
					__hcd.dev[num].hub_port, num);

	__hcd.dev_state[num] = HCD_DEV_DISCONNECT;
	return;
}

/**
  * @brief  Get hub port.
  * @param  inst: Instance.
  * @retval Port.
  */
uint8_t usb_hcd_dev_hub_port(uint32_t inst)
{
    uint32_t idx;

    idx = hcd_inst_to_dev_idx(inst);

    if (idx == 0xff)
        return idx;

    return __hcd.dev[idx].hub_port;
}

/**
  * @brief  Get the device address.
  * @param  inst: Instance.
  * @retval Address.
  */
uint8_t usb_hcd_dev_addr(uint32_t inst)
{
	uint32_t idx;

	idx = hcd_inst_to_dev_idx(inst);

	if (idx == 0xff)
		return idx;

	return __hcd.dev[idx].addr;
}

/**
  * @brief  Get the device class.
  * @param  inst: Instance.
  * @param  interface: Interface.
  * @retval Class.
  */
uint8_t usb_hcd_dev_class(uint32_t inst, uint32_t interface)
{
	uint32_t idx;
	interface_desc_t *iface;

	idx = hcd_inst_to_dev_idx(inst);

	if (idx == 0xff)
		return(USB_CLASS_DEVICE);

	iface = usb_desc_get_interface(__hcd.dev[idx].desc_config, __hcd.dev[idx].interface, interface);

	if (iface)
		return iface->bInterfaceClass;

	return USB_CLASS_DEVICE;
}

/**
  * @brief  Get the device sub-class.
  * @param  inst: Instance.
  * @param  interface: Interface.
  * @retval Sub-class.
  */
uint8_t usb_hcd_dev_sub_class(uint32_t inst, uint32_t interface)
{
	uint32_t idx;
	interface_desc_t *iface;

	idx = hcd_inst_to_dev_idx(inst);

	if (idx == 0xff)
		return USB_SUBCLASS_UNDEFINED;

	iface = usb_desc_get_interface(__hcd.dev[idx].desc_config, __hcd.dev[idx].interface, interface);

	if (iface)
		return iface->bInterfaceSubClass;

	return USB_SUBCLASS_UNDEFINED;
}

/**
  * @brief  Get the device protocol.
  * @param  inst: Instance.
  * @param  interface: Interface.
  * @retval Protocol.
  */
uint8_t usb_hcd_dev_protocol(uint32_t inst, uint32_t interface)
{
	uint32_t idx;
	interface_desc_t *iface;

	idx = hcd_inst_to_dev_idx(inst);

	if (idx == 0xff)
		return USB_PROTOCOL_UNDEFINED;

	iface = usb_desc_get_interface(__hcd.dev[idx].desc_config, __hcd.dev[idx].interface, interface);

	if (iface)
		return iface->bInterfaceProtocol;

	return USB_PROTOCOL_UNDEFINED;
}

/**
  * @brief  Set the device feature.
  * @param  idx: Index of USB controller.
  * @param  feature: Feature.
  * @param  arg: Parameter.
  * @retval Status.
  */
uint8_t usb_hcd_feature_set(uint32_t idx, uint32_t feature, void *arg)
{
	uint8_t ret = 1;
	lpm_feature_t *lpm = (lpm_feature_t *)arg;

	switch (feature) {
	case USB_FEATURE_LPM:
		if (lpm->feature & USB_FEATURE_LPM_EN) {
			__hcd.feature |= USB_FEATURE_LPM_EN;

			if (lpm->feature & USB_FEATURE_LPM_RMT_WAKE)
				__hcd.feature |= USB_FEATURE_LPM_RMT_WAKE;

			__hcd.lpm = lpm->hird;
		}
		else {
			lpm->feature &= ~USB_FEATURE_LPM_EN;
		}
		break;

	default:
		ret = 0;
		break;
	}

	return ret;
}

/**
  * @brief  Get the status of the LPM.
  * @param  dev: Device.
  * @retval Status.
  */
uint32_t usb_hcd_lpm_status(usbh_device_t *dev)
{
	uint32_t ret = USBHCD_LPM_AVAIL;

	assert_param(dev != 0);
	assert_param((dev->flag & (USBHDEV_FLAG_LPMERROR | USBHDEV_FLAG_LPMPEND)) !=
			(USBHDEV_FLAG_LPMERROR | USBHDEV_FLAG_LPMPEND));

	if (dev->flag & USBHDEV_FLAG_LPMERROR)
		ret = USBHCD_LPM_ERROR;
	else if (dev->flag & USBHDEV_FLAG_LPMPEND)
		ret = USBHCD_LPM_PENDING;

	return ret;
}

/**
  * @brief  Generates an LPM request enter sleep.
  * @param  dev: Device.
  * @retval Status.
  */
uint32_t usb_hcd_lpm_sleep(usbh_device_t *dev)
{
	uint32_t ret;

	assert_param(dev != 0);

	map_usb_int_unregister();

	if ((__hcd.int_event & INT_EVENT_LPM_PEND) || (dev->flag & USBHDEV_FLAG_LPMPEND)) {
		ret = USBHCD_LPM_PENDING;
	}
	else {
		__hcd.int_event |= INT_EVENT_LPM_PEND;
		dev->flag |= USBHDEV_FLAG_LPMPEND;
		dev->flag &= ~USBHDEV_FLAG_LPMERROR;
		map_usb_host_lpm_send(dev->addr, USB_EP_0);
		ret = USBHCD_LPM_AVAIL;
	}

	map_usb_int_register();
	return ret;
}

/**
  * @brief  Generates an LPM request exit sleep.
  * @param  idx: Index of USB controller.
  * @retval None
  */
void usb_hcd_lpm_resume(uint32_t idx)
{
	assert_param(idx == 0);

	map_usb_host_lpm_resume();
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
