/**
  *********************************************************************************
  *
  * @file    usbh_bulk.c
  * @brief   Functions related to host BULK driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbh_bulk.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @defgroup Host_BULK BULK
  * @brief Host BULK driver
  * @{
  */
/** @defgroup Host_BULK_Private_Functions Private Functions
  * @{
  */
static void *usbh_bulk_open(usbh_device_t *dev);
static void usbh_bulk_close(void *inst);
/**
  * @}
  */
/** @defgroup Host_BULK_Private_Variables Private Variables
  * @{
  */
static usbh_bulk_inst_t __bulk_dev = {
	0
};

const usbh_class_driver_t __usbh_bulk_driver = {
	USB_CLASS_VEND_SPECIFIC,
	usbh_bulk_open,
	usbh_bulk_close,
	0
};
/**
  * @}
  */

/** @addtogroup Host_BULK_Private_Functions
  * @{
  */
/**
  * @brief  Open an instance of a BULK device.
  * @param  dev: Device information structure.
  * @retval None
  */
static void *usbh_bulk_open(usbh_device_t *dev)
{
	int32_t i;
	endpoint_desc_t *ep;
	interface_desc_t *iface;

	if (__bulk_dev.dev)
		return 0;

	__bulk_dev.dev = dev;
	iface = usb_desc_get_interface(dev->desc_config, 0, 0);

	for (i = 0; i < 3; ++i) {
		ep = usb_desc_get_ep(iface, i, dev->size_config);

		if (ep == NULL)
			break;

		if ((ep->bmAttributes & USB_EP_ATTR_TYPE_M) != USB_EP_ATTR_BULK)
			continue;

		if (ep->bEndpointAddress & USB_EP_DESC_IN) {
			__bulk_dev.in_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_BULK_IN, dev, ep->wMaxPacketSize, 0);
			usb_hcd_pipe_config(__bulk_dev.in_pipe, ep->wMaxPacketSize, 0, (ep->bEndpointAddress & USB_EP_DESC_NUM_M));
		}
		else {
			__bulk_dev.out_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_BULK_OUT, dev, ep->wMaxPacketSize, 0);
			usb_hcd_pipe_config(__bulk_dev.out_pipe, ep->wMaxPacketSize, 0, (ep->bEndpointAddress & USB_EP_DESC_NUM_M));
		}
	}

	if (__bulk_dev.cbk)
		__bulk_dev.cbk(&__bulk_dev, USBH_BULK_EVENT_OPEN, 0);

	return &__bulk_dev;
}

/**
  * @brief  Close an instance of a BULK device.
  * @param  inst: Device information structure.
  * @retval None
  */
static void usbh_bulk_close(void *inst)
{
	if (__bulk_dev.dev == NULL)
		return;

	__bulk_dev.dev = NULL;

	if (__bulk_dev.in_pipe != 0)
		usb_hcd_pipe_free(__bulk_dev.in_pipe);
	if (__bulk_dev.out_pipe != 0)
		usb_hcd_pipe_free(__bulk_dev.out_pipe);
	if (__bulk_dev.cbk)
		__bulk_dev.cbk(&__bulk_dev, USBH_BULK_EVENT_CLOSE, 0);

	return;
}
/**
  * @}
  */

/** @defgroup Host_BULK_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Open an instance of a BULK device.
  * @param  cbk: Callback that will be called when any event occurs.
  * @retval Instance value
  */
usbh_bulk_inst_t *usbh_bulk_driver_open(usbh_bulk_cbk cbk)
{
	if (__bulk_dev.cbk)
		return 0;

	__bulk_dev.cbk = cbk;
	return &__bulk_dev;
}

/**
  * @brief  Release an instance of a BULK device.
  * @param  inst: Instance value.
  * @retval None
  */
void usbh_bulk_driver_close(usbh_bulk_inst_t *inst)
{
	usbh_bulk_close((void *)inst);
	inst->cbk = NULL;

	return;
}

/**
  * @brief  Gets the speed of the target endpoint.
  * @param  inst: Instance value.
  * @retval Speed.
  */
uint32_t usbh_bulk_target_speed_get(usbh_bulk_inst_t *inst)
{
	if (inst->dev == NULL)
		return 0xFFFFFFFF;

	return inst->dev->speed;
}

/**
  * @brief  Read data from an BULK device.
  * @param  inst: Device to use for this read.
  * @param  data: The buffer.
  * @param  size: Size of the buffer.
  * @retval Status
  */
int32_t usbh_bulk_read(usbh_bulk_inst_t *inst, uint8_t *data, uint32_t size)
{
	if (inst->dev == NULL || data == NULL)
		return -1;
	if (size == 0)
		return 0;

	return usb_hcd_pipe_read(__bulk_dev.in_pipe, data, size);
}

/**
  * @brief  Write data to an BULK device.
  * @param  inst: Device to use for this write.
  * @param  data: The buffer.
  * @param  size: Size of the buffer.
  * @retval Status
  */
int32_t usbh_bulk_write(usbh_bulk_inst_t *inst, uint8_t *data, uint32_t size)
{
	if (inst->dev == NULL || data == NULL)
		return -1;
	if (size == 0)
		return 0;

	return usb_hcd_pipe_write(__bulk_dev.out_pipe, data, size);
}

/**
  * @brief  Generates an LPM request enter sleep.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_bulk_lpm_sleep(usbh_bulk_inst_t *inst)
{
	return usb_hcd_lpm_sleep(inst->dev);
}

/**
  * @brief  Get the status of the LPM.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_bulk_lpm_status(usbh_bulk_inst_t *inst)
{
	return usb_hcd_lpm_status(inst->dev);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
