/**
  *********************************************************************************
  *
  * @file    usbh_audio.h
  * @brief   Header file for the audio[HOST].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBH_AUDIO_H__
#define __USBH_AUDIO_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbh_core.h"
#include "usb_audio.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @addtogroup Host_Audio
  * @{
  */
/**
  * @defgroup Host_Audio_Public_Macros Public Macros
  * @{
  */
#define USBH_AUDIO_EVENT_OPEN	(USBH_AUDIO_EVENT_BASE + 0)
#define USBH_AUDIO_EVENT_CLOSE	(USBH_AUDIO_EVENT_BASE + 1)
#define USBH_AUDIO_FORMAT_IN	0x1U
#define USBH_AUDIO_FORMAT_OUT	0x0U
#define INTERFACE_NUM_M		0xFFU
#define INTERFACE_ALTSETTING_M	0xFF00U
#define INTERFACE_ALTSETTING_S	0x8U
#define INVALID_INTERFACE	0xffffffffU
#define AUDIO_FLAG_OUT_ACTIVE	0x1U
#define AUDIO_FLAG_IN_ACTIVE	0x2U
/**
  * @}
  */

/**
  * @defgroup Host_Audio_Public_Types Public Types
  * @{
  */
struct usbh_audio_inst_s;
typedef void (*usbh_audio_cbk)(struct usbh_audio_inst_s *inst, uint32_t event, uint32_t param, void *data);

/**
  * @brief Instance of a USB host audio driver
  */
typedef struct usbh_audio_inst_s {
	usbh_device_t *dev;	/**< Device instance */
	usbh_audio_cbk cbk;	/**< Callback */
	uint8_t _ctrl;		/**< Control interface */
	uint8_t out_interface;	/**< Output streaming interface */
	uint8_t out_alt;	/**< Currently selected active output interface */
	uint8_t in_interface;	/**< Input streaming interface */
	uint8_t in_alt;		/**< Currently selected active output interface */
	uint8_t in_addr;	/**< In endpoint */
	uint8_t out_addr;	/**< Out endpoint */
	ac_input_t *in_term;	/**< In terminal */
	ac_output_t *out_term;	/**< Out terminal */
	uint8_t volume;		/**< Volume identifier */
	ac_feature_t *feature;	/**< Feature Unit for controlling volume */
	uint16_t ctrl[3];	/**< Controls */
	uint32_t in_pipe;	/**< Isochronous IN pipe */
	uint16_t in_pipe_size;	/**< Size of IN pipe */
	usbh_audio_cbk in_cbk;	/**< Callback of IN pipe */
	void *in_buf;		/**< Buffer of IN pipe */
	uint32_t out_pipe;	/**< Isochronous OUT pipe */
	uint16_t out_pipe_size;	/**< Size of OUT pipe */
	usbh_audio_cbk out_cbk;	/**< Callback of OUT pipe */
	void *out_buf;		/**< Buffer of OUT pipe */
	uint32_t flag;		/**< State flags */
} usbh_audio_inst_t;

/**
  * @brief Format
  */
typedef struct {
	uint8_t channel;	/**< Channel */
	uint8_t bits;		/**< Numbers of bits */
	uint32_t rate;		/**< Sampling rate */
} usbh_audio_format_t;
/**
  * @}
  */

/** @addtogroup Host_Audio_Public_Functions
  * @{
  */
extern usbh_audio_inst_t *usbh_audio_open(uint32_t idx, usbh_audio_cbk cbk);
extern void usbh_audio_close(usbh_audio_inst_t *inst);
extern int32_t usbh_audio_play(usbh_audio_inst_t *inst, void *buf, uint32_t size, usbh_audio_cbk cbk);
extern int32_t usbh_audio_record(usbh_audio_inst_t *inst, void *buf, uint32_t size, usbh_audio_cbk cbk);
extern uint32_t usbh_audio_format_get(usbh_audio_inst_t *inst, uint32_t rate, uint32_t bits, uint32_t channel, uint32_t flag);
extern uint32_t usbh_audio_format_set(usbh_audio_inst_t *inst, uint32_t rate, uint32_t bits, uint32_t channel, uint32_t flag);
extern uint32_t usbh_audio_volume_get(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel);
extern void usbh_audio_volume_set(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel, uint32_t value);
extern uint32_t usbh_audio_volume_max_get(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel);
extern uint32_t usbh_audio_volume_min_get(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel);
extern uint32_t usbh_audio_volume_res_get(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel);
extern uint32_t usbh_audio_lpm_sleep(usbh_audio_inst_t *inst);
extern uint32_t usbh_audio_lpm_status(usbh_audio_inst_t *inst);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
