/**
  *********************************************************************************
  *
  * @file    usbh_audio.c
  * @brief   Functions related to host audio driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbh_audio.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup HOST
  * @{
  */
/** @defgroup Host_Audio Audio
  * @brief Host Audio driver
  * @{
  */
/** @defgroup Host_Audio_Private_Functions Private Functions
  * @{
  */
static void *_usb_audio_open(usbh_device_t *dev);
static void _usb_audio_close(void *instance);
/**
  * @}
  */
/** @defgroup Host_Audio_Private_Variables Private Variables
  * @{
  */
static usbh_audio_inst_t __audio_dev =
{
	0
};

const usbh_class_driver_t __usbh_audio_driver =
{
	USB_CLASS_AUDIO,
	_usb_audio_open,
	_usb_audio_close,
	0
};
/**
  * @}
  */

/** @addtogroup Host_Audio_Private_Functions
  * @{
  */
/**
  * @brief  Handles callbacks from the USB IN pipe.
  * @param  pipe: The Pipe.
  * @param  event: The event.
  * @retval None
  */
static void in_pipe_handle(uint32_t pipe, uint32_t event)
{
	if (event != USB_EVENT_RX_AVAILABLE)
		return;
	if (__audio_dev.in_cbk == NULL)
		return;

	__audio_dev.in_cbk(&__audio_dev, USB_EVENT_RX_AVAILABLE, usb_hcd_pipe_transfer_size_get(pipe), __audio_dev.in_buf);
	return;
}

/**
  * @brief  Handles callbacks from the USB OUT pipe.
  * @param  pipe: The Pipe.
  * @param  event: The event.
  * @retval None
  */
static void out_pipe_handle(uint32_t pipe, uint32_t event)
{
	if (event != USB_EVENT_TX_COMPLETE)
		return;
	if (__audio_dev.out_cbk == NULL)
		return;

	__audio_dev.out_cbk(&__audio_dev, USB_EVENT_TX_COMPLETE, 0, __audio_dev.out_buf);
	return;
}

/**
  * @brief  Get terminal in an audio configuration descriptor.
  * @param  config: Configuration descriptor.
  * @param  term: Terminal.
  * @param  type: Type.
  * @retval Terminal descriptor.
  */
static desc_head_t *audio_term_get(config_desc_t *config, uint32_t term, uint32_t type)
{
	int32_t len;
	ac_output_t *out;
	desc_head_t *hdr;

	hdr = (desc_head_t *)config;
	len = config->wTotalLength;

	while (len > 0) {
		out = (ac_output_t *)hdr;

		if ((hdr->bDescriptorType == USB_DTYPE_CS_INTERFACE) && (term == out->bDescriptorSubtype)) {
			if ((out->bDescriptorSubtype == USB_AI_OUTPUT_TERMINAL) || (out->bDescriptorSubtype == USB_AI_INPUT_TERMINAL)) {
				if (out->wTerminalType == type)
					return hdr;
			}
			else if (out->bDescriptorSubtype == USB_AI_FEATURE_UNIT) {
				return hdr;
			}
		}

		len -=  hdr->bLength;
		hdr  = (desc_head_t *)((uint32_t)hdr + hdr->bLength);
	}

	return NULL;
}

/**
  * @brief  Get interface number for the control interface.
  * @param  config: Configuration descriptor.
  * @retval Interface descriptor number.
  */
static uint32_t audio_ctrl_get(config_desc_t *config)
{
	desc_head_t *hdr;
	interface_desc_t *iface;
	uint32_t interface;
	int32_t len;

	hdr = (desc_head_t *)config;
	len = config->wTotalLength;
	interface = INVALID_INTERFACE;

	while (len > 0) {
		if (hdr->bDescriptorType == USB_DTYPE_INTERFACE) {
			iface = (interface_desc_t *)hdr;

			if (iface->bInterfaceSubClass == USB_ASC_AUDIO_CONTROL){
				interface = iface->bInterfaceNumber;
				break;
			}
		}

		len -= hdr->bLength;
		hdr  = (desc_head_t*)((uint32_t)hdr + hdr->bLength);
	}

	return interface;
}

/**
  * @brief  Get the correct audio interface.
  * @param  inst: Instance value.
  * @param  format: Format.
  * @param  rate: Sampling rate.
  * @param  size: Size of the interface.
  * @param  channel: Channel.
  * @param  flag: Flag.
  * @retval Audio interface.
  */
static uint32_t audio_interface_get(usbh_audio_inst_t *inst, uint16_t format, uint32_t rate, uint32_t size, uint32_t channel, uint32_t flag)
{
	desc_head_t *hdr;
	interface_desc_t *iface;
	endpoint_desc_t *iep, *oep;
	ac_head_t *ac_hdr;
	ac_general_t *gerl;
	ac_format_t *fmt;
	endpoint_desc_t *ep;
	uint8_t *p;
	uint32_t value;
	int32_t len, i;

	iface = NULL;
	iep   = NULL;
	oep   = NULL;
	hdr   = (desc_head_t *)inst->dev->desc_config;
	len   = inst->dev->desc_config->wTotalLength;

	while (len > 0) {
		if (hdr->bDescriptorType == USB_DTYPE_INTERFACE) {
			if (iface)
				break;

			iface = (interface_desc_t *)hdr;
			iep   = NULL;
			oep   = NULL;

			if ((iface->bNumEndpoints == 0) ||
						(iface->bInterfaceClass != USB_CLASS_AUDIO) ||
						(iface->bInterfaceSubClass != USB_ASC_AUDIO_STREAMING)) {
				iface = NULL;
			}
		}
		if ((iface) && (hdr->bDescriptorType == USB_DTYPE_CS_INTERFACE)) {
			ac_hdr = (ac_head_t *)hdr;

			if (ac_hdr->bDescriptorSubtype == USB_AS_GENERAL) {
				gerl = (ac_general_t *)hdr;

				if(gerl->wFormatTag != format)
					iface = NULL;
			}
			else if (ac_hdr->bDescriptorSubtype == USB_AS_FORMAT_TYPE) {
				fmt = (ac_format_t *)hdr;

				if ((fmt->bNrChannels != channel) || (fmt->bSubFrameSize != size)) {
					iface = 0;
				}
				else {
					p = &fmt->tSamFreq;

					for (i = 0; i < fmt->bSamFreqType; ++i) {
						value = (*((uint32_t *)&p[i * 3]) & 0xffffff);

						if (value == rate)
							break;
					}

					if (i == fmt->bSamFreqType)
						iface = NULL;
				}
			}
		}
		else if ((iface) && (hdr->bDescriptorType == USB_DTYPE_ENDPOINT)) {
			ep = (endpoint_desc_t *)hdr;

			if (flag & USBH_AUDIO_FORMAT_IN) {
				if (ep->bEndpointAddress & USB_EP_DESC_IN) {
					if ((ep->bmAttributes & USB_EP_ATTR_USAGE_M) == USB_EP_ATTR_USAGE_FEEDBACK)
						iface = NULL;
					else
						iep = ep;
				}
			}
			else {
				if ((ep->bEndpointAddress & USB_EP_DESC_IN) == 0) {
					if ((ep->bmAttributes & USB_EP_ATTR_USAGE_M) == USB_EP_ATTR_USAGE_FEEDBACK)
						iface = 0;
					else
						oep = ep;
				}
			}
		}

		len -= hdr->bLength;
		hdr = (desc_head_t *)((uint32_t)hdr + hdr->bLength);
	}

	if (iface) {
		if (iep) {
			__audio_dev.in_addr = iep->bEndpointAddress & USB_EP_DESC_NUM_M;

			if (__audio_dev.in_pipe == 0) {
				__audio_dev.in_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_ISOC_IN_DMA, 
								__audio_dev.dev, iep->wMaxPacketSize, in_pipe_handle);
			}
			else if (__audio_dev.in_pipe_size < iep->wMaxPacketSize) {
				usb_hcd_pipe_free(__audio_dev.in_pipe);
				__audio_dev.in_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_ISOC_IN_DMA,
								__audio_dev.dev, iep->wMaxPacketSize, in_pipe_handle);
				__audio_dev.in_pipe_size = iep->wMaxPacketSize;
			}

			usb_hcd_pipe_config(__audio_dev.in_pipe, iep->wMaxPacketSize, 0, __audio_dev.in_addr);
		}

		if (oep) {
			__audio_dev.out_addr = oep->bEndpointAddress & USB_EP_DESC_NUM_M;

			if (__audio_dev.out_pipe == 0) {
				__audio_dev.out_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_ISOC_OUT_DMA,
								__audio_dev.dev, oep->wMaxPacketSize, out_pipe_handle);
			}
			else if (__audio_dev.out_pipe_size < oep->wMaxPacketSize) {
				usb_hcd_pipe_free(__audio_dev.out_pipe);
				__audio_dev.out_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_ISOC_OUT_DMA,
								__audio_dev.dev, oep->wMaxPacketSize, out_pipe_handle);
				__audio_dev.out_pipe_size = oep->wMaxPacketSize;
			}

			usb_hcd_pipe_config(__audio_dev.out_pipe, oep->wMaxPacketSize, 0, __audio_dev.out_addr);
		}

		return (iface->bInterfaceNumber | (iface->bAlternateSetting << INTERFACE_ALTSETTING_S));
	}

	return INVALID_INTERFACE;
}

/**
  * @brief  Open an instance of the USB host audio driver.
  * @param  dev: Device information structure.
  * @retval None
  */
static void *_usb_audio_open(usbh_device_t *dev)
{
	uint32_t tmp;
	config_desc_t *config;

	if (__audio_dev.dev)
		return NULL;

	__audio_dev.dev = dev;
	config = dev->desc_config;
	__audio_dev.in_term  = (ac_input_t *)audio_term_get(config, USB_AI_INPUT_TERMINAL, USB_TTYPE_STREAMING);
	__audio_dev.out_term = (ac_output_t *)audio_term_get(config, USB_AI_OUTPUT_TERMINAL, USB_TTYPE_STREAMING);
	__audio_dev.feature  = (ac_feature_t *)audio_term_get(config, USB_AI_FEATURE_UNIT, 0);

	if ((__audio_dev.out_term == 0) && (__audio_dev.in_term == 0))
		return NULL;

	tmp = audio_ctrl_get(config);

	if (tmp == INVALID_INTERFACE)
		return NULL;

	__audio_dev._ctrl = (uint8_t)tmp;

	if (__audio_dev.cbk != 0)
		__audio_dev.cbk(&__audio_dev, USBH_AUDIO_EVENT_OPEN, 0, 0);
	if (__audio_dev.feature != 0)
		__audio_dev.volume = __audio_dev.feature->bUnitID;

	__audio_dev.in_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_ISOC_IN_DMA, __audio_dev.dev, 256, in_pipe_handle);
	__audio_dev.in_pipe_size = 256;
	__audio_dev.out_pipe = usb_hcd_pipe_alloc_size(0, USBHCD_PIPE_ISOC_OUT_DMA, __audio_dev.dev, 256, out_pipe_handle);
	__audio_dev.out_pipe_size = 256;
	__audio_dev.flag = 0;

	return &__audio_dev;
}

/**
  * @brief  Release an instance of the USB host audio driver.
  * @param  device: Device information structure.
  * @retval None
  */
static void _usb_audio_close(void *device)
{
	usbh_audio_inst_t *inst = (usbh_audio_inst_t *)device;

	if (inst->dev == NULL)
		return;

	inst->dev = NULL;

	if (inst->in_pipe != 0)
		usb_hcd_pipe_free(inst->in_pipe);
	if (inst->out_pipe != 0)
		usb_hcd_pipe_free(inst->out_pipe);
	if (inst->cbk)
		inst->cbk(inst, USBH_AUDIO_EVENT_CLOSE, 0, 0);

	return;
}

/**
  * @brief  Request to set volume.
  * @param  inst: Instance value.
  * @param  interface: Interface.
  * @param  channel: Channel.
  * @param  request: Audio device request.
  * @retval Requested value.
  */
static uint32_t audio_volume_setting_get(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel, uint32_t request)
{
	uint32_t value = 0;
	usb_request_t req;

	req.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_CLASS | USB_RTYPE_INTERFACE;
	req.bRequest      = (request & 0xff);
	req.wValue        = VOLUME_CONTROL | (channel & 0xff);
	req.wIndex        = (inst->volume << 8) | (interface & 0xff);
	req.wLength       = 2;

	usb_hcd_ctrl_transfer(0, &req, inst->dev, (uint8_t *)&value, 4, inst->dev->desc_device.bMaxPacketSize0);
	return value;
}
/**
  * @}
  */

/** @defgroup Host_Audio_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Enable the host audio class driver.
  * @param  idx: Index of the USB cotroller.
  * @param  cbk: Callback function.
  * @retval Audio instance.
  */
usbh_audio_inst_t *usbh_audio_open(uint32_t idx, usbh_audio_cbk cbk)
{
	if ((idx != 0) || (__audio_dev.cbk))
		return(0);

	__audio_dev.cbk = cbk;
	return &__audio_dev;
}

/**
  * @brief  Disable the host audio class driver.
  * @param  inst: Instance value.
  * @retval None
  */
void usbh_audio_close(usbh_audio_inst_t *inst)
{
	_usb_audio_close(inst);
	inst->cbk = NULL;

	return;
}

/**
  * @brief  Send an audio buffer to the audio device.
  * @param  inst: Instance value.
  * @param  buf: Data buffer.
  * @param  size: Size of the buffer.
  * @param  cbk: Callback function.
  * @retval Length.
  */
int32_t usbh_audio_play(usbh_audio_inst_t *inst, void *buf, uint32_t size, usbh_audio_cbk cbk)
{
	uint32_t len;

	if (inst->dev == NULL)
		return 0;

	if ((inst->flag & AUDIO_FLAG_OUT_ACTIVE) == 0) {
		inst->flag |= AUDIO_FLAG_OUT_ACTIVE;
		usb_hcd_set_interface(0, inst->dev, inst->out_interface, inst->out_alt);
	}

	inst->out_cbk = cbk;
	inst->out_buf = (void *)buf;
	len = usb_hcd_pipe_schedule(inst->out_pipe, buf, size);

	return len;
}

/**
  * @brief  Provide an audio buffer to the audio device for audio input.
  * @param  inst: Instance value.
  * @param  buf: Data buffer.
  * @param  size: Size of the buffer.
  * @param  cbk: Callback function.
  * @retval Length.
  */
int32_t usbh_audio_record(usbh_audio_inst_t *inst, void *buf, uint32_t size, usbh_audio_cbk cbk)
{
	uint32_t len;

	if (inst->dev == NULL)
		return 0;

	if ((inst->flag & AUDIO_FLAG_IN_ACTIVE) == 0) {
		inst->flag |= AUDIO_FLAG_IN_ACTIVE;
		usb_hcd_set_interface(0, inst->dev, inst->in_interface, inst->in_alt);
	}

	inst->in_cbk = cbk;
	inst->in_buf = (void *)buf;
	len = usb_hcd_pipe_schedule(inst->in_pipe, buf, size);

	return len;
}

/**
  * @brief  Determine which an audio format is supported.
  * @param  inst: Instance value.
  * @param  rate: sample rate of the audio stream.
  * @param  bits: Number of bits per sample.
  * @param  channel: Channel.
  * @param  flag: Flags.
  * @retval Status.
  */
uint32_t usbh_audio_format_get(usbh_audio_inst_t *inst, uint32_t rate, uint32_t bits, uint32_t channel, uint32_t flag)
{
	if (audio_interface_get(inst, USB_ADF_PCM, rate, bits >> 3, channel, flag) != INVALID_INTERFACE)
		return 0;

	return 1;
}

/**
  * @brief  Set the current format on an audio interface.
  * @param  inst: Instance value.
  * @param  rate: sample rate of the audio stream.
  * @param  bits: Number of bits per sample.
  * @param  channel: Channel.
  * @param  flag: Flags.
  * @retval Status.
  */
uint32_t usbh_audio_format_set(usbh_audio_inst_t *inst, uint32_t rate, uint32_t bits, uint32_t channel, uint32_t flag)
{
	uint32_t interface;

	interface = audio_interface_get(inst, USB_ADF_PCM, rate, bits >> 3, channel, flag);

	if (interface == INVALID_INTERFACE)
		return 1;

	if (flag & USBH_AUDIO_FORMAT_IN) {
		inst->in_interface = (uint8_t)(interface & INTERFACE_NUM_M);
		inst->in_alt = (uint8_t)((interface & INTERFACE_ALTSETTING_M) >> INTERFACE_ALTSETTING_S);
	}
	else {
		inst->out_interface = (uint8_t)(interface & INTERFACE_NUM_M);
		inst->out_alt = (uint8_t)((interface & INTERFACE_ALTSETTING_M) >> INTERFACE_ALTSETTING_S);
	}

	return 0;
}

/**
  * @brief  Get the current volume.
  * @param  inst: Instance value.
  * @param  interface: Interface.
  * @param  channel: Channel.
  * @retval Status.
  */
uint32_t usbh_audio_volume_get(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel)
{
	return audio_volume_setting_get(inst, interface, channel, USB_AC_GET_CUR);
}

/**
  * @brief  Set the current volume.
  * @param  inst: Instance value.
  * @param  interface: Interface.
  * @param  channel: Channel.
  * @param  value: Volume.
  * @retval None
  */
void usbh_audio_volume_set(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel, uint32_t value)
{
	usb_request_t req;

	req.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS | USB_RTYPE_INTERFACE;
	req.bRequest      = USB_AC_SET_CUR;
	req.wValue        = VOLUME_CONTROL | (channel & 0xff);
	req.wIndex        = inst->volume << 8;
	req.wLength       = 2;

	usb_hcd_ctrl_transfer(0, &req, inst->dev, (uint8_t *)&value, 2, inst->dev->desc_device.bMaxPacketSize0);
	return;
}

/**
  * @brief  Get the maximum volume.
  * @param  inst: Instance value.
  * @param  interface: Interface.
  * @param  channel: Channel.
  * @retval Maximum volume.
  */
uint32_t usbh_audio_volume_max_get(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel)
{
	return audio_volume_setting_get(inst, interface, channel, USB_AC_GET_MAX);
}

/**
  * @brief  Get the minimum volume.
  * @param  inst: Instance value.
  * @param  interface: Interface.
  * @param  channel: Channel.
  * @retval Minimum volume.
  */
uint32_t usbh_audio_volume_min_get(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel)
{
	return audio_volume_setting_get(inst, interface, channel, USB_AC_GET_MIN);
}

/**
  * @brief  Get the volume control resolution.
  * @param  inst: Instance value.
  * @param  interface: Interface.
  * @param  channel: Channel.
  * @retval Minimum volume.
  */
uint32_t usbh_audio_volume_res_get(usbh_audio_inst_t *inst, uint32_t interface, uint32_t channel)
{
	return audio_volume_setting_get(inst, interface, channel, USB_AC_GET_RES);
}

/**
  * @brief  Generates an LPM request enter sleep.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_audio_lpm_sleep(usbh_audio_inst_t *inst)
{
	return usb_hcd_lpm_sleep(inst->dev);
}

/**
  * @brief  Get the status of the LPM.
  * @param  inst: Device.
  * @retval Status.
  */
uint32_t usbh_audio_lpm_status(usbh_audio_inst_t *inst)
{
	return usb_hcd_lpm_status(inst->dev);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
