/**
  *********************************************************************************
  *
  * @file    usbd_printer.h
  * @brief   Header file for the PRINTER driver[Device].
  *
  * @version V1.0
  * @date    02 Nov 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          02 Nov 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBD_PRINTER_H__
#define __USBD_PRINTER_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbd_core.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_PRINTER
  * @{
  */
/**
  * @defgroup Device_PRINTER_Public_Macros Public Macros
  * @{
  */
#define PRINTER_INTERFACE_SIZE	23
#define COMPOSITE_PRINTER_SIZE	PRINTER_INTERFACE_SIZE
#define USB_RX_ERROR_FLAGS	(USBERR_DEV_RX_DATA_ERROR | USBERR_DEV_RX_OVERRUN | USBERR_DEV_RX_FIFO_FULL)
#define PRINTER_DO_PACKET_RX	5
#define PRINTER_DATA_IN_EP	PRINTER_EP_IN
#define PRINTER_DATA_OUT_EP	PRINTER_EP_OUT

#define USBD_PRINTER_EVENT_GET_DEVICE_ID	(USBD_PRINTER_EVENT_BASE + 0)
#define USBD_PRINTER_EVENT_GET_PORT_STATUS	(USBD_PRINTER_EVENT_BASE + 1)
#define USBD_PRINTER_EVENT_SOFT_RESET		(USBD_PRINTER_EVENT_BASE + 2)
#define USBD_PRINTER_EVENT_SYSTEM_RESET		(USBD_PRINTER_EVENT_BASE + 3)

/* Class-specific request GET_DEVICE_ID */
#define USB_DEVICE_PRINTER_GET_DEVICE_ID	(0x00U)
/* Class-specific request GET_PORT_STATUS */
#define USB_DEVICE_PRINTER_GET_PORT_STATUS	(0x01U)
/* Class-specific request SOFT_RESET */
#define USB_DEVICE_PRINTER_SOFT_RESET		(0x02U)
/* Vendor-specific request system reset */
#define USB_DEVICE_PRINTER_SYSTEM_RESET		(0x03U)

/* Paper empty bit mask for GET_PORT_STATUS */
#define USB_DEVICE_PRINTER_PORT_STATUS_PAPER_EMPTRY_MASK	(0x20U)
/* Select bit mask for GET_PORT_STATUS */
#define USB_DEVICE_PRINTER_PORT_STATUS_SELECT_MASK	(0x10U)
/* Error bit mask for GET_PORT_STATUS */
#define USB_DEVICE_PRINTER_PORT_STATUS_NOT_ERROR_MASK	(0x08U)
/* Default value of the printer status */
#define USB_DEVICE_PRINTER_PORT_STATUS_DEFAULT_VALUE	(0x18)

/* Maximum size of the ID buffer */
#define USB_DEVICE_PRINTER_DEVICE_ID_BUFFER_SIZE   (200)
/**
  * @}
  */

/**
  * @defgroup Device_PRINTER_Public_Types Public Types
  * @{
  */
/**
  * @brief PRINTER status
  */
typedef enum {
	PRINTER_STATE_UNCONFIG    = 0,	/**< Unconfigure */
	PRINTER_STATE_IDLE        = 1,	/**< Idle */
	PRINTER_STATE_WAIT_DATA   = 2,	/**< Waiting data */
} printer_state_t;

/**
  * @brief Define the private instance data.
  */
typedef struct {
	uint32_t usb_idx;			/**< Index of the USB controller */
	device_info_t info;			/**< Device information */
	volatile printer_state_t rx_state;	/**< State of the PRINTER receive */
	volatile printer_state_t tx_state;	/**< State of the PRINTER transmit */
	volatile uint16_t flag;			/**< Flags */
	uint16_t size;				/**< Size */
	volatile uint8_t conn;			/**< Connection status */
	uint8_t iep;				/**< IN endpoint */
	uint8_t oep;				/**< OUT endpoint */
	uint8_t interface;			/**< Interface */
} printer_inst_t;

/**
  * @brief Define operating parameters for the PRINTER device
  */
typedef struct {
	const uint16_t vid;		/**< Vendor ID */
	const uint16_t pid;		/**< Product ID */
	const uint16_t max_power;	/**< Maximum power consumption */
	const uint8_t attr_pwr;		/**< Attribute of the power */
	const usb_cbk rx_cbk;		/**< Callback function */
	void *rx_arg;			/**< Parameter */
	const usb_cbk tx_cbk;		/**< Callback function */
	void *tx_arg;			/**< Parameter */
	const uint8_t * const *desc_str;/**< String descriptor */
	const uint32_t num_str;		/**< Number of the string descriptors */     
	const uint8_t *id;		/**< Using for responsing GET_DEVICE_ID request */
	uint32_t id_size;		/**< Size of the ID buffer */
	printer_inst_t inst;		/**< Device instance */
} usbd_printer_dev_t;
/**
  * @}
  */

/** @addtogroup Device_PRINTER_Public_Functions
  * @{
  */
extern void *usbd_printer_init(uint32_t idx, usbd_printer_dev_t *dev);
extern void *usbd_printer_init_comp(uint32_t idx, usbd_printer_dev_t *dev, comp_entry_t *entry);
extern void usbd_printer_term(usbd_printer_dev_t *dev);
extern void *usbd_printer_set_rx_param(usbd_printer_dev_t *dev, void *arg);
extern void *usbd_printer_set_tx_param(usbd_printer_dev_t *dev, void *arg);
extern uint32_t usbd_printer_packet_write(usbd_printer_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_printer_packet_read(usbd_printer_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_printer_tx_packet_avail(usbd_printer_dev_t *dev);
extern uint32_t usbd_printer_rx_packet_avail(usbd_printer_dev_t *dev);
extern int usbd_printer_remote_wakeup_req(usbd_printer_dev_t *dev);
extern void usbd_printer_power_status_set(usbd_printer_dev_t *dev, uint8_t power);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
