/**
  *********************************************************************************
  *
  * @file    usbd_printer.c
  * @brief   USB PRINTER device driver.
  *
  * @version V1.0
  * @date    02 Nov 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          02 Nov 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include <string.h>
#include "usbd_printer.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_PRINTER PRINTER
  * @brief Device PRINTER driver
  * @{
  */
/** @defgroup Device_PRINTER_Private_Variables Private Variables
  * @{
  */
static uint16_t __max_packet_size = 64;

/**
  * @brief Device descriptor
  */
uint8_t __printer_device_desc[] = {
	18,                       /**< Size of the descriptor */
	USB_DTYPE_DEVICE,         /**< Type of the descriptor */
	USBShort(0x200),          /**< Version 2.0 */
	USB_CLASS_VEND_SPECIFIC,  /**< USB device class */
	0,                        /**< USB device sub-class */
	0,                        /**< USB device protocol */
	64,                       /**< Maximum packet size */
	USBShort(0),              /**< VID */
	USBShort(0),              /**< PID */
	USBShort(0x200),          /**< Device version */
	1,                        /**< Manufacturer string */
	2,                        /**< Product string */
	3,                        /**< Manufacturer serial number */
	1,                        /**< number of the configurations */
};

/**
  * @brief Configuration descriptor header
  */
uint8_t __printer_desc[] = {
	9,                       /**< Size of the descriptor */
	USB_DTYPE_CONFIGURATION, /**< Type of the descriptor */
	USBShort(32),            /**< Total size of the descriptor */
	1,                       /**< Number of the inerfaces */
	1,                       /**< Unique value for this configuration */
	5,                       /**< Index of the string */
	USB_CONF_ATTR_SELF_PWR,  /**< Type of the power */
	250,                     /**< Maximum power in 2mA increments */
};

/**
  * @brief Printer interface descriptor
  */
const uint8_t __printer_interface[PRINTER_INTERFACE_SIZE] = {
	9,                                     /**< Size of the interface */
	USB_DTYPE_INTERFACE,                   /**< Type of the interface */
	0,                                     /**< Index of the interface */
	0,                                     /**< Index of the interface */
	2,                                     /**< Number of endpoints in the interface */
	USB_CLASS_PRINTER,                     /**< Interface class */
	1,                                     /**< Interface sub-class */
	2,                                     /**< Interface protocol */
	4,                                     /**< Index of the string */
	7,                                     /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                    /**< Type is an endpoint */
	USB_EP_DESC_IN | PRINTER_DATA_IN_EP,   /**< Endpoint direction */
	USB_EP_ATTR_BULK,                      /**< Endpoint type */
	USBShort(64),                          /**< Maximum packet size */
	0,                                     /**< Polling interval */
	7,                                     /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                    /**< Type is an endpoint */
	USB_EP_DESC_OUT | PRINTER_DATA_OUT_EP, /**< Endpoint direction */
	USB_EP_ATTR_BULK,                      /**< Endpoint type */
	USBShort(64),                          /**< Maximum packet size */
	0,                                     /**< Polling interval */
};

/**
  * @brief Printer interface descriptor for High-Speed mode
  */
const uint8_t __printer_interface_hs[PRINTER_INTERFACE_SIZE] = {
	9,                                     /**< Size of the interface */
	USB_DTYPE_INTERFACE,                   /**< Type of the interface */
	0,                                     /**< Index of the interface */
	0,                                     /**< Index of the interface */
	2,                                     /**< Number of endpoints in the interface */
	USB_CLASS_VEND_SPECIFIC,               /**< Interface class */
	0,                                     /**< Interface sub-class */
	0,                                     /**< Interface protocol */
	4,                                     /**< Index of the string */
	7,                                     /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                    /**< Type is an endpoint */
	USB_EP_DESC_IN | PRINTER_DATA_IN_EP,   /**< Endpoint direction */
	USB_EP_ATTR_BULK,                      /**< Endpoint type */
	USBShort(512),                         /**< Maximum packet size */
	0,                                     /**< Polling interval */
	7,                                     /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                    /**< Type is an endpoint */
	USB_EP_DESC_OUT | PRINTER_DATA_OUT_EP, /**< Endpoint direction */
	USB_EP_ATTR_BULK,                      /**< Endpoint type */
	USBShort(512),                         /**< Maximum packet size */
	0,                                     /**< Polling interval */
};

/**
  * @brief Printer configuration section
  */
const config_section_t __printer_config_sec = {
	sizeof(__printer_desc),
	__printer_desc
};

/**
  * @brief Printer interface section
  */
const config_section_t __printer_interface_sec = {
	sizeof(__printer_interface),
	__printer_interface
};

/**
  * @brief Printer High-Speed interface section
  */
const config_section_t __printer_interface_hs_sec = {
	sizeof(__printer_interface_hs),
	__printer_interface_hs
};

/**
  * @brief Printer section
  */
const config_section_t *__printer_sec[] = {
	&__printer_config_sec,
	&__printer_interface_sec
};

/**
  * @brief Printer High-Speed section
  */
const config_section_t *__printer_hs_sec[] = {
	&__printer_config_sec,
	&__printer_interface_hs_sec
};

/**
  * @brief Number of the section
  */
#define NUM_PRINTER_SECTIONS	(sizeof(__printer_sec) / sizeof(__printer_sec[0]))

/**
  * @brief BULK configuration header
  */
const config_head_t __printer_config_head = {
	NUM_PRINTER_SECTIONS,
	__printer_sec
};

/**
  * @brief Printer High-Speed configuration header
  */
const config_head_t __printer_config_head_hs = {
	NUM_PRINTER_SECTIONS,
	__printer_hs_sec
};

/**
  * @brief Printer configuration descriptor
  */
const config_head_t * const __printer_config_desc[] = {
	&__printer_config_head
};

/**
  * @brief Printer High-Speed configuration descriptor
  */
const config_head_t * const __printer_config_desc_hs[] = {
	&__printer_config_head_hs
};
/**
  * @}
  */

/** @defgroup Device_PRINTER_Private_Functions Private Functions
  * @{
  */
static void handle_request(void *device, usb_request_t *request);
static void handle_config_change(void *device, uint32_t info);
static void handle_disconnect(void *device);
static void handle_ep(void *device, uint32_t status);
static void handle_suspend(void *device);
static void handle_resume(void *device);
static void handle_device(void *device, uint32_t request, void *data);
/**
  * @}
  */

/** @addtogroup Device_PRINTER_Private_Variables
  * @{
  */
const base_handler_t printer_handler = {
	0,                    /**< Get descriptor */
	handle_request,       /**< Request handler */
	0,                    /**< Inerface change */
	handle_config_change, /**< Configuration change */
	0,                    /**< Data receive */
	0,                    /**< Data sent callback */
	0,                    /**< Reset handler */
	handle_suspend,       /**< Suspend handler */
	handle_resume,        /**< Resume handler */
	handle_disconnect,    /**< Disconnect handler */
	handle_ep,            /**< Endpoint handler */
	handle_device,        /**< Device handler */
};
/**
  * @}
  */

/** @addtogroup Device_PRINTER_Private_Functions
  * @{
  */
/**
  * @brief  Set/Reset the flag.
  * @param  flag: Flag.
  * @param  bit: indicates which bit number is to be set or reset.
  * @param  set: Set/Reset.
  * @retval None
  */
static void config_printer_flag(volatile uint16_t *flag, uint16_t bit, uint8_t set)
{
	/* Set/Reset the flag bit */
	if (set)
		SET_BIT(*flag, 1u << bit);
	else
		CLEAR_BIT(*flag, 1u << bit);

	return;
}

/**
  * @brief  Process receives notifications from the host.
  * @param  dev: PRINTER device stucture.
  * @param  status: Status.
  * @retval Status.
  */
static int process_data_from_host(usbd_printer_dev_t *dev, uint32_t status)
{
	uint32_t ep_status, size;
	printer_inst_t *inst;

	/* Create a pointer to the device instance */
	inst      = &dev->inst;
	/* Get the endpoint status */
	ep_status = map_usb_ep_status(inst->oep);
	/* Clear the endpoint status */
	map_usb_dev_ep_status_clear(inst->oep, ep_status);

	/* Has receive a packet */
	if (ep_status & USB_DEV_RX_PKT_RDY) {
		/* Set flag bit */
		config_printer_flag(&inst->flag, PRINTER_DO_PACKET_RX, 1);
		/* Get the size of the data */
		size = map_usb_ep_data_avail(inst->oep);
		/* Call the callback function */
		dev->rx_cbk(dev->rx_arg, USB_EVENT_RX_AVAILABLE, size, NULL);
	}
	else {
		/* No packet was received */
		if (ep_status & USB_RX_ERROR_FLAGS)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_ERROR, (ep_status & USB_RX_ERROR_FLAGS), NULL);

		return -1;
	}

	return 0;
}

/**
  * @brief  Process send data to the host.
  * @param  dev: PRINTER device stucture.
  * @param  status: Status.
  * @retval Status.
  */
static int process_data_to_host(usbd_printer_dev_t *dev, uint32_t status)
{
	uint32_t ep_status, size;
	printer_inst_t *inst;

	/* Create a pointer to the device instance */
	inst = &dev->inst;
	/* Get the endpoint status */
	ep_status = map_usb_ep_status(inst->iep);
	/* Clear the endpoint status */
	map_usb_dev_ep_status_clear(inst->iep, ep_status);
	/* Set the TX_STATE as IDLE */
	inst->tx_state = PRINTER_STATE_IDLE;

	/* Get the size */
	size = inst->size;
	inst->size = 0;

	/* Call the callback function */
	dev->tx_cbk(dev->tx_arg, USB_EVENT_TX_COMPLETE, size, NULL);
	return 0;
}

/**
  * @brief  Handle the interrupts on the endpoints.
  * @param  device: PRINTER device.
  * @param  status: Current status.
  * @retval None
  */
static void handle_ep(void *device, uint32_t status)
{
	usbd_printer_dev_t *dev = (usbd_printer_dev_t *)device;

	assert_param(dev);

	/* Handle OUT data endpoint */
	if (status & (0x10000 << (dev->inst.oep)))
		process_data_from_host(dev, status);

	/* Handle IN data endpoint */
	if (status & (1 << (dev->inst.iep)))
		process_data_to_host(dev, status);

	return;
}

/**
  * @brief  Handle device configuration changes.
  * @param  device: PRINTER device.
  * @param  ui32Info: New value.
  * @retval None
  */
static void handle_config_change(void *device, uint32_t ui32Info)
{
	usbd_printer_dev_t *dev = (usbd_printer_dev_t *)device;

	assert_param(dev);

	/* Set all endpoint state to IDLE */
	dev->inst.rx_state = PRINTER_STATE_IDLE;
	dev->inst.tx_state = PRINTER_STATE_IDLE;

	/* Call the callback function */
	if (dev->rx_cbk)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_CONNECTED, 0, NULL);

	/* Save the connected state */
	dev->inst.conn = 1;
	return;
}

/**
  * @brief  Handle device envent.
  * @param  device: PRINTER device.
  * @param  request: Request.
  * @param  data: Data buffer.
  * @retval None
  */
static void handle_device(void *device, uint32_t request, void *data)
{
	uint8_t *buf = (uint8_t *)data;
	usbd_printer_dev_t *dev = (usbd_printer_dev_t *)device;

	switch (request) {
	case USB_EVENT_COMP_IFACE_CHANGE:
		/* Interface change event */
		dev->inst.interface = buf[1];
		break;

	case USB_EVENT_COMP_EP_CHANGE:
		/* Endpint change event */
		if (buf[0] & USB_EP_DESC_IN)
			dev->inst.iep = buf[1] & 0x7f;
		else
			dev->inst.oep = buf[1] & 0x7f;

		break;
	case USB_EVENT_LPM_RESUME:
		/* LPM resume event */
		if (dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_LPM_RESUME, 0, NULL);

		break;
	case USB_EVENT_LPM_SLEEP:
		/* LPM sleep event */
		if (dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_LPM_SLEEP, 0, NULL);

		break;
	case USB_EVENT_LPM_ERROR:
		/* LPM error event */
		if (dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_LPM_ERROR, 0, NULL);

		break;
	default:
		break;
	}
}

/**
  * @brief  Handle non-standard request.
  * @param  device: printer device.
  * @param  request: Non-standard request.
  * @retval None
  */
static void handle_request(void *device, usb_request_t *request)
{
	uint32_t i, len;
	uint8_t buf[USB_DEVICE_PRINTER_DEVICE_ID_BUFFER_SIZE + 1];
	uint8_t port_status;
	uint32_t ep_status;
	usbd_printer_dev_t *dev;
	printer_inst_t *inst;
    
	assert_param(device);

	/* Create a pointer to the device instance */
	dev  = (usbd_printer_dev_t *)device;
	inst = &dev->inst;

	/* Check the index of the interface */
	if (request->wIndex != inst->interface)
		return;

	/* Handle the request */
	switch (request->bRequest) {
	case USB_DEVICE_PRINTER_GET_DEVICE_ID:
		/* Handle GET_DEVICE_ID request */
		if (dev->inst.conn && dev->rx_cbk) {
			/* Call the callback */
			dev->rx_cbk(dev->rx_arg, USBD_PRINTER_EVENT_GET_DEVICE_ID, 0, 0);

			/* Check the buffer of ID */
			if (dev->id == NULL)
				break;

			/* Copy the data */
			for (i = 0; i < dev->id_size; ++i)
				buf[i] = dev->id[i];

			/* Set the length */
			len = dev->id_size - 1;
			buf[0] = (uint8_t)((len >> 8) & 0xFF);
			buf[1] = (uint8_t)(len & 0xFF);

			/* Adjust the length */
			if (len > request->wLength)
				len = request->wLength;

			/* ACK the request */
			map_usb_dev_ep_data_ack(USB_EP_0, true);
			/* Send the response */
			usb_dcd_ep0_data_send(0, buf, len);
		}
		break;

	case USB_DEVICE_PRINTER_GET_PORT_STATUS:
		/* Handle GET_PORT_STATUS request */
		if (dev->inst.conn && dev->rx_cbk) {
			/* Call the callback */
			dev->rx_cbk(dev->rx_arg, USBD_PRINTER_EVENT_GET_PORT_STATUS, 1, &port_status);
			/* ACK the request */
			map_usb_dev_ep_data_ack(USB_EP_0, true);
			/* Send the response */
			usb_dcd_ep0_data_send(0, &port_status, 1);
		}
		break;

	case USB_DEVICE_PRINTER_SOFT_RESET:
		/* Handle GET_PORT_STATUS request.
		 * reset BULK_IN/OUT endpoint and inform application
		 */
		ep_status = map_usb_ep_status(inst->oep);
		map_usb_dev_ep_status_clear(inst->oep, ep_status);
		ep_status = map_usb_ep_status(inst->iep);
		map_usb_dev_ep_status_clear(inst->iep, ep_status);

		/* Handle SOFT_RESET request */
		if (dev->inst.conn && dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USBD_PRINTER_EVENT_SOFT_RESET, 0, NULL);
		break;
		
	case USB_DEVICE_PRINTER_SYSTEM_RESET:
		/* Handle System reset request */
		if (dev->inst.conn && dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USBD_PRINTER_EVENT_SYSTEM_RESET, 0, NULL);
		break;

	default:
		break;
	}
}

/**
  * @brief  Handle disconnect envent.
  * @param  device: PRINTER device.
  * @retval None
  */
static void handle_disconnect(void *device)
{
	usbd_printer_dev_t *dev = (usbd_printer_dev_t *)device;

	assert_param(device);

	/* Call the callback function */
	if (dev->inst.conn && dev->rx_cbk)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_DISCONNECTED, 0, NULL);

	/* Save the disconnected state */
	dev->inst.conn= 0;
}

/**
  * @brief  Called whenever the device is suspend.
  * @param  device: PRINTER device.
  * @retval None
  */
static void handle_suspend(void *device)
{
	usbd_printer_dev_t *dev = (usbd_printer_dev_t *)device;

	assert_param(device);

	/* Call callback function */
	if (dev->rx_cbk)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_SUSPEND, 0, NULL);

	return;
}

/**
  * @brief  Called whenever the device is resume.
  * @param  device: PRINTER device.
  * @retval None
  */
static void handle_resume(void *device)
{
	usbd_printer_dev_t *dev = (usbd_printer_dev_t *)device;

	assert_param(device);

	/* Call callback function */
	if (dev->rx_cbk)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_RESUME, 0, NULL);

	return;
}

/**
  * @brief  Provides us with a time reference.
  * @param  device: PRINTER device.
  * @param  time: Elapsed time in milliseconds.
  * @retval None
  */
static void printer_tick_handler(void *device, uint32_t time)
{
	uint32_t size;
	usbd_printer_dev_t *dev = (usbd_printer_dev_t *)device;

	assert_param(dev);

	/* Have a deferred receive waiting */
	if ((dev->inst.flag & (1 << PRINTER_DO_PACKET_RX)) && (dev->rx_cbk))  {
		/* Get size of the data */
		size = map_usb_ep_data_avail(dev->inst.oep);
		/* Call the callback function */
		dev->rx_cbk(dev->rx_arg, USB_EVENT_RX_AVAILABLE, size, NULL);
	}

	return;
}
/**
  * @}
  */

/** @defgroup Device_PRINTER_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes PRINTER device.
  * @param  idx: Index of the USB controller.
  * @param  dev: PRINTER device.
  * @retval Device structure.
  */
void *usbd_printer_init(uint32_t idx, usbd_printer_dev_t *dev)
{
	void *ret;
	device_desc_t *device_desc;
	config_desc_t *config_desc;

	assert_param(idx == 0);
	assert_param(dev);

	/* Composite initialization */
	ret = usbd_printer_init_comp(idx, dev, 0);

	if (ret) {
		/* Fill the device descriptor */
		device_desc = (device_desc_t *)__printer_device_desc;
		device_desc->idVendor  = dev->vid;
		device_desc->idProduct = dev->pid;
		/* Fill the configuration descriptor */
		config_desc = (config_desc_t *)__printer_desc;
		config_desc->bmAttributes = dev->attr_pwr;
		config_desc->bMaxPower    = dev->max_power >> 1;

		/* Initialze device controler */
		usb_dcd_init(idx, &dev->inst.info, (void *)dev);
	}

	return ret;
}

/**
  * @brief  Initializes PRINTER device.
  * @param  idx: Index of the USB controller.
  * @param  dev: PRINTER device.
  * @param  entry: Composite entry.
  * @retval Device structure.
  */
void *usbd_printer_init_comp(uint32_t idx, usbd_printer_dev_t *dev, comp_entry_t *entry)
{
	printer_inst_t *inst;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->rx_cbk);
	assert_param(dev->tx_cbk);

	/* Create a pointer to the device instance */
	inst = &dev->inst;

	/* Initialze the composite entry */
	if (entry != 0) {
		entry->info = &inst->info;
		entry->inst = (void *)dev;
	}

	/* Initialize the device information */
	inst->info.cbk         = &printer_handler;
	inst->info.device_desc = __printer_device_desc;
	inst->info.config_desc = __printer_config_desc;
	__max_packet_size      = 64;
#ifdef USBD_PRINTER_HS
	inst->info.config_desc = __printer_config_desc_hs;
	__max_packet_size      = 512;
#endif
	inst->info.string_desc    = 0;
	inst->info.nr_string_desc = 0;

	/* Initialize the device instance structure */
	inst->usb_idx  = 0;
	inst->rx_state = PRINTER_STATE_UNCONFIG;
	inst->tx_state = PRINTER_STATE_UNCONFIG;
	inst->flag     = 0;
	inst->conn     = 0;

	/* Initialize the device information */
	usb_device_info_init(0, &inst->info);
	/* Initialize device instance endpoint */
	inst->iep = PRINTER_EP_IN;
	inst->oep = PRINTER_EP_OUT;
	inst->interface = 0;
	/* Initialize the device instance structure */
	inst->info.string_desc    = dev->desc_str;
	inst->info.nr_string_desc = dev->num_str;

	/* Initialze tick */
	usb_tick_init();
	usb_tick_handler_register(printer_tick_handler, (void *)dev);

	return (void *)dev;
}

/**
  * @brief  Terminal the PRINTER device.
  * @param  dev: PRINTER device.
  * @retval None
  */
void usbd_printer_term(usbd_printer_dev_t *dev)
{
	assert_param(dev);

	/* Terminate the device controler */
	usb_dcd_term(dev->inst.usb_idx);
	dev->inst.usb_idx = 0;

	return;
}

/**
  * @brief  Set RX callback parameter.
  * @param  dev: PRINTER device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_printer_set_rx_param(usbd_printer_dev_t *dev, void *arg)
{
	void *old;

	assert_param(dev);

	/* Set the RX callback parameter */
	old = dev->rx_arg;
	dev->rx_arg = arg;

	return old;
}

/**
  * @brief  Set TX callback parameter.
  * @param  dev: PRINTER device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_printer_set_tx_param(usbd_printer_dev_t *dev, void *arg)
{
	void *old;

	assert_param(dev);

	/* Set the TX callback parameter */
	old = dev->tx_arg;
	dev->tx_arg = arg;

	return old;
}

/**
  * @brief  Transmits a packet to the USB host.
  * @param  dev: PRINTER device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Indicates whether more data is to be written.
  * @retval Length.
  */
uint32_t usbd_printer_packet_write(usbd_printer_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last)
{
	int32_t ret;
	printer_inst_t *inst = &dev->inst;

	assert_param(dev);

	/* Check the length and state */
	if ((len > __max_packet_size) || (inst->tx_state != PRINTER_STATE_IDLE))
		return 0;

	/* Put the data into the FIFO */
	ret = map_usb_ep_data_put(inst->iep, data, len);

	if (ret == -1)
		return 0;

	/* Save sent bytes */
	inst->size += len;

	/* Last call for a packet. Send the data now */
	if (last) {
		/* Set the TX_STATE to WAIT_DATA */
		inst->tx_state = PRINTER_STATE_WAIT_DATA;
		/* Send the data */
		ret = map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
	}

	return ret != -1 ? len : 0;
}

/**
  * @brief  Received a packet from the USB host.
  * @param  dev: PRINTER device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Indicates whether more data is to be written.
  * @retval Status.
  */
uint32_t usbd_printer_packet_read(usbd_printer_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last)
{
	int ret;
	uint32_t status, cnt, _len;
	printer_inst_t *inst = &dev->inst;

	assert_param(dev);

	/* Get the endpoint status */
	status = map_usb_ep_status(inst->oep);

	/* Check had receive a pecket */
	if ((status & USB_DEV_RX_PKT_RDY) == 0)
		return 0;

	/* Save the length */
	cnt  = len;
	/* Get the size of the data */
	_len = map_usb_ep_data_avail(inst->oep);
	/* Get data from FIFO */
	ret  = map_usb_ep_data_get(inst->oep, data, &cnt);

	if (cnt == _len) {
		/* Clear endpoint status */
		map_usb_dev_ep_status_clear(inst->oep, status);
		/* ACK the data */
		map_usb_dev_ep_data_ack(inst->oep, true);
		/* Reset the flag bit */
		config_printer_flag(&inst->flag, PRINTER_DO_PACKET_RX, 0);
	}

	return ret != -1 ? cnt : 0;
}

/**
  * @brief  Get number of free bytes in the transmit buffer.
  * @param  dev: PRINTER device.
  * @retval Length.
  */
uint32_t usbd_printer_tx_packet_avail(usbd_printer_dev_t *dev)
{
	assert_param(dev);

	/* Check the TX_STATE */
	if (dev->inst.tx_state == PRINTER_STATE_IDLE)
		return __max_packet_size;

	return 0;
}

/**
  * @brief  Determine whether a packet is available.
  * @param  dev: PRINTER device.
  * @retval Length.
  */
uint32_t usbd_printer_rx_packet_avail(usbd_printer_dev_t *dev)
{
	uint32_t status;

	assert_param(dev);

	/* Get the endpoint status */
	status = map_usb_ep_status(dev->inst.oep);

	/* Had received a packet? */
	if (status & USB_DEV_RX_PKT_RDY)
		return map_usb_ep_data_avail(dev->inst.oep);

	return 0;
}

/**
  * @brief  Requests a remote wake up to resume communication.
  * @param  dev: PRINTER device.
  * @retval Status.
  */
int usbd_printer_remote_wakeup_req(usbd_printer_dev_t *dev)
{
	assert_param(dev);

	return usb_dcd_remote_wakeup_req(0);
}

/**
  * @brief  Set power status.
  * @param  dev: PRINTER device.
  * @param  power: Status of the power.
  * @retval None
  */
void usbd_printer_power_status_set(usbd_printer_dev_t *dev, uint8_t power)
{
	assert_param(dev);

	usb_dcd_power_status_set(0, power);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
