/**
  *********************************************************************************
  *
  * @file    usbd_msc.h
  * @brief   Header file for the MSC driver[Device].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBD_MSC_H__
#define __USBD_MSC_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbd_core.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_MSC
  * @{
  */
/**
  * @defgroup Device_MSC_Public_Macros Public Macros
  * @{
  */
#define MSC_BLOCK_SIZE		512U
#define MSC_INTERFACE_SIZE	23U
#define COMP_DMSC_SIZE		(MSC_INTERFACE_SIZE)
#define USBD_MSC_EVENT_IDLE     (USBD_MSC_EVENT_BASE + 0)
#define USBD_MSC_EVENT_READING  (USBD_MSC_EVENT_BASE + 1)
#define USBD_MSC_EVENT_WRITING  (USBD_MSC_EVENT_BASE + 2)
#define USBD_MSC_EVENT_DMA_ERR  (USBD_MSC_EVENT_BASE + 3)

#define USBD_MSC_DMA_IN		0x1U
#define USBD_MSC_DMA_OUT	0x2U
#define USBD_MSC_ALLOW_REMOVAL	0x4U
#define USBD_MSC_SUBCLASS_SCSI	0x6U
#define USBD_MSC_PROTO_BULKONLY	0x50U

#ifdef USBD_MSC_HS
#define MSC_EP_IN_MAX_SIZE	512U
#define MSC_EP_OUT_MAX_SIZE	512U
#else
#define MSC_EP_IN_MAX_SIZE	64U
#define MSC_EP_OUT_MAX_SIZE	64U
#endif
/**
  * @}
  */

/**
  * @defgroup Device_MSC_Public_Types Public Types
  * @{
  */
/**
  * @brief Media access functions
  */
typedef struct {
	void *(*open)(uint32_t drv);	/**< open the media */
	void (*close)(void *drv);	/**< Cpen the media */
	uint32_t (*read_block)(void *drv, uint8_t *buf, uint32_t sector, uint32_t num);	/**< Read block */
	uint32_t (*write_block)(void *drv, uint8_t *buf, uint32_t sector, uint32_t num);/**< Write block */
	uint32_t (*get_block_num)(void *drv);	/**< Get block number */
	uint32_t (*get_block_size)(void *drv);	/**< Get block size */
} msc_media_t;

/**
  * @brief Media status
  */
typedef enum {
	USBD_MSC_MEDIA_PRESENT,		/**< Present */
	USBD_MSC_MEDIA_PRESENT_NO,	/**< No present */
	USBD_MSC_MEDIA_STOP,		/**< Stop */
	USBD_MSC_MEDIA_UNKNOWN,		/**< Unknown */
} msc_media_status_t;

/**
  * @brief SCSI status
  */
typedef enum {
	USBD_SCSI_IDLE,		/**< Idle */
	USBD_SCSI_SEND_BLOCK,	/**< Read block */
	USBD_SCSI_RECV_BLOCK,	/**< Write block */
	USBD_SCSI_SEND_STAT,	/**< Send status */
	USBD_SCSI_RECV_STAT,	/**< Receive status */
	USBD_SCSI_SENT_STAT,	/**< Sent status */
} scsi_status_t;

/**
  * @brief Private instance
  */
typedef struct {
	uint32_t usb_idx;		/**< Index of the USB controller */
	device_info_t info;		/**< Device information */
	uint8_t err;			/**< Error code */
	uint8_t sense;			/**< Sense key */
	uint16_t sense_code;		/**< Sense code */
	void *media;			/**< Pointer to media */
	volatile bool conn;		/**< Connection status */
	uint32_t flag;			/**< Flags */
	uint32_t buf[1024];		/**< Buffer */
	uint32_t size;			/**< Number of bytes to transfer */
	uint32_t _size;			/**< Temporary variable for USB_DMA_N_SUPPORT */
	uint32_t lba;			/**< Logical block address */
	uint8_t iep;			/**< IN endpoint */
	uint8_t idmach;			/**< IN DMA channel */
	uint8_t oep;			/**< OUT endpoint */
	uint8_t odmach;			/**< OUT DMA channel */
	uint8_t interface;		/**< Interface */
	uint8_t scsi_state;		/**< SCSI state */
	msc_media_status_t status;	/**< Media state */
	uint8_t use_dma;                /**< Enable/Disable DMA transfer data */
} msc_inst_t;

/**
  * @brief Define operating parameters for the mass storage device
  */
typedef struct {
	const uint16_t vid;			/**< Vendor ID */
	const uint16_t pid;			/**< Product ID */
	const uint8_t vendor[8];		/**< Vendor string */
	const uint8_t product[16];		/**< Product string */
	const uint8_t version[4];		/**< Version string */
	const uint16_t max_power;		/**< Maximum power consumption */
	const uint8_t attr_pwr;			/**< Attribute of the power */
	const uint8_t * const *desc_str;	/**< String descriptor */
	const uint32_t num_str;			/**< Number of the string descriptors */
	const msc_media_t func;			/**< Media access functions */
	const usb_cbk cbk;			/**< Callback */
	msc_inst_t inst;			/**< Device instance */
} usbd_msc_dev_t;
/**
  * @}
  */

/** @addtogroup Device_MSC_Public_Functions
  * @{
  */
extern void *usbd_msc_init(uint32_t idx, usbd_msc_dev_t *dev);
extern void *usbd_msc_init_comp(uint32_t idx, usbd_msc_dev_t *dev, comp_entry_t *entry);
extern void usbd_msc_term(void *device);
extern void usbd_msc_media_change(void *device, msc_media_status_t status);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
