/**
  *********************************************************************************
  *
  * @file    usbd_msc.c
  * @brief   USB MSC device driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include <string.h>
#include "usb_msc.h"
#include "usbd_msc.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_MSC MSC
  * @brief Device MSC driver
  * @{
  */
/** @defgroup Device_MSC_Private_Variables Private Variables
  * @{
  */
uint32_t _block_size = MSC_BLOCK_SIZE;
uint8_t _command[64];
msc_csw_t _scsi_csw;
#ifdef USB_DMA_N_SUPPORT
usb_dma_callback_t msc_tx_cbk;
usb_dma_callback_t msc_rx_cbk;
#else
usb_dma_callback_t msc_cbk;
#endif

/**
  * @brief Device descriptor
  */
static uint8_t msc_dev_desc[] = {
	18,                       /**< Size of the descriptor */
	USB_DTYPE_DEVICE,         /**< Type of the descriptor */
#ifdef USBD_MSC_HS
	USBShort(0x200),          /**< Version 2.0 */
#else
	USBShort(0x110),          /**< Version 1.1 */
#endif
	0,                        /**< USB device class */
	0,                        /**< USB device sub-class */
	0,                        /**< USB device protocol */
	64,                       /**< Maximum packet size */
	USBShort(0),              /**< VID */
	USBShort(0),              /**< PID */
	USBShort(0x100),          /**< Device version */
	1,                        /**< Manufacturer string */
	2,                        /**< Product string */
	3,                        /**< Manufacturer serial number */
	1                         /**< number of the configurations */
};

/**
  * @brief Configuration descriptor header
  */
static uint8_t msc_desc[] = {
	9,                       /**< Size of the descriptor */
	USB_DTYPE_CONFIGURATION, /**< Type of the descriptor */
	USBShort(32),            /**< Total size of the descriptor */
	1,                       /**< Number of the inerfaces */
	1,                       /**< Unique value for this configuration */
	0,                       /**< Index of the string */
	USB_CONF_ATTR_SELF_PWR,  /**< Type of the power */
	250,                     /**< Maximum power in 2mA increments */
};

/**
  * @brief MSC interface descriptor
  */
const uint8_t msc_interface_desc[MSC_INTERFACE_SIZE] = {
	9,                             /**< Size of the interface */
	USB_DTYPE_INTERFACE,           /**< Type of the interface */
	0,                             /**< Index of the interface */
	0,                             /**< Alternate setting */
	2,                             /**< Number of endpoints in the interface */
	USB_CLASS_MASS_STORAGE,        /**< Interface class */
	USBD_MSC_SUBCLASS_SCSI,        /**< Interface sub-class */
	USBD_MSC_PROTO_BULKONLY,       /**< Interface protocol */
	0,                             /**< Index of the string */

	7,                             /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,            /**< Type is an endpoint */
	USB_EP_DESC_IN | MSC_EP_IN,    /**< Endpoint direction */
	USB_EP_ATTR_BULK,              /**< Endpoint type */
	USBShort(MSC_EP_IN_MAX_SIZE),  /**< Maximum packet size */
	0,                             /**< Polling interval */

	7,                             /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,            /**< Type is an endpoint */
	USB_EP_DESC_OUT | MSC_EP_OUT,  /**< Endpoint direction */
	USB_EP_ATTR_BULK,              /**< Endpoint type */
	USBShort(MSC_EP_OUT_MAX_SIZE), /**< Maximum packet size */
	0,                             /**< Polling interval */
};

/**
  * @brief MSC configuration section
  */
const config_section_t msc_config_section = {
	sizeof(msc_desc),
	msc_desc
};

/**
  * @brief MSC interface section
  */
const config_section_t msc_interface_section = {
	sizeof(msc_interface_desc),
	msc_interface_desc
};

/**
  * @brief MSC section
  */
const config_section_t *msc_section[] = {
	&msc_config_section,
	&msc_interface_section
};

/**
  * @brief Number of the section
  */
#define NUM_MSC_SECTIONS	(sizeof(msc_section) / sizeof(msc_section[0]))

/**
  * @brief MSC configuration header
  */
const config_head_t msc_config_header = {
	NUM_MSC_SECTIONS,
	msc_section
};

/**
  * @brief MSC configuration descriptor
  */
const config_head_t * const msc_config_desc[] =
{
	&msc_config_header
};
/**
  * @}
  */

/** @defgroup Device_MSC_Private_Functions Private Functions
  * @{
  */
static void ep_handle(void *device, uint32_t status);
static void device_handle(void *device, uint32_t request, void *data);
static void suspend_handle(void *device);
static void disconnect_handle(void *device);
static void config_change_handle(void *device, uint32_t value);
static void request_handle(void *device, usb_request_t *req);
static void usbd_scsi_status_send(usbd_msc_dev_t *dev);
static uint32_t usbd_scsi_cmd(usbd_msc_dev_t *dev, msc_cbw_t *cbw);
/**
  * @}
  */

/** @addtogroup Device_MSC_Private_Variables
  * @{
  */
const base_handler_t msc_handler = {
	0,                    /**< Get descriptor */
	request_handle,       /**< Request handler */
	0,                    /**< Inerface change */
	config_change_handle, /**< Configuration change */
	0,                    /**< Data receive */
	0,                    /**< Data sent callback */
	0,                    /**< Reset handler */
	suspend_handle,       /**< Suspend handler */
	0,                    /**< Resume handler */
	disconnect_handle,    /**< Disconnect handler */
	ep_handle,            /**< Endpoint handler */
	device_handle         /**< Device handler */
};
/**
  * @}
  */

/** @addtogroup Device_MSC_Private_Functions
  * @{
  */
#ifdef USB_DMA_N_SUPPORT
/**
  * @brief  Callback function when external DMA transfer complete.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_msc_dma_tx_cplt(void *arg)
{
	usbd_msc_dev_t *dev = (usbd_msc_dev_t *)arg;

	map_usb_ep_data_send(dev->inst.iep, USB_TRANS_IN);
	return;
}

/**
  * @brief  Callback function when external DMA transfer complete.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_msc_dma_rx_cplt(void *arg)
{
	usbd_msc_dev_t *dev = (usbd_msc_dev_t *)arg;
	msc_inst_t *inst = &dev->inst;

#ifdef USBD_MSC_HS
	inst->size -= _block_size;
	dev->func.write_block(dev->inst.media, (uint8_t *)dev->inst.buf, dev->inst.lba, 1);
	map_usb_dev_ep_data_ack(dev->inst.oep, false);

	if (inst->size == 0) {
		_scsi_csw.bCSWStatus = 0;
		_scsi_csw.dCSWDataResidue = 0;
		dev->inst.flag &= ~USBD_MSC_DMA_OUT;

		if (dev->cbk)
			dev->cbk(0, USBD_MSC_EVENT_IDLE, 0, 0);

		usbd_scsi_status_send(dev);
		return;
	}

	++inst->lba;
#else
	map_usb_dev_ep_data_ack(inst->oep, false);
	inst->_size += MSC_EP_OUT_MAX_SIZE;

	if (inst->_size != _block_size)
		return;

	inst->size -= _block_size;
	dev->func.write_block(dev->inst.media, (uint8_t *)dev->inst.buf, dev->inst.lba, 1);

	if (inst->size == 0) {
		_scsi_csw.bCSWStatus = 0;
		_scsi_csw.dCSWDataResidue = 0;
		dev->inst.flag &= ~USBD_MSC_DMA_OUT;

		if (dev->cbk)
			dev->cbk(0, USBD_MSC_EVENT_IDLE, 0, 0);

		usbd_scsi_status_send(dev);
		return;
	}

	++dev->inst.lba;
	inst->_size = 0;
#endif
	return;
}

/**
  * @brief  Callback function when external DMA send error.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_msc_dma_e_err(void *arg)
{
	usbd_msc_dev_t *dev = (usbd_msc_dev_t *)arg;

	dev->cbk(dev, USBD_MSC_EVENT_DMA_ERR, 0, NULL);
	return;
}
#else

/**
  * @brief  Callback function when DMA transfer complete.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_msc_dma_cplt(uint8_t ch, void *arg)
{
	usbd_msc_dev_t *dev = (usbd_msc_dev_t *)arg;

	if (ch == dev->inst.idmach) {
#ifdef USBD_MSC_HS
		map_usb_ep_data_send(dev->inst.iep, USB_TRANS_IN);
#else
		dev->inst.size -= _block_size;

		if (dev->inst.size == 0) {
			_scsi_csw.bCSWStatus = 0;
			_scsi_csw.dCSWDataResidue = 0;
			dev->inst.flag &= (uint32_t)(~USBD_MSC_DMA_IN);
			map_usb_ep_dma_config(dev->inst.iep, USB_DMA_EP_CFG_TX, DISABLE);

			if (dev->cbk)
				dev->cbk(0, USBD_MSC_EVENT_IDLE, 0, 0);

			if (map_usb_ep_status(dev->inst.iep) & USB_DEV_TX_TXPKTRDY)
				dev->inst.scsi_state = USBD_SCSI_SEND_STAT;
			else
				usbd_scsi_status_send(dev);

			return;
		}

		++dev->inst.lba;
		dev->func.read_block(dev->inst.media, (uint8_t *)dev->inst.buf, dev->inst.lba, 1);
		usb_dma_config(dev->inst.idmach, (uint32_t)&dev->inst.buf, _block_size);
#endif
	}
	else {
		dev->inst.size -= _block_size;
		dev->func.write_block(dev->inst.media, (uint8_t *)dev->inst.buf, dev->inst.lba, 1);
#ifdef USBD_MSC_HS
		map_usb_dev_ep_data_ack(dev->inst.oep, false);
#endif
		if (dev->inst.size == 0) {
			_scsi_csw.bCSWStatus = 0;
			_scsi_csw.dCSWDataResidue = 0;
			dev->inst.flag &= (uint32_t)(~USBD_MSC_DMA_OUT);
#ifndef USBD_MSC_HS
			map_usb_ep_dma_config(dev->inst.oep, USB_DMA_EP_CFG_RX_DEV, DISABLE);
#endif
			if (dev->cbk)
				dev->cbk(0, USBD_MSC_EVENT_IDLE, 0, 0);

			usbd_scsi_status_send(dev);
			return;
		}

		++dev->inst.lba;
#ifndef USBD_MSC_HS
		usb_dma_config(dev->inst.odmach, (uint32_t)&dev->inst.buf, _block_size);
#endif
	}

	return;
}

/**
  * @brief  Callback function when DMA send error.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_msc_dma_err(uint8_t ch, void *arg)
{
	usbd_msc_dev_t *dev = (usbd_msc_dev_t *)arg;

	dev->cbk(dev, USBD_MSC_EVENT_DMA_ERR, 0, NULL);
	return;
}
#endif

/**
  * @brief  Handle the interrupts on the Bulk endpoints.
  * @param  device: MSC device.
  * @param  status: Current status.
  * @retval None
  */
static void ep_handle(void *device, uint32_t status)
{
	usbd_msc_dev_t *dev;
	msc_inst_t *inst;
	msc_cbw_t *_cbw;
	uint32_t ep_status, size;

	assert_param(device != 0);

	dev  = (usbd_msc_dev_t *)device;
	inst = &dev->inst;

	if (status & (1 << inst->iep)) {
		switch (inst->scsi_state) {
		case USBD_SCSI_SEND_BLOCK:
			if (inst->use_dma) {
#ifdef USB_DMA_N_SUPPORT
#ifdef USBD_MSC_HS
				dev->inst.size -= _block_size;
		
				if (dev->inst.size == 0) {
					_scsi_csw.bCSWStatus = 0;
					_scsi_csw.dCSWDataResidue = 0;
					dev->inst.flag &= ~USBD_MSC_DMA_IN;
		
					if (dev->cbk)
						dev->cbk(0, USBD_MSC_EVENT_IDLE, 0, 0);
		
					if (map_usb_ep_status(dev->inst.iep) & USB_DEV_TX_TXPKTRDY)
						dev->inst.scsi_state = USBD_SCSI_SEND_STAT;
					else
						usbd_scsi_status_send(dev);
		
					break;
				}
		
				++dev->inst.lba;
				dev->func.read_block(dev->inst.media, (uint8_t *)dev->inst.buf, dev->inst.lba, 1);
				usb_dma_config(USB_E_DMA_TYPE_MSC_TX, (uint32_t)&dev->inst.buf, _block_size);
#else
				if (inst->_size != _block_size) {
					usb_dma_config(USB_E_DMA_TYPE_MSC_TX, (uint32_t)((uint8_t *)inst->buf + inst->_size), MSC_EP_IN_MAX_SIZE);
					inst->_size += MSC_EP_IN_MAX_SIZE;
				}
				else {
					dev->inst.size -= _block_size;

					if (dev->inst.size == 0) {
						_scsi_csw.bCSWStatus = 0;
						_scsi_csw.dCSWDataResidue = 0;
						dev->inst.flag &= ~USBD_MSC_DMA_IN;

						if (dev->cbk)
							dev->cbk(0, USBD_MSC_EVENT_IDLE, 0, 0);

						if (map_usb_ep_status(dev->inst.iep) & USB_DEV_TX_TXPKTRDY)
							dev->inst.scsi_state = USBD_SCSI_SEND_STAT;
						else
							usbd_scsi_status_send(dev);
		
						break;
					}

					++dev->inst.lba;
					dev->func.read_block(dev->inst.media, (uint8_t *)dev->inst.buf, dev->inst.lba, 1);
					usb_dma_config(USB_E_DMA_TYPE_MSC_TX, (uint32_t)&inst->buf, MSC_EP_IN_MAX_SIZE);
					inst->_size = MSC_EP_IN_MAX_SIZE;
				}
#endif
#else
#ifdef USBD_MSC_HS
				dev->inst.size -= _block_size;
		
				if (dev->inst.size == 0) {
					_scsi_csw.bCSWStatus = 0;
					_scsi_csw.dCSWDataResidue = 0;
					dev->inst.flag &= ~USBD_MSC_DMA_IN;
		
					if (dev->cbk)
						dev->cbk(0, USBD_MSC_EVENT_IDLE, 0, 0);
		
					if (map_usb_ep_status(dev->inst.iep) & USB_DEV_TX_TXPKTRDY)
						dev->inst.scsi_state = USBD_SCSI_SEND_STAT;
					else
						usbd_scsi_status_send(dev);
		
					break;
				}
		
				++dev->inst.lba;
				dev->func.read_block(dev->inst.media, (uint8_t *)dev->inst.buf, dev->inst.lba, 1);
				usb_dma_config(dev->inst.idmach, (uint32_t)&dev->inst.buf, _block_size);
#endif
#endif
			}
			else {
				if (inst->_size != _block_size) {
					map_usb_ep_data_put(inst->iep, (uint8_t *)((uint8_t *)inst->buf + inst->_size), MSC_EP_IN_MAX_SIZE);
					map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
					inst->_size += MSC_EP_IN_MAX_SIZE;
				}
				else {
					dev->inst.size -= _block_size;

					if (dev->inst.size == 0) {
						_scsi_csw.bCSWStatus = 0;
						_scsi_csw.dCSWDataResidue = 0;
						dev->inst.flag &= (uint32_t)(~USBD_MSC_DMA_IN);

						if (dev->cbk)
							dev->cbk(0, USBD_MSC_EVENT_IDLE, 0, 0);

						if (map_usb_ep_status(dev->inst.iep) & USB_DEV_TX_TXPKTRDY)
							dev->inst.scsi_state = USBD_SCSI_SEND_STAT;
						else
							usbd_scsi_status_send(dev);
		
						break;
					}

					++dev->inst.lba;
					dev->func.read_block(dev->inst.media, (uint8_t *)dev->inst.buf, dev->inst.lba, 1);
					map_usb_ep_data_put(inst->iep, (uint8_t *)inst->buf, MSC_EP_IN_MAX_SIZE);
					map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
					inst->_size = MSC_EP_IN_MAX_SIZE;
				}
			}

			break;

		case USBD_SCSI_SEND_STAT:
			usbd_scsi_status_send(dev);
			break;

		case USBD_SCSI_SENT_STAT:
			inst->scsi_state = USBD_SCSI_IDLE;
			break;
		case USBD_SCSI_IDLE:
		default:
			break;
		}
	}

	if (status & (0x10000 << inst->oep)) {
		ep_status = map_usb_ep_status(inst->oep);

		switch (inst->scsi_state) {
		case USBD_SCSI_RECV_BLOCK:
			if (inst->use_dma) {
#ifdef USB_DMA_N_SUPPORT
				size = map_usb_ep_data_avail(inst->oep);
#ifdef USBD_MSC_HS
				usb_dma_config(USB_E_DMA_TYPE_MSC_RX, (uint32_t)&inst->buf, size);
#else
				usb_dma_config(USB_E_DMA_TYPE_MSC_RX, ((uint32_t)&inst->buf) + inst->_size, size);
#endif
#else
#ifdef USBD_MSC_HS
				size = map_usb_ep_data_avail(inst->oep);
				usb_dma_config(inst->odmach, (uint32_t)&inst->buf, size);
#else
				map_usb_dma_mult_recv_start(inst->oep);
#endif
#endif
			}
			else {
				size = map_usb_ep_data_avail(inst->oep);
				map_usb_ep_data_get(inst->oep, (uint8_t *)((uint8_t *)inst->buf + inst->_size), &size);
				map_usb_dev_ep_data_ack(inst->oep, false);
				inst->_size += size;
	
				if (inst->_size != _block_size)
					break;
	
				inst->size -= _block_size;
				dev->func.write_block(dev->inst.media, (uint8_t *)dev->inst.buf, dev->inst.lba, 1);
			
				if (inst->size == 0) {
					_scsi_csw.bCSWStatus = 0;
					_scsi_csw.dCSWDataResidue = 0;
					dev->inst.flag &= (uint32_t)(~USBD_MSC_DMA_OUT);
	
					if (dev->cbk)
						dev->cbk(0, USBD_MSC_EVENT_IDLE, 0, 0);
	
					usbd_scsi_status_send(dev);
					break;
				}
			
				++dev->inst.lba;
				inst->_size = 0;
			}

			break;

		case USBD_SCSI_IDLE:
			size = 64;
			map_usb_ep_data_get(inst->oep, _command, &size);
			_cbw = (msc_cbw_t *)_command;
			map_usb_dev_ep_data_ack(inst->oep, false);

			if(_cbw->dCBWSignature == CBW_SIGNATURE) {
				_scsi_csw.dCSWSignature   = CSW_SIGNATURE;
				_scsi_csw.dCSWTag         = _cbw->dCBWTag;
				_scsi_csw.dCSWDataResidue = 0;
				_scsi_csw.bCSWStatus      = 0;

				usbd_scsi_cmd(dev, _cbw);
			}
			else {
				inst->scsi_state = USBD_SCSI_IDLE;
			}

			break;
		default:
			break;
		}

		map_usb_dev_ep_status_clear(inst->oep, ep_status);
	}
}

/**
  * @brief  Handle device envent.
  * @param  device: MSC device.
  * @param  request: Request.
  * @param  data: Data buffer.
  * @retval None
  */
static void device_handle(void *device, uint32_t request, void *data)
{
	msc_inst_t *inst;
	uint8_t *buf;
	usbd_msc_dev_t *dev;

	/* Create a pointer to the device instance */
	dev  = (usbd_msc_dev_t *)device;
	inst = &dev->inst;
	/* Create a pointer to the data */
	buf  = (uint8_t *)data;

	/* Which request? */
	switch (request) {
	case USB_EVENT_COMP_IFACE_CHANGE:
		/* Records the new interface */
		inst->interface = buf[1];
		break;

	case USB_EVENT_COMP_EP_CHANGE:
		break;
	case USB_EVENT_LPM_RESUME:
		/* Send a LPM_RESUME event to the client */
		if (dev->cbk)
			dev->cbk(0, USB_EVENT_LPM_RESUME, 0, (void *)0);

		break;
	case USB_EVENT_LPM_SLEEP:
		/* Send a LPM_SLEEP event to the client */
		if (dev->cbk)
			dev->cbk(0, USB_EVENT_LPM_SLEEP, 0, (void *)0);

		break;
	case USB_EVENT_LPM_ERROR:
		/* Send a LPM_ERROR event to the client */
		if(dev->cbk)
			dev->cbk(0, USB_EVENT_LPM_ERROR, 0, (void *)0);

		break;
	default:
		break;
	}
}

/**
  * @brief  Handle disconnect envent.
  * @param  device: MSC device.
  * @retval None
  */
static void disconnect_handle(void *device)
{
	usbd_msc_dev_t *dev = (usbd_msc_dev_t *)device;

	/* If the media is initialized */
	if (dev->inst.media != 0) {
		/* Clear the media */
		dev->inst.media = 0;
		/* Close the device */
		dev->func.close(0);
	}

	/* Send a DISCONNECTED event to the client */
	if (dev->cbk)
		dev->cbk(dev, USB_EVENT_DISCONNECTED, 0, 0);
}

/**
  * @brief  Handle suspend envent.
  * @param  device: MSC device.
  * @retval None
  */
static void suspend_handle(void *device)
{
	/* Handle disconnected event */
	disconnect_handle(device);
	return;
}

/**
  * @brief  Handle device configuration changes.
  * @param  device: MSC device.
  * @param  value: New value.
  * @retval None
  */
static void config_change_handle(void *device, uint32_t value)
{
	usbd_msc_dev_t *dev = (usbd_msc_dev_t *)device;

	/* Send a CONNECTED event to the client */
	if (dev->cbk)
		dev->cbk(dev, USB_EVENT_CONNECTED, 0, 0);
}

/**
  * @brief  Handle non-standard request.
  * @param  device: MSC device.
  * @param  req: Non-standard request.
  * @retval None
  */
static void request_handle(void *device, usb_request_t *req)
{
	static const uint8_t lun = 0;

	assert_param(device != 0);

	/* Which request? */
	switch (req->bRequest) {
	case USBREQ_GET_MAX_LUN:
		/* ACK the data */
		map_usb_dev_ep_data_ack(USB_EP_0, true);
		/* Send the lun to the host */
		usb_dcd_ep0_data_send(0, (uint8_t *)&lun, 1);
		break;

	case USBREQ_BULK_ONLY_RESET:
		/* ACK the data */
		map_usb_dev_ep_data_ack(USB_EP_0, true);
		/* Send the lun to the host */
		usb_dcd_ep0_data_send(0, (uint8_t *)&lun, 0);
		break;

	default:
		usb_dcd_ep0_stall(0);
		break;
	}
}

/**
  * @brief  Send out the response data.
  * @param  dev: MSC device.
  * @retval None
  */
static void usbd_scsi_status_send(usbd_msc_dev_t *dev)
{
	msc_inst_t *inst = &dev->inst;

	/* Put the data into the FIFO */
	map_usb_ep_data_put(inst->iep, (uint8_t *)&_scsi_csw, 13);
	/* Send the data to the host */
	map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
	/* Set the SCSI_STATE */
	inst->scsi_state = USBD_SCSI_SENT_STAT;
}

/**
  * @brief  Handle the SCSI Inquiry command.
  * @param  dev: MSC device.
  * @retval None
  */
static void usbd_scsi_inquiry(usbd_msc_dev_t *dev)
{
	uint32_t *cmd, i;
	msc_inst_t *inst;

	inst   = &dev->inst;
	cmd    = (uint32_t *)_command;
	cmd[0] = SCSI_INQ_PDT_SBC | (SCSI_INQ_RMB << 8);
	cmd[1] = 31;

	/* Build the command */
	for (i = 0; i < 8; ++i)
		_command[i + 8] = dev->vendor[i];
	for (i = 0; i < 16; ++i)
		_command[i + 16] = dev->product[i];
	for(i = 0; i < 4; ++i)
		_command[i + 32] = dev->version[i];

	/* Put the data into the FIFO */
	map_usb_ep_data_put(inst->iep, _command, 36);
	/* Send the data to the host */
	map_usb_ep_data_send(inst->iep, USB_TRANS_IN);

	/* Set the CSW information */
	_scsi_csw.bCSWStatus      = 0;
	_scsi_csw.dCSWDataResidue = 0;
	inst->scsi_state          = USBD_SCSI_SEND_STAT;
}

/**
  * @brief  Handle the SCSI Capacities command.
  * @param  dev: MSC device.
  * @retval None
  */
static void usbd_scsi_read_caps(usbd_msc_dev_t *dev)
{
	uint32_t *cmd, num;
	msc_inst_t *inst;

	inst = &dev->inst;
	cmd  = (uint32_t *)_command;

	if (inst->media != 0) {
		/* Get the block size and block numbers from media */
		if (dev->func.get_block_size)
			_block_size = dev->func.get_block_size(inst->media);
		num = dev->func.get_block_num(inst->media);

		cmd[0] = 0x08000000;

		/* Build the command */
		_command[4]  = num >> 24;
		_command[5]  = 0xff & (num >> 16);
		_command[6]  = 0xff & (num >> 8);
		_command[7]  = 0xff & (num);
		_command[8]  = 0x2;
		_command[9]  = 0xff & (_block_size >> 16);
		_command[10] = 0xff & (_block_size >> 8);
		_command[11] = 0xff & _block_size;

		/* Put the data into the FIFO */
		map_usb_ep_data_put(inst->iep, _command, 12);
		/* Send the data to the host */
		map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 0;
		_scsi_csw.dCSWDataResidue = 0;
	}
	else {
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 1;
		_scsi_csw.dCSWDataResidue = 0;

		/* Stall the IN endpoint */
		map_usb_dev_ep_stall(inst->iep, USB_EP_DEV_IN);

		/* Fill the instance state */
		inst->err        = SCSI_RS_VALID | SCSI_RS_CUR_ERRORS;
		inst->sense      = SCSI_RS_KEY_NOT_READY;
		inst->sense_code = SCSI_RS_MED_NOT_PRSNT;
	}

	/* Set the SCSI state */
	inst->scsi_state = USBD_SCSI_SEND_STAT;
}

/**
  * @brief  Handle the SCSI Capacity command.
  * @param  dev: MSC device.
  * @retval None
  */
static void usbd_scsi_read_cap(usbd_msc_dev_t *dev)
{
	uint32_t num;
	msc_inst_t *inst = &dev->inst;

	/* Get the block size and block numbers from media */
	if (dev->func.get_block_size)
		_block_size = dev->func.get_block_size(inst->media);
	num = dev->func.get_block_num(inst->media);

	/* Check the number */
	if (num != 0)
		--num;

	if (inst->media != 0) {
		/* Build the command */
		_command[0] = 0xff & (num >> 24);
		_command[1] = 0xff & (num >> 16);
		_command[2] = 0xff & (num >> 8);
		_command[3] = 0xff & num;
		_command[4] = 0;
		_command[5] = 0xff & (_block_size >> 16);
		_command[6] = 0xff & (_block_size >> 8);
		_command[7] = 0xff & _block_size;

		/* Put the data into the FIFO */
		map_usb_ep_data_put(inst->iep, _command, 8);
		/* Send the data to the host */
		map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 0;
		_scsi_csw.dCSWDataResidue = 0;
	}
	else {
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 1;
		_scsi_csw.dCSWDataResidue = 0;

		/* Stall the IN endpoint */
		map_usb_dev_ep_stall(inst->iep, USB_EP_DEV_IN);

		/* Fill the instance state */
		inst->err        = SCSI_RS_VALID | SCSI_RS_CUR_ERRORS;
		inst->sense      = SCSI_RS_KEY_NOT_READY;
		inst->sense_code = SCSI_RS_MED_NOT_PRSNT;
	}

	/* Set the SCSI state */
	inst->scsi_state = USBD_SCSI_SEND_STAT;
}

/**
  * @brief  Handle the SCSI Request Sense command.
  * @param  dev: MSC device.
  * @retval None
  */
static void usbd_scsi_request_sense(usbd_msc_dev_t *dev)
{
	uint32_t i;
	msc_inst_t *inst = &dev->inst;

	/* Clear the command */
	for (i = 0; i < 18; ++i)
		_command[i] = 0;

	/* Build the command */
	_command[0]  = inst->err;
	_command[2]  = inst->sense;
	_command[7]  = 10;
	_command[12] = (uint8_t)inst->sense_code;
	_command[13] = (uint8_t)(inst->sense_code >> 8);

	/* Put the data into the FIFO */
	map_usb_ep_data_put(inst->iep, _command, 18);
	/* Send the data to the host */
	map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
	/* Set the error state */
	inst->err = SCSI_RS_CUR_ERRORS;
	/* Set the CSW information */
	_scsi_csw.bCSWStatus      = 0;
	_scsi_csw.dCSWDataResidue = 0;
	/* Set the SCSI state */
	inst->scsi_state          = USBD_SCSI_SEND_STAT;
}

/**
  * @brief  Handle the SCSI Read(10) command.
  * @param  dev: MSC device.
  * @param  cbw: Command Block Wrapper(CBW).
  * @retval None
  */
static void usbd_scsi_read10(usbd_msc_dev_t *dev, msc_cbw_t *cbw)
{
	uint16_t num;
	msc_inst_t *inst = &dev->inst;

	if (inst->media != 0) {
		/* Get the starting read blocks */
		inst->lba = (cbw->CBWCB[2] << 24) | (cbw->CBWCB[3] << 16) |
			    (cbw->CBWCB[4] << 8)  | (cbw->CBWCB[5] << 0);
		/* Get the number of the read blocks */
		num = (cbw->CBWCB[7] << 8) | cbw->CBWCB[8];

		/* Read block date from media */
		if (dev->func.read_block(inst->media, (uint8_t *)inst->buf, inst->lba, 1) == 0) {
			inst->media = 0;
			dev->func.close(0);
		}

		/* Set the instance information */
		inst->size       = (_block_size * num);
		inst->flag      |= USBD_MSC_DMA_IN;
		inst->scsi_state = USBD_SCSI_SEND_BLOCK;

		if (inst->use_dma) {
#ifdef USB_DMA_N_SUPPORT
			usb_dma_config(USB_E_DMA_TYPE_MSC_TX, (uint32_t)&inst->buf, MSC_EP_IN_MAX_SIZE);
			inst->_size = MSC_EP_IN_MAX_SIZE;
#else
			usb_dma_config(inst->idmach, (uint32_t)&inst->buf, _block_size);
#endif
		}
		else {
			/* Put the data into the FIFO */
			map_usb_ep_data_put(inst->iep, (uint8_t *)inst->buf, MSC_EP_IN_MAX_SIZE);
			/* Send the data to the host */
			map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
			/* Set the maximum size of the endpoint */
			inst->_size = MSC_EP_IN_MAX_SIZE;
		}

		/* Send a READING event to the client */
		if (dev->cbk)
			dev->cbk(0, USBD_MSC_EVENT_READING, 0, 0);
	}
	else {
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 1;
		_scsi_csw.dCSWDataResidue = 0;

		/* Stall the IN endpoint */
		map_usb_dev_ep_stall(inst->iep, USB_EP_DEV_IN);

		/* Set the instance information */
		inst->err        = SCSI_RS_VALID | SCSI_RS_CUR_ERRORS;
		inst->sense      = SCSI_RS_KEY_NOT_READY;
		inst->sense_code = SCSI_RS_MED_NOT_PRSNT;
		inst->scsi_state = USBD_SCSI_SEND_STAT;
	}
}

/**
  * @brief  Handle the SCSI Write(10) command.
  * @param  dev: MSC device.
  * @param  cbw: Command Block Wrapper(CBW).
  * @retval None
  */
static void usbd_scsi_write10(usbd_msc_dev_t *dev, msc_cbw_t *cbw)
{
	uint16_t num;
	msc_inst_t *inst = &dev->inst;

	if (inst->media != 0) {
		/* Get the starting write blocks */
		inst->lba = (cbw->CBWCB[2] << 24) | (cbw->CBWCB[3] << 16) |
			    (cbw->CBWCB[4] << 8)  | (cbw->CBWCB[5] << 0);
		/* Get the number of the read blocks */
		num = (cbw->CBWCB[7] << 8) | cbw->CBWCB[8];
		/* Set the instance information */
		inst->size       = _block_size * num;
		inst->scsi_state = USBD_SCSI_RECV_BLOCK;
		inst->flag      |= USBD_MSC_DMA_OUT;

		if (inst->use_dma) {
#ifdef USB_DMA_N_SUPPORT
			inst->_size = 0;
#else
#ifndef USBD_MSC_HS
			usb_dma_config(inst->odmach, (uint32_t)&inst->buf, _block_size);
#endif
#endif
		}
		else {
			inst->_size = 0;
		}

		if (dev->cbk)
			dev->cbk(0, USBD_MSC_EVENT_WRITING, 0, 0);
	}
	else {
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 1;
		_scsi_csw.dCSWDataResidue = 0;

		/* Stall the OUT endpoint */
		map_usb_dev_ep_stall(inst->oep, USB_EP_DEV_OUT);

		/* Set the instance information */
		inst->err        = SCSI_RS_VALID | SCSI_RS_CUR_ERRORS;
		inst->sense      = SCSI_RS_KEY_NOT_READY;
		inst->sense_code = SCSI_RS_MED_NOT_PRSNT;
		inst->scsi_state = USBD_SCSI_SEND_STAT;
	}
}

/**
  * @brief  Handle the SCSI Sense(6) command.
  * @param  dev: MSC device.
  * @param  cbw: Command Block Wrapper(CBW).
  * @retval None
  */
static void usbd_scsi_mode_sense6(usbd_msc_dev_t *dev, msc_cbw_t *cbw)
{
	msc_inst_t *inst = &dev->inst;

	if (inst->media != 0) {
		/* Build the command */
		_command[0] = 3;
		_command[1] = 0;
		_command[2] = 0;
		_command[3] = 0;

		/* Send the command to the host */
		map_usb_ep_data_put(inst->iep, _command, 4);
		map_usb_ep_data_send(inst->iep, USB_TRANS_IN);

		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 0;
		_scsi_csw.dCSWDataResidue = cbw->dCBWDataTransferLength - 4;
	}
	else {
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 1;
		_scsi_csw.dCSWDataResidue = 0;

		/* Stall the IN endpoint */
		map_usb_dev_ep_stall(inst->iep, USB_EP_DEV_IN);

		/* Set the instance information */
		inst->err        = SCSI_RS_VALID | SCSI_RS_CUR_ERRORS;
		inst->sense      = SCSI_RS_KEY_NOT_READY;
		inst->sense_code = SCSI_RS_MED_NOT_PRSNT;
	}

	/* Set the SCSI state */
	inst->scsi_state = USBD_SCSI_SEND_STAT;
}

/**
  * @brief  Handle the SCSI Prevent/Allow Medium Removal command.
  * @param  dev: MSC device.
  * @param  cbw: Command Block Wrapper(CBW).
  * @retval None
  */
static void usbd_scsi_medium_removal(usbd_msc_dev_t *dev, msc_cbw_t *cbw)
{
	msc_inst_t *inst = &dev->inst;

	if (inst->media != 0) {
		if ((cbw->CBWCB[4] & SCSI_PE_MEDRMV_M) == SCSI_PE_MEDRMV_ALLOW)
			inst->flag |= USBD_MSC_ALLOW_REMOVAL;
		else
			inst->flag &= (uint32_t)(~USBD_MSC_ALLOW_REMOVAL);

		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 0;
		_scsi_csw.dCSWDataResidue = 0;
	}
	else {
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 1;
		_scsi_csw.dCSWDataResidue = 0;

		/* Stall the IN endpoint */
		map_usb_dev_ep_stall(inst->iep, USB_EP_DEV_IN);

		/* Set the instance information */
		inst->err        = SCSI_RS_VALID | SCSI_RS_CUR_ERRORS;
		inst->sense      = SCSI_RS_KEY_NOT_READY;
		inst->sense_code = SCSI_RS_MED_NOT_PRSNT;
	}

	/* Set the SCSI state */
	inst->scsi_state = USBD_SCSI_SEND_STAT;
}

/**
  * @brief  Handle the SCSI Start/Stop Unit command.
  * @param  dev: MSC device.
  * @param  cbw: Command Block Wrapper(CBW).
  * @retval None
  */
static void usbd_scsi_start_stop_unit(usbd_msc_dev_t *dev, msc_cbw_t *cbw)
{
	msc_inst_t *inst = &dev->inst;

	if (inst->media != 0) {
		switch (cbw->CBWCB[4] & (SCSI_SS_UNIT_START | SCSI_SS_UNIT_LOEJ)) {
		case 0:
			/* Media state is stopped */
			inst->status = USBD_MSC_MEDIA_STOP;
			_scsi_csw.bCSWStatus = 0;
			break;

		case SCSI_SS_UNIT_START:
			/* Media state is present */
			inst->status = USBD_MSC_MEDIA_PRESENT;
			_scsi_csw.bCSWStatus = 0;
			break;

		case SCSI_SS_UNIT_LOEJ:
			/* Media state is eject */
			if(inst->flag & USBD_MSC_ALLOW_REMOVAL) {
				inst->status = USBD_MSC_MEDIA_PRESENT_NO;
				dev->func.close(0);
				inst->media = 0;
				_scsi_csw.bCSWStatus = 0;
			}
			else {
				_scsi_csw.bCSWStatus = 1;
			}

			break;

		case SCSI_SS_UNIT_START | SCSI_SS_UNIT_LOEJ:
			/* Check if there is no media */
			inst->media = dev->func.open(0);

			if (inst->media != 0)
				_scsi_csw.bCSWStatus = 0;
			else
				_scsi_csw.bCSWStatus = 1;

			break;

		default:
			break;
		}

		/* No further data to send */
		_scsi_csw.dCSWDataResidue = 0;
	}
	else {
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 1;
		_scsi_csw.dCSWDataResidue = 0;

		/* Stall the IN endpoint */
		map_usb_dev_ep_stall(inst->iep, USB_EP_DEV_IN);

		/* Set the instance information */
		inst->err        = SCSI_RS_VALID | SCSI_RS_CUR_ERRORS;
		inst->sense      = SCSI_RS_KEY_NOT_READY;
		inst->sense_code = SCSI_RS_MED_NOT_PRSNT;
	}

	/* Set the SCSI state */
	inst->scsi_state = USBD_SCSI_SEND_STAT;
}

/**
  * @brief  Handle all SCSI commands.
  * @param  dev: MSC device.
  * @param  cbw: Command Block Wrapper(CBW).
  * @retval Status
  */
static uint32_t usbd_scsi_cmd(usbd_msc_dev_t *dev, msc_cbw_t *cbw)
{
	uint32_t len;
	msc_inst_t *inst = &dev->inst;

	/* Save the length */
	len = cbw->dCBWDataTransferLength;

	switch (cbw->CBWCB[0]) {
	case SCSI_INQUIRY_CMD:
		/* Handle SCSI Inquiry command */
		usbd_scsi_inquiry(dev);
		break;

	case SCSI_TEST_UNIT_READY:
		/* Handle SCSI Test-Unit-Ready command */
		_scsi_csw.dCSWDataResidue = 0;

		if (inst->media != 0) {
			_scsi_csw.bCSWStatus = 0;
		}
		else if (inst->status == USBD_MSC_MEDIA_PRESENT_NO) {
			_scsi_csw.bCSWStatus = 1;
			inst->err        = SCSI_RS_VALID | SCSI_RS_CUR_ERRORS;
			inst->sense      = SCSI_RS_KEY_NOT_READY;
			inst->sense_code = SCSI_RS_MED_NOT_PRSNT;
		}
		else {
			inst->media = dev->func.open(0);

			if (inst->media != 0)
				_scsi_csw.bCSWStatus = 0;
			else
				_scsi_csw.bCSWStatus = 1;
		}

		break;

	case SCSI_READ_CAPACITIES:
		/* Handle SCSI Read-Capacities command */
		usbd_scsi_read_caps(dev);
		break;

	case SCSI_READ_CAPACITY:
		/* Handle SCSI Read-Capacity command */
		usbd_scsi_read_cap(dev);
		break;

	case SCSI_REQUEST_SENSE:
		/* Handle SCSI Request-Sense command */
		usbd_scsi_request_sense(dev);
		break;

	case SCSI_READ_10:
		/* Handle SCSI Read-10 command */
		usbd_scsi_read10(dev, cbw);
		break;

	case SCSI_WRITE_10:
		/* Handle SCSI Write-10 command */
		usbd_scsi_write10(dev, cbw);
		break;

	case SCSI_MODE_SENSE_6:
		/* Handle SCSI Mode-Sense-6 command */
		usbd_scsi_mode_sense6(dev, cbw);
		break;

	case SCSI_MEDIUM_REMOVAL:
		/* Handle SCSI Prevent-Allow Medium-Removal command */
		usbd_scsi_medium_removal(dev, cbw);
		break;

	case SCSI_START_STOP_UNIT:
		/* Handle SCSI Stop-Unit command */
		usbd_scsi_start_stop_unit(dev, cbw);
		break;

	default:
		/* Set the CSW information */
		_scsi_csw.bCSWStatus      = 1;
		_scsi_csw.dCSWDataResidue = cbw->dCBWDataTransferLength;

		/* Stall the IN or OUT endpoint */
		if (cbw->dCBWDataTransferLength != 0) {
			if (cbw->bmCBWFlags & CBWFLAGS_DIR_IN)
				map_usb_dev_ep_stall(inst->iep, USB_EP_DEV_IN);
			else
				map_usb_dev_ep_stall(inst->oep, USB_EP_DEV_OUT);

			inst->scsi_state = USBD_SCSI_IDLE;
		}

		/* Set the instance information */
		inst->err        = SCSI_RS_VALID | SCSI_RS_CUR_ERRORS;
		inst->sense      = SCSI_RS_KEY_ILGL_RQST;
		inst->sense_code = SCSI_RS_PV_INVALID;
		break;
	}

	/* Send the current SCSI status */
	if (len == 0)
		usbd_scsi_status_send(dev);

	return 0;
}
/**
  * @}
  */

/** @defgroup Device_MSC_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initialized basic operation and prepare for enumeration.
  * @param  idx: Index of the USB controller.
  * @param  dev: MSC device.
  * @retval Device structure.
  */
void *usbd_msc_init(uint32_t idx, usbd_msc_dev_t *dev)
{
	device_desc_t *desc_dev;
	config_desc_t *desc_conf;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->desc_str);

	/* Composite initialization */
	usbd_msc_init_comp(idx, dev, 0);

	/* Fill the device descriptor */
	desc_dev = (device_desc_t *)msc_dev_desc;
	desc_dev->idVendor  = dev->vid;
	desc_dev->idProduct = dev->pid;
	/* Fill the configuration descriptor */
	desc_conf = (config_desc_t *)msc_desc;
	desc_conf->bmAttributes = dev->attr_pwr;
	desc_conf->bMaxPower    = (uint8_t)(dev->max_power / 2);

	/* Initialze device controler */
	usb_dcd_init(idx, &dev->inst.info, (void *)dev);
	return((void *)dev);
}

/**
  * @brief  Initialized basic operation and prepare for enumeration.
  * @param  idx: Index of the USB controller.
  * @param  dev: MSC device.
  * @param  entry: Composite entry.
  * @retval Device structure.
  */
void *usbd_msc_init_comp(uint32_t idx, usbd_msc_dev_t *dev, comp_entry_t *entry)
{
	msc_inst_t *inst;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->desc_str);

	/* Create a pointer to the device instance */
	inst = &dev->inst;
	/* Set the device instance information */
	inst->usb_idx = 0;
	inst->conn    = false;
	inst->status  = USBD_MSC_MEDIA_UNKNOWN;

	/* Initialze the composite entry */
	if (entry != 0) {
		entry->info = &inst->info;
		entry->inst = (void *)dev;
	}

	/* Initialize the device information */
	inst->info.cbk            = &msc_handler;
	inst->info.device_desc    = msc_dev_desc;
	inst->info.config_desc    = msc_config_desc;
	inst->info.string_desc    = 0;
	inst->info.nr_string_desc = 0;
	usb_device_info_init(0, &inst->info);

	/* Initialize the device instance structure */
	inst->interface = 0;
	inst->oep = MSC_EP_OUT;
	inst->iep = MSC_EP_IN;
	inst->scsi_state = USBD_SCSI_IDLE;
	inst->info.string_desc    = dev->desc_str;
	inst->info.nr_string_desc = dev->num_str;
	inst->media = dev->func.open(0);
	inst->use_dma = 1;

	if (inst->media == 0) {
		inst->sense      = SCSI_RS_KEY_NOT_READY;
		inst->sense_code = SCSI_RS_MED_NOT_PRSNT;
	}
	else {
		inst->sense      = SCSI_RS_KEY_UNIT_ATTN;
		inst->sense_code = SCSI_RS_MED_NOTRDY2RDY;
	}

	/* Initialze DMA */
	usb_dma_init(0);

	/* Configure DMA structure */
#ifdef USB_DMA_N_SUPPORT
	msc_tx_cbk.cplt_cbk = usbd_msc_dma_tx_cplt;
	msc_tx_cbk.err_cbk  = usbd_msc_dma_e_err;
	msc_tx_cbk.cplt_arg = dev;
	msc_tx_cbk.err_arg  = dev;
	msc_rx_cbk.cplt_cbk = usbd_msc_dma_rx_cplt;
	msc_rx_cbk.err_cbk  = usbd_msc_dma_e_err;
	msc_rx_cbk.cplt_arg = dev;
	msc_rx_cbk.err_arg  = dev;

	usb_dma_channel_alloc(dev->inst.iep, USB_E_DMA_TYPE_MSC_TX, &msc_tx_cbk);
	usb_dma_channel_alloc(dev->inst.oep, USB_E_DMA_TYPE_MSC_RX, &msc_rx_cbk);
#else
	msc_cbk.cplt_cbk = usbd_msc_dma_cplt;
	msc_cbk.err_cbk  = usbd_msc_dma_err;
	msc_cbk.cplt_arg = dev;
	msc_cbk.err_arg  = dev;

	dev->inst.idmach = usb_dma_channel_alloc(dev->inst.iep, USB_DMA_EP_CFG_TX, &msc_cbk);
	dev->inst.odmach = usb_dma_channel_alloc(dev->inst.oep, USB_DMA_EP_CFG_RX_DEV, &msc_cbk);
#endif
	return dev;
}

/**
  * @brief  Terminal the MSC device.
  * @param  device: MSC device.
  * @retval None
  */
void usbd_msc_term(void *device)
{
	usbd_msc_dev_t *dev;

	assert_param(device != 0);

	/* Terminate the device controler */
	dev = (usbd_msc_dev_t *)device;
	usb_dcd_term(0);

	/* Free DMA channel */
	usb_dma_channel_free(dev->inst.iep, USB_DMA_EP_CFG_TX);
	usb_dma_channel_free(dev->inst.oep, USB_DMA_EP_CFG_RX_DEV);

	/* Close the media */
	if (dev->inst.media != 0) {
		dev->inst.media = 0;
		dev->func.close(0);
	}
}

/**
  * @brief  Detect insertion or removal of the media.
  * @param  device: MSC device.
  * @param  status: Update status for the media.
  * @retval None
  */
void usbd_msc_media_change(void *device, msc_media_status_t status)
{
	/* Save the new state */
	((usbd_msc_dev_t *)device)->inst.status = status;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
