/**
  *********************************************************************************
  *
  * @file    usbd_hid_mouse.c
  * @brief   USB HID mouse device driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbd_hid_mouse.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_Mouse Mouse
  * @brief Device Mouse driver
  * @{
  */
/** @defgroup Device_Mouse_Private_Variables Private Variables
  * @{
  */
/**
  * @brief Configuration descriptor header
  */
static uint8_t __mouse_desc[] = {
	9,                       /**< Size of the descriptor */
	USB_DTYPE_CONFIGURATION, /**< Type of the descriptor */
	USBShort(34),            /**< Total size of the descriptor */
	1,                       /**< Number of the inerfaces */
	1,                       /**< Unique value for this configuration */
	5,                       /**< Index of the string */
	USB_CONF_ATTR_SELF_PWR,  /**< Type of the power */
	250,                     /**< Maximum power in 2mA increments */
};

/**
  * @brief HID interface descriptor
  */
static uint8_t __hid_interface[HID_INTERFACE_SIZE] = {
	9,                      /**< Size of the interface */
	USB_DTYPE_INTERFACE,    /**< Type of the interface */
	0,                      /**< Index of the interface */
	0,                      /**< Alternate setting */
	1,                      /**< Number of endpoints in the interface */
	USB_CLASS_HID,          /**< Interface class */
	USB_HID_SCLASS_BOOT,    /**< Interface sub-class */
	USB_HID_PROTOCOL_MOUSE, /**< Interface protocol */
	4,                      /**< Index of the string */
};

/**
  * @brief HID IN endpoint descriptor
  */
static const uint8_t __hid_endpoint[HID_IN_ENDPOINT_SIZE] = {
	7,                                  /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                 /**< Type is an endpoint */
	USB_EP_DESC_IN | MOUSE_IN_ENDPOINT, /**< Endpoint direction */
	USB_EP_ATTR_INT,                    /**< Endpoint type */
	USBShort(64),                       /**< Maximum packet size */
	16,                                 /**< Polling interval */
};

/**
  * @brief Mouse report descriptor
  */
static const uint8_t __mouse_report_desc[] = {
	USAGE_PAGE(USB_HID_GENERIC_DESKTOP),
	USAGE(USB_HID_MOUSE),
	COLLECTION(USB_HID_APPLICATION),
		USAGE(USB_HID_POINTER),
		COLLECTION(USB_HID_PHYSICAL),

			/* Button */
			USAGE_PAGE(USB_HID_BUTTONS),
			USAGE_MIN(1),
			USAGE_MAX(3),
			LOGICAL_MIN(0),
			LOGICAL_MAX(1),

			/* 3 - 1 bit values for the button */
			REPORT_SIZE(1),
			REPORT_COUNT(3),
			INPUT(USB_HID_INPUT_DATA | USB_HID_INPUT_VARIABLE | USB_HID_INPUT_ABS),

			/* 1 - 5 bit unused constant value to fill the 8 bits */
			REPORT_SIZE(5),
			REPORT_COUNT(1),
			INPUT(USB_HID_INPUT_CONSTANT | USB_HID_INPUT_ARRAY | USB_HID_INPUT_ABS),

			/* The X and Y axis */
			USAGE_PAGE(USB_HID_GENERIC_DESKTOP),
			USAGE(USB_HID_X),
			USAGE(USB_HID_Y),
			LOGICAL_MIN(-127),
			LOGICAL_MAX(127),

			/* 2 - 8 bit Values for x and y */
			REPORT_SIZE(8),
			REPORT_COUNT(2),
			INPUT(USB_HID_INPUT_DATA | USB_HID_INPUT_VARIABLE | USB_HID_INPUT_RELATIVE),

		END_COLLECTION,
	END_COLLECTION,
};

/**
  * @brief HID descriptor
  */
static const hid_desc_t __mouse_hid_desc = {
	9,                 /**< Size of the endpoint descriptor */
	USB_HID_DTYPE_HID, /**< Type is an endpoint */
	0x111,             /**< Version 1.11 */
	0,                 /**< Country code */
	1,                 /**< Number of the descriptors */
	{
		{
			USB_HID_DTYPE_REPORT,       /**< Report descriptor */
			sizeof(__mouse_report_desc) /**< Size of thr report descriptor */
		}
	}
};

/**
  * @brief HID configuration section
  */
static const config_section_t __hid_config_sec = {
	sizeof(__mouse_desc),
	__mouse_desc
};

/**
  * @brief HID interface section
  */
static const config_section_t __hid_interface_sec = {
	sizeof(__hid_interface),
	__hid_interface
};

/**
  * @brief HID IN endpoint section
  */
static const config_section_t __hid_in_ep_sec = {
	sizeof(__hid_endpoint),
	__hid_endpoint
};

/**
  * @brief HID descriptor section
  */
static config_section_t __hid_desc_sec = {
	sizeof(__mouse_hid_desc),
	(const uint8_t *)&__mouse_hid_desc
};

/**
  * @brief HID section
  */
static const config_section_t *__hid_sec[] = {
	&__hid_config_sec,
	&__hid_interface_sec,
	&__hid_desc_sec,
	&__hid_in_ep_sec,
};

/**
  * @brief Number of the section
  */
#define NUM_HID_SECTIONS        (sizeof(__hid_sec) / sizeof(__hid_sec[0]))

/**
  * @brief HID configuration header
  */
static config_head_t __hid_config_head = {
	NUM_HID_SECTIONS,
	__hid_sec
};

/**
  * @brief HID configuration descriptor
  */
static const config_head_t * const __hid_config_desc[] = {
	&__hid_config_head
};

/**
  * @brief Keyboad class descriptor
  */
static const uint8_t * const __mouse_class_desc[] = {
	__mouse_report_desc
};
/**
  * @}
  */

/** @defgroup Device_Mouse_Private_Functions Private Functions
  * @{
  */
static uint32_t hid_mouse_rx_handler(void *device, uint32_t event, uint32_t param, void *data);
static uint32_t hid_mouse_tx_handler(void *device, uint32_t event, uint32_t param, void *data);

/**
  * @brief  Handle HID device class event.
  * @param  device: Mouse device.
  * @param  event: Event.
  * @param  param: Event-specific value.
  * @param  data: Event-specific data.
  * @retval Status.
  */
static uint32_t hid_mouse_rx_handler(void *device, uint32_t event, uint32_t param, void *data)
{
	hid_mouse_inst_t *inst;
	usbd_hid_mouse_dev_t *dev;

	assert_param(device);

	/* Create a pointer to the device instance */
	dev = (usbd_hid_mouse_dev_t *)device;
	inst = &dev->inst;

	/* Which event? */
	switch (event) {
	case USB_EVENT_CONNECTED:
		inst->config = 1;
		/* Send a CONNECTED event to the client */
		dev->cbk(dev->arg, USB_EVENT_CONNECTED, 0, NULL);
		break;

	case USB_EVENT_DISCONNECTED:
		inst->config = 0;
		/* Send a DISCONNECTED event to the client */
		dev->cbk(dev->arg, USB_EVENT_DISCONNECTED, 0, NULL);
		break;

	case USBD_HID_EVENT_IDLE_TIMEOUT:
	case USBD_HID_EVENT_GET_REPORT:
		/* Set the report pointer */
		*(uint8_t **)data = inst->report;
		/* Return the size of the report */
		return 8;

	case USBD_HID_EVENT_REPORT_SENT:
		break;

	case USBD_HID_EVENT_GET_REPORT_BUFFER:
		return 0;

	case USBD_HID_EVENT_SET_PROTOCOL:
		/* Set the report protocol */
		inst->protocol = param;
		break;

	case USBD_HID_EVENT_GET_PROTOCOL:
		/* Get the report protocol */
		return inst->protocol;

	case USB_EVENT_ERROR:
	case USB_EVENT_SUSPEND:
	case USB_EVENT_RESUME:
	case USB_EVENT_LPM_RESUME:
	case USB_EVENT_LPM_SLEEP:
	case USB_EVENT_LPM_ERROR:
		return dev->cbk(dev->arg, event, param, data);

	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle HID device class transmit event.
  * @param  device: Mouse device.
  * @param  event: Event.
  * @param  param: Event-specific value.
  * @param  data: Event-specific data.
  * @retval Status.
  */
static uint32_t hid_mouse_tx_handler(void *device, uint32_t event, uint32_t param, void *data)
{
	hid_mouse_inst_t *inst;
	usbd_hid_mouse_dev_t *dev;

	assert_param(device);

	/* Create a pointer to the device instance */
	dev = (usbd_hid_mouse_dev_t *)device;
	inst = &dev->inst;

	/* Which event? */
	switch (event) {
	case USB_EVENT_TX_COMPLETE:
		/* Set the state at idle */
		inst->state = HID_MOUSE_STATE_IDLE;
		/* Send the TX_COMPLETE event to the client */
		dev->cbk(dev->arg, USB_EVENT_TX_COMPLETE, param, NULL);
		break;

	default:
		break;
	}

	return 0;
}
/**
  * @}
  */

/** @defgroup Device_Mouse_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes HID mouse device.
  * @param  idx: Index of the USB controller.
  * @param  dev: Mouse device.
  * @retval Device structure.
  */
void *usbd_hid_mouse_init(uint32_t idx, usbd_hid_mouse_dev_t *dev)
{
	void *ret;
	usbd_hid_dev_t *_dev;
	config_desc_t *config;

	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->cbk);

	/* Create a pointer to the device */
	_dev = &dev->inst.dev;

	/* Fill the configuration descriptor */
	config = (config_desc_t *)__mouse_desc;
	config->bmAttributes = dev->attr_pwr;
	config->bMaxPower    = dev->max_power >> 1;
	/* Composite initialization */
	ret = usbd_hid_mouse_init_comp(idx, dev, 0);

	if (ret) {
		/* Initialze device controler */
		usbd_hid_init(idx, _dev);
		return (void *)dev;
	}

	return NULL;
}

/**
  * @brief  Initializes HID mouse device.
  * @param  idx: Index of the USB controller.
  * @param  dev: Mouse device.
  * @param  entry: Composite entry.
  * @retval Device structure.
  */
void *usbd_hid_mouse_init_comp(uint32_t idx, usbd_hid_mouse_dev_t *dev, comp_entry_t *entry)
{
	hid_mouse_inst_t *inst;
	usbd_hid_dev_t *_dev;

	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->cbk);

	/* Create a pointer to the device instance */
	inst = &dev->inst;
	_dev = &dev->inst.dev;

	/* Initialize the device instance information */
	inst->config      = 0;
	inst->protocol    = USB_HID_PROTOCOL_REPORT;
	inst->idle.dur4ms = 0;
	inst->idle.id     = 0;
	inst->idle.since  = 0;
	inst->idle.next   = 0;
	inst->state       = HID_MOUSE_STATE_UNCONFIG;

	/* Initialize the device information */
	_dev->vid             = dev->vid;
	_dev->pid             = dev->pid;
	_dev->max_power       = dev->max_power;
	_dev->attr_pwr        = dev->attr_pwr;
	_dev->sub_class       = USB_HID_SCLASS_BOOT;
	_dev->protocol        = USB_HID_PROTOCOL_MOUSE;
	_dev->nr_input_report = 1;
	_dev->idle            = &inst->idle;
	_dev->rx_cbk          = hid_mouse_rx_handler;
	_dev->rx_arg          = (void *)dev;
	_dev->tx_cbk          = hid_mouse_tx_handler;
	_dev->tx_arg          = (void *)dev;
	_dev->use_oep         = 0;
	_dev->desc_hid        = &__mouse_hid_desc;
	_dev->desc_class      = __mouse_class_desc;
	_dev->desc_str        = dev->desc_str;
	_dev->num_str         = dev->num_str;
	_dev->desc_config     = __hid_config_desc;

	/* Call the lower layer composite initialization */
	return usbd_hid_init_comp(idx, _dev, entry);
}

/**
  * @brief  Terminal the HID mouse device.
  * @param  device: Mouse device.
  * @retval None
  */
void usbd_hid_mouse_term(void *device)
{
	usbd_hid_mouse_dev_t *dev;
	usbd_hid_dev_t *_dev;

	assert_param(device);

	/* Create a pointer to the device */
	dev  = (usbd_hid_mouse_dev_t *)device;
	_dev = &dev->inst.dev;
	/* Clear the flag */
	dev->inst.config = 0;
	/* Call the lower layer function */
	usbd_hid_term(_dev);

	return;
}

/**
  * @brief  Set callback parameter.
  * @param  device: Mouse device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_hid_mouse_set_param(void *device, void *arg)
{
	void *old;
	usbd_hid_mouse_dev_t *dev;

	assert_param(device);

	/* Create a pointer to the device */
	dev = (usbd_hid_mouse_dev_t *)device;
	/* Set the parameter */
	old = dev->arg;
	dev->arg = arg;

	return old;
}

/**
  * @brief  Reports a mouse state change, pointer movement or button press, to the USB host.
  * @param  device: Mouse device.
  * @param  x: Relative horizontal pointer movement.
  * @param  y: Relative vertical pointer movement.
  * @param  button: Map of the button.
  * @retval Status.
  */
uint32_t usbd_hid_mouse_state_change(void *device, int8_t x, int8_t y, uint8_t button)
{
	uint32_t size;
	hid_mouse_inst_t *inst;
	usbd_hid_mouse_dev_t *dev;
	usbd_hid_dev_t *_dev;

	/* Create a pointer to the device instance */
	dev  = (usbd_hid_mouse_dev_t *)device;
	inst = &dev->inst;
	_dev = &dev->inst.dev;

	/* Store the pos_x, pos_y, and button */
	inst->report[MOUSE_REPORT_BUTTON] = button;
	inst->report[MOUSE_REPORT_X]      = (uint8_t)x;
	inst->report[MOUSE_REPORT_Y]      = (uint8_t)y;

	/* Check the flag */
	if (!inst->config)
		return MOUSE_ERR_NOT_CONFIG;
	/* Check if the transmitter is currently free */
	if ((usbd_hid_tx_packet_avail((void *)_dev)) == 0)
		return MOUSE_ERR_TX_ERROR;

	/* Set the state */
	inst->state = HID_MOUSE_STATE_SEND;
	/* Send the report to the host */
	size = usbd_hid_report_write((void *)_dev, inst->report, MOUSE_REPORT_SIZE, 1);

	if (size != 0)
		return MOUSE_SUCCESS;

	return MOUSE_ERR_TX_ERROR;
}

/**
  * @brief  Set power status.
  * @param  device: Mouse device.
  * @param  power: Status of the power.
  * @retval None
  */
void usbd_hid_mouse_power_status_set(void *device, uint8_t power)
{
	usbd_hid_mouse_dev_t *dev;
	usbd_hid_dev_t *_dev;

	assert_param(device);

	/* Create a pointer to the device */
	dev  = (usbd_hid_mouse_dev_t *)device;
	_dev = &dev->inst.dev;
	/* Set the status of the power */
	usbd_hid_power_status_set((void *)_dev, power);

	return;
}

/**
  * @brief  Requests a remote wake up to resume communication.
  * @param  device: Mouse device.
  * @retval Status.
  */
int usbd_hid_mouse_remote_wakeup_req(void *device)
{
	usbd_hid_mouse_dev_t *dev;
	usbd_hid_dev_t *_dev;

	assert_param(device);

	/* Create a pointer to the device */
	dev  = (usbd_hid_mouse_dev_t *)device;
	_dev = &dev->inst.dev;

	/* Call the lower layer function */
	return usbd_hid_remote_wakeup_req((void *)_dev);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
