/**
  *********************************************************************************
  *
  * @file    usbd_hid_keyb.h
  * @brief   Header file for the Keyboard driver[Device].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBD_HID_KEYB_H__
#define __USBD_HID_KEYB_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbd_hid.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_Keyb
  * @{
  */
/**
  * @defgroup Device_Keyb_Public_Macros Public Macros
  * @{
  */
#define KEYB_SUCCESS			0U
#define KEYB_ERR_TOO_MANY_KEYS		1U
#define KEYB_ERR_TX_ERROR		2U
#define KEYB_ERR_NOT_FOUND		3U
#define KEYB_ERR_NOT_CONFIGURED		4U
#define KEYB_IN_REPORT_SIZE		8U
#define KEYB_OUT_REPORT_SIZE		1U
#define KEYB_MAX_CHARS_PER_REPORT	6U
#define USBD_HID_KEYB_EVENT_SET_LEDS	USBD_HID_KEYB_EVENT_BASE
#define KEYB_IN_ENDPOINT	HID_KEYB_EP_IN
#define KEYB_OUT_ENDPOINT	HID_KEYB_EP_OUT
/**
  * @}
  */

/**
  * @defgroup Device_Keyb_Public_Types Public Types
  * @{
  */
/**
  * @brief Keyboard status
  */
typedef enum {
	HID_KEYB_STATE_UNCONFIG  = 0,	/**< Unconfigured */
	HID_KEYB_STATE_IDLE      = 1,	/**< Idle */
	HID_KEYB_STATE_WAIT_DATA = 2,	/**< Waiting data */
	HID_KEYB_STATE_SEND      = 3,	/**< Send */
} keyb_state_t;

/**
  * @brief Define the private instance data.
  */
typedef struct {
	uint8_t config;					/**< Configuration number */
	uint8_t protocol;				/**< Protocol */
	volatile uint8_t led_state;			/**< States of the keyboard LEDs */
	uint8_t key_cnt;				/**< Number of keys currently pressed */
	volatile keyb_state_t state;			/**< State of the keyboard */
	volatile uint8_t change;			/**< Change mode */
	uint8_t data[KEYB_OUT_REPORT_SIZE];		/**< Data buffer */
	uint8_t report[KEYB_IN_REPORT_SIZE];		/**< IN report */
	uint8_t press[KEYB_MAX_CHARS_PER_REPORT];	/**< Pressed state */
	hid_report_idle_t idle;				/**< Idle */
	usbd_hid_dev_t dev;				/**< HID device */
} hid_keyb_inst_t;

/**
  * @brief Define operating parameters for the HID device
  */
typedef struct {
	const uint16_t vid;			/**< Vendor ID */
	const uint16_t pid;			/**< Product ID */
	const uint16_t max_power;		/**< Maximum power consumption */
	const uint8_t attr_pwr;			/**< Attribute of the power */
	const usb_cbk cbk;			/**< Callback function */
	void *arg;				/**< Parameter */
	const uint8_t * const *desc_str;	/**< String descriptor */
	const uint32_t num_str;			/**< Number of the string descriptors */
	hid_keyb_inst_t inst;			/**< Device instance */
} usbd_hid_keyb_dev_t;
/**
  * @}
  */

/** @addtogroup Device_Keyb_Public_Functions
  * @{
  */
extern void *usbd_hid_keyb_init(uint32_t idx, usbd_hid_keyb_dev_t *dev);
extern void *usbd_hid_keyb_init_comp(uint32_t idx, usbd_hid_keyb_dev_t *dev, comp_entry_t *entry);
extern void usbd_hid_keyb_term(void *device);
extern void *usbd_hid_keyb_set_param(void *device, void *arg);
extern uint32_t usbd_hid_keyb_state_change(void *device, uint8_t modify, uint8_t usage, uint8_t press);
extern void usbd_hid_keyb_power_status_set(void *device, uint8_t power);
extern int usbd_hid_keyb_remote_wakeup_req(void *device);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
